from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem
from qgis.core import QgsProject, QgsMapLayer, QgsWkbTypes, QgsFeatureRequest, QgsField, QgsVectorLayer, QgsFeature
from PyQt5.QtCore import QVariant
from .hatagu_water1_dialog import HataguWater1Dialog

class HataguWater1:
    def __init__(self, iface):
        self.iface = iface
        self.dlg = None
        self.first_start = True
        self.actions = []
        self.menu = self.tr(u'&hatagu_water1')

    def tr(self, message):
        return QCoreApplication.translate('HataguWater1', message)

    def add_action(self, icon_path, text, callback, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        self.iface.addToolBarIcon(action)
        self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = ':/plugins/hatagu_water1/icon.png'
        self.add_action(icon_path, text=self.tr(u'配水ブロック重要度評価を実行'), callback=self.run, parent=self.iface.mainWindow())

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&hatagu_water1'), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        if self.first_start:
            self.first_start = False
            self.dlg = HataguWater1Dialog()
            self.dlg.buttonRun.clicked.connect(self.evaluate_priority_score)
            self.dlg.buttonAddRow.clicked.connect(self.add_table_row)
            self.dlg.buttonRemoveRow.clicked.connect(self.remove_selected_rows)

        self.dlg.comboPolygonLayer.clear()
        self.dlg.comboAttributeField.clear()
        self.dlg.comboPointLayer.clear()

        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                self.dlg.comboPolygonLayer.addItem(layer.name(), layer)

        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PointGeometry:
                self.dlg.comboPointLayer.addItem(layer.name())

        self.dlg.tableFacilities.setRowCount(0)
        self.dlg.tableFacilities.setColumnCount(2)

        self.dlg.comboPolygonLayer.currentIndexChanged.connect(self.populate_attribute_fields)
        self.populate_attribute_fields()

        self.dlg.show()
        result = self.dlg.exec_()
        if result:
            pass

    def populate_attribute_fields(self):
        self.dlg.comboAttributeField.clear()
        polygon_layer = self.dlg.comboPolygonLayer.currentData()
        if polygon_layer:
            fields = polygon_layer.fields()
            for field in fields:
                self.dlg.comboAttributeField.addItem(field.name())

    def add_table_row(self):
        layer_name = self.dlg.comboPointLayer.currentText()
        if not layer_name:
            return
        row = self.dlg.tableFacilities.rowCount()
        self.dlg.tableFacilities.insertRow(row)
        self.dlg.tableFacilities.setItem(row, 0, QTableWidgetItem(layer_name))
        self.dlg.tableFacilities.setItem(row, 1, QTableWidgetItem("1"))

    def remove_selected_rows(self):
        selected = self.dlg.tableFacilities.selectionModel().selectedRows()
        for index in sorted(selected, reverse=True):
            self.dlg.tableFacilities.removeRow(index.row())

    def evaluate_priority_score(self):
        polygon_layer = self.dlg.comboPolygonLayer.currentData()
        output_name = self.dlg.lineOutputName.text()
        field_name = self.dlg.comboAttributeField.currentText()

        if not polygon_layer or not field_name:
            self.iface.messageBar().pushWarning("警告", "ポリゴンレイヤまたは属性フィールドが未選択です")
            return

        fields = polygon_layer.fields()
        fields.append(QgsField("score", QVariant.Int))
        scored_layer = QgsVectorLayer(f"Polygon?crs={polygon_layer.crs().authid()}", output_name, "memory")
        provider = scored_layer.dataProvider()
        provider.addAttributes(fields)
        scored_layer.updateFields()

        row_count = self.dlg.tableFacilities.rowCount()
        facility_layers = []
        for row in range(row_count):
            name_item = self.dlg.tableFacilities.item(row, 0)
            weight_item = self.dlg.tableFacilities.item(row, 1)
            if not name_item or not weight_item:
                continue
            layer_name = name_item.text()
            weight = int(weight_item.text())
            layers = QgsProject.instance().mapLayersByName(layer_name)
            if layers:
                facility_layers.append((layers[0], weight))

        for feature in polygon_layer.getFeatures():
            geom = feature.geometry()
            score = 0
            for layer, weight in facility_layers:
                for pt in layer.getFeatures(QgsFeatureRequest().setFilterRect(geom.boundingBox())):
                    if geom.contains(pt.geometry()):
                        score += weight
            new_feature = QgsFeature(scored_layer.fields())
            new_feature.setGeometry(geom)
            new_feature.setAttributes(feature.attributes() + [score])
            provider.addFeature(new_feature)

        scored_layer.updateExtents()
        QgsProject.instance().addMapLayer(scored_layer)
