# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GSMCoverBuilder
                                 A QGIS plugin
 GSM Cover Builder allows you to quickly generate coverage plans based on localities and a defined coverage radius
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-02-11
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Fadil Kristof GNANKAMBARY
        email                : krisfadil@live.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.core import QgsMapLayerProxyModel, QgsProject, QgsVectorLayer, QgsDistanceArea, QgsFeature, QgsGeometry, QgsField, QgsPointXY, QgsCoordinateReferenceSystem, QgsCoordinateTransform
from .resources import *
from .GSM_Cover_Builder_dialog import GSMCoverBuilderDialog
import os.path
import processing  # Import du module processing
from collections import defaultdict
import math

class GSMCoverBuilder:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GSMCoverBuilder_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&GSM Cover Builder')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        return QCoreApplication.translate('GSMCoverBuilder', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = ':/plugins/GSM_Cover_Builder/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'GSM Cover Builder'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&GSM Cover Builder'), action)
            self.iface.removeToolBarIcon(action)

    def on_field_changed(self):
        """Gérer les changements dans le champ sélectionné."""
        selected_field = self.dlg.mFieldComboBox.currentField()
        print(f"Champ sélectionné : {selected_field}")

    def run(self):
        """Run method that performs all the real work"""

        # Initialisation du dialogue si c'est la première fois que l'on lance le plugin
        if self.first_start:
            self.first_start = False
            self.dlg = GSMCoverBuilderDialog()

            # Connecter la couche et le champ avec les événements
            self.dlg.mMapLayerComboBox.layerChanged.connect(self.dlg.update_fields)
            self.dlg.mFieldComboBox.fieldChanged.connect(self.on_field_changed)

        # Afficher le dialogue
        self.dlg.show()
        # Exécuter la boucle d'événements du dialogue
        result = self.dlg.exec_()
        
        # Vérifier si l'utilisateur a validé le dialogue
        if result:
            # Récupérer la couche sélectionnée
            input_layer = self.dlg.mMapLayerComboBox.currentLayer()
            
            # Récupérer le champ sélectionné
            loc_name_field = self.dlg.mFieldComboBox.currentField()

            # Récupérer la valeur du DoubleSpinBox (distance seuil)
            distance_threshold = self.dlg.doubleSpinBox.value()

            # Récupérer l'état de la CheckBox (générer Voronoï)
            generate_voronoi = "oui" if self.dlg.checkBox.isChecked() else "non"

            # Afficher les variables pour vérifier si elles ont bien été récupérées
            #print(f"Shapefile Path: {input_layer}")
            print(f"Location Name Field: {loc_name_field}")
            print(f"Distance Threshold: {distance_threshold}")
            print(f"Generate Voronoi: {generate_voronoi}")

            # Passer ces paramètres à la méthode de traitement des données
            self.process_data(input_layer, loc_name_field, distance_threshold, generate_voronoi)

    def process_data(self, input_layer, loc_name_field, distance_threshold, generate_voronoi):
        """Traitement géospatial pour la gestion des pivots et satellites"""
        
        # Créer une nouvelle couche vide nommée "Localités"
        localites_vide = QgsVectorLayer("Point?crs=EPSG:4326", "Localités", "memory")
        localites_provider = localites_vide.dataProvider()

        # Ajouter des champs à la couche Localités
        localites_provider.addAttributes([
            QgsField("Name", QVariant.String),  # Nom de la localité
            QgsField("LAT", QVariant.Double),   # Latitude
            QgsField("LON", QVariant.Double)    # Longitude
        ])
        localites_vide.updateFields()

        # Copier les noms des localités de la couche d'origine dans la couche Localités
        for feature in input_layer.getFeatures():
            localite_feature = QgsFeature()
            localite_feature.setGeometry(feature.geometry())  # Copier la géométrie
            localite_feature.setAttributes([
                feature[loc_name_field],  # Remplacer par le champ du nom des localités
                feature.geometry().asPoint().y(),  # Latitude
                feature.geometry().asPoint().x()   # Longitude
            ])
            localites_provider.addFeature(localite_feature)
    
        # Ajouter la couche Localités au projet
        # QgsProject.instance().addMapLayer(localites_vide)

        # ---- Identifier les pivots et leurs satellites ----

        # Créer une nouvelle couche pour stocker les pivots et leurs satellites
        output_layer = QgsVectorLayer("Point?crs=EPSG:4326", "Pivots et Satellites", "memory")
        output_provider = output_layer.dataProvider()

        # Ajouter des champs pour stocker les informations
        output_provider.addAttributes([
            QgsField("pivot", QVariant.String),      # Nom de la localité pivot
            QgsField("satellite", QVariant.String),  # Nom de la localité satellite
            QgsField("LAT_piv", QVariant.Double),    # Latitude du pivot
            QgsField("LON_piv", QVariant.Double),    # Longitude du pivot
            QgsField("LAT_sat", QVariant.Double),    # Latitude du satellite
            QgsField("LON_sat", QVariant.Double),    # Longitude du satellite
            QgsField("dist_m", QVariant.Double),     # Distance entre pivot et satellite
            QgsField("Type_loc", QVariant.String)    # "Pivot" ou "Satellite"
        ])
        output_layer.updateFields()

        # Initialiser l'outil de mesure des distances
        distance_calculator = QgsDistanceArea()
        distance_calculator.setEllipsoid("WGS84")

        # Liste des localités traitées comme satellites
        treated_localities = set()

        # Liste des localités qui seront des pivots
        features = list(localites_vide.getFeatures())  # Utiliser la nouvelle couche Localités

        # Identifier les pivots et leurs satellites
        for pivot in features:
            if pivot.id() in treated_localities:
                continue

            pivot_geometry = pivot.geometry()
            pivot_point = pivot_geometry.asPoint()
            pivot_nom = pivot["Name"]
            pivot_lat = pivot["LAT"]
            pivot_lon = pivot["LON"]

            # Ajouter ce pivot à la couche de sortie
            for satellite in features:
                satellite_geometry = satellite.geometry()
                satellite_point = satellite_geometry.asPoint()
                distance = distance_calculator.measureLine(pivot_point, satellite_point)

                # Si la localité est dans le rayon spécifié, l'ajouter comme satellite
                if distance <= distance_threshold:
                    satellite_nom = satellite["Name"]
                    satellite_lat = satellite["LAT"]
                    satellite_lon = satellite["LON"]

                    # Ajouter la ligne pour le satellite
                    satellite_feature = QgsFeature()
                    satellite_feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(satellite_point)))
                    satellite_feature.setAttributes([
                        pivot_nom,           # Nom du pivot
                        satellite_nom,       # Nom du satellite
                        pivot_lat,           # Latitude du pivot
                        pivot_lon,           # Longitude du pivot
                        satellite_lat,       # Latitude du satellite
                        satellite_lon,       # Longitude du satellite
                        distance,            # Distance en mètres
                        "Satellite" if distance != 0 else "Pivot"  # Type de localisation
                    ])
                    output_provider.addFeature(satellite_feature)

                    # Marquer cette localité comme traitée (couverte par un pivot)
                    treated_localities.add(satellite.id())

        # ---- Créer la couche Loc_pivots ----

        # Créer une couche temporaire directement en EPSG:4326 pour les pivots
        loc_pivots = QgsVectorLayer("Point?crs=EPSG:4326", "Loc_pivots", "memory")
        loc_pivots_provider = loc_pivots.dataProvider()

        # Ajouter des champs à la couche temporaire
        loc_pivots_provider.addAttributes([
            QgsField("pivot", QVariant.String),  # Nom de la localité pivot
            QgsField("LAT_piv", QVariant.Double),
            QgsField("LON_piv", QVariant.Double),
            QgsField("zone_utm", QVariant.Int),   # Zone UTM
            QgsField("hemisphere", QVariant.String) # Hémisphère
        ])
        loc_pivots.updateFields()

        # Ajouter les pivots directement dans la couche loc_pivots
        for feature in output_layer.getFeatures():
            if feature["Type_loc"] == "Pivot":
                loc_pivot_feature = QgsFeature()
                loc_pivot_feature.setGeometry(feature.geometry())  # Copier la géométrie
                loc_pivot_feature.setAttributes([
                    feature["pivot"],  # Nom de la localité
                    feature["LAT_piv"],  # Latitude
                    feature["LON_piv"],   # Longitude
                    None,                 # Zone UTM (à remplir plus tard)
                    None                  # Hémisphère (à remplir plus tard)
                ])
                loc_pivots_provider.addFeature(loc_pivot_feature)

        # ---- Déterminer la zone UTM et l'hémisphère pour chaque pivot ----

        def get_utm_zone_and_hemisphere(longitude, latitude):
            zone = math.floor((longitude + 180) / 6) + 1
            hemisphere = "Nord" if latitude >= 0 else "Sud"
            return zone, hemisphere

        # Remplir les zones UTM et hémisphères
        for feature in loc_pivots.getFeatures():
            longitude = feature["LON_piv"]
            latitude = feature["LAT_piv"]
            zone, hemisphere = get_utm_zone_and_hemisphere(longitude, latitude)

            # Mettre à jour l'attribut de zone UTM et d'hémisphère
            loc_pivots_provider.changeAttributeValues({
                feature.id(): {
                    loc_pivots.fields().indexOf("zone_utm"): zone,
                    loc_pivots.fields().indexOf("hemisphere"): hemisphere
                }
            })

        # ---- Projeter les Loc_pivots en créant plusieurs couches par zone UTM ----

        utm_layers = {}

        for feature in loc_pivots.getFeatures():
            zone = feature["zone_utm"]
            
            # Créer une couche pour chaque zone UTM si elle n'existe pas
            if zone not in utm_layers:
                utm_layers[zone] = QgsVectorLayer("Point?crs=EPSG:326" + f"{zone:02d}", f"Loc_pivots_utm_zone_{zone}", "memory")
                utm_layers[zone].dataProvider().addAttributes([
                    QgsField("pivot", QVariant.String),
                    QgsField("LAT_piv", QVariant.Double),
                    QgsField("LON_piv", QVariant.Double)
                ])
                utm_layers[zone].updateFields()

            # Transformer les coordonnées
            utm_crs = QgsCoordinateReferenceSystem(f"EPSG:326{zone:02d}" if feature["hemisphere"] == "Nord" else f"EPSG:327{zone:02d}")
            transformer = QgsCoordinateTransform(QgsCoordinateReferenceSystem("EPSG:4326"), utm_crs, QgsProject.instance())
            
            point = feature.geometry().asPoint()
            transformed_point = transformer.transform(point)

            # Ajouter le pivot à la couche correspondante
            new_feature = QgsFeature()
            new_feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(transformed_point)))
            new_feature.setAttributes([
                feature["pivot"],  # Nom de la localité
                feature["LAT_piv"],  # Latitude
                feature["LON_piv"]   # Longitude
            ])
            utm_layers[zone].dataProvider().addFeature(new_feature)

        # ---- Créer des buffers pour chaque couche UTM ----

        buffer_layers = []

        for zone, layer in utm_layers.items():
            # Utilisation de l'algorithme de traitement 'native:buffer'
            params = {
                'INPUT': layer,                # Couche d'entrée
                'DISTANCE': distance_threshold, # Distance du buffer en mètres
                'SEGMENTS': 5,                # Nombre de segments pour la création du buffer
                'DISSOLVE': False,            # Ne pas dissoudre les résultats
                'OUTPUT': 'memory:'           # Sortie en couche temporaire
            }

            # Exécuter l'algorithme de buffer
            buffer_result = processing.run("native:buffer", params)['OUTPUT']
            buffer_result.setName(f"buffer_zone_{zone}")
            buffer_layers.append(buffer_result)  # Conserver la référence pour la fusion

        # ---- Reprojeter les buffers en WGS 84 et fusionner dans range_pivot ----

        # Créer une couche pour stocker le résultat fusionné
        merged_buffer = QgsVectorLayer("Polygon?crs=EPSG:4326", "range_pivot_buffer", "memory")
        merged_buffer_provider = merged_buffer.dataProvider()

        # Ajouter les champs nécessaires
        merged_buffer_provider.addAttributes([
            QgsField("Name", QVariant.String),  # Nom de la localité
            QgsField("LAT", QVariant.Double),    # Latitude
            QgsField("LON", QVariant.Double)     # Longitude
        ])
        merged_buffer.updateFields()

        # Reprojeter et ajouter les buffers à la couche fusionnée
        for buffer_layer in buffer_layers:
            # Reprojeter le buffer en WGS 84
            params = {
                'INPUT': buffer_layer,
                'TARGET_CRS': 'EPSG:4326',
                'OUTPUT': 'memory:'  # Sortie en couche mémoire
            }
            reproj_buffer = processing.run("native:reprojectlayer", params)['OUTPUT']
            
            # Ajouter les features reprojetées à la couche fusionnée
            for feature in reproj_buffer.getFeatures():
                merged_buffer_provider.addFeature(feature)

        # ---- Contrôle de génération des polygones de Voronoï ----

        if generate_voronoi.lower() == "non":
            # Ajouter la couche fusionnée au projet
            QgsProject.instance().addMapLayer(merged_buffer)
            
        else:
            # ---- Créer des polygones de Voronoï autour des localités pivots ----

            # Utilisation de l'algorithme de traitement 'qgis:voronoipolygons'
            voronoi_params = {
                'INPUT': loc_pivots,              # Couche d'entrée
                'BUFFER': 10,                     # Appliquer un buffer de 10% (si nécessaire)
                'OUTPUT': 'memory:'               # Sortie en couche temporaire
            }

            voronoi_result = processing.run("qgis:voronoipolygons", voronoi_params)['OUTPUT']
            voronoi_result.setName("voronoi_polygons")
            #QgsProject.instance().addMapLayer(voronoi_result)

            # ---- Utiliser l'outil "clip" sur les polygones de Voronoï et la couche range_pivot ----
            clip_params = {
                'INPUT': voronoi_result,             # Couche de polygones Voronoï
                'OVERLAY': merged_buffer,            # Couche de buffer fusionné
                'OUTPUT': 'memory:'                  # Sortie en couche temporaire
            }

            clipped_result = processing.run("native:clip", clip_params)['OUTPUT']
            clipped_result.setName("range_pivot_veronoï")
            QgsProject.instance().addMapLayer(clipped_result)  # Ajouter la couche au projet

        # ---- Gestion des doublons parmi les satellites ----

        # Dictionnaire pour stocker les doublons
        duplicates = defaultdict(list)

        # Parcourir les caractéristiques de la couche de sortie
        for feature in output_layer.getFeatures():
            if feature["Type_loc"] == "Satellite":
                # Créer une clé unique pour chaque satellite basé sur ses attributs
                key = (feature["satellite"], feature["LAT_sat"], feature["LON_sat"])
                duplicates[key].append(feature)

        # Liste pour stocker les résultats finaux
        filtered_features = []

        # Traiter les doublons
        for key, features in duplicates.items():
            if len(features) > 1:
                # Classer les doublons par distance
                features.sort(key=lambda f: f["dist_m"])
                
                # Conserver uniquement la localité avec la distance minimale
                filtered_features.append(features[0])
            else:
                # Si pas de doublon, ajouter directement
                filtered_features.append(features[0])

        # Créer une nouvelle couche pour stocker les résultats filtrés
        filtered_layer = QgsVectorLayer("Point?crs=EPSG:4326", "Loc_satellites", "memory")
        filtered_provider = filtered_layer.dataProvider()

        # Ajouter les champs nécessaires
        filtered_provider.addAttributes(output_layer.fields())
        filtered_layer.updateFields()

        # Ajouter les caractéristiques filtrées à la nouvelle couche
        for feature in filtered_features:
            filtered_provider.addFeature(feature)

        # Ajouter la couche filtrée au projet
        filtered_layer.updateExtents()

        # ---- Remplacer les satellites dans Pivots et Satellites par ceux de Localités Filtrées ----

        # Supprimer les satellites de la couche Pivots et Satellites
        output_provider.deleteFeatures([f.id() for f in output_layer.getFeatures() if f["Type_loc"] == "Satellite"])

        # Ajouter les satellites filtrés à la couche Pivots et Satellites
        for feature in filtered_features:
            output_provider.addFeature(feature)

        # Mettre à jour la couche de sortie
        output_layer.updateExtents()

        # Suppression des champs "zone_utm" et "hemisphere" de la couche loc_pivots
        loc_pivots_provider.deleteAttributes([loc_pivots.fields().indexOf("zone_utm"), loc_pivots.fields().indexOf("hemisphere")])
        loc_pivots.updateFields()  # Mettre à jour les champs de la couche

        # Ajouter les couches de sortie au projet
        QgsProject.instance().addMapLayer(output_layer)     # Ajouter la couche Localités et Pivots au projet
        QgsProject.instance().addMapLayer(filtered_layer)   # Ajouter la couche Loc_satellites au projet
        QgsProject.instance().addMapLayer(loc_pivots)       # Ajouter la couche loc_pivots au projet

        print("Le traitement est terminé. Les couches 'Loc_pivots', 'Loc_pivots_utm', et 'Loc_satellites' ont été créées, et chaque buffer a été ajouté au projet.")
