# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Growth region - Raster

                             -------------------
        begin                : 2025-09-02
        copyright            : (C) 2025 by Luiz Motta
        email                : motta.luiz@gmail.com

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 """

from osgeo import gdal, osr, gdal_array
gdal.UseExceptions()

import numpy as np
from numpy.typing import NDArray

from scipy.ndimage import label, binary_fill_holes

from typing import Tuple, Callable, List, Optional, Union

from qgis.core import (
    QgsProject, QgsApplication, Qgis,
    QgsMapLayer, QgsRasterLayer,
    QgsGeometry,
    QgsMapRendererSequentialJob, QgsPalettedRasterRenderer,
    QgsTask
)
from qgis.gui import (
    QgsGui,
    QgsMapCanvas,
    QgsLayerTreeEmbeddedWidgetProvider    
)
from qgis.utils import iface

from qgis.PyQt.QtCore import (
    Qt, QObject,
    QPoint,
    pyqtSlot, pyqtSignal,
    QEvent
)
from qgis.PyQt.QtGui import QImage, QColor, QPainter
from qgis.PyQt.QtWidgets import (
    QWidget,
    QLabel,
    QToolButton, QRadioButton,
    QSpinBox,
    QGroupBox, QButtonGroup,
    QHBoxLayout, QVBoxLayout
)

from .gr_vector_layer import GrowthRegionVectorLayer

from .translate import tr

# from .debugtask import DebugTask # DEBUG

def createImageGrowthRegion(
        source_arr:NDArray[np.uint8],
        seed:Tuple[int, int], # ( line_id, column_id ),
        rgb_threshold:List[int],
        region_value_in: int,
        region_value_out: int,
        is_connect_4: bool,
        isCanceled: Callable[[None], bool]
    )->Optional[NDArray[np.uint8]]:

    def get4Neighbors(row:int, col:int)->List[Tuple[int, int]]:
        return [
            (row + 1, col),
            (row - 1, col),
            (row, col + 1),
            (row, col - 1)
        ]

    def get8Neighbors(row:int, col:int)->List[int]:
        return get4Neighbors(row, col) + [
            (row + 1, col + 1),
            (row - 1, col - 1),
            (row + 1, col - 1),
            (row - 1, col + 1)
        ]

    def fillImage(
            region_value:int,
            region_arry:NDArray[np.uint16],
            getNeighbors:Callable[[int, int], List[int]]
        )->NDArray[np.uint16]:
        # Adaptation from https://github.com/python-pillow/Pillow/src/PIL/ImageDraw.py
        def isSameValue(row:int, col:int)->bool:
            for idx in range( n_bands ):
                # perc = abs( value_seed[ idx ] - region_arry[ idx, row, col ].item() )*100.0 / value_seed[ idx ]
                diff = abs( value_seed[ idx ].item() - region_arry[ idx, row, col ].item() )
                if diff > rgb_threshold[ idx ]:
                    return False
            return True

        def process()->NDArray[np.uint16]:
            ( row, col )= seed
            for idx in range( n_bands ):
                value_seed.append( region_arry[ idx, row, col ] )
                region_arry[ idx, row, col ] = region_value
            edge = { ( row, col ) }
            full_edge = set()
            while edge:
                new_edge = set()
                for ( row, col ) in edge:
                    if isCanceled():
                        return None
                    for (s, t) in getNeighbors( row, col ):
                        # If already processed, or if a coordinate is negative, or a coordinate greather image limit, skip
                        if (s, t) in full_edge or s < 0 or t < 0 or s > (rows-1) or t > (cols-1):
                            continue
                        coord = ( s, t )
                        full_edge.add( coord )
                        if isSameValue( coord[0], coord[1] ):
                            region_arry[ :, coord[0], coord[1] ] = region_value
                            new_edge.add((s, t))
                full_edge = edge  # discard pixels processed
                edge = new_edge

            return region_arry

        ( n_bands, rows, cols ) = source_arr.shape
        value_seed = []

        return process()

    region_value = 1000 # RGB value 0 - 255
    region_bands_arr = source_arr.astype('uint16')
    region_bands_arr = fillImage( region_value, region_bands_arr, get4Neighbors if is_connect_4 else get8Neighbors )
    if region_bands_arr is None:
        return None # Cancel

    region_band_arr = region_bands_arr[0:1, :, :]

    bool_out = ~(region_band_arr == region_value)
    region_band_arr[ bool_out ] = region_value_out
    region_band_arr[ region_band_arr == region_value ] = region_value_in

    return region_band_arr.astype('uint8')

def removeHolesRegion(
        source_arr:NDArray[np.uint8],
        region_value_in: int,
        region_value_out: int,
        max_hole_size: int,
        is_connect_4:bool
    )->Optional[NDArray[np.uint8]]:
    holes_arr = source_arr == region_value_out
    args = {
        'input': holes_arr
    }
    if is_connect_4:
        args['structure'] = np.array([ [0,1,0], [1,1,1], [0,1,0] ])
    labeled_array, num_labels = label( **args )
    # Borders labels
    border_labels = set()
    border_labels.update(labeled_array[0, :])
    border_labels.update(labeled_array[-1, :])
    border_labels.update(labeled_array[:, 0])
    border_labels.update(labeled_array[:, -1])
    if 0 in border_labels: # Remove background
        border_labels.remove(0)

    arr = source_arr.copy()
    for label_id in range(1, num_labels + 1):
        if label_id in border_labels:
            continue
        
        area = np.sum(labeled_array == label_id)
        if area <= max_hole_size:
            arr[labeled_array == label_id] = region_value_in

    return arr

def removeHolesRegionAll(source_arr:NDArray[np.uint8], region_value_in:int)->NDArray[np.uint8]:
    mask = source_arr == region_value_in
    filled_mask = binary_fill_holes(mask)
    arr = source_arr.copy()
    arr[ filled_mask ] = region_value_in

    return arr

def registerGRRLWidgetProvider()->int:
    provider = GrowthRegionRasterLayerWidgetProvider()
    id = provider.id()
    registry = QgsGui.layerTreeEmbeddedWidgetRegistry()
    if bool( registry.provider( id ) ):
        registry.removeProvider( id )
    registry.addProvider( provider )

    return id

def getCanvasRGB(canvas:QgsMapCanvas, pixel_point:QPoint)->Tuple[int, int, int]:
    canvas_image = canvas.grab().toImage()
    if not canvas_image.rect().contains( pixel_point ):
        return
    
    pixel_color = canvas_image.pixelColor( pixel_point )
    return ( pixel_color.red(), pixel_color.green(), pixel_color.blue() )


class GrowthRegionRasterLayer(QObject):
    KEYPROPERTY = 'GrowthRegionRasterLayerWidget'
    showRGB = pyqtSignal(tuple)
    seedRGB = pyqtSignal(tuple)
    changeToRegionButton = pyqtSignal()
    def __init__(self, map_canvas:QgsMapCanvas):
        super().__init__()

        #self.layer_tree_view = iface.layerTreeView()
        self.canvas = map_canvas
        
        self.project = QgsProject.instance()
        self.tree_root = self.project.layerTreeRoot()
        self.taskManager = QgsApplication.taskManager()

        self._layer_name = 'growth_region'
        self._layer_id = None
        self._layer = None
        self._vsipath = f"/vsimem/{self._layer_name}.tif"
        self._task_id = None
        self._id_widget_provider = registerGRRLWidgetProvider()

        self._vector_layer = GrowthRegionVectorLayer( map_canvas )
        self._vector_layer.filepath_raster = self._vsipath

        self._seed_img_point = None
        self._seed_map_point = None # ( x, y)
        self._color_values = {
            'region_value_in': 255,
            'region_value_out': 0
        }

        self.update_region = True
        self.rgb_threshold = ( 50, 50, 50 )
        self.rgb_values = False
        self.is_connect_4 = True
        self.hole_size_max = 0
        self.hole_all = False

        self._last_extent = None

    def updateLayer(self)->None:
        def on_finished(exception, is_cancel=None)->None:
            def addRegisterWidgetProvider(layer):
                layer.growth_region = self # Use by GrowthRegionRasterLayerWidgetProvider.createWidget
                layer.setCustomProperty( self.KEYPROPERTY, True )
                total_ew = int( layer.customProperty('embeddedWidgets/count', 0) )
                layer.setCustomProperty('embeddedWidgets/count', total_ew + 1 )
                layer.setCustomProperty(f"embeddedWidgets/{total_ew}/id", self._id_widget_provider )

            def replaceLayer()->None:
                layer = self.project.mapLayer( self._layer_id  )
                prov = layer.dataProvider()
                prov.reloadData()
                layer.setExtent( prov.extent() )
                layer.triggerRepaint()

                if is_visibled_layer:
                    node = self.tree_root.findLayer( self._layer_id )
                    node.setItemVisibilityChecked(True)

            def createLayer()->None:
                def setRenderer(layer:QgsRasterLayer)->None:
                    band = 1
                    entries = [
                        QgsPalettedRasterRenderer.Class(
                            self._color_values['region_value_in'],
                            QColor(255, 0, 0, 255),
                            f"Red ({self._color_values['region_value_in']})"
                        )
                    ]
                    renderer = QgsPalettedRasterRenderer(layer.dataProvider(), band, entries)
                    layer.setRenderer( renderer )

                layer = QgsRasterLayer( self._vsipath, self._layer_name, 'gdal')
                addRegisterWidgetProvider( layer )
                setRenderer( layer )
                self.project.addMapLayer( layer )
                self._layer_id = layer.id()
                self._layer = layer

            self._task_id = None

            if exception:
                iface.messageBar().pushMessage('Growth Region', str(exception), Qgis.Critical, 3 )
                return

            createLayer() if not existsLayer() else replaceLayer()

            if is_cancel:
                iface.messageBar().pushMessage('Growth Region', 'Canceled by user', Qgis.Critical, 3 )
                return

            provider = self._layer.dataProvider()
            provider.setNoDataValue( 1,  self._color_values['region_value_out'] )

            if not self.update_region:
                self.changeToRegionButton.emit()

        def run(task:QgsTask):
            def getImage()->QImage:
                job = QgsMapRendererSequentialJob( map_settings )
                job.start()
                job.waitForFinished()
                img = job.renderedImage()
                del job

                return img

            def createArrayFromImage(image:QImage)->NDArray[np.uint8]:
                ptr = image.bits()
                ptr.setsize( image.byteCount() )
                arr = np.array(ptr).reshape( (image.height(), image.width(), 4))[:, :, :3]  # Bandss = BGR
                arr_rgb = arr[:, :, [2, 1, 0]] # bands = RGB
                arr_rgb = arr_rgb.transpose((2, 0, 1)) # (height, width, bands) -> (bands, height, width)

                return arr_rgb

            def setGeorefDataset(
                    dataset:gdal.Dataset,
                    rows:int, cols:int
                )->None:
                gt = (
                    canvas_extent.xMinimum(),
                    canvas_extent.width() / cols,
                    0.0,
                    canvas_extent.yMaximum(),
                    0.0,
                    -1*canvas_extent.height() / rows,
                )
                dataset.SetGeoTransform( gt )
                sr = osr.SpatialReference()
                sr.SetFromUserInput( map_settings.destinationCrs().authid() )  # ex.: "EPSG:3857"
                dataset.SetProjection( sr.ExportToWkt() )

            # self.debug.active() # DEBUG

            canvas_extent = self.canvas.extent()
            if self._last_extent and not self._last_extent == canvas_extent:
                point_f = self.canvas.getCoordinateTransform().transform( self._seed_map_point ) # (X,Y)
                self._seed_img_point = QPoint( int(point_f.x()), int(point_f.y()) )
            self._last_extent = canvas_extent

            map_settings = self.canvas.mapSettings()
            # Canvas Array
            img = getImage()
            canvas_arr = createArrayFromImage( img )
            # Region Array
            seed_rgb = getCanvasRGB( self.canvas, self._seed_img_point )
            if not seed_rgb is None:
                self.seedRGB.emit( seed_rgb )
            seed_arr = ( int( self._seed_img_point.y() ), int( self._seed_img_point.x() ) ) # ( line, column )
            args = {
                'source_arr': canvas_arr,
                'seed': seed_arr,
                'rgb_threshold': self.rgb_threshold,
                'is_connect_4': self.is_connect_4,
                'isCanceled': task.isCanceled
            }
            args.update( self._color_values )
            region_arr = createImageGrowthRegion( **args )
            del canvas_arr
            
            if region_arr is None:
                return 1 # Cancel
                
            # Remove holes
            if self.hole_all or self.hole_size_max:
                args_remove = {
                    'source_arr': region_arr[0],
                    'max_hole_size': self.hole_size_max,
                    'is_connect_4': self.is_connect_4                }
                args_remove.update( self._color_values )
                args_remove_all = {
                    'source_arr': region_arr[0],
                    'region_value_in': self._color_values['region_value_in']
                }
                region_arr[0] = removeHolesRegionAll( **args_remove_all ) if self.hole_all else removeHolesRegion( **args_remove )
            # Region Dataset
            ( _, rows, cols) = region_arr.shape
            ds = gdal_array.OpenArray( region_arr )
            setGeorefDataset( ds, rows, cols )
            ds_warp = gdal.Warp(
                "", ds,
                format="MEM",
                dstSRS="EPSG:4326",
            )
            ds = None
            # Memory GeoTif
            band = ds_warp.GetRasterBand(1)
            band.SetDescription('Growth Region')
            ds_ = gdal.GetDriverByName("GTiff").CreateCopy(self._vsipath, ds_warp, options=[])
            ds_ = None
            ds_warp = None

        def createTask():
            name = f"Growth Region raster layer"
            task = QgsTask.fromFunction( name, run, on_finished=on_finished )
            self.taskManager.addTask( task )
            self._task_id = self.taskManager.taskId(task)

            # self.debug = DebugTask() # DEBUG

        def on_RenderComplete(painter: QPainter)->None:
            self.canvas.renderComplete.disconnect( on_RenderComplete )
            createTask()
            return None
            
        existsLayer = lambda: not self._layer_id is None and any(l.id() == self._layer_id for l in self.project.mapLayers().values() )

        is_visibled_layer = False
        if existsLayer():
            node = self.tree_root.findLayer( self._layer_id )
            if node.itemVisibilityChecked():
                is_visibled_layer = True
                self.canvas.renderComplete.connect( on_RenderComplete )
                node.setItemVisibilityChecked(False)
                return
        
        createTask()

    def freeLayer(self)->None:
        if not gdal.VSIStatL( self._vsipath ) is None:
            gdal.Unlink( self._vsipath )

    def flashSeed(self)->None:
        #crs = self.canvas.mapSettings().destinationCrs()
        geom = QgsGeometry.fromPointXY( self._seed_map_point )
        self.canvas.flashGeometries( [ geom] )
        

class GrowthRegionRasterLayerWidget(QWidget):
    def __init__(self, growth_region:GrowthRegionRasterLayer):
        def initGuit(lyt_main:Union[QHBoxLayout, QVBoxLayout]):
            def createSpinBoxRGB(
                    value:int,
                    color:str,
                    layout:Union[QHBoxLayout, QVBoxLayout]
                )->QSpinBox:
                spbx = QSpinBox()
                spbx.setRange(0, 100)
                spbx.setValue( value )
                spbx.setSingleStep(10)
                #spbx.valueChanged.connect(self.on_ValueChangedRGB)
                spbx.setStyleSheet(f"""
                    background-color: {color};
                    color: black;
                    border: 1px solid #CCCCCC;
                    padding-left: 5px;
                """)
                layout.addWidget( spbx )

                return spbx
    
            def createButtonRun(
                    slot:Callable[[None], None],
                    layout:Union[QHBoxLayout, QVBoxLayout]
                )->QToolButton:
                btn = QToolButton()
                btn.setAutoRaise(True)
                btn.setIcon( QgsApplication.getThemeIcon('mTaskRunning.svg') )
                btn.released.connect( slot )
                btn.setStyleSheet("""
                    background-color: #D3D3D3;
                    color: black;
                    border: 1px solid #CCCCCC;
                    padding-left: 5px;
                """)
                layout.addWidget( btn )

            def createLabelChannel(
                    value:str, 
                    color:str,
                    layout:Union[QHBoxLayout, QVBoxLayout]
                )->None:
                label = QLabel( value )
                label.setAlignment(Qt.AlignCenter)
                #label.setMinimumSize(80, 40)
                label.setStyleSheet(f"""
                    background-color: {color};
                    color: black;
                    border: 1px solid #CCCCCC;
                    padding-left: 5px;
                """)
                layout.addWidget( label )

                return label

            # RGB values (seed, cursor)
            grpbox_rgb_values = QGroupBox( tr('RGB Values') )
            grpbox_rgb_values.toggled.connect( self.on_ToggleRGBValues )
            lyt_main.addWidget( grpbox_rgb_values )
            grpbox_rgb_values.setCheckable(True)
            lyt_rgb_labels = QVBoxLayout( grpbox_rgb_values )

            self.format_rgb = '{}--'

            len_label = 10
            lyt_seed = QHBoxLayout()
            lbl_seed = QLabel(tr('Seed:').ljust( len_label ) )
            lbl_seed.enterEvent = self.enterEventSeed
            lyt_seed.addWidget( lbl_seed )
            
            self.lbl_seed_r = createLabelChannel( self.format_rgb.format('R'), '#FF9999', lyt_seed)
            self.lbl_seed_g = createLabelChannel( self.format_rgb.format('G'), '#99CC99', lyt_seed)
            self.lbl_seed_b = createLabelChannel( self.format_rgb.format('B'), '#99CCFF', lyt_seed)
            lbls = ( self.lbl_seed_r, self.lbl_seed_g, self.lbl_seed_b )
            self._setTextRGB( lbls, self._growth_region.seed_rgb )
            lyt_seed.addStretch(1)
            lyt_rgb_labels.addLayout( lyt_seed )

            self._growth_region.seedRGB.connect( self.on_SeedRGBChanged )

            lyt_cursor = QHBoxLayout()
            lyt_cursor.addWidget( QLabel(tr('Cursor:').ljust( len_label ) ) )
            self.lbl_cursor_r = createLabelChannel( self.format_rgb.format('R'), '#FF9999', lyt_cursor)
            self.lbl_cursor_g = createLabelChannel( self.format_rgb.format('G'), '#99CC99', lyt_cursor)
            self.lbl_cursor_b = createLabelChannel( self.format_rgb.format('B'), '#99CCFF', lyt_cursor)
            lyt_cursor.addStretch(1)
            lyt_rgb_labels.addLayout( lyt_cursor )

            self._growth_region.showRGB.connect( self.on_ShowRGB )
            if not self._growth_region.rgb_values:
                grpbox_rgb_values.setChecked(False) # Disconnect self._growth_region.showRGB

            # RGB Threshold
            grpbox_rgb = QGroupBox( tr('RGB Threshold (pixels)') )
            lyt_main.addWidget( grpbox_rgb )
            lyt_rgb = QHBoxLayout( grpbox_rgb )
            self.spbx_r = createSpinBoxRGB( self._growth_region.rgb_threshold[0], '#FF9999', lyt_rgb )
            self.spbx_g = createSpinBoxRGB( self._growth_region.rgb_threshold[1], '#99CC99', lyt_rgb )
            self.spbx_b = createSpinBoxRGB( self._growth_region.rgb_threshold[2], '#99CCFF', lyt_rgb )
            lyt_rgb.addStretch(1)

            # Connect 4 or 8
            grpbox_conn = QGroupBox( tr('Connectivity') )
            lyt_main.addWidget( grpbox_conn )
            lyt_conn = QHBoxLayout( grpbox_conn )

            self.rdbtn_4conn = QRadioButton( tr('4-Cardinal') )
            self.rdbtn_4conn.setChecked( self._growth_region.is_connect_4 )
            rdbtn_8conn = QRadioButton( tr('8-Diagonal') )
            rdbtn_8conn.setChecked( not self._growth_region.is_connect_4 )

            btngrp_conn = QButtonGroup(self)
            btngrp_conn.addButton( self.rdbtn_4conn )
            btngrp_conn.addButton( rdbtn_8conn )

            lyt_conn.addWidget( self.rdbtn_4conn )
            lyt_conn.addWidget( rdbtn_8conn )
            lyt_conn.addStretch(1)

            # Remove hole
            grpbox_hole = QGroupBox( tr('Remove holes(pixels/canvas)') )
            lyt_main.addWidget(grpbox_hole)
            lyt_hole = QHBoxLayout( grpbox_hole )

            self.rbtn_hole_all = QRadioButton( tr('All') )
            self.rbtn_hole_all.setChecked( self._growth_region.hole_all )

            rbtn_hole_size = QRadioButton( tr('Until') )
            rbtn_hole_size.setChecked( not self._growth_region.hole_all )

            btngrp_hole = QButtonGroup(self)
            btngrp_hole.addButton( self.rbtn_hole_all )
            btngrp_hole.addButton( rbtn_hole_size )

            self.spbx_hole_size = QSpinBox()
            self.spbx_hole_size.setRange(0, 100)
            self.spbx_hole_size.setValue(self._growth_region.hole_size_max)
            self.spbx_hole_size.setSingleStep(10)
            self.spbx_hole_size.setEnabled( not self._growth_region.hole_all )

            lyt_hole_size = QHBoxLayout()
            lyt_hole_size.addWidget(rbtn_hole_size)
            lyt_hole_size.addWidget(self.spbx_hole_size)
            lyt_hole_size.addStretch(1)
            
            lyt_hole.addWidget( self.rbtn_hole_all )
            lyt_hole.addLayout( lyt_hole_size )
            lyt_hole.addStretch(1)
            
            # Run
            grpbox_run = QGroupBox( tr('Run') )
            lyt_main.addWidget( grpbox_run )
            lyt_run = QHBoxLayout( grpbox_run )

            self.rdbtn_region = QRadioButton( tr('Update Region') )
            self.rdbtn_region.setChecked( self._growth_region.update_region )
            rdbtn_polygonize = QRadioButton( tr('Region to Vector') )
            rdbtn_polygonize.setChecked( not self._growth_region.update_region )

            self._growth_region.changeToRegionButton.connect( self.on_ChangeToRegionButton )

            btngrp_run = QButtonGroup(self)
            btngrp_run.addButton( self.rdbtn_region )
            btngrp_run.addButton( rdbtn_polygonize )

            lyt_radio_buttons = QVBoxLayout()
            lyt_radio_buttons.addWidget( self.rdbtn_region )
            lyt_radio_buttons.addWidget( rdbtn_polygonize )
            lyt_radio_buttons.addStretch(1)

            lyt_run.addLayout( lyt_radio_buttons )
            createButtonRun( self.on_Run, lyt_run )
            lyt_run.addStretch(1)

        super().__init__()

        self._growth_region = growth_region

        lyt_main = QVBoxLayout()
        initGuit( lyt_main )

        self.setLayout( lyt_main )

    def _setTextRGB(self, w_labels:Tuple[QLabel, QLabel, QLabel], values:Tuple[int, int,int])->None:
        for i in range( len( w_labels ) ):
            w_labels[i].setText(f"{values[i]:03d}")

    # RGB Values
    @pyqtSlot(QEvent)
    def enterEventSeed(self, e:QEvent)->None:
        if self._growth_region.rgb_values:
            self._growth_region.flashSeed()

    @pyqtSlot(bool)
    def on_ToggleRGBValues(self, checked:bool)->None:
        self._growth_region.rgb_values = checked
        if checked:
            self._growth_region.showRGB.connect( self.on_ShowRGB )
            return
        
        self._growth_region.showRGB.disconnect( self.on_ShowRGB )
        self.lbl_cursor_r.setText( self.format_rgb.format('R') )
        self.lbl_cursor_g.setText( self.format_rgb.format('G') )
        self.lbl_cursor_b.setText( self.format_rgb.format('B') )

    @pyqtSlot(tuple)
    def on_SeedRGBChanged(self, values:Tuple[int, int, int])->None:
        self._setTextRGB( ( self.lbl_seed_r, self.lbl_seed_g, self.lbl_seed_b ) , values )

    @pyqtSlot(tuple)
    def on_ShowRGB(self, values:Tuple[int, int, int])->None:
        self._setTextRGB( ( self.lbl_cursor_r, self.lbl_cursor_g, self.lbl_cursor_b ) , values )

    @pyqtSlot()
    def on_Run(self)->None:
        self._growth_region.update_region = self.rdbtn_region.isChecked()
        if self._growth_region.update_region:
            rgb = ( self.spbx_r.value(), self.spbx_g.value(), self.spbx_b.value() )
            self._growth_region.rgb_threshold = rgb

            self._growth_region.is_connect_4 = self.rdbtn_4conn.isChecked()

            self._growth_region.hole_all = self.rbtn_hole_all.isChecked()
            self._growth_region.hole_size_max = self.spbx_hole_size.value()
            
            self._growth_region.updateLayer()

            return

        self._growth_region._vector_layer.updateLayer()

    @pyqtSlot()
    def on_ChangeToRegionButton(self)->None:
        self.rdbtn_region.setChecked(True)


class GrowthRegionRasterLayerWidgetProvider(QgsLayerTreeEmbeddedWidgetProvider):
    def __init__(self):
        super().__init__()

    def id(self)->str:
        return self.__class__.__name__

    def name(self)->str:
        return 'Growth region raster'

    def createWidget(self, layer:QgsMapLayer, widgetIndex:int)->QWidget:
        return GrowthRegionRasterLayerWidget( layer.growth_region )

    def supportsLayer(self, layer):
        return layer.customProperty( GrowthRegionRasterLayer.KEYPROPERTY, False )
