# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Growth region - Vector

                             -------------------
        begin                : 2025-09-02
        copyright            : (C) 2025 by Luiz Motta
        email                : motta.luiz@gmail.com

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 """

from osgeo import gdal, ogr, osr
gdal.UseExceptions()

from typing import List, Union

from qgis.core import (
    QgsProject, QgsApplication,
    QgsMapLayer, QgsVectorLayer,
    QgsField,
    QgsFeatureRequest,
    QgsGeometry, QgsFeature,
    QgsFillSymbol,
    QgsTask    
)
from qgis.gui import (
    QgsGui,
    QgsMapCanvas,
    QgsLayerTreeEmbeddedWidgetProvider    
)
from qgis.utils import iface

from qgis.PyQt.QtCore import (
    QObject,
    QVariant,
    pyqtSlot,
)
from qgis.PyQt.QtWidgets import (
    QWidget,
    QRadioButton,
    QGroupBox, QButtonGroup,
    QHBoxLayout, QVBoxLayout, QGridLayout
)

from .translate import tr

#from .debugtask import DebugTask # DEBUG

def createGeometriesFromRaster(
        filepath:str
    )->List[QgsGeometry]:
    smooth_iter = 1
    smooth_offset = 0.25

    ds_r = gdal.Open( filepath, gdal.GA_ReadOnly )
    proj = ds_r.GetProjection()
    srs = osr.SpatialReference()
    srs.ImportFromWkt( proj )

    band = ds_r.GetRasterBand(1)
    ds_v = ogr.GetDriverByName('MEMORY').CreateDataSource('mem_data')
    layer_v = ds_v.CreateLayer( name='memLayer', srs=srs, geom_type=ogr.wkbPolygon )
    gdal.Polygonize( srcBand=band, maskBand=band, outLayer=layer_v, iPixValField=-1 )
    ds_r = None

    geoms = []

    total = layer_v.GetFeatureCount()
    if not total:
        return geoms

    for feat in layer_v:
        g = QgsGeometry()
        g.fromWkb( feat.GetGeometryRef().ExportToIsoWkb() )
        g = g.smooth( smooth_iter, smooth_offset )
        geoms.append( g )
    ds_v = None

    return geoms

def registerGRVLWidgetProvider()->int:
    provider = GrowthRegionVectorLayerWidgetProvider()
    id = provider.id()
    registry = QgsGui.layerTreeEmbeddedWidgetRegistry()
    if bool( registry.provider( id ) ):
        registry.removeProvider( id )
    registry.addProvider( provider )

    return id


class GrowthRegionVectorLayer(QObject):
    KEYPROPERTY = 'GrowthRegionVectorLayerWidget'
    def __init__(self, map_canvas:QgsMapCanvas):
        super().__init__()

        self.canvas = map_canvas

        self.project = QgsProject.instance()
        self.tree_root = self.project.layerTreeRoot()
        self.taskManager = QgsApplication.taskManager()

        self._layer_name = 'growth_region'
        self._layer_id = None
        self._layer = None
        self._task_id = None
        self._id_widget_provider = registerGRVLWidgetProvider()

        self._field_gr = QgsField('growth_region_id', QVariant.Int )
        self._value_gr = 0

        self.existsLayer = lambda:  not self._layer_id is None and any(l.id() == self._layer_id for l in self.project.mapLayers().values() )

        self.operations = {
            'Add':   { 'id': 1, 'label': tr('Add'), 'function': self._addFeatures },
            'Cut':   { 'id': 2, 'label': tr('Cut'), 'function': self._cutFeatures},
            'Diff':  { 'id': 3, 'label': tr('Difference'), 'function': self._differenceFeatures},
            'Union': { 'id': 4, 'label': tr('Union'), 'function': self._unionFeatures},
        }

        self.operation = 'Add' # Widget
        self.filepath_raster = None

    def _addAttributeGR(self, feat:QgsFeature)->None:
        self._value_gr += 1
        feat.setAttributes( [ self._value_gr ] )

    def _addFeatures(self, geoms:List[QgsGeometry])->dict:
        data = []
        for g in geoms:
            f = QgsFeature()
            f.setGeometry( g )
            self._addAttributeGR( f )
            data.append( f )
        
        return { 'addFeatures': data }

    def _cutFeatures(self, geoms:List[QgsGeometry])->dict:
        geom_union = QgsGeometry.unaryUnion( geoms )
        data = {}
        request = QgsFeatureRequest().setFilterRect( geom_union.boundingBox() )
        for f in self._layer.getFeatures( request ):
            g = f.geometry()
            if g.overlaps( geom_union ):
                g = g.difference( geom_union )
                if not g.isEmpty():
                    data[ f.id() ] = g

        return { 'changeGeometryValues': data }

    def _differenceFeatures(self, geoms:List[QgsGeometry])->dict:
        geom_union = QgsGeometry.unaryUnion( geoms )
        request = QgsFeatureRequest().setFilterRect( geom_union.boundingBox() )
        for f in self._layer.getFeatures( request ):
            g = f.geometry()
            if g.overlaps( geom_union ):
                geom_union = geom_union.difference( g )
                if geom_union.isEmpty():
                    return { 'addFeatures': [] }

        f = QgsFeature()
        f.setGeometry( geom_union )
        self._addAttributeGR( f )

        return { 'addFeatures': [ f ] }

    def _unionFeatures(self, geoms:List[QgsGeometry])->dict:
        geom_union = QgsGeometry.unaryUnion( geoms )
        delete_feats = []
        request = QgsFeatureRequest().setFilterRect( geom_union.boundingBox() )
        for f in self._layer.getFeatures( request ):
            g = f.geometry()
            if g.overlaps( geom_union ):
                geom_union = geom_union.combine( g )
                delete_feats.append( f.id() )

        f = QgsFeature()
        f.setGeometry( geom_union )
        self._addAttributeGR( f )
        return { 'addFeatures': [ f ], 'deleteFeatures': delete_feats }

    def updateLayer(self)->None:
        def on_finished(exception, data:dict)->None:
            def createLayer()->None:
                layer = QgsVectorLayer('Polygon?crs=EPSG:4326', 'growth_region', 'memory')
                
                field = self._field_gr
                field.setReadOnly( True )
                layer.dataProvider().addAttributes( [ field ] )

                symbol = QgsFillSymbol.createSimple({
                    'outline_color': 'blue',
                    'outline_width': '0.46',
                    'style': 'no'
                })
                layer.renderer().setSymbol( symbol )

                # Add Widget in layer
                layer.growth_region = self # Use by GrowthRegionVectorLayerWidgetProvider.createWidget
                layer.setCustomProperty( self.KEYPROPERTY, True )
                total_ew = int( layer.customProperty('embeddedWidgets/count', 0) )
                layer.setCustomProperty('embeddedWidgets/count', total_ew + 1 )
                layer.setCustomProperty(f"embeddedWidgets/{total_ew}/id", self._id_widget_provider )

                self.project.addMapLayer( layer )
                self._layer_id = layer.id()
                self._layer = layer

            self._task_id = None

            if exception:
                return

            if not  self.existsLayer():
                createLayer()
            
            # Update layer
            isEditable = self._layer.isEditable()
            if not isEditable:
                self._layer.startEditing()
            #provider = self._layer.dataProvider()
            if 'addFeatures' in data:
                self._layer.addFeatures( data['addFeatures'] )
                #provider.addFeatures( data['addFeatures'] )
            if 'changeGeometryValues' in data:
                for fid, geom in data['changeGeometryValues'].items():
                    self._layer.changeGeometry( fid, geom )
            if 'deleteFeatures' in data:
                self._layer.deleteFeatures( data['deleteFeatures'] )
            self._layer.commitChanges()
            if isEditable:
                self._layer.startEditing()
            ##
            self._layer.updateExtents()
            self._layer.triggerRepaint()

        def run(task:QgsTask)->List[QgsGeometry]:
            #self.debug.active() # DEBUG
            return self.operations[ self.operation ]['function']( createGeometriesFromRaster(self.filepath_raster ) )

        name = f"Growth Region vector layer"
        task = QgsTask.fromFunction( name, run, on_finished=on_finished )
        self.taskManager.addTask( task )
        self._task_id = self.taskManager.taskId(task)

        # self.debug = DebugTask() # DEBUG


class GrowthRegionVectorLayerWidget(QWidget):
    def __init__(self, growth_region:GrowthRegionVectorLayer):
        def initGuit(lyt_main:Union[QHBoxLayout, QVBoxLayout]):
            grpbox = QGroupBox( tr('Operation') )
            lyt_main.addWidget( grpbox )
            lyt_oper = QGridLayout( grpbox )

            btngrp = QButtonGroup(self)
            rbtn_s = []
            for k, v in self._growth_region.operations.items():
                rdbtn = QRadioButton( v['label'] )
                currentLabel = self._growth_region.operations[self._growth_region.operation]['label']
                rdbtn.setChecked( currentLabel ==  v['label'] )
                btngrp.addButton( rdbtn, v['id'] )
                rbtn_s.append( rdbtn )
            
            row, col = 0, 0
            for w in rbtn_s[:2]:
                lyt_oper.addWidget( w, row, col )
                col += 1

            row, col = 1, 0
            for w in rbtn_s[2:]:
                lyt_oper.addWidget( w, row, col )
                col += 1

            lyt_oper.setColumnStretch(2, 1)
            btngrp.buttonClicked[int].connect( self.on_ButtonGroupClicked )

        super().__init__()

        self._growth_region = growth_region

        lyt_main = QVBoxLayout()
        initGuit( lyt_main )

        self.setLayout( lyt_main )

    @pyqtSlot(int)
    def on_ButtonGroupClicked(self, id:int)->None:
        for k, v in self._growth_region.operations.items():
            if v['id'] == id:
                break
        self._growth_region.operation = k
        # self._growth_region.setVisible( k == 'Add' )


class GrowthRegionVectorLayerWidgetProvider(QgsLayerTreeEmbeddedWidgetProvider):
    def __init__(self):
        super().__init__()

    def id(self)->str:
        return self.__class__.__name__

    def name(self)->str:
        return 'Growth region vector'

    def createWidget(self, layer:QgsMapLayer, widgetIndex:int)->QWidget:
        return GrowthRegionVectorLayerWidget( layer.growth_region )

    def supportsLayer(self, layer):
        return layer.customProperty( GrowthRegionVectorLayer.KEYPROPERTY, False )
