# -*- coding: utf-8 -*-

"""
/***************************************************************************
 gridindex
                                 A QGIS plugin
 This plugin generates a grid of rectangular polygons for map book page indexing.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-07
        copyright            : (C) 2025 by Kapildev Adhikari
        email                : kapildevadk@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Kapildev Adhikari'
__date__ = '2025-06-07'
__copyright__ = '(C) 2025 by Kapildev Adhikari'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import math
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterDistance,
                       QgsProcessingParameterNumber,
                       QgsRectangle,
                       QgsGeometry,
                       QgsFeature,
                       QgsField,
                       QgsFields,
                       QgsWkbTypes,
                       QgsSpatialIndex,
                       QgsProject,
                       QgsProcessingException,
                       QgsFeatureRequest,
                       QgsCoordinateReferenceSystem,
                       QgsCoordinateTransform)
from qgis.PyQt.QtGui import QIcon
import os


class gridindexAlgorithm(QgsProcessingAlgorithm):
    """
    This is a simplified algorithm to create an intersecting grid index, with optional overrides for grid dimensions.
    """

    # Parameter definitions
    INPUT_LAYER = 'INPUT_LAYER'
    CELL_WIDTH = 'CELL_WIDTH'
    CELL_HEIGHT = 'CELL_HEIGHT'
    NUM_ROWS = 'NUM_ROWS'
    NUM_COLS = 'NUM_COLS'
    START_PAGE = 'START_PAGE'
    OUTPUT = 'OUTPUT'

    def initAlgorithm(self, config=None):
        """Define the parameters for the tool."""
        
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_LAYER,
                self.tr('Input Layer (Area of Interest)'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        self.addParameter(
            QgsProcessingParameterDistance(
                self.CELL_WIDTH,
                self.tr('Grid Cell Width (in layer units, or meters for geographic CRS)'),
                parentParameterName=self.INPUT_LAYER,
                defaultValue=1000.0
            )
        )
        
        self.addParameter(
            QgsProcessingParameterDistance(
                self.CELL_HEIGHT,
                self.tr('Grid Cell Height (in layer units, or meters for geographic CRS)'),
                parentParameterName=self.INPUT_LAYER,
                defaultValue=1000.0
            )
        )
        
        self.addParameter(
            QgsProcessingParameterNumber(
                self.NUM_ROWS,
                self.tr('Number of rows (optional override)'),
                QgsProcessingParameterNumber.Integer,
                optional=True,
                defaultValue=0 
            )
        )
        self.parameterDefinitions()[-1].setHelp("Leave empty or 0 to auto-calculate from layer extent. Enter a value to override.")
        
        self.addParameter(
            QgsProcessingParameterNumber(
                self.NUM_COLS,
                self.tr('Number of columns (optional override)'),
                QgsProcessingParameterNumber.Integer,
                optional=True,
                defaultValue=0 
            )
        )
        self.parameterDefinitions()[-1].setHelp("Leave empty or 0 to auto-calculate from layer extent. Enter a value to override.")

        self.addParameter(
            QgsProcessingParameterNumber(
                self.START_PAGE,
                self.tr('Starting page number'),
                QgsProcessingParameterNumber.Integer,
                optional=True,
                defaultValue=1
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output Grid Index')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # 1. Get parameters and input layer
        source = self.parameterAsSource(parameters, self.INPUT_LAYER, context)
        if source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT_LAYER))
            
        cell_width = self.parameterAsDouble(parameters, self.CELL_WIDTH, context)
        cell_height = self.parameterAsDouble(parameters, self.CELL_HEIGHT, context)
        
        num_rows_override = self.parameterAsInt(parameters, self.NUM_ROWS, context)
        num_cols_override = self.parameterAsInt(parameters, self.NUM_COLS, context)
        start_page = self.parameterAsInt(parameters, self.START_PAGE, context)

        # 2. Prepare the output sink
        fields = QgsFields()
        fields.append(QgsField('PageNumber', QVariant.Int))
        fields.append(QgsField('PageName', QVariant.String))

        (sink, dest_id) = self.parameterAsSink(
            parameters, self.OUTPUT, context, fields, QgsWkbTypes.Polygon, source.sourceCrs()
        )
        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        # --- MODIFICATION START: Handle Geographic CRS ---
        source_crs = source.sourceCrs()
        extent = source.sourceExtent()
        is_geographic = source_crs.isGeographic()
        transform_to_calc = None
        transform_to_source = None

        if is_geographic:
            feedback.pushInfo(f"Geographic CRS '{source_crs.authid()}' detected. Grid will be calculated in meters.")
            # Use a global projected CRS for calculations, e.g., Web Mercator (EPSG:3857)
            # This ensures that cell width/height are treated as meters.
            calc_crs = QgsCoordinateReferenceSystem('EPSG:3857')
            if not calc_crs.isValid():
                raise QgsProcessingException("Could not create calculation CRS (EPSG:3857).")

            transform_to_calc = QgsCoordinateTransform(source_crs, calc_crs, QgsProject.instance())
            transform_to_source = QgsCoordinateTransform(calc_crs, source_crs, QgsProject.instance())
            
            # Reproject the source extent to the calculation CRS
            extent = transform_to_calc.transform(extent)
        # --- MODIFICATION END ---
            
        # 3. Get the extent and calculate grid dimensions
        origin_x = extent.xMinimum()
        origin_y = extent.yMinimum()

        if num_rows_override > 0:
            num_rows = num_rows_override
            feedback.pushInfo(f"Using user-provided number of rows: {num_rows}")
        else:
            num_rows = math.ceil(extent.height() / cell_height)
            feedback.pushInfo(f"Calculated number of rows: {num_rows}")

        if num_cols_override > 0:
            num_cols = num_cols_override
            feedback.pushInfo(f"Using user-provided number of columns: {num_cols}")
        else:
            num_cols = math.ceil(extent.width() / cell_width)
            feedback.pushInfo(f"Calculated number of columns: {num_cols}")
        
        # 4. Build spatial index for fast intersection checks (always on original layer)
        feedback.pushInfo("Building spatial index for input layer...")
        spatial_index = QgsSpatialIndex(source.getFeatures())
        
        # 5. Loop through the grid and create features
        page_counter = start_page
        total_potential_features = num_rows * num_cols
        current_feature_index = 0
        
        request = QgsFeatureRequest()

        def get_row_label(n):
            label = ""
            if n < 0: return ""
            while True:
                label = chr(n % 26 + 65) + label
                n = n // 26 - 1
                if n < 0:
                    break
            return label

        # This loop iterates through rows from TOP to BOTTOM
        for r in range(num_rows - 1, -1, -1):
            
            col_name_counter = 1
            
            # This loop iterates through columns from LEFT to RIGHT
            for c in range(num_cols):
                if feedback.isCanceled():
                    return {}
                
                current_feature_index += 1
                feedback.setProgress(int((current_feature_index / total_potential_features) * 100))
                
                # --- MODIFICATION START: Calculate cell in projected space ---
                min_x = origin_x + (c * cell_width)
                min_y = origin_y + (r * cell_height)
                # This rectangle is always in a projected CRS (either original or EPSG:3857)
                calc_rect = QgsRectangle(min_x, min_y, min_x + cell_width, min_y + cell_height)

                # The geometry to be saved must be in the source CRS.
                # The rectangle used for querying the index must also be in the source CRS.
                query_rect = calc_rect
                cell_geom = QgsGeometry.fromRect(calc_rect)

                if is_geographic:
                    # Reproject the calculated cell back to the original geographic CRS
                    query_rect = transform_to_source.transform(calc_rect)
                    cell_geom.transform(transform_to_source)
                # --- MODIFICATION END ---
                
                # Use the (potentially reprojected) query rectangle to find candidates
                candidate_ids = spatial_index.intersects(query_rect)
                
                found_intersection = False
                if candidate_ids:
                    request = QgsFeatureRequest()
                    # Iterate through features in the original layer (in original CRS)
                    for feat_id in candidate_ids:
                        request.setFilterFid(feat_id)
                        input_feat = next(source.getFeatures(request))
                        
                        # Intersect the reprojected cell geometry with the original feature geometry
                        if cell_geom.intersects(input_feat.geometry()):
                            found_intersection = True
                            break
                
                if found_intersection:
                    feat = QgsFeature(fields)
                    feat.setGeometry(cell_geom)
                    
                    top_down_row_index = (num_rows - 1) - r
                    row_letter = get_row_label(top_down_row_index)
                    
                    page_name = f"{row_letter}{col_name_counter}"
                    
                    feat.setAttributes([page_counter, page_name])
                    sink.addFeature(feat, QgsFeatureSink.FastInsert)
                    
                    page_counter += 1
                    col_name_counter += 1

        return {self.OUTPUT: dest_id}

    def description(self):
        """
        Returns a detailed description of the algorithm.
        """
        return self.tr(
            """
            <p>This algorithm creates a grid of rectangular polygon features, designed for creating a map book or atlas index.</p>
            <p>The grid is generated over the full extent of a required <b>Input Layer</b>. However, only grid cells that actually intersect with the features of the input layer will be saved in the final output.</p>
            
            <p><b><u>Coordinate Reference System (CRS) Handling:</u></b></p>
            <p>The tool works with both projected (e.g., UTM) and geographic (e.g., WGS 84) coordinate systems.</p>
            <ul>
                <li>If the input layer has a <b>projected CRS</b>, the cell width and height are interpreted in the layer's native units (e.g., meters, feet).</li>
                <li>If the input layer has a <b>geographic CRS</b> (latitude/longitude), the grid calculations are performed in a temporary projected system (EPSG:3857). This means you should <b>always specify the cell width and height in meters</b>, regardless of the input CRS. The final grid is correctly reprojected back to the original geographic CRS.</li>
            </ul>

            <p><b><u>Grid Dimensions:</u></b></p>
            <p>The size of each grid cell is defined by the <b>Grid Cell Width</b> and <b>Grid Cell Height</b> parameters.</p>
            <p>You can optionally override the automatic calculation of the grid dimensions by providing a specific <b>Number of rows</b> and/or <b>Number of columns</b>. If these are left as 0, the tool will calculate them automatically based on the input layer's extent and the specified cell size.</p>
            
            <p><b><u>Output Attributes:</u></b></p>
            <p>The output grid layer will contain two attribute fields:</p>
            <ul>
                <li><b>PageNumber</b>: A sequential integer, starting from 1 (or the specified <b>Starting page number</b>).</li>
                <li><b>PageName</b>: A grid-style label (e.g., A1, A2, B1...). The number increments sequentially for created cells within each row, starting from the top-left corner.</li>
            </ul>
            """
        )

    def name(self):
        return 'Grid Index'

    def displayName(self):
        return self.tr(self.name())

    def group(self):
        return ''

    def groupId(self):
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return gridindexAlgorithm()
    
    def icon(self):
        # The icon path needs to be correct relative to the plugin's main file.
        # This assumes the icon is in the same directory as the script file.
        return QIcon(os.path.join(os.path.dirname(__file__), 'icon.png'))