# -*- coding: utf-8 -*-
# ============================================================================
# Grid Coordinate Generator - dockwidget
# Author: Aziz T. | Copyright (C) 2025 | License: GPLv3 | Version: 1.0.0
# ============================================================================
from qgis.PyQt import QtWidgets, QtCore
from qgis.PyQt.QtGui import QColor
from qgis.core import (
    QgsProject,
    QgsMapLayerType,
    QgsWkbTypes,
    QgsRectangle
)
from qgis.gui import QgsRubberBand
from .grid_utils import (
    extract_main_polygon,
    get_feature_bbox,
    get_crs_info,
    nice_number,
    convert_meters_to_degrees,
    degrees_to_dms,
    format_coordinate,
    draw_grid,
    clear_grid
)
import math


def create_separator():
    """Crée une ligne de séparation horizontale."""
    line = QtWidgets.QFrame()
    line.setFrameShape(QtWidgets.QFrame.HLine)
    line.setFrameShadow(QtWidgets.QFrame.Sunken)
    return line


class GridGeneratorDock(QtWidgets.QDockWidget):
    """Widget principal pour la génération automatique de grilles."""
    
    def __init__(self, iface):
        """Constructeur.
        
        Args:
            iface: Interface QGIS
        """
        super().__init__(self.tr("Grid Coordinate Generator"))
        self.iface = iface
        self.setAllowedAreas(
            QtCore.Qt.RightDockWidgetArea | QtCore.Qt.LeftDockWidgetArea
        )
        
        self.init_ui()
        self.load_layers()
        self.log_msg(self.tr("✓ Interface initialized. Select one or more layers to begin."))
    
    def tr(self, message):
        """Traduction."""
        return QtCore.QCoreApplication.translate('GridGeneratorDock', message)
    
    def init_ui(self):
        """Initialise l'interface utilisateur."""
        container = QtWidgets.QWidget()
        self.setWidget(container)
        layout = QtWidgets.QVBoxLayout(container)
        
        # Section 1 : Sélection de couches
        layout.addWidget(QtWidgets.QLabel(f"<b>{self.tr('1. Layer Selection')}</b>"))
        
        self.layer_list = QtWidgets.QListWidget()
        self.layer_list.setSelectionMode(QtWidgets.QAbstractItemView.MultiSelection)
        self.layer_list.setMaximumHeight(120)
        layout.addWidget(self.layer_list)
        
        self.btn_refresh = QtWidgets.QPushButton(f"🔄 {self.tr('Refresh Layers')}")
        self.btn_refresh.clicked.connect(self.load_layers)
        layout.addWidget(self.btn_refresh)
        
        layout.addWidget(create_separator())
        
        # Section 2 : Mode de mise en page
        layout.addWidget(QtWidgets.QLabel(f"<b>{self.tr('2. Layout Mode')}</b>"))
        
        self.radio_general = QtWidgets.QRadioButton(self.tr("General (all features together)"))
        self.radio_atlas = QtWidgets.QRadioButton(self.tr("Atlas (by identifier)"))
        self.radio_general.setChecked(True)
        layout.addWidget(self.radio_general)
        layout.addWidget(self.radio_atlas)
        
        self.label_id_field = QtWidgets.QLabel(self.tr("Identifier field:"))
        self.combo_id_field = QtWidgets.QComboBox()
        layout.addWidget(self.label_id_field)
        layout.addWidget(self.combo_id_field)
        
        self.label_id_field.setEnabled(False)
        self.combo_id_field.setEnabled(False)
        
        layout.addWidget(create_separator())
        
        # Section 3 : Unités
        layout.addWidget(QtWidgets.QLabel(f"<b>{self.tr('3. Display Units')}</b>"))
        
        self.combo_units = QtWidgets.QComboBox()
        self.combo_units.addItems([
            self.tr("meters"),
            self.tr("km"),
            self.tr("decimal degrees"),
            self.tr("DMS")
        ])
        layout.addWidget(self.combo_units)
        
        layout.addWidget(create_separator())
        
        # Section 4 : Actions
        btn_layout = QtWidgets.QHBoxLayout()
        
        self.btn_run = QtWidgets.QPushButton(f"▶ {self.tr('Execute')}")
        self.btn_run.setStyleSheet("background-color: #4CAF50; color: white; font-weight: bold; padding: 8px;")
        
        self.btn_clear = QtWidgets.QPushButton(f"🗑 {self.tr('Clear Grid')}")
        
        btn_layout.addWidget(self.btn_run)
        btn_layout.addWidget(self.btn_clear)
        layout.addLayout(btn_layout)
        
        # Navigation
        nav_layout = QtWidgets.QHBoxLayout()
        self.btn_prev = QtWidgets.QPushButton(f"◀ {self.tr('Previous')}")
        self.btn_next = QtWidgets.QPushButton(f"{self.tr('Next')} ▶")
        self.btn_prev.setEnabled(False)
        self.btn_next.setEnabled(False)
        nav_layout.addWidget(self.btn_prev)
        nav_layout.addWidget(self.btn_next)
        layout.addLayout(nav_layout)
        
        layout.addWidget(create_separator())
        
        # Section 5 : Journal
        layout.addWidget(QtWidgets.QLabel(f"<b>{self.tr('Execution Log')}</b>"))
        
        self.log = QtWidgets.QTextEdit()
        self.log.setReadOnly(True)
        self.log.setMaximumHeight(200)
        self.log.setStyleSheet("background-color: #f5f5f5; font-family: monospace;")
        layout.addWidget(self.log)
        
        # Connexions
        self.btn_run.clicked.connect(self.run)
        self.btn_clear.clicked.connect(self.clear_all)
        self.btn_prev.clicked.connect(self.prev_item)
        self.btn_next.clicked.connect(self.next_item)
        self.radio_atlas.toggled.connect(self.toggle_atlas_mode)
        self.layer_list.itemSelectionChanged.connect(self.update_id_fields)
        
        # Variables d'état
        self.canvas = self.iface.mapCanvas()
        self.items = []
        self.current_idx = 0
    
    def log_msg(self, msg):
        """Ajoute un message au journal."""
        self.log.append(msg)
        self.log.ensureCursorVisible()
    
    def load_layers(self):
        """Charge toutes les couches vectorielles."""
        self.layer_list.clear()
        self.layers = []
        
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayerType.VectorLayer:
                geom_type = layer.geometryType()
                icon = ""
                if geom_type == QgsWkbTypes.PolygonGeometry:
                    icon = "🟦"
                elif geom_type == QgsWkbTypes.PointGeometry:
                    icon = "📍"
                elif geom_type == QgsWkbTypes.LineGeometry:
                    icon = "📏"
                
                self.layers.append(layer)
                item = QtWidgets.QListWidgetItem(f"{icon} {layer.name()}")
                item.setData(QtCore.Qt.UserRole, layer.id())
                self.layer_list.addItem(item)
        
        self.log_msg(self.tr(f"✓ {len(self.layers)} vector layer(s) detected."))
    
    def toggle_atlas_mode(self, checked):
        """Active/désactive le mode Atlas."""
        self.label_id_field.setEnabled(checked)
        self.combo_id_field.setEnabled(checked)
        if checked:
            self.update_id_fields()
    
    def update_id_fields(self):
        """Met à jour la liste des champs."""
        self.combo_id_field.clear()
        
        if not self.radio_atlas.isChecked():
            return
        
        selected = self.layer_list.selectedItems()
        if not selected:
            return
        
        layer_id = selected[0].data(QtCore.Qt.UserRole)
        layer = QgsProject.instance().mapLayer(layer_id)
        
        if layer:
            self.combo_id_field.addItems([f.name() for f in layer.fields()])
    
    def clear_all(self):
        """Efface la grille."""
        clear_grid(self.canvas)
        self.items = []
        self.current_idx = 0
        self.btn_prev.setEnabled(False)
        self.btn_next.setEnabled(False)
        self.log.clear()
        self.log_msg(self.tr("✓ Grid cleared."))
    
    def run(self):
        """Exécute le calcul de la grille."""
        self.log.clear()
        self.log_msg(self.tr("▶ Starting processing..."))
        
        selected = self.layer_list.selectedItems()
        if not selected:
            self.log_msg(self.tr("❌ No layer selected."))
            return
        
        self.items = []
        mode_atlas = self.radio_atlas.isChecked()
        
        if mode_atlas:
            id_field = self.combo_id_field.currentText()
            if not id_field:
                self.log_msg(self.tr("❌ No identifier field selected for Atlas mode."))
                return
            
            self.log_msg(self.tr(f"ℹ Atlas mode enabled - Grouping by '{id_field}'"))
            
            for item in selected:
                layer_id = item.data(QtCore.Qt.UserRole)
                layer = QgsProject.instance().mapLayer(layer_id)
                
                groups = {}
                for feat in layer.getFeatures():
                    key = feat[id_field]
                    if key not in groups:
                        groups[key] = []
                    groups[key].append(feat)
                
                for group_id, features in groups.items():
                    self.items.append({
                        'layer': layer,
                        'features': features,
                        'group_id': group_id,
                        'mode': 'atlas'
                    })
            
            self.log_msg(self.tr(f"✓ {len(self.items)} group(s) created."))
        
        else:
            self.log_msg(self.tr("ℹ General mode enabled - All features together"))
            
            all_features = []
            first_layer = None
            for item in selected:
                layer_id = item.data(QtCore.Qt.UserRole)
                layer = QgsProject.instance().mapLayer(layer_id)
                if not first_layer:
                    first_layer = layer
                all_features.extend(list(layer.getFeatures()))
            
            if not all_features:
                self.log_msg(self.tr("❌ No features found in selected layers."))
                return
            
            self.items = [{
                'layer': first_layer,
                'features': all_features,
                'group_id': self.tr('All features'),
                'mode': 'general'
            }]
            
            self.log_msg(self.tr(f"✓ {len(all_features)} feature(s) loaded."))
        
        self.current_idx = 0
        self.show_item(self.current_idx)
        
        if len(self.items) > 1:
            self.btn_prev.setEnabled(True)
            self.btn_next.setEnabled(True)
    
    def show_item(self, idx):
        """Affiche un item avec sa grille."""
        if idx < 0 or idx >= len(self.items):
            return
        
        item = self.items[idx]
        features = item['features']
        
        self.log_msg(f"\n{'='*60}")
        self.log_msg(self.tr(f"📍 Item {idx + 1}/{len(self.items)} : {item['group_id']}"))
        self.log_msg(f"{'='*60}")
        
        # Calculer l'emprise
        bboxes = []
        valid_features = 0
        
        for feat in features:
            bbox = get_feature_bbox(feat)
            if bbox:
                bboxes.append(bbox)
                valid_features += 1
        
        if not bboxes:
            self.log_msg(self.tr("❌ No valid geometry found."))
            return
        
        xmin = min(b[0] for b in bboxes)
        ymin = min(b[1] for b in bboxes)
        xmax = max(b[2] for b in bboxes)
        ymax = max(b[3] for b in bboxes)
        
        width = xmax - xmin
        height = ymax - ymin
        
        self.log_msg(self.tr(f"✓ {valid_features} valid feature(s) analyzed"))
        self.log_msg(self.tr(f"Extent: ({xmin:.6f}, {ymin:.6f}) - ({xmax:.6f}, {ymax:.6f})"))
        
        # Détection CRS
        crs_type, dim_unit = get_crs_info(item['layer'])
        
        if crs_type == 'geographic':
            width_km = width * 111
            height_km = height * 111
            self.log_msg(self.tr(f"Dimensions: {width:.4f}° × {height:.4f}° (~{width_km:.0f} km × {height_km:.0f} km)"))
        else:
            self.log_msg(self.tr(f"Dimensions: {width:.2f} m × {height:.2f} m ({width/1000:.2f} km × {height/1000:.2f} km)"))
        
        if item['layer']:
            crs = item['layer'].crs()
            self.log_msg(self.tr(f"CRS: {crs.authid()} - {crs.description()}"))
        
        # Calcul de l'intervalle
        unit = self.combo_units.currentText()
        target_divisions = 10
        raw_interval = max(width, height) / target_divisions
        interval = None
        interval_display = ""
        
        if crs_type == 'geographic':
            if self.tr("km") in unit:
                interval_km = raw_interval * 111
                interval_km_nice = nice_number(interval_km)
                interval = interval_km_nice / 111
                interval_display = self.tr(f"{interval_km_nice:.2f} km (= {interval:.6f}°)")
            elif self.tr("meters") in unit:
                interval_m = raw_interval * 111000
                interval_m_nice = nice_number(interval_m)
                interval = interval_m_nice / 111000
                interval_display = self.tr(f"{interval_m_nice:.0f} m (= {interval:.6f}°)")
            elif self.tr("decimal degrees") in unit:
                interval = nice_number(raw_interval)
                interval_display = f"{interval:.6f}°"
            elif self.tr("DMS") in unit:
                interval = nice_number(raw_interval)
                d, m, s = degrees_to_dms(interval)
                interval_display = f"{d}° {m}' {s:.2f}\""
        else:
            if self.tr("decimal degrees") in unit or self.tr("DMS") in unit:
                lat_center = (ymin + ymax) / 2
                interval = nice_number(convert_meters_to_degrees(raw_interval, lat_center))
                if self.tr("decimal degrees") in unit:
                    interval_display = f"{interval:.6f}°"
                else:
                    d, m, s = degrees_to_dms(interval)
                    interval_display = f"{d}° {m}' {s:.2f}\""
            elif self.tr("km") in unit:
                interval_km = nice_number(raw_interval / 1000)
                interval = interval_km * 1000
                interval_display = self.tr(f"{interval_km:.2f} km")
            else:
                interval = nice_number(raw_interval)
                interval_display = self.tr(f"{interval:.0f} m")
        
        self.log_msg(self.tr(f"Calculated interval: {interval_display}"))
        
        # Zoom et grille
        rect = QgsRectangle(xmin, ymin, xmax, ymax)
        self.canvas.setExtent(rect)
        self.canvas.refresh()
        
        draw_grid(self.canvas, xmin, ymin, xmax, ymax, interval)
        
        self.log_msg(self.tr("✓ Grid generated and displayed on canvas."))
        self.log_msg(self.tr("\n💡 Parameters for QGIS:"))
        self.log_msg(self.tr(f"   Interval X = {interval:.12f}"))
        self.log_msg(self.tr(f"   Interval Y = {interval:.12f}"))
        self.log_msg(self.tr(f"   Offset X = 0.000000000000"))
        self.log_msg(self.tr(f"   Offset Y = 0.000000000000"))
    
    def prev_item(self):
        """Item précédent."""
        if not self.items:
            return
        self.current_idx = (self.current_idx - 1) % len(self.items)
        self.show_item(self.current_idx)
    
    def next_item(self):
        """Item suivant."""
        if not self.items:
            return
        self.current_idx = (self.current_idx + 1) % len(self.items)
        self.show_item(self.current_idx)
