# -*- coding: utf-8 -*-
# ============================================================================
# Grid Coordinate Generator - main
# Author: Aziz T. | Copyright (C) 2025 | License: GPLv3 | Version: 1.0.0
# ============================================================================
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.core import QgsApplication
import os.path

class GridCoordinateGeneratorPlugin:
    """Plugin principal pour la génération de grilles de coordonnées."""
    
    def __init__(self, iface):
        """Constructeur.
        
        Args:
            iface: Interface QGIS (QgisInterface)
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        # Initialiser la traduction
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            f'gcg_{locale}.qm'
        )
        
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)
        
        # Variables
        self.actions = []
        self.menu = self.tr('&KAT Tools')
        self.dock = None
    
    def tr(self, message):
        """Traduction du texte.
        
        Args:
            message: Texte à traduire
            
        Returns:
            Texte traduit
        """
        return QCoreApplication.translate('GridCoordinateGenerator', message)
    
    def initGui(self):
        """Initialise l'interface graphique."""
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        
        # Créer l'action
        self.action = QAction(
            QIcon(icon_path),
            self.tr('Grid Coordinate Generator'),
            self.iface.mainWindow()
        )
        self.action.setObjectName('gridCoordinateGeneratorAction')
        self.action.setWhatsThis(self.tr('Generate coordinate grids with automatic interval calculation'))
        self.action.setStatusTip(self.tr('Open Grid Coordinate Generator'))
        self.action.triggered.connect(self.run)
        
        # Ajouter à la barre d'outils et au menu
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu(self.menu, self.action)
        
        self.actions.append(self.action)
    
    def unload(self):
        """Décharge le plugin."""
        for action in self.actions:
            self.iface.removeToolBarIcon(action)
            self.iface.removePluginMenu(self.menu, action)
        
        if self.dock:
            self.iface.removeDockWidget(self.dock)
            self.dock.close()
            self.dock = None
    
    def run(self):
        """Exécute le plugin."""
        if not self.dock:
            from .grid_dockwidget import GridGeneratorDock
            self.dock = GridGeneratorDock(self.iface)
        
        self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dock)
        self.dock.show()
