# -*- coding: utf-8 -*-
# ============================================================================
# Grid Coordinate Generator - utils
# Author: Aziz T. | Copyright (C) 2025 | License: GPLv3 | Version: 1.0.0
# ============================================================================
from qgis.core import QgsWkbTypes, QgsPointXY
from qgis.gui import QgsRubberBand
from qgis.PyQt.QtGui import QColor
import math


def extract_main_polygon(geom):
    """Extrait l'anneau extérieur d'un polygone."""
    if geom.isEmpty() or geom.type() != QgsWkbTypes.PolygonGeometry:
        return None
    
    wkb = geom.wkbType()
    
    if QgsWkbTypes.isSingleType(wkb):
        poly = geom.asPolygon()
        return poly[0] if poly else None
    elif QgsWkbTypes.isMultiType(wkb):
        mp = geom.asMultiPolygon()
        return mp[0][0] if mp else None
    
    return None


def get_feature_bbox(feat):
    """Retourne la bbox d'une entité."""
    geom = feat.geometry()
    if geom.isEmpty():
        return None
    
    bbox = geom.boundingBox()
    return (bbox.xMinimum(), bbox.yMinimum(), bbox.xMaximum(), bbox.yMaximum())


def get_crs_info(layer):
    """Détecte si le CRS est géographique ou projeté."""
    if not layer:
        return None, None
    
    crs = layer.crs()
    is_geographic = crs.isGeographic()
    
    if is_geographic:
        return 'geographic', '°'
    else:
        return 'projected', 'm'


def nice_number(value):
    """Retourne un nombre "propre" (1, 2, 5 × 10^n)."""
    if value <= 0:
        return 1
    
    exponent = int(math.floor(math.log10(value)))
    fraction = value / (10 ** exponent)
    
    if fraction < 1.5:
        nice = 1
    elif fraction < 3:
        nice = 2
    elif fraction < 7:
        nice = 5
    else:
        nice = 10
    
    return nice * (10 ** exponent)


def convert_meters_to_degrees(meters, latitude):
    """Convertit mètres en degrés."""
    lat_degree = meters / 111000.0
    lon_degree = meters / (111000.0 * math.cos(math.radians(latitude)))
    return (lat_degree + lon_degree) / 2


def degrees_to_dms(decimal_degrees):
    """Convertit degrés décimaux en DMS."""
    is_negative = decimal_degrees < 0
    decimal_degrees = abs(decimal_degrees)
    
    degrees = int(decimal_degrees)
    minutes_decimal = (decimal_degrees - degrees) * 60
    minutes = int(minutes_decimal)
    seconds = (minutes_decimal - minutes) * 60
    
    if is_negative:
        degrees = -degrees
    
    return (degrees, minutes, seconds)


def format_coordinate(value, unit):
    """Formate une coordonnée selon l'unité."""
    if "meter" in unit.lower() or "mètre" in unit.lower():
        return f"{value:.2f} m"
    elif "km" in unit.lower():
        return f"{value/1000:.3f} km"
    elif "decimal" in unit.lower() or "décimal" in unit.lower():
        return f"{value:.6f}°"
    elif "dms" in unit.lower():
        d, m, s = degrees_to_dms(value)
        return f"{d}° {m}' {s:.2f}\""
    return f"{value}"


def draw_grid(canvas, xmin, ymin, xmax, ymax, interval_x, interval_y=None):
    """Dessine une grille sur le canvas."""
    if interval_y is None:
        interval_y = interval_x
    
    for rb in getattr(canvas, "_grid_rbs", []):
        canvas.scene().removeItem(rb)
    canvas._grid_rbs = []
    
    # Lignes verticales
    x = math.ceil(xmin / interval_x) * interval_x
    while x <= xmax:
        rb = QgsRubberBand(canvas, QgsWkbTypes.LineGeometry)
        rb.setColor(QColor(0, 150, 255, 150))
        rb.setWidth(1)
        rb.addPoint(QgsPointXY(x, ymin))
        rb.addPoint(QgsPointXY(x, ymax))
        canvas._grid_rbs.append(rb)
        x += interval_x
    
    # Lignes horizontales
    y = math.ceil(ymin / interval_y) * interval_y
    while y <= ymax:
        rb = QgsRubberBand(canvas, QgsWkbTypes.LineGeometry)
        rb.setColor(QColor(0, 150, 255, 150))
        rb.setWidth(1)
        rb.addPoint(QgsPointXY(xmin, y))
        rb.addPoint(QgsPointXY(xmax, y))
        canvas._grid_rbs.append(rb)
        y += interval_y


def clear_grid(canvas):
    """Supprime toutes les lignes de grille."""
    for rb in getattr(canvas, "_grid_rbs", []):
        canvas.scene().removeItem(rb)
    canvas._grid_rbs = []
