# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QGribDownloaderDockWidget
                                 A QGIS plugin
 download gribs from opengribs.org
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2019-02-26
        git sha              : $Format:%H$
        copyright            : (C) 2019 by enrico ferreguti
        email                : enricofer@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import urllib
import tempfile
import json
import webbrowser

from PyQt5 import QtGui, QtWidgets, uic
from PyQt5.QtCore import QUrl, pyqtSignal, QJsonDocument, Qt
from PyQt5.QtWidgets import QProgressBar, QApplication, QPushButton, QFileDialog
from PyQt5.QtNetwork import QNetworkRequest

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsNetworkAccessManager,
    QgsMeshLayer,
    QgsProject,
    QgsRectangle,
    Qgis,
    QgsWkbTypes,
    QgsGeometry,
    QgsMessageLog,
    QgsMeshDatasetIndex,
)

from qgis.gui import QgsMessageBar, QgsRubberBand

from .QGribDownloader_defs import (
    atm_models,
    atm_model_capabilities,
    surface_data_decode,
    resolution_decode,
    altitude_decode,
    intv_decode,
    cyc_decode,
    days_decode,
    wave_models,
    wave_model_capabilities,
    wavedata_decode,
)
from .QGribDownloader_server_status import *

MAX_LON_LAT = [-180,-85,180,85]

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'QGribDownloader_dockwidget_base.ui'))


def build_url(baseurl, path, args_dict):
    # Returns a list in the structure of urlparse.ParseResult
    url_parts = list(urlparse.urlparse(baseurl))
    url_parts[2] = path
    url_parts[4] = urllib.urlencode(args_dict)
    return urlparse.urlunparse(url_parts)

class progressBar:
    def __init__(self, parent, title = ''):
        '''
        progressBar class instatiation method. It creates a QgsMessageBar with provided msg and a working QProgressBar
        :param parent:
        :param msg: string
        '''
        self.iface = parent.iface
        self.title = title

    def start(self, msg = '', max=0):
        self.widget = self.iface.messageBar().createMessage(self.title,msg)
        self.progressBar = QProgressBar()
        self.progressBar.setRange(0,max)
        self.progressBar.setValue(0)
        self.progressBar.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.widget.layout().addWidget(self.progressBar)
        QApplication.processEvents()
        self.iface.messageBar().pushWidget(self.widget, Qgis.Info, 50)
        QApplication.processEvents()

    def setProgress(self,value):
        try:
            self.progressBar.setValue(value)
            QApplication.processEvents()
        except:
            pass

    def setMsg(self,msg):
        self.widget.setText(msg)
        QApplication.processEvents()

    def stop(self, msg = '',level=Qgis.Info,duration=2):
        '''
        the progressbar is stopped with a succes message
        :param msg: string
        :return:
        '''
        self.iface.messageBar().clearWidgets()
        message = self.iface.messageBar().createMessage(self.title,msg)
        self.iface.messageBar().pushWidget(message, level, duration)

class QGribDownloaderDockWidget(QtWidgets.QDockWidget, FORM_CLASS):

    closingPlugin = pyqtSignal()

    def __init__(self, iface, parent=None):
        """Constructor."""
        super(QGribDownloaderDockWidget, self).__init__(parent)
        # Set up the user interface from Designer.
        # After setupUI you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.iface = iface
        self.context = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.PolygonGeometry)
        self.context.setFillColor(QtGui.QColor("#000000ff"))
        self.context.setStrokeColor(QtGui.QColor("#fdb100"))
        self.context.setWidth(2)
        self.modelLimits = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.PolygonGeometry)
        self.modelLimits.setFillColor(QtGui.QColor("#000000ff"))
        self.modelLimits.setStrokeColor(QtGui.QColor("#69c1ff"))
        self.context.setWidth(2)
        #self.modelLimits.setStrokeColor(QtGui.QColor("#000000ff"))
        self.modelsComboBox.addItems(atm_models)
        self.wavemodelComboBox.addItems(wave_models)
        self.modelsComboBox.currentIndexChanged.connect(self.model_changed)
        self.download_pushButton.clicked.connect(self.download)
        self.status_pushButton.clicked.connect(self.status)
        self.mExtentGroupBox.setMapCanvas(self.iface.mapCanvas())
        self.mExtentGroupBox.setCurrentExtent(self.iface.mapCanvas().extent(),self.iface.mapCanvas().mapSettings().destinationCrs())
        self.mExtentGroupBox.setOriginalExtent(QgsRectangle(6,36,18,47),QgsCoordinateReferenceSystem(4326))
        self.mExtentGroupBox.setOutputCrs(QgsCoordinateReferenceSystem(4326))
        self.mExtentGroupBox.extentChanged.connect(self.drawContext)
        self.visibilityChanged.connect(self.dockWidgetChangedVisibility)
        self.lim_rect = None
        self.model_changed(0)
        self.infoButton.clicked.connect(self.loadModelInfo)
        self.status_dialog = QGribDownloaderServerStatusDialog(self.iface)

        #mButtonDrawOnCanvas causes crash
        for button in self.mExtentGroupBox.findChildren(QPushButton):
            if button.objectName() == "mButtonDrawOnCanvas":
                button.hide()

    def loadModelInfo(self):
        info_url = 'https://opengribs.org/en/gribs'
        webbrowser.open(info_url)


    def model_changed(self,current_model):
        self.capabilities = atm_model_capabilities[self.modelsComboBox.currentText()]
        for field in ["altitude", "surface_data"]:
            field_decode = globals()[field +"_decode"]
            for attr in field_decode.keys():
                field_widget = getattr(self, "_checkBox_" + attr.replace(";",""))
                field_widget.setEnabled(attr in self.capabilities[field])
        for field in ["resolution", "intv", "cyc", "days"]:
            field_widget =  getattr(self,field +"_comboBox")
            field_widget.clear()
            field_decode = globals()[field +"_decode"]
            for attr in self.capabilities[field]:
                field_widget.addItem(field_decode[attr],attr)
        self.drawLimits()


    def status(self):
        self.status_dialog.updateServerStatus()
        self.status_dialog.exec_()

    def download(self):

        self.progress = progressBar(self,"QGribDownloader")
        self.progress.start("Sending request to opengribs.org")
        ext = self.mExtentGroupBox.outputExtent().intersect(QgsRectangle(*MAX_LON_LAT))

        url_params = {
            "osys": "Unknown",
            "ver": atm_model_capabilities["version"],
            "la1": str(ext.yMinimum()),
            "la2": str(ext.yMaximum()),
            "lo1": str(ext.xMinimum()),
            "lo2": str(ext.xMaximum()),
            "model": self.capabilities["parameter"]+self.resolution_comboBox.currentData(),
            "wmdl": wave_model_capabilities[self.wavemodelComboBox.currentText()]["wmdl"]
        }

        for param in ["intv", "days", "cyc"]:
            wdgt = getattr(self,param + "_comboBox")
            url_params[param] = str(wdgt.currentData())

        atm_params = ""
        for param in list(surface_data_decode.keys())+list(altitude_decode.keys()):
            wdgt = getattr(self, "_checkBox_" + param.replace(";",""))
            if wdgt.isEnabled():
                if wdgt.isChecked():
                    atm_params += param

        wave_params = ""
        for param in ["s;", "H;D;P;", "h;d;p;"]:
            wdgt = getattr(self, "_checkBox__" + param.replace(";",""))
            if wdgt.isEnabled():
                if wdgt.isChecked():
                    wave_params += param

        url_params["par"] = atm_params
        url_params["wpar"] = wave_params

        url = "http://grbsrv.opengribs.org/getmygribs2.php/?" + urllib.parse.urlencode(url_params)
        QgsMessageLog.logMessage("Request: " + url, 'QGribDownloader', level=Qgis.Info)

        request = QNetworkRequest()
        request.setUrl(QUrl(url))
        self.reply_stage1 = QgsNetworkAccessManager.instance().get(request)
        self.reply_stage1.finished.connect(self.asyncManageResponse_stage1)

    def asyncManageResponse_stage1(self):
        self.progress.stop()
        stringReply = str(self.reply_stage1.readAll(),'utf-8')
        QgsMessageLog.logMessage("Reply: " + stringReply, 'QGribDownloader', level=Qgis.Info)
        replyObject = json.loads(stringReply)
        if replyObject["status"]:
            self.mesh_name = QUrl(replyObject["message"]["url"]).fileName()
            self.targetFolder = QFileDialog.getExistingDirectory(self, self.tr("Save Grib"))
            request = QNetworkRequest()
            request.setUrl(QUrl(replyObject["message"]["url"]))
            self.reply_stage2 = QgsNetworkAccessManager.instance().get(request)
            self.reply_stage2.finished.connect(self.asyncManageResponse_stage2)
            self.reply_stage2.downloadProgress.connect(self.asyncDownloadProgress)
            self.progress.start("Downloading " + self.mesh_name, replyObject["message"]["size"])
        else:
            self.iface.messageBar().pushMessage("QGribDownloader", "Download error: " + replyObject["message"], level=Qgis.Critical, duration=5)
            QgsMessageLog.logMessage("Download error: " + replyObject["message"], 'QGribDownloader', level=Qgis.Warning)

    def asyncManageResponse_stage2(self):
        targetFile = os.path.join(self.targetFolder,self.mesh_name)
        raw_data = self.reply_stage2.readAll()
        with open(targetFile, 'wb') as f:
            f.write(raw_data)
        self.progress.stop("Downloaded: " + targetFile, level=Qgis.Success, duration=15 )
        QgsMessageLog.logMessage("Downloaded: " + targetFile, 'QGribDownloader', level=Qgis.Info)
        meshLayer = QgsMeshLayer(targetFile, self.mesh_name, "mdal")
        dp = meshLayer.dataProvider()
        gprCount = dp.datasetGroupCount()
        for i in range(gprCount):
            meta = dp.datasetGroupMetadata(QgsMeshDatasetIndex(i, 0))
            isVector = meta.isVector()
            name = meta.name()
            if isVector:
                break

        s = meshLayer.rendererSettings()
        s.setActiveVectorDatasetGroup(i)
        s.setActiveScalarDatasetGroup(i)
        meshLayer.setRendererSettings(s)
        QgsProject.instance().addMapLayer(meshLayer)
        self.iface.mapCanvas().setDestinationCrs(QgsCoordinateReferenceSystem(4326))
        QgsProject.instance().setCrs(QgsCoordinateReferenceSystem(4326))

    def asyncDownloadProgress(self,downloaded,total):
        self.progress.setProgress(downloaded)

    def dockWidgetChangedVisibility(self, visible):
        if visible:
            self.drawContext(self.mExtentGroupBox.currentExtent())
            self.drawLimits()
        else:
            self.context.reset()
            self.modelLimits.reset()

    def updateExtent(self):
        self.mExtentGroupBox.setOutputExtentFromCurrent()

    def drawContext(self,ext):
        ext = self.mExtentGroupBox.outputExtent().intersect(QgsRectangle(*MAX_LON_LAT))
        ext_geom = QgsGeometry.fromWkt(ext.asWktPolygon())
        self.context.reset()
        self.context.addGeometry(ext_geom,self.mExtentGroupBox.outputCrs())
        self.checkModelLimits()

    def drawLimits(self):
        self.modelLimits.reset()
        if self.capabilities["limits"]:
            self.limits = self.capabilities["limits"]
        else:
            self.limits = [-180,-90,180,90]
        self.lim_rect = QgsRectangle(*self.limits)
        if self.capabilities["limits"]:
            lim_geom = QgsGeometry.fromWkt(self.lim_rect.asWktPolygon())
            self.modelLimits.addGeometry(lim_geom,QgsCoordinateReferenceSystem(4326))
        self.checkModelLimits()

    def checkModelLimits(self):
        if self.lim_rect:
            if self.lim_rect.intersects(self.mExtentGroupBox.outputExtent().intersect(QgsRectangle(*MAX_LON_LAT))):
                self.download_pushButton.setEnabled(True)
            else:
                self.download_pushButton.setEnabled(False)

    def closeEvent(self, event):
        self.closingPlugin.emit()
        event.accept()
