# -*- coding: utf-8 -*-
"""
/******************************************************************************
        grd2stream – generates streamlines from velocity grids using GMT6
                                 A QGIS plugin
        This plugin utilizes the CLI tool 'grd2stream' by Thomas Kleiner
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-12-01
        copyright            : (C) 2013-2025 by Thomas Kleiner & SFP-Team
        email                : scienceflightplanner@gmail.com
        git sha              : $Format:%H$
 ******************************************************************************/

/******************************************************************************
 *                                                                            *
 *    This program is free software; you can redistribute it and/or modify    *
 *    it under the terms of the GNU General Public License as published by    *
 *    the Free Software Foundation; either version 3 of the License, or       *
 *    (at your option) any later version.                                     *
 *                                                                            *
 ******************************************************************************/
"""
import os

from qgis.PyQt.QtWidgets import QWidget
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis._core import QgsSettings

# Initialize Qt resources from file resources.py (DON'T DELETE IT)
from .resources import *
# Import the code for the dialog
from .grd_2_stream_dialog import Grd2StreamDialog
import os.path

from .flowline_module import FlowlineModule
from .help_widget import add_help_menu_action

plugin_instance = None


def is_dark_theme():
    """Detect if QGIS is using a dark theme by sampling colors from the UI"""
    from qgis.PyQt.QtWidgets import QApplication
    from qgis.PyQt.QtGui import QPalette
    app = QApplication.instance()
    background_color = app.palette().color(QPalette.Window)
    luminance = (0.299 * background_color.red() +
                 0.587 * background_color.green() +
                 0.114 * background_color.blue()) / 255
    return luminance < 0.5


class Grd2Stream:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        global plugin_instance
        if plugin_instance is not None:
            raise RuntimeError("Another instance of the plugin is already running.")
        plugin_instance = self
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        self.icon_dir = os.path.join(self.plugin_dir, "resources", "icons")
        self.toolbar = self.iface.addToolBar("grd2stream Toolbar")
        self.toolbar.setObjectName("grd2stream")
        self.flowline_module = FlowlineModule(iface)
        self.flowline_action = None
        self.help_action = None

    def add_action(
        self,
        icon: str,
        text: str,
        callback,
        enabled_flag: bool = True,
    ) -> QAction:
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """
        icon_path = os.path.join(self.icon_dir, icon)
        if not os.path.exists(icon_path):
            self.iface.messageBar().pushWarning(
                "Warning",
                f"Icon file not found: {icon_path}. Using a placeholder."
            )
            icon = QIcon()
        else:
            icon = QIcon(icon_path)

        action = QAction(icon, text, self.iface.mainWindow())
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        self.toolbar.addAction(action)
        return action

    def update_icon_theme(self):
        """Update the icon based on the current theme"""
        if is_dark_theme():
            icon_path = os.path.join(self.icon_dir, "flowline_darkmode.png")
        else:
            icon_path = os.path.join(self.icon_dir, "flowline.png")
        if os.path.exists(icon_path):
            self.flowline_action.setIcon(QIcon(icon_path))

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = "flowline.png"
        self.flowline_action = self.add_action(
            icon=icon_path,
            text="Calculate Flowlines",
            callback=self.flowline_module.open_selection_dialog
        )
        self.help_action = add_help_menu_action(self.iface, self.plugin_dir)
        self.update_icon_theme()
        from qgis.PyQt.QtWidgets import QApplication
        QApplication.instance().paletteChanged.connect(self.update_icon_theme)

    def unload(self):
        """Properly unloads the plugin, ensuring no lingering instances."""
        if self.flowline_action:
            self.flowline_action.triggered.disconnect()
            self.toolbar.removeAction(self.flowline_action)

        if hasattr(self, 'help_action'):
            self.iface.pluginHelpMenu().removeAction(self.help_action)
            self.help_action = None

        if self.toolbar:
            del self.toolbar

        global plugin_instance
        plugin_instance = None
