# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Graphab
                                 A QGIS plugin
 This plugin allows you to use Graphab software directly into QGis
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-04-20
        git sha              : $Format:%H$
        copyright            : (C) 2020 by ThéMA
        email                : robin.marlinl@gmail.com
        author               : Robin Marlin-Lefebvre
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os # This is is needed in the pyqgis console also
import subprocess

import processing

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QInputDialog, QMessageBox, QFileDialog

from .processing.GraphabProvider import GraphabProvider
# Initialize Qt resources from file resources.py
from .resources import *

# To add contextuel menu (UNSTABLE)
#from .context_menu_graphab import CustomMenuProvider

from qgis.core import QgsApplication, QgsProject, QgsMessageLog, Qgis

# Import necessary objects
from .GraphabProject import GraphabProject
from .GraphabStyle import GraphabStyle
from .graph_symbology_dialog import GraphSymbologyDialog
from .create_graph_dialog import CreateGraphDialog
from .create_linkset_dialog import CreateLinksetDialog
from .calculate_metrics_dialog import CalculateMetricDialog
from .corridor_dialog import CorridorDialog


class GraphabPlugin:
    """QGIS Plugin Implementation."""

    PROBA_DIST = ['0.5', '0.05']
    GMETRICS = ['PC', 'EC', 'IIC']
    LMETRICS = ['F', 'IF', 'BC']
    UNITS = []

    graphabProvider = None
    java = True

    projects = {}

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        
        # all available styles for Graph circles
        self.stylesTabUnabled = ["Red","Blue"]
        
        # csv prefix that is important to know because it's in imported fieldnames
        self.prefix = '_'

        self.UNITS = [self.translate('py', 'Meter'), self.translate('py', 'Cost')]

        # All necessary objects
        self.GraphabStyle = GraphabStyle(self)
        self.GraphSymbologyDialog = GraphSymbologyDialog(self)
        self.CreateGraphDialog = CreateGraphDialog(self)
        self.CreateLinksetDialog = CreateLinksetDialog(self)
        self.CalculateMetricDialogGlobal = CalculateMetricDialog(self.GMETRICS, 0, self)
        self.CalculateMetricDialogLocal = CalculateMetricDialog(self.LMETRICS, 1, self)
        self.CorridorDialog = CorridorDialog(self)

        self.graphabProvider = GraphabProvider(self)
        QgsApplication.processingRegistry().addProvider(self.graphabProvider)

        javaExec = self.graphabProvider.getJavaCommand()
        ret = subprocess.run([javaExec, '-version'], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        if ret.returncode != 0:
            QgsMessageLog.logMessage("Java not found for Graphab plugin.\n" + javaExec + "\n", 'Extensions', Qgis.Warning)
            self.java = False


    #--------------------------------------------------------------------------
    @staticmethod
    def translate(context, string):
        return QCoreApplication.translate(context, string)

    #--------------------------------------------------------------------------

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip:
            action.setStatusTip(status_tip)

        if whats_this:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    #--------------------------------------------------------------------------

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        self.menu = u'&Graphab'

        self.toolbar = self.iface.addToolBar(u'Graphab')
        self.toolbar.setObjectName(u'Graphab')
        self.actions = []

        # Load Graphab project button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_open_project.png',
            text=self.translate('py', u'Load a project'),
            callback=self.openProject,
            parent=self.iface.mainWindow())


        # Create Graphab Project button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_project.png',
            text=self.translate('py', u'Create Project'),
            callback=self.createNewProject,
            parent=self.iface.mainWindow())

        # Create Linkset button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_linkset.png',
            text=self.translate('py', u'Create Linkset'),
            callback=self.CreateLinksetDialog.showDialog,
            parent=self.iface.mainWindow())
        # Remove Linkset button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_linkset_rem.png',
            text=self.translate('py', u'Remove Linkset'),
            callback=self.removeLinkset,
            parent=self.iface.mainWindow())
        # Create Graph button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_graph.png',
            text=self.translate('py', u'Create Graph'),
            callback=self.CreateGraphDialog.showDialog,
            parent=self.iface.mainWindow())
        # Remove Graph button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_graph_rem.png',
            text=self.translate('py', u'Remove Graph'),
            callback=self.removeGraph,
            parent=self.iface.mainWindow())
        # Calculate corridors
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_corridor.png',
            text=self.translate('py', u'Calculate Corridors'),
            callback=self.CorridorDialog.showDialog,
            parent=self.iface.mainWindow())
        # Calculate Local Metrics
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_mlocal.png',
            text=self.translate('py', u'Calculate Local Metric'),
            callback=self.CalculateMetricDialogLocal.showDialog,
            parent=self.iface.mainWindow())
        # Calculate Global Metrics
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_mglobal.png',
            text=self.translate('py', u'Calculate Global Metric'),
            callback=self.CalculateMetricDialogGlobal.showDialog,
            parent=self.iface.mainWindow())

        # Change symbology button
        self.add_action(
            ':/plugins/graphab-for-qgis/icons/icon_symbo.png',
            text=self.translate('py', u'Display metric'),
            callback=self.GraphSymbologyDialog.showDialog,
            parent=self.iface.mainWindow())

    # --------------------------------------------------------------------------

    def showJavaNotFound(self):
        QMessageBox.warning(self.iface.mainWindow(), "Graphab plugin",
                            "Java executable not found, you can only display Graphab project. "
                            "Please install Java from https://adoptopenjdk.net to access all features of the plugin")

    # --------------------------------------------------------------------------

    def openProject(self):
        # open a window to select a project xlm of Graphab a
        filename, _filter = QFileDialog.getOpenFileName(None, "Select Graphab project", "", '*.xml')

        # when you close the window but you haven't select a file
        if filename is None or filename == '':
            return

        self.loadProject(filename)


    # --------------------------------------------------------------------------

    def loadProject(self, filename):
        filename = os.path.normcase(filename)
        self.updateLoadedProject()
        # if the file is already loaded the plugin stop
        if filename in self.projects.keys():
            return

        self.projects[filename] = GraphabProject(self, filename)

    # --------------------------------------------------------------------------

    def createNewProject(self):
        if not self.java:
            self.showJavaNotFound()
            return
        processing.execAlgorithmDialog("graphab:create_project")

    # --------------------------------------------------------------------------

    def removeLinkset(self):
        if not self.java:
            self.showJavaNotFound()
            return

        linksetNames = []
        for prj in self.getProjects():
            for linkset in prj.project.costLinks:
                linksetNames.append(prj.project.name + ' -> ' + linkset.name)

        res, ok = QInputDialog().getItem(self.iface.mainWindow(), self.translate('py', "Remove linkset"),
                                         self.translate('py', "Select the linkset to remove :"), linksetNames, 0, False)
        if ok and res:
            project, linkset = res.split(' -> ')
            prj = self.getProject(project)
            if not prj.removeLinkset(linkset):
                QMessageBox.warning(self.iface.mainWindow(), self.translate('py', "Remove linkset"), "Impossible to remove linkset " + linkset)

    # --------------------------------------------------------------------------

    def removeGraph(self):
        if not self.java:
            self.showJavaNotFound()
            return

        graphNames = []
        for prj in self.getProjects():
            for graph in prj.project.graphs:
                graphNames.append(prj.project.name + ' -> ' + graph.name)

        res, ok = QInputDialog().getItem(self.iface.mainWindow(), self.translate('py', "Remove graph"),
                                         self.translate('py', "Select the graph to remove :"), graphNames, 0, False)
        if ok and res:
            project, graph = res.split(' -> ')
            prj = self.getProject(project)
            if not prj.removeGraph(graph):
                QMessageBox.warning(self.iface.mainWindow(), self.translate('py', "Remove graph"), "Impossible to remove graph " + graph)

    # --------------------------------------------------------------------------

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        if self.graphabProvider:
            QgsApplication.processingRegistry().removeProvider(self.graphabProvider)

        for action in self.actions:
            self.iface.removePluginMenu(
                self.menu,
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    # --------------------------------------------------------------------------

    def getProject(self, projectName):
        for project in self.projects.values():
            if project.project.name == projectName:
                return project

        return None

    # --------------------------------------------------------------------------

    def updateLoadedProject(self):
        projects = {}
        root = QgsProject.instance().layerTreeRoot()
        for group in root.findGroups():
            if "GraphabProject" in group.customProperties():
                prjFile = group.customProperty("GraphabProject")
                if os.path.isfile(prjFile):
                    if prjFile in self.projects.keys():
                        projects[prjFile] = self.projects[prjFile]
                    else:
                        projects[prjFile] = GraphabProject(self, prjFile, group)
                else:
                    group.removeCustomProperty("GraphabProject")


        self.projects = projects

    # --------------------------------------------------------------------------

    def getProjects(self):
        self.updateLoadedProject()
        return list(self.projects.values())

