# -*- coding: utf-8 -*-

__author__ = 'gaspard.quentin1905@gmail.com'
__date__ = '2024-04-26'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

# This will get replaced with a git SHA1 when you do a git archive
__revision__ = '$Format:%H%'


import os

from qgis.core import (QgsProcessingParameterString,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingContext,
                       QgsProcessingFeedback,
                       QgsProcessingParameterFile, QgsProcessingParameterEnum)

from ..GraphabProject import GraphabProject


from .GraphabAlgoProcessing import GraphabAlgoProcessing


class CreateVectorHabitatFromFile(GraphabAlgoProcessing):
    
    def __init__(self, plugin):
        super().__init__(plugin)

    INPUT = 'INPUT'
    FILE = 'FILE'
    HABITATNAME = 'HABITATNAME'
    MINCAPA = 'MINCAPA'
    CAPA = 'CAPA'
    NOVORONOI = 'NOVORONOI'

    def initAlgorithm(self, config):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.translate('py', 'XML project file'),
                fileFilter = 'xml(*.xml)',
                optional = False
            )
        )

        self.addParameter(
            QgsProcessingParameterFile(
                self.FILE,
                self.translate('py', 'Vector file'),
                fileFilter = 'Vector layer (*.gpkg *.geojson *.shp);; All files (*)',
                optional = False
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.HABITATNAME,
                self.translate('py', 'Name of the habitat that will be created'),
                optional = False
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MINCAPA,
                self.translate('py', 'Minimum capacity'),
                type = QgsProcessingParameterNumber.Double,
                minValue = 0.0,
                defaultValue = 0.0,
                optional = True
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.CAPA,
                self.translate('py', 'Capacity field'),
                optional = True
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NOVORONOI,
                self.translate('py', 'No voronoi'),
                defaultValue = False,
                optional = False
            )
        )



    def processAlgorithm(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> dict:
        """
        Create a command line depending of parameters set by the user and send it to its parent 
        to execute it
        """
        project = self.parameterAsFile(parameters, self.INPUT, context)
        self.projectPath = os.path.normcase(r'' + project)
        source = self.parameterAsFile(parameters, self.FILE, context)
        source_path = os.path.normcase(r'' + source)
        self.name = self.parameterAsString(parameters, self.HABITATNAME, context)
        min_capa = self.parameterAsDouble(parameters, self.MINCAPA, context)
        capa = self.parameterAsString(parameters, self.CAPA, context)
        novoronoi = self.parameterAsBool(parameters, self.NOVORONOI, context)

        commandToExec = ['--project', self.projectPath, '--vhabitat', 'name=' + self.name, 'file=' + source_path]

        if min_capa > 0.0:
            commandToExec.append(f'mincapa={min_capa}')
        if capa:
            commandToExec.append(f'capa={capa}')
        if novoronoi:
            commandToExec.append('novoronoi')

        super().processAlgorithm(commandToExec, context, feedback)
        return {}

    def postProcessAlgorithm(self, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> dict:
        """After the execution of the processing, this function
        add and show the new Graph in the map. """
        project: GraphabProject = self.plugin.getProjectFromPath(self.projectPath)

        if self.executionError or project is None:
            return {}

        project.add_habitat_project(self.name)
        return {}


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'create_vector_habitat_from_file'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.translate('py', 'Create Vector Habitat (from file)')

    def createInstance(self):
        return CreateVectorHabitatFromFile(self.plugin)


class CreateVectorHabitatFromHabitat(GraphabAlgoProcessing):
    
    def __init__(self, plugin):
        super().__init__(plugin)

    INPUT = 'INPUT'
    HABITATSRC = 'HABITATSRC'
    HABITATNAME = 'HABITATNAME'
    MINCAPA = 'MINCAPA'
    CAPA = 'CAPA'
    NOVORONOI = 'NOVORONOI'

    def initAlgorithm(self, config):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.translate('py', 'XML project file'),
                fileFilter = 'xml(*.xml)',
                optional = False
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.HABITATSRC,
                self.translate('py', 'Name of the habitat source'),
                optional = False
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.HABITATNAME,
                self.translate('py', 'Name of the habitat that will be created'),
                optional = False
            )
        )


        self.addParameter(
            QgsProcessingParameterNumber(
                self.MINCAPA,
                self.translate('py', 'Minimum capacity'),
                type = QgsProcessingParameterNumber.Double,
                minValue = 0.0,
                defaultValue = 0.0,
                optional = True
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.CAPA,
                self.translate('py', 'Capacity field'),
                optional = True
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NOVORONOI,
                self.translate('py', 'No voronoi'),
                defaultValue = False,
                optional = False
            )
        )



    def processAlgorithm(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> dict:
        """
        Create a command line depending of parameters set by the user and send it to its parent 
        to execute it
        """ 
        project = self.parameterAsFile(parameters, self.INPUT, context)
        self.projectPath = os.path.normcase(r'' + project)
        habitat_source = self.parameterAsString(parameters, self.HABITATSRC, context)
        self.name = self.parameterAsString(parameters, self.HABITATNAME, context)
        min_capa = self.parameterAsDouble(parameters, self.MINCAPA, context)
        capa = self.parameterAsString(parameters, self.CAPA, context)
        novoronoi = self.parameterAsBool(parameters, self.NOVORONOI, context)

        commandToExec = ['--project', self.projectPath, '--vhabitat', 'name=' + self.name, 'from=' + habitat_source]

        if min_capa > 0.0:
            commandToExec.append(f'mincapa={min_capa}')
        if capa:
            commandToExec.append(f'capa={capa}')
        if novoronoi:
            commandToExec.append('novoronoi')


        super().processAlgorithm(commandToExec, context, feedback)
        return {}

    def postProcessAlgorithm(self, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> dict:
        """After the execution of the processing, this function
        add and show the new Graph in the map. """
        project: GraphabProject = self.plugin.getProjectFromPath(self.projectPath)

        if self.executionError or project is None:
            return {}

        project.add_habitat_project(self.name)
        return {}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'create_vector_habitat_from_habitat'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.translate('py', 'Create Vector Habitat (from habitat)')

    def createInstance(self):
        return CreateVectorHabitatFromHabitat(self.plugin)
