# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CalculateCorridor
                                 A QGIS plugin
 Process that able you to create a Graph for Graphab
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-07-02
        copyright            : (C) 2020 by ThéMA
        email                : robin.marlinl@gmail.com
        author               : Robin Marlin-Lefebvre
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'gilles.vuidel@univ-fcomte.fr'
__date__ = '2020-07-02'
__copyright__ = '(C) 2020 by ThéMA'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from PyQt5.QtCore import QFileInfo
from qgis.core import (QgsProcessingParameterString,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterFile, QgsProcessingParameterEnum, QgsProject, QgsRasterLayer, QgsVectorLayer)

import os

from ..utils import get_last_modified_file

from .GraphabAlgoProcessing import GraphabAlgoProcessing

class CalculateCorridor(GraphabAlgoProcessing):
    """
    Processing algorithm use to calculate corridors. It inherit of GraphabAlgoProcessing
    that execute the command send by this class and it inherit of QgsProcessingAlgorithm
    """
    
    def __init__(self, plugin):
        super().__init__(plugin)

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    NAMEGRAPH = 'NAMEGRAPH'
    MAXDIST = 'MAXDIST'
    MAXDISTUNIT = 'MAXDISTUNIT'
    METRIC = 'METRICS'
    FORMAT = 'FORMAT'
    FORMATS = ['raster', 'vector']
    # only for raster
    DIST = 'DIST'
    DISTUNIT = 'DISTUNIT'
    PROBA = 'PROBA'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Input for the XML file that represent the Graphab Project
        self.addParameter(
                QgsProcessingParameterFile(
                    self.INPUT,
                    self.translate('py', 'XML project file'),
                    fileFilter = 'xml(*.xml)',
                    optional=False
                    )
                )

        # Input for the graph name
        self.addParameter(
                QgsProcessingParameterString(
                    self.NAMEGRAPH,
                    self.translate('py', 'Graph name'),
                    optional=False
                    )
                )

        # Unit of the distance max
        self.addParameter(
                QgsProcessingParameterEnum(
                    self.MAXDISTUNIT,
                    self.translate('py', 'Maximum Distance unit (vector only)'),
                    self.plugin.UNITS,
                    allowMultiple=False,
                    defaultValue=0,
                    optional=False
                    )
                )

        self.addParameter(
                QgsProcessingParameterNumber(
                    self.MAXDIST,
                    self.translate('py', 'Maximum distance (vector only)'),
                    type=QgsProcessingParameterNumber.Double,
                    minValue=0.0,
                    defaultValue=0.0,
                    optional=False
                    )
                )

        self.addParameter(
                QgsProcessingParameterString(
                    self.METRIC,
                    self.translate('py', 'Metric'),
                    defaultValue='',
                    optional=True
                    )
                )

        self.addParameter(
                QgsProcessingParameterEnum(
                    self.FORMAT,
                    self.translate('py', 'raster or vector output'),
                    self.FORMATS,
                    allowMultiple=False,
                    defaultValue=0,
                    optional=False
                    )
                )
        # Unit of the distance (raster only)
        self.addParameter(
                QgsProcessingParameterEnum(
                    self.DISTUNIT,
                    self.translate('py', 'Distance unit (raster only)'),
                    self.plugin.UNITS,
                    allowMultiple=False,
                    defaultValue=0,
                    optional=False
                    )
                )

        # distance (raster only)
        self.addParameter(
            QgsProcessingParameterNumber(
                self.DIST,
                self.translate('py', 'distance (raster only)'),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                defaultValue=0.0,
                optional=False
                )
            )

        # Input for the proba (p parameter)
        self.addParameter(
            QgsProcessingParameterEnum(
                self.PROBA,
                self.translate('py', 'Probability (raster only)'),
                self.plugin.PROBA_DIST,
                allowMultiple=False,
                defaultValue=0
            )
        )


    def processAlgorithm(self, parameters, context, feedback):
        """
        Create a command line depending of parameters set by the user and send it to its parent 
        to execute it
        """

        # Get all parameter set
        project = self.parameterAsFile(parameters, self.INPUT, context)
        self.graph_name = self.parameterAsString(parameters, self.NAMEGRAPH, context)
        distanceMax = self.parameterAsDouble(parameters, self.MAXDIST, context)
        distMaxUnit = self.parameterAsInt(parameters, self.MAXDISTUNIT, context)
        metric = self.parameterAsString(parameters, self.METRIC, context)
        self.is_raster: bool = self.parameterAsString(parameters, self.FORMAT, context) == str(0)
        distance = self.parameterAsDouble(parameters, self.DIST, context)
        distanceUnit = self.parameterAsInt(parameters, self.DISTUNIT, context)
        proba = self.plugin.PROBA_DIST[self.parameterAsInt(parameters, self.PROBA, context)]

        self.projectPath = os.path.normcase(r''+project)
        commandToExec = ['--project', self.projectPath, '--usegraph', self.graph_name]

        if self.is_raster:
            # raster corridor
            if metric != '':
                commandToExec += ['--usemetric', metric]
            maxDist: float = 6*distance if str(proba) == '0.5' else 1.5*distance
            strDist = str(distance)
            strMaxDist = str(maxDist)
            if distanceUnit == 0:
                strDist = '{' + strDist + '}'
                strMaxDist = '{' + strMaxDist + '}'
            commandToExec += [
                '--corridor',
                'maxcost=' + strMaxDist,
                'format=raster',
                'd=' + strDist,
                'p=' + str(proba)
            ]
        else:
            # vector corridor
            strDistMax = str(distanceMax)
            if distMaxUnit == 0:
                strDistMax = '{' + strDistMax + '}'
            commandToExec += [
                '--corridor',
                'maxcost=' + strDistMax,
                'format=vector'
            ]


        super().processAlgorithm(commandToExec, context, feedback)
        return {}


    def postProcessAlgorithm(self, context, feedback):

        project = self.plugin.getProjectFromPath(self.projectPath)

        if self.executionError and not project:
            return {}

        projectGroup = project.projectGroup

        directory = os.path.dirname(self.projectPath)
        extension = '.tif' if self.is_raster else '.gpkg'
        regex_str = ''
        if extension == '.gpkg':
            regex_str = self.loglines[-2].split(' : ')[0].split(' ')[-1]
        else:
            maxCost = self.loglines[-1].split(' : ')[1]
            regex_str = f'{self.graph_name}-corridor-{maxCost}.*{extension}'

        corridor_file = get_last_modified_file(regex_str, directory)
        if corridor_file == "":
            return {}

        fileInfo = QFileInfo(corridor_file)

        if fileInfo.exists():
            # Load the file and add it to the project group
            if extension == '.tif':
                layer = QgsRasterLayer(fileInfo.filePath(), fileInfo.baseName())
            else:
                layer = QgsVectorLayer(fileInfo.filePath(), fileInfo.baseName())
            QgsProject.instance().addMapLayer(layer, False)
            projectGroup.insertLayer(len(projectGroup.children()), layer)
        return {}


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'calc_corridor'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.translate('py', 'Create corridors')

    def createInstance(self):
        return CalculateCorridor(self.plugin)
