# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CalculateMetrics
                                 A QGIS plugin
 Process that able you to calculate metrics for Graphab Graph
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-07-02
        copyright            : (C) 2020 by ThéMA
        email                : robin.marlinl@gmail.com
        author               : Robin Marlin-Lefebvre
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'robin.marlinl@gmail.com'
__date__ = '2020-07-02'
__copyright__ = '(C) 2020 by ThéMA'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.core import (QgsProcessingParameterString,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterEnum, QgsProject)

# import subprocess
import os

from ..GraphabProject import GraphabProject

from .GraphabAlgoProcessing import GraphabAlgoProcessing
from ..utils import getLayerMinMaxValue

class CalculateLocalMetric(GraphabAlgoProcessing):
    """
    Processing algorithm use to calculate metrics. It inherit of GraphabAlgoProcessing
    that execute the command send by this class and it inherit of QgsProcessingAlgorithm
    """
    
    def __init__(self, plugin):
        super().__init__(plugin)

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    GRAPHNAME = 'GRAPHNAME'
    METRICSNAME = 'METRICSNAME'
    DPARAMETER = 'DPARAMETER'
    PPARAMETER = 'PPARAMETER'
    DISTUNIT = 'DISTUNIT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        if self.plugin is None:
            return
        # Input for the XML file that represent the Graphab Project
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.translate('py', 'XML project file'),
                fileFilter = 'xml(*.xml)',
                optional=False
            )
        )
        
        # Input for the name of the selected Graph
        self.addParameter(
            QgsProcessingParameterString(
                self.GRAPHNAME,
                self.translate('py', 'Graph name'),
                optional=False
            )
        )
        
        # Input for the shortcut name of metrics
        self.addParameter(
            QgsProcessingParameterEnum(
                self.METRICSNAME,
                self.translate('py', 'Local metric'),
                self.plugin.LMETRICS,
                allowMultiple=False,
                defaultValue=0
            )
        )

        # Unit of the distance
        self.addParameter(
            QgsProcessingParameterEnum(
                self.DISTUNIT,
                self.translate('py', 'Distance unit'),
                self.plugin.UNITS,
                allowMultiple=False,
                defaultValue=0,
                optional=False
            )
        )

        # Input for the d parameter
        self.addParameter(
            QgsProcessingParameterNumber(
                self.DPARAMETER,
                self.translate('py', 'Distance'),
                type=QgsProcessingParameterNumber.Double,
                minValue=0.0,
                defaultValue=1000.0,
                optional=False
            )
        )

        # Input for the p parameter
        self.addParameter(
            QgsProcessingParameterEnum(self.PPARAMETER,
                self.translate('py', 'Probability'),
                self.plugin.PROBA_DIST,
                allowMultiple=False,
                defaultValue=0
            )
        )


    def processAlgorithm(self, parameters, context, feedback):
        """
        Create a command line depending of parameters set by the user and send it to its parent 
        to execute it
        """
        if self.plugin is None:
            return {}
        # Get all parameter set
        project = self.parameterAsFile(parameters, self.INPUT, context)
        self.graph: str = self.parameterAsString(parameters, self.GRAPHNAME, context)
        self.metrics = self.plugin.LMETRICS[self.parameterAsInt(parameters, self.METRICSNAME, context)]
        dValue = self.parameterAsDouble(parameters, self.DPARAMETER, context)
        distUnit = self.parameterAsInt(parameters, self.DISTUNIT, context)
        pValue = self.plugin.PROBA_DIST[self.parameterAsInt(parameters, self.PPARAMETER, context)]

        self.projectPath = os.path.normcase(r''+project)
        strDist = str(dValue)
        if distUnit == 0:
            strDist = '{' + strDist + '}'
        commandToExec = ['--project', self.projectPath, '--usegraph', self.graph, '--lmetric', self.metrics,
                        'd=' + strDist, 'p=' + pValue, 'beta=1']

        super().processAlgorithm(commandToExec, context, feedback)
        return {}

    def postProcessAlgorithm(self, context, feedback):
        """After the execution of the processing, depending of the command executed, this function
        can reload CSV file and show the new metric with its graph"""
        if self.executionError or self.plugin is None:
            return {}

        project: GraphabProject = self.plugin.getProjectFromPath(self.projectPath)
        if project is None:
            return {}

        # Here we just want to get the layer to change its symbology
        if not project.reload_graph(self.graph):
            raise Exception('Graph reloading failed')
        # Try to find the Graph group
        graphGroup = project.get_graphs_group().findGroup(self.graph)
        # get patches of the specific graphGroup
        layerGraph = graphGroup.children()[0] 
        fieldname =  (self.loglines[10].split(' ')[0])
        nodes = [lt.layer() for lt in graphGroup.children() if lt.layer().name() == 'Nodes']
        min, max = getLayerMinMaxValue(nodes, fieldname)
        for layer in nodes:
            self.plugin.GraphabStyle.prepare_style(fieldname, layer, self.plugin.stylesTabUnabled[0], min_value=min, max_value=max)


        # Set visible on the QGIS layer window the edited Graph
        layerGraph.setItemVisibilityChecked(True)
        graphGroup.setItemVisibilityChecked(True)

        # Expand the layer group
        graphGroup.setExpanded(True)

        return {}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'local_metric'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.translate('py', 'Calculate Local Metric')

    def createInstance(self):
        return CalculateLocalMetric(self.plugin)
