# -*- coding: utf-8 -*-
"""
GraphabProject
All functions that set and create a Graphab Project
"""
from __future__ import annotations

import os
from os.path import isfile # This is is needed in the pyqgis console also
import subprocess
import warnings
from typing import TYPE_CHECKING, cast

from .GraphabStyle import GraphabStyleLoadingFailed
if TYPE_CHECKING:
    from .GraphabPlugin import GraphabPlugin

from PyQt5.QtCore import QFileInfo, QCoreApplication
from qgis import processing
from qgis.core import (
    Qgis,
    QgsProject,
    QgsVectorLayer,
    QgsMapLayer,
    QgsRasterLayer,
    QgsSingleBandGrayRenderer,
    QgsRasterBandStats,
    QgsContrastEnhancement,
    QgsVectorLayerJoinInfo,
    QgsFillSymbol,
    QgsLayerTreeGroup,
    QgsLayerTreeLayer,
    QgsMessageLog
)

from .utils import getLayerMinMaxValue
# Module to convert graphab xml file to python object
from .xml_to_python.xmlToPython import (
    AbstractHabitat,
    AbstractGraph,
    AbstractLinkset,
    Graph,
    GraphabVersionTooLowException,
    MultiGraph,
    Project,
    convertXmlToPy
)

__author__ = 'Gaspard QUENTIN, Robin MARLIN-LEFEBVRE'
__email__ = 'gaspard.quentin1905@gmail.com'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

# type definition for quite complex data structure used in this file
HabitatNameLayers = dict[str, list[QgsMapLayer]]
GraphsData = tuple[AbstractGraph, HabitatNameLayers]

class GraphabProject:

    """
    This class describes the behavior of a Graphab project loaded in QGIS.
    """

    def __init__(self, plugin: GraphabPlugin, projectFile: str, projectGroup: QgsLayerTreeGroup|None = None):
        """Initialize a Graphab project into QGIS.

        :param plugin: the running instance of the plugin.
        :type plugin: GraphabPlugin
        
        :param projectFile:  the path of the xml file which represents the Graphab project.
        :type projectFile: str

        :param projectGroup: the group of layers where this project will be created
            or None if the layer group has to be created.
        :type projectGroup: QgsLayerTreeGroup or None
        
        Description:
        It make some verifications, get the XML file
        of a Graphab project and parse it into a Python object and load all required files like :
        patches, linkset... and creates the layers.
        """
        # Link the object to the main object (Graphab) for shared variables
        self.plugin = plugin
        self.projectFile = os.path.normcase(projectFile)
        self.projectDir = os.path.dirname(self.projectFile)

        # creation of a python object named 'project' using my lib 'xmlToPython'
        try:
            self.project: Project = convertXmlToPy(projectFile)
        except GraphabVersionTooLowException:
            self.plugin.iface.messageBar().pushCritical(
                QCoreApplication.translate('py', "ERROR"),
                QCoreApplication.translate('py', 'The project loaded does not come from a version >= 3 of Graphab.' 
                '\nPlease use the appropriate plugin for earlier versions.')
            )
            return


        self.projectGroup: QgsLayerTreeGroup = projectGroup
        self.source: QgsRasterLayer = None
        graphsGroup = None
        habitatsGroup = None

        if projectGroup is None:

            # creation of a layers group with the name of the Graphab project
            root = QgsProject.instance().layerTreeRoot()
            self.projectGroup = root.addGroup(self.project.name)
            self.projectGroup.setCustomProperty("Graphab3Project", projectFile)

            path_to_landtif = os.path.join(self.projectDir, "land.tif")
            self.set_source(path_to_landtif)

            # get all graphab habitats and create a layer group for each of them
            habitatsGroup = self.projectGroup.insertGroup(0, 'Habitats')
            for habitat in self.project.habitats:
                self.set_habitat(habitat, habitatsGroup)

            # get all graphab linksets and add them in qgis layers group 'Linksets'
            linksetsGroup = self.projectGroup.insertGroup(0, 'Linksets')
            for linkset in self.project.linksets:
                self.set_linkset(linkset.name, linkset, linksetsGroup)

            # get all graphab graph and add them in qgis layers group 'Graphs' with for each a new layers group
            graphsGroup = self.projectGroup.insertGroup(0, 'Graphs')

            #when you active one graph group; others graph group are disabled
            graphsGroup.setIsMutuallyExclusive(True)

        #load each graph (will always be done to avoid AttributeError)
        if graphsGroup is None:
            graphsGroup = self.projectGroup.findGroup('Graphs')
            graphsGroup.removeAllChildren()
        if habitatsGroup is None:
            habitatsGroup = self.get_habitats_group()

    
        self.projectGraphs: dict[str, GraphsData]  = {}
        self.habitatsLayers: list[QgsLayerTreeLayer] = habitatsGroup.findLayers()
        for graph in self.project.graphs:
            graphGroup = graphsGroup.addGroup(graph.name)
            QgsProject.instance().layerTreeRoot().findGroup(graphGroup.name()).setItemVisibilityChecked(False)
            self.load_graph(graph, graphGroup)



    #--------------------------------------------------------------------------
    def load_graph(self, graph: AbstractGraph, graphGroup: QgsLayerTreeGroup) -> None:
        """Procedure that sets the graph in the project, links to it the metrics and stores it in the project graphs dictionary.

        :param graph: the graph that will be loaded in QGIS
        :type graph: AbstractGraph (Graph or MultiGraph)
        
        :param graphGroup: the QGIS layer group in which the graph will be added
        :type graphGroup: QgsLayerTreeGroup
        """

        graphLayers_ = self.set_graph(graphGroup, graph)

        # build the dictionary entry 
        graphLayers = dict([(e.name, graphLayers_[e]) for e in graphLayers_ if e.name])
        
        # store the entry in the project graphs dictionnary
        self.projectGraphs[graph.name] = (graph, graphLayers)
        self.link_graph_metrics(graphLayers, graph)



    #--------------------------------------------------------------------------
    def reload_graph(self, graphName: str) -> bool:
        """Method that relinks the graph to it's metrics and reloads it's layer.

        :param graphName: the name of the graph that we want to reload.
        :type graphName: str

        :return: True if the graph exists otherwise False
        :rtype: bool
        """
        if graphName not in self.projectGraphs:
            return False 
        for layerTree in self.habitatsLayers:
            layerTree.layer().reload()
        self.link_graph_metrics(self.projectGraphs[graphName][1], self.projectGraphs[graphName][0])
        return True

    #--------------------------------------------------------------------------
    def get_habitatID_of_layer(self, layer: QgsMapLayer) -> int:
        """Method that searches for the ID of the habitat of the layer given as argument.

        :param layer: the layer we want to get it's habitat ID
        :type layer: QgsMapLayer

        :return: The habitat ID if it has been found, otherwise -1.
        :rtype: int
        """
        for graph_name in self.projectGraphs:
            for habitat in self.projectGraphs[graph_name][1]:
                for layer_tmp in self.projectGraphs[graph_name][1][habitat]:
                    if layer_tmp == layer:
                        habitatID = next((e.id for e in self.project.habitats if e.name == habitat), -1)
                        return habitatID if habitatID else -1
        return -1
    
    #--------------------------------------------------------------------------
    def set_source(self, raster: str):
        """Method that loads the source.tif/land.tif file of a loaded Project and change its symbology.

        :param raster : the file's path of the source.tif or land.tif that needs to be loaded.
        :type raster: str
        """
        
        # Get the path and the name of the file to be sure the file can be load
        fileInfo = QFileInfo(raster)
        path = fileInfo.filePath()
        baseName = fileInfo.baseName()

        # Load the file and add it to a specific group to have the same structure of a Graphab project
        layer = QgsRasterLayer(path, baseName)
        self.source: QgsRasterLayer = layer
        QgsProject.instance().addMapLayer(layer, False)
        self.projectGroup.insertLayer(0, layer)

        # change color of layer that was add
        myGrayRenderer = QgsSingleBandGrayRenderer(layer.dataProvider(), 1)
        layer.setRenderer(myGrayRenderer)

        renderer = layer.renderer()
        provider = layer.dataProvider()

        layer_extent = layer.extent()
        uses_band = renderer.usesBands()

        myType = renderer.dataType(uses_band[0])

        stats = provider.bandStatistics(uses_band[0],
                                        QgsRasterBandStats.All,
                                        layer_extent,
                                        0)

        myEnhancement = QgsContrastEnhancement(myType)

        contrast_enhancement = QgsContrastEnhancement.StretchToMinimumMaximum

        myEnhancement.setContrastEnhancementAlgorithm(contrast_enhancement,True)
        myEnhancement.setMinimumValue(stats.minimumValue)
        myEnhancement.setMaximumValue(stats.maximumValue)

        layer.renderer().setContrastEnhancement(myEnhancement)
        # Refresh the symbology
        layer.triggerRepaint()

        # Don't set visible the layer such as the loading of a Graphab project
        QgsProject.instance().layerTreeRoot().findLayer(layer.id()).setItemVisibilityChecked(False)

    #--------------------------------------------------------------------------
    def get_source(self) -> QgsRasterLayer :
        """Methods that returns the raster layer of the land.

        :return: the layer of the land
        :rtype: QgsRasterLayer
        """
        if self.source:
            return self.source
        for child in self.projectGroup.children():
            if child.name() == 'land':
                return child.layer()
        raise Exception('The project contains no source')

    #--------------------------------------------------------------------------

    def set_voronoi(self, voronoi: str, group: QgsLayerTreeGroup):
        """Method that loads the Graph voronoi file into a VectorLayer, change its symbology and adds it 
        into the desired group.

        :param voronoi: the file's path of the voronoi file
        :type voronoi: str
        
        :param group: the layer's group in which we want to add the future voronoi layer.
        :type group: QgsLayerTreeGroup
        """
        
        # Get the path and the name of the file to be sure the file can be load
        path = QFileInfo(voronoi).filePath()
        # Load the file and add it to a specific group to have the same structure of a Graphab project
        layer = QgsVectorLayer(path, "Components")
        QgsProject.instance().addMapLayer(layer, False)
        group.addLayer(layer)
            
        # Create a new symbology
        symbol = QgsFillSymbol.createSimple({'line_style': 'solid', 'style': 'no'})

        # apply symbol to layer renderer
        layer.renderer().setSymbol(symbol)

        # repaint the layer
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())

        # Don't set visible the layer such as the loading of a Graphab project
        QgsProject.instance().layerTreeRoot().findLayer(layer.id()).setItemVisibilityChecked(False)

    
    #--------------------------------------------------------------------------
    def _load_graph_topo(self, habitatName: str, group: QgsLayerTreeGroup) -> QgsMapLayer:
        """Submethod of set_graph that loads the topo patches of an habitat if it wasn't already done and adds the 
        layer to the desired group.

        :param habitatName: the name of the/one of the habitat of one of the Graph's Linkset.
        :type habitatName: str

        :param group: the group where the layer will be added
        :type group: QgsLayerTreeGroup

        :return: the created layer
        :rtype: QgsMapLayer
        """
        path_to_patches_gpkg = os.path.join(self.projectDir, habitatName, 'patches-topo.gpkg')
        if not os.path.isfile(path_to_patches_gpkg):
            self.create_topo_patches(habitatName)
        
        pathPatches = QFileInfo(path_to_patches_gpkg).filePath()
        
        # Load the file and add it to a specific group to have the same structure of a Graphab project
        layer = QgsVectorLayer(pathPatches, "Nodes")
        QgsProject.instance().addMapLayer(layer, False)
        group.addLayer(layer)

        return layer

    def _set_symbology(self, layers: HabitatLayers, graph_name: str, fieldname: str = 'capacity', isMultiLayer: bool = False, isColorLikeHabitat: bool = False) -> None:
        """Submethod that does all the colorization/sizing for the graph

        :param layers: the layers of the graph's habitat.
        :type layers: HabitatLayers == dict[AbstractHabitat, list[QgsMapLayer]]
        
        :param fieldname: The name of the field/metric that will be used to draw the graph symbology
        :type fieldname: str

        :param isMultiLayer: , defaults to False
        :type isMultiLayer: bool, optional

        :param isColorLikeHabitat: if set to True, all the graph's node will be colored 
            the same as the habitat, otherwise the color will depend on the field values.
            , defaults to False
        :type isColorLikeHabitat: bool, optional
        """
        min = -1
        max = -1
        if isMultiLayer:
            # calculate the min and max value
            min, max = getLayerMinMaxValue(layers.get_all_layers(), fieldname)
        for habitat in layers:
            for layer in layers[habitat]:
                habitatID = habitat.id if isColorLikeHabitat else -1
                habitatID = cast(int, habitatID)

                try:
                    self.plugin.GraphabStyle.prepare_style(fieldname, layer, self.plugin.stylesTabUnabled[1], min_value=min, max_value=max, habitat_id=habitatID)
                except GraphabStyleLoadingFailed:
                    QgsMessageLog.logMessage(f"Failed to display metric {fieldname} for layer {layer.name()} of graph {graph_name}")

                QgsProject.instance().layerTreeRoot().findLayer(layer.id()).setItemVisibilityChecked(True)


    def _set_multigraph(self, group: QgsLayerTreeGroup, multigraph: MultiGraph) -> HabitatLayers:
        """Submethod that loads every graph of a multigraph and change the symbology of them.

        Parameters
        :param group: the group where we want to add the multigraph (often the project graph group)
        :type group: QgsLayerTreeGroup

        :param multigraph: Multigraph that we want to load and add to the qgis project.
        :type multigraph: MultiGraph

        :return: The layers of all the graphs of the multigraph that has been created. Those are stored in a way to
            easily access to the habitat of each layer, which helps for searching operations on the layers and
            all the graphs data in general.
        :rtype: HabitatLayers == dict[AbstractHabitat, list[QgsMapLayer]]
        """
        d = self.HabitatLayers()
        for e in multigraph.graphs:
            e = cast(AbstractGraph, e)
            d.update(self.set_graph(group, e))
        self._set_symbology(d, multigraph.name, isMultiLayer=True, isColorLikeHabitat=True)
        return d

    def _load_habitat_layers(self, group: QgsLayerTreeGroup, graph: Graph) -> HabitatLayers:
        if not graph.linkset:
            return self.HabitatLayers()

        layers = self.HabitatLayers()
        habitats = [graph.linkset.habitat] if not graph.linkset.inter else graph.linkset.habitats
        for habitat in habitats:
            habitat = cast(AbstractHabitat, habitat)
            layer = self._load_graph_topo(str(habitat.name), group)
            if not layer.isValid():
                return self.HabitatLayers()
            layers.insert_layer(habitat, layer)
        return layers

    def _load_graph_linksets(self, group: QgsLayerTreeGroup, graph: Graph) -> None:
        if not graph.linkset:
            return

        topoLinkset = os.path.join(
            self.projectDir,
            graph.linkset.habitatName,
            graph.name + '-topo_links.gpkg'
        )
        if not os.path.isfile(topoLinkset):
            was_created = self.create_topo_linkset(graph)
            if not was_created:
                warnings.warn(f"failed to create topo linkset for graph {graph.name}")

        self.set_linkset("Edges", graph.linkset, group, graph.name, True)

        # Load Linkset with filter or not depending of the type of the Graph loaded
        if graph.threshold and graph.threshold > 0:
            self.set_linkset("Paths", graph.linkset, group, "", False, graph.threshold)
        else:
            self.set_linkset("Paths", graph.linkset, group, "",  False)

        # Load voronoi file
        path_to_voronoi = os.path.join(
            self.projectDir,
            graph.linkset.habitatName,
            graph.name+"-voronoi.gpkg"
        )
        if isfile(path_to_voronoi):
            self.set_voronoi(path_to_voronoi, group)

    def set_graph(self, group: QgsLayerTreeGroup, graph: AbstractGraph) -> HabitatLayers:
        """Method that loads every file that compose a Graph and change symboly of them.
        
        Parameters
        :param group: the group where we want to add the graph (often the project graph group)
        :type group: QgsLayerTreeGroup
        
        :param graph: Graph or MultiGraph that we want to load and add to the qgis project.
        :type graph: AbstractGraph

        :return: The layers of the graph that have been created. Those are stored in a way to easily access to
            the habitat of each layer, which helps for searching operations on the layers and all the graphs data in 
            general.
        :rtype: HabitatLayers
        """
        if graph.isMulti and isinstance(graph, MultiGraph):
            return self._set_multigraph(group, graph)
        if not isinstance(graph, Graph) or not graph.linkset:
            return self.HabitatLayers()

        layers = self._load_habitat_layers(group, graph)

        self._load_graph_linksets(group, graph)

        # Set default symbology of the current graph
        self._set_symbology(layers, graph.name, isMultiLayer=len(layers) > 1, isColorLikeHabitat=True)

        # Collapse the layer group
        group.setExpanded(False)
        return layers

    #--------------------------------------------------------------------------

    def set_habitat(self, habitat: AbstractHabitat, habitatsGroup: QgsLayerTreeGroup) -> None:
        """Method that loads the habitat and places the created habitat layer into the habitatsGroup

        :param habitat: the habitat that will be loaded into the
            QGIS project
        :type habitat: AbstractHabitat (MonoHabitat or MonoVectorHabitat)

        :param habitatsGroup: the group where all the layers created from habitats are placed.
        :type habitatsGroup: QgsLayerTreeGroup
        """

        # Make sure the habitat directory exists
        dir_path = os.path.join(self.projectDir, str(habitat.name))
        if not os.path.isdir(dir_path):
            warnings.warn(
                "GraphabProject.py: set_habitat failed: no {} directory found."
                .format(str(habitat.name))
            )
            return None

        path_to_patches_gpkg = os.path.join(dir_path, "patches.gpkg")
        layer = self.set_patches(
            path_to_patches_gpkg,
            str(habitat.name),
            habitatsGroup)

        #change the symbology color
        habitat.id = cast(int, habitat.id)
        self.plugin.GraphabStyle.colorizeHabitatLayer(layer, habitat.id)


    #--------------------------------------------------------------------------
    def set_patches(self, patchesPath: str, habitatName: str, habitatsGroup: QgsLayerTreeGroup) -> QgsVectorLayer | None:
        """Method that loads patches.gpkg file into an habitat layer and change its symbology.

        :param patchesPath: file path of the patches.gpkg
        :type patchesPath: str

        :param habitatName: name of the future habitat layer that will be created.
        :type habitatName: str

        :param habitatsGroup: the group where all the layers created from habitats are placed. 
        :type habitatsGroup: QgsLayerTreeGroup

        :return: the created habitat layer from the patches.gpkg or None if the operation failed.
        :rtype: QgsVectorLayer | None
        """

        # Get the path and the name of the file to be sure the file can be load
        fileInfo = QFileInfo(patchesPath)
        path = fileInfo.filePath()

        # Load the file and add it to a specific group to have the same structure of a Graphab project
        layer = QgsVectorLayer(path, habitatName)
        QgsProject.instance().addMapLayer(layer, False)
        habitatsGroup.insertLayer(0, layer).setItemVisibilityChecked(False)

        # If the file is loaded
        if not layer.isValid():
            return None
        return layer


    #--------------------------------------------------------------------------
    
    def set_linkset(self, name: str, linkset: AbstractLinkset, group: QgsLayerTreeGroup, graphName: str = "", visible: bool = False, threshold: int = 0)->None:
        """Method that loads a linkset into a layer and change it's symbology depending on it's type.

        :param name: the future layer name
        :type name: str

        :param linkset: the linkset that needs to be loaded.
        :type linkset:  AbstractLinkset (Euclide|Cost|Circuit Linkset)

        :param group: the group where the created linkset's layer will be placed
        :type group: QgsLayerTreeGroup

        :param graphName: the name of the graph that has this linkset, defaults to ""
        :type graphName: str, optional

        :param visible: if set to True will show the created layer, set to False by default to
            hide it when all the project is loading, defaults to False
        :type visible: bool, optional

        :param threshold: the graph's threshold, defaults to 0
        :type threshold: int, optional
        """
        if graphName:
            linksetFile = graphName+"-topo_links.gpkg" 
        else:
            linksetFile = linkset.name+"-links.gpkg"


        # Get the path and the name of the file to be sure the file can be load
        fileInfo = QFileInfo(os.path.join(self.projectDir, linkset.habitatName, linksetFile))
        path = fileInfo.filePath()
        if not fileInfo.exists():
            warnings.warn(f'Linkset file of path : {path} doesn\'t exit')
            return 

        # Load the file and add it to a specific group to have the same structure of a Graphab project
        layer = QgsVectorLayer(path, name, "ogr")
        QgsProject.instance().addMapLayer(layer, False)
        group.addLayer(layer)
                
        # Set up of the color which depends of the type of the linkset
        if linkset.topology == "COMPLETE":
            self.plugin.GraphabStyle.colorizeLine(layer, '#B8C45D')
        elif linkset.topology == "PLANAR":
            self.plugin.GraphabStyle.colorizeLine(layer, '#25372B')
        else:
            self.plugin.GraphabStyle.colorizeLine(layer, '#BCC3AC')

        if threshold > 0:
            #set filter ("Dist"<Graph.threshold)
            equation = '"Dist" <= %d' % (threshold,)
            layer.setSubsetString(equation)

        # Don't set visible the layer such as the loading of a Graphab project
        QgsProject.instance().layerTreeRoot().findLayer(layer.id()).setItemVisibilityChecked(visible)


    # --------------------------------------------------------------------------

    def add_graph_project(self, graphName: str) -> None:
        """This method loads a Graph that had been created while the project was already loaded

        :param graphName: the name of the graph that needs to be loaded
        :type graphName: str
        """
        # replace the previous stocked project by the new one
        self.upgrade_loaded_project()

        # get 'Graphs' group
        graphsGroup = self.projectGroup.findGroup('Graphs')

        # try to find the new graph with its name and load it
        for graph in self.project.graphs:
            if graph.name == graphName:
                graphGroup = graphsGroup.addGroup(graph.name)
                self.load_graph(graph, graphGroup)

    #--------------------------------------------------------------------------
    
    def add_linkset_project(self, linksetName: str) -> None:
        """This method loads a Linkset that had been created while the project was already loaded

        :param linksetName: the name of the linkset that needs to be loaded
        :type linksetName: str
        """
        # replace the previous stocked project by the new one
        self.upgrade_loaded_project()

        # get 'Linksets' group
        linksetsGroup = self.get_linksets_group()

        # try to find the new linkset with its name and load it
        for linkset in self.project.linksets:
            if linkset.name == linksetName:
                self.set_linkset(linksetName, linkset, linksetsGroup, visible=True)

    #--------------------------------------------------------------------------
    
    def add_habitat_project(self, habitatName: str) -> None:
        """This method loads an habitat that had been created while the project was already loaded

        :param habitatName: the name of the linkset that needs to be loaded
        :type habitatName: str
        """
        self.upgrade_loaded_project()
        habitatsGroup = self.get_habitats_group()
        for habitat in self.project.habitats:
            if habitat.name == habitatName:
                self.set_habitat(habitat, habitatsGroup)


    #--------------------------------------------------------------------------
    
    def link_graph_metrics(self, graphLayers: HabitatNameLayers, graph: AbstractGraph) -> None:
        """Method that does link each graph layer to the metric's data that is stored in 
        the habitat layer.

        :param graphLayers: all the layers of the graph stored with a dict
            mapping each graph's habitat name with all the linkset's layers linked to the habitat and the graph.
        :type graphLayers: HabitatNameLayers == dict[str, list[QgsMapLayer]]
        
        :param graph: the graph that will be linked with the habitat's metrics data
        :type graph: AbstractGraph (Graph or MultiGraph)
        """
        for layerTree in self.habitatsLayers:
            if graph.isMulti:
                for g in graph.graphs:
                    self._link_graph_layers_metrics(g, layerTree, graphLayers)
            else:
                self._link_graph_layers_metrics(graph, layerTree, graphLayers)
    
    def _link_graph_layers_metrics(self, graph: Graph, layerTree: QgsLayerTreeLayer, graphLayers: HabitatNameLayers) -> None:
        """Submethod of .linkGraphMetrics that is used to avoid redundancy.

        :param graph: the graph that will be linked with the habitat's metrics data
        :type graph: Graph

        :param layerTree: the habitat tree layer.
        :type layerTree: QgsLayerTreeLayer

        :param graphLayers: all the layers of the graph stored with a dict
            mapping each graph's habitat name with all the linkset's layers linked to the habitat and the graph.
        :type graphLayers: HabitatNameLayers == dict[str, list[QgsMapLayer]]
        """

        habitats = [graph.linkset.habitat] if not graph.linkset.inter else graph.linkset.habitats

        for hab in habitats:
            if layerTree.name() != hab.name:
                continue

            for layer in graphLayers[hab.name]:
                self.join_graph_habitat(layer, layerTree.layer(), [metric.name for metric in graph.metrics])

    # --------------------------------------------------------------------------
    def join_graph_habitat(self, graphLayer: QgsMapLayer, habitatLayer: QgsMapLayer, graphMetrics: list[str]) -> None:
        """Method that joins the data of a graph's layer to it's habitat's layer.
        Indeed, all the metrics data is stored in the habitat's layer.

        :param graphLayer: the layer of the graph that needs it's data to be joined with it's habitat layer.
        :type graphLayer: QgsMapLayer

        :param habitatLayer: the layer habitat of the graph that will give some of it's data to the graphLayer.
        :type habitatLayer: QgsMapLayer
        """
        def _get_metric_full_name(metric):
                return metric.name + '_' + metric.graph.name
        joinObject = QgsVectorLayerJoinInfo()
        joinObject.setJoinFieldName('Id')
        joinObject.setTargetFieldName('Id')
        joinObject.setUsingMemoryCache(True)
        notGraphMetrics = [_get_metric_full_name(metric) for metric in self.project.localMetrics if metric.name not in graphMetrics ]
        joinObject.setJoinFieldNamesBlockList(['area', 'perim'] + notGraphMetrics)
        joinObject.setPrefix('') # or self.plugin.prefix
        joinObject.setJoinLayer(habitatLayer)
        graphLayer.addJoin(joinObject)

    
    # --------------------------------------------------------------------------
    
    def upgrade_loaded_project(self) -> None:
        """Method that replaces an old project by a new project after a project modif"""
        # reload project
        self.project = convertXmlToPy(self.projectFile)

    # --------------------------------------------------------------------------
    def create_topo_patches(self, habitat_of_graph_linkset: str) -> None:
        """Method that calls the native:centroids processing algorithms to create topo patches.
        It creates the file habitat_of_graph_linkset/patches-topo.gpkg.

        :param habitat_of_graph_linkset: the name of graph's linkset habitat
        :type habitat_of_graph_linkset: str
        """
        processing.run("native:centroids", {'INPUT': os.path.join(self.projectDir, habitat_of_graph_linkset, 'patches.gpkg'),
                                         'ALL_PARTS': False,
                                         'OUTPUT': os.path.join(self.projectDir, habitat_of_graph_linkset,'patches-topo.gpkg')})


    # --------------------------------------------------------------------------
    def create_topo_linkset(self, graph: AbstractGraph) -> bool:
        """Method that creates a topo linkset for the graph given as argument.

        :param graph: the graph that we will use to create the topo linkset
        :type graph: AbstractGraph (Graph or MultiGraph)

        :return: True if the operation succeeded, False otherwise
        :rtype: bool
        """
        try:
            cmd = self.plugin.graphabProvider.getGraphabCommand() + ['--project', self.projectFile, '--usegraph', graph.name, '--topo']
        except:
            QgsMessageLog.logMessage("Unable to get Graphab", 'Processing', Qgis.Critical)
            return False

        return subprocess.run(cmd).returncode == 0

    # --------------------------------------------------------------------------
    def remove_habitat(self, habitatName: str) -> bool:
        """Method that removes a linkset from the Graphab Project and QGIS.

        :param habitatName: the name of the habitat to remove
        :type habitatName: str

        :return: True if the operation succeeded, False otherwise
        :rtype: bool
        """
        cmd = self.plugin.graphabProvider.getGraphabCommand() + ['--project', self.projectFile, '--removehabitat', habitatName]
        if subprocess.run(cmd).returncode == 0:
            self.upgrade_loaded_project()
            habitatGroup = self.get_habitats_group()
            for layer in habitatGroup.children():
                if layer.name() == habitatName:
                    habitatGroup.removeChildNode(layer)
                    break
            return True
        return False

    # --------------------------------------------------------------------------
    def remove_linkset(self, linksetName: str) -> bool:
        """Method that removes a linkset from the Graphab project and QGIS.

        :param linksetName: the name of the linkset to remove
        :type linksetName: str

        :return: True if the operation succeeded, False otherwise
        :rtype: bool
        """
        cmd = self.plugin.graphabProvider.getGraphabCommand() + ['--project', self.projectFile, '--removelinkset', linksetName]
        ret = subprocess.run(cmd).returncode
        if ret == 0:
            self.upgrade_loaded_project()
            linksetGroup = self.get_linksets_group()
            for layer in linksetGroup.children():
                if layer.name() == linksetName:
                    linksetGroup.removeChildNode(layer)
                    break
            return True
        else:
            return False

    # --------------------------------------------------------------------------
    def remove_graph(self, graphName: str) -> bool: 
        """Method that removes a graph from the Graphab project and QGIS.

        :param graphName: the name of the graph to remove
        :type graphName: str

        :return: True if the operation succeeded, False otherwise
        :rtype: bool
        """
        cmd = self.plugin.graphabProvider.getGraphabCommand() + ['--project', self.projectFile, '--removegraph', graphName]
        ret = subprocess.run(cmd).returncode
        if ret == 0:
            self.upgrade_loaded_project()
            graphGroup = self.projectGroup.findGroup('Graphs')
            for layer in graphGroup.children():
                if layer.name() == graphName:
                    graphGroup.removeChildNode(layer)
                    break
            return True
        else:
            return False

    # --------------------------------------------------------------------------
    def get_habitats_layers(self) -> list[QgsMapLayer]:
        """Method that returns the layers of the habitats

        :return: the habitats layers
        :rtype: list[QgsMapLayer]
        """
        return [ltl.layer() for ltl in self.get_habitats_group().children()]

    # --------------------------------------------------------------------------
    def get_habitat_layer(self, habitatName: str) -> QgsMapLayer:
        """Method that returns the layer of an habitat.

        :param habitatName: the habitat's name
        :type habitatName: str

        :return: the layer representing the habitat or None if nothing was found
        :rtype: QgsMapLayer
        """
        habitatsLayers = self.get_habitats_layers()
        for layer in habitatsLayers:
            if layer.name() == habitatName:
                return layer
        return None
    #--------------------------------------------------------------------------
    def get_group(self, groupName: str) -> QgsLayerTreeGroup:
        """method returns the layer group of name groupName.

        :param groupName: the name of the group we want to get
        :type groupName: str

        :return: the group or None if not found
        :rtype: QgsLayerTreeGroup
        """
        return self.projectGroup.findGroup(groupName)
    #--------------------------------------------------------------------------
    def get_habitats_group(self) -> QgsLayerTreeGroup:
        """Searches for and returns the habitats layer group of the project.
        Raise an exception if the group is not found

        :raises Exception: raises an exception if the group is not found
        :return: the group if found
        :rtype: QgsLayerTreeGroup
        """
        if (hg := self.get_group('Habitats')) is not None:
            return hg
        raise Exception('Habitats group not found in project layer group')
     #--------------------------------------------------------------------------
    def get_linksets_group(self) -> QgsLayerTreeGroup:
        """Searches for and returns the linksets layer group of the project.
        Raise an exception if the group is not found

        :raises Exception: raises an exception if the group is not found
        :return: the group if found
        :rtype: QgsLayerTreeGroup
        """
        if (lg := self.get_group('Linksets')) is not None:
            return lg
        raise Exception('Linksets group not found in project layer group')
    #--------------------------------------------------------------------------
    def get_graphs_group(self) -> QgsLayerTreeGroup:
        """Searches for and returns the graphs layer group of the project.
        Raise an exception if the group is not found

        :raises Exception: raises an exception if the group is not found
        :return: the group if found
        :rtype: QgsLayerTreeGroup
        """
        if (gg := self.get_group('Graphs')) is not None:
            return gg
        raise Exception('Graphs group not found in project layer group')


    class HabitatLayers(dict[AbstractHabitat, list[QgsMapLayer]]):

        """
        Data structure only used in GraphabProject class.

        This structure maps each habitat of the project to all it's layers.

        It is a dictionnary (and can be used like it), but that can accept only abstract habitats 
        as keys and QgsMapLayers as values.
        """

        def __init__(self) -> None:
            super().__init__()


        def get_all_layers(self) -> list[QgsMapLayer]:
            return [ layer for layers in self.values() for layer in layers ]

        def update(self, other_layers: GraphabProject.HabitatLayers):
            for hab in other_layers:
                for layer in other_layers[hab]:
                    self.insert_layer(hab, layer)

        def insert_layer(self, habitat: AbstractHabitat, layer: QgsMapLayer):
            if habitat in self:
                self[habitat].append(layer)
            else:
                self[habitat] = layer # calls __setitem__ and so layer will become [layer]

        def __setitem__(self, key: _KT, value: _VT, /) -> None:
            """
            Overload the  dict[key] = newValue  operation so that key can only be an Habitat and value a layer.
            """
            if not isinstance(key, AbstractHabitat) or not isinstance(value, QgsMapLayer):
                raise TypeError()
            return super().__setitem__(key, [value])

