# -*- coding: utf-8 -*-

"""

utils.py

This file contains every helper functions.

"""


__author__ = 'gaspard.quentin1905@gmail.com'
__date__ = '2024-02-11'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

"""
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import re
import os

from qgis.core import QgsVectorLayer, QgsProcessingFeedback

def getLayerMinMaxValue(layers: list[QgsVectorLayer], fieldname: str) -> tuple[int, int]:
    """ 
        Helper function that returns the couple min max value found in the layers with the field name given.
    """
    min = -1
    max = -1
    for layer in layers:
        idx = layer.fields().indexFromName(fieldname)
        if idx == -1:
            raise Exception(f"field {fieldname} not found in the layer of fields : \n {[field.name() for field in layer.fields()]}")
        tmp_min = layer.minimumValue(idx)
        tmp_max = layer.maximumValue(idx)
        if min == -1 or tmp_min < min:
            min = tmp_min
        if max == -1 or tmp_max > max:
            max = tmp_max
    return (min, max)


def get_last_modified_file(regex_str: str, directory: str) -> str:
    """
    get the last modified file with the name that matches regex_str in the directory directory.

    :param regex_str: The regex for the file name.
    :type regex_str: str

    :param directory: The directory path to search for the file
    :type directory: str

    :return: The filepath of the last modified file that matches the regex
    in the directory or an empty string if no file matching the regex_str was found.
    :rtype: str
    """
    matches: list[str] = []
    regex = re.compile(regex_str)
    for _, _, files in os.walk(directory):
        for file in files:
            if regex.match(file):
                matches.append(directory + '/' + file)

    matches.sort(key=os.path.getmtime, reverse=True)

    return matches[0] if len(matches) > 0 else ""


class NoFeedback(QgsProcessingFeedback):
    """
    This class can be used in case we don't want to have
    any feedback printed on the console.
    """
    def pushInfo(self, info: str):
        pass

    def reportError(self, error: str, fatalError: bool = False):
        pass

