# -*- coding: utf-8 -*-

__author__ = 'gaspard.quentin1905@gmail.com'
__date__ = '2024-04-26'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

# This will get replaced with a git SHA1 when you do a git archive
__revision__ = '$Format:%H%'


import os

from qgis.core import (QgsProcessingParameterString,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingContext,
                       QgsProcessingFeedback,
                       QgsProcessingParameterFile, QgsProcessingParameterEnum)

from ..GraphabProject import GraphabProject


from .GraphabAlgoProcessing import GraphabAlgoProcessing


class CreateRasterHabitat(GraphabAlgoProcessing):
    
    def __init__(self, plugin):
        super().__init__(plugin)

    INPUT = 'INPUT'
    HABITATNAME = 'HABITATNAME'
    CODES = 'CODES'
    MINAREA = 'MINAREA'
    MAXSIZE = 'MAXSIZE'
    CON8 = 'CON8'
    NOVORONOI = 'NOVORONOI'

    def initAlgorithm(self, config):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.translate('py', 'XML project file'),
                fileFilter = 'xml(*.xml)',
                optional = False
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.HABITATNAME,
                self.translate('py', 'Habitat name'),
                defaultValue = '',
                optional = False
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.CODES,
                self.translate('py', 'Habitat patch codes (separated by commas ex: 1,4,3)'),
                optional=False
            )
        )

        #minarea
        self.addParameter(
            QgsProcessingParameterNumber(
                self.MINAREA,
                self.translate('py', 'Minimum patches area (hectare)'),
                type = QgsProcessingParameterNumber.Double,
                defaultValue=0.0,
                minValue=0.0,
                optional = True
            )
        )

        #maxsize
        self.addParameter(
            QgsProcessingParameterNumber(
                self.MAXSIZE,
                self.translate('py', 'Maximum size of patches in meter'),
                type = QgsProcessingParameterNumber.Double,
                defaultValue = 0.0,
                minValue = 0.0,
                optional = True
            )
        )

        #con8
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CON8,
                self.translate('py', '8 connexity for patch definition'),
                defaultValue = False,
                optional = False
            )
        )

        #novoronoi
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NOVORONOI,
                self.translate('py', 'No voronoi'),
                defaultValue = False,
                optional = False
            )
        )

    def processAlgorithm(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> dict:
        """
        Create a command line depending of parameters set by the user and send it to its parent 
        to execute it
        """
        project = self.parameterAsFile(parameters, self.INPUT, context)
        self.projectPath = os.path.normcase(r'' + project)

        self.name = self.parameterAsString(parameters, self.HABITATNAME, context)

        codes = self.parameterAsString(parameters, self.CODES, context)
        minarea = self.parameterAsDouble(parameters, self.MINAREA, context)
        maxsize = self.parameterAsDouble(parameters, self.MAXSIZE, context)
        con8 = self.parameterAsBool(parameters, self.CON8, context)
        novoronoi = self.parameterAsBool(parameters, self.NOVORONOI, context)
        
        commandToExec = ['--project', self.projectPath, '--habitat', 'name=' + self.name, 'codes=' + codes]

        if minarea > 0.0:
            commandToExec.append(f'minarea={minarea}')
        if maxsize > 0.0:
            commandToExec.append(f'maxsize={maxsize}')
        if con8:
            commandToExec.append('con8')
        if novoronoi:
            commandToExec.append('novoronoi')

        super().processAlgorithm(commandToExec, context, feedback)
        return {}

    def postProcessAlgorithm(self, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> dict:
        """After the execution of the processing, this function
        add and show the new Graph in the map. """
        project: GraphabProject = self.plugin.getProjectFromPath(self.projectPath)

        if self.executionError or project is None:
            return {}

        project.add_habitat_project(self.name)
        return {}


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'create_raster_habitat'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.translate('py', 'Create Raster Habitat')

    def createInstance(self):
        return CreateRasterHabitat(self.plugin)




