# -*- coding: utf-8 -*-

__author__ = 'gaspard.quentin1905@gmail.com'
__date__ = '2024-05-06'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

import os

from PyQt5 import QtWidgets
import processing

from qgis.PyQt import uic
from PyQt5.QtWidgets import QTableWidgetItem, QMessageBox, QHeaderView
from PyQt5.QtCore import QCoreApplication, Qt

from qgis.core import QgsProcessing, QgsMapLayerProxyModel, QgsMapLayer, QgsProject, QgsProcessingException

from ..utils import NoFeedback

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'ui_files', 'create_project_dialog.ui'))

class CreateProjectDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, plugin) -> None:
        super(CreateProjectDialog, self).__init__(None)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        self.maxUniquesCodePossible = 1000

        # keep this variable : if windowProcess isn't stored, postprocessalgoritm can't be called.
        self.windowProcess = None

        self.landscapeLCB.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.landscapeLCB.layerChanged.connect(self.landscapeSelectionLayer)

        self.destFW.setStorageMode(1)
        self.destWidgets = [self.destFW, self.label_dest]

        self.habitatCB.clicked.connect(lambda: self.habitatGB.setEnabled(not self.habitatGB.isEnabled()))
        self.sizePatchesCB.clicked.connect(lambda: self.sizePatchesSB.setEnabled(self.sizePatchesCB.isChecked()))
        self.maxSizeCB.clicked.connect(lambda: self.maxSizeSB.setEnabled(self.maxSizeCB.isChecked()))
        self.tempCB.clicked.connect(self.toggleDestinationWidget)

        self.landscapeFW.fileChanged.connect(lambda: self.landscapeSelectionFilePath(self.landscapeFW.filePath()))
        self.codesTable.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)

        self.plugin = plugin

    # -------------------------------------------------------------------
    def landscapeSelectionFilePath(self, filePath: str):
        if not self.isVisible():
            return
        if self.codesTable.rowCount() > 0:
            self.populateCodesTables()

        if not os.path.exists(filePath) or not os.path.isfile(filePath):
            return

        try:
            self.populateCodesTables(filePath)
        except QgsProcessingException:
            pass
        # -------------------------------------------------------------------
    def landscapeSelectionLayer(self, layer: QgsMapLayer):
        if not self.isVisible():
            return
        if layer is None:
            return

        try:
            self.populateCodesTables(layer.source())
        except QgsProcessingException:
            pass
        self.landscapeFW.setFilePath("")

    # -------------------------------------------------------------------
    def toggleDestinationWidget(self):
        for e in self.destWidgets:
            e.setEnabled(not e.isEnabled())

    # -------------------------------------------------------------------
    def showDialog(self):
        """Initialize basic window and waiting for any action of the user"""
        if not self.plugin.java:
            self.plugin.showJavaNotFound()
            return

        self.habitatCB.setChecked(False)
        self.habitatGB.setEnabled(False)
        self.tempCB.setChecked(True)
        self.destFW.setEnabled(False)
        self.label_dest.setEnabled(False)
        self.sizePatchesCB.setChecked(False)
        self.sizePatchesSB.setEnabled(self.sizePatchesCB.isChecked())
        self.maxSizeSB.setEnabled(self.maxSizeCB.isChecked())

        base_name = 'NewProject'
        i = 1
        names = [group.name() for group in QgsProject
                 .instance()
                 .layerTreeRoot()
                 .children()]
        while base_name + str(i) in names:
            i+=1
        self.projectNameLE.setText(base_name + str(i))

        if len(landscape := self.landscapeFW.filePath()) > 0:
            self.landscapeSelectionFilePath(landscape)
        elif self.landscapeLCB.count() > 0:
            self.landscapeSelectionLayer(self.landscapeLCB.currentLayer())
        else:
            self.populateCodesTables()

        while self.exec_() and not self.create_project():
            pass
    # --------------------------------------------------------------------------
    def getUniquesValuesFromRaster(self, raster: str):
        """
        Returns the codes from a given raster file
        """
        params = {
            "INPUT": raster,
            "BAND": 1,
            "OUTPUT_TABLE": QgsProcessing.TEMPORARY_OUTPUT
        }

        vectorLayer = processing.run(
            "native:rasterlayeruniquevaluesreport",
            params,
            feedback=NoFeedback()
        )["OUTPUT_TABLE"]

        params = {
            "INPUT": vectorLayer,
            "FIELDS": 'value',
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT
        }
        stringValues = processing.run("qgis:listuniquevalues", params, feedback=NoFeedback())["UNIQUE_VALUES"]
        tabValues = stringValues.split(';')
        if tabValues.count("\"") == 0: #for a strange reason, when there is no result, tabValues == ["\""]
            return sorted(tabValues, key=lambda x: int(float(x)))
        return []

    # -------------------------------------------------------------------
    def populateCodesTables(self, raster=None):
        """Initialize the encode tab and the nodata list with the selected raster,
        if raster is None it clears the tab"""
        #self.codesTable.setRowCount(0)

        while (count := self.noDataCB.count()) > 1:
            self.noDataCB.removeItem(count - 1)

        if raster is None or raster == '':
            self.codesTable.setRowCount(0)
            return

        unique_values = self.getUniquesValuesFromRaster(raster)
        if len(unique_values) > self.maxUniquesCodePossible:
            QMessageBox.information(
                None,
                self.translate('osraster_raster', "ERROR : Raster unique codes"),
                self.translate('osraster_raster', "This raster has been ignored because there is too much unique code values. (MAX=%d)") % self.maxUniquesCodePossible
            )
            self.resetNoFile()
            return

        # Add a number of row depending of the number of unique values
        self.codesTable.setRowCount(len(unique_values))
        self.codesTable.setColumnCount(1)
        # Fill lines
        counter = 0
        for code in unique_values:
            intCode = int(float(code))
            item = QTableWidgetItem(str(intCode))
            item.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
            self.codesTable.setItem(counter, 0, item)
            self.noDataCB.addItem(str(intCode))
            counter += 1


    def resetNoFile(self):
        """TODO: implement this"""
        return
    # -------------------------------------------------------------------
    def translate(self, context, string):
        return QCoreApplication.translate(context, string)

    # -------------------------------------------------------------------
    def showMessageBox(self, title: str, text: str) -> None:
        """Displays a popup. (information QMessageBox)

        :param title: the title of the popup
        :type title: str

        :param text: the main content of the popup
        :type text: str
        """
        QMessageBox.information(
            None,
            self.translate('create_project_dialog', title),
            self.translate('create_project_dialog', text)
        )

    # -------------------------------------------------------------------
    def create_project_habitat(self, params: dict[str, str]) -> bool:
        if len(self.codesTable.selectedItems()) == 0:
            self.showMessageBox(
                'ERROR: no raster Codes for habitat definition',
                "Please select raster codes or don't create an habitat for now"
            )
            return False
        params['LANDCODE'] = ''.join(item.text() + ',' for item in self.codesTable.selectedItems())[:-1]
        if self.sizePatchesCB.isChecked():
            params['SIZEPATCHES'] = self.sizePatchesSB.value()
        if self.maxSizeCB.isChecked():
            params['MAXSIZE'] = self.maxSizeSB.value()
        return True

    # -------------------------------------------------------------------
    def create_project(self) -> bool:
        """Method that call the CreateProject processing algorithm
        with the arguments provided by the user using this dialog.
        (to call once OK is pressed)

        :return: True if the algorithm has been launched, False if a parameter is missing.
        :rtype: bool
        """
        alg = 'graphab3:create_project'
        params = {}


        if len((filePath := self.landscapeFW.filePath())) > 0:
            if not os.path.isfile(filePath):
                self.showMessageBox(
                        'ERROR: bad file provided',
                        'The provided file does not exist on your filesystem or is invalid, please provide a valid one'
                        )
                self.landscapeFW.setFilePath("")
                return False
            params['INPUT'] = filePath
        elif self.landscapeLCB.count() > 0 and (layer := self.landscapeLCB.currentLayer()):
            params['INPUT'] = layer.source()
        else:
            self.showMessageBox(
                'ERROR: no landscape map selected',
                'Please select a landscape map'
            )
            return False

        name = self.projectNameLE.text()
        if len(name) == 0:
            self.showMessageBox( 
                'ERROR: no name for the project provided',
                'Please enter a project name'
            )
            return False

        params['NAMEPROJECT'] = name

        if self.habitatCB.isChecked() and not self.create_project_habitat(params):
            return False

        params['DIRPATH'] = ''
        if not self.tempCB.isChecked():
            if len(self.destFW.filePath()) == 0:
                self.showMessageBox(
                    'ERROR: no directory path provided',
                    'Please provide a directory to store the project or use temporary directory'
                )
                return False
            params['DIRPATH'] = self.destFW.filePath()

        if (nodata := self.noDataCB.currentText()) != "(none)":
            params['NODATA'] = nodata

        # call the process to create a project with all the parameters
        self.windowProcess = processing.createAlgorithmDialog("graphab3:create_project", params)
        self.windowProcess.show()
        self.windowProcess.runAlgorithm()
        return True
