# -*- coding: utf-8 -*-
from __future__ import annotations
"""
 Graphab3 for QGIS
 This plugin allows you to use Graphab version 3 software directly into QGIS
"""
import os # This is needed in the pyqgis console also
import subprocess

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QInputDialog, QMessageBox, QFileDialog

from qgis.core import QgsApplication, QgsProject, QgsMessageLog, Qgis, QgsVectorLayer

from .processing.Graphab3Provider import Graphab3Provider
# Initialize Qt resources from file resources.py
from .resources import *
# Import necessary objects
from .GraphabProject import GraphabProject
from .GraphabStyle import GraphabStyle
from .gui.graph_symbology_dialog import GraphSymbologyDialog
from .gui.create_graph_dialog import CreateGraphDialog
from .gui.create_linkset_dialog import CreateLinksetDialog
from .gui.calculate_metrics_dialog import CalculateMetricDialog
from .gui.corridor_dialog import CorridorDialog
from .gui.create_habitat_dialog import CreateHabitatDialog
from .gui.create_project_dialog import CreateProjectDialog
# Import os raster project
from .OsRaster.OsRaster import OsRaster

__author__ = 'Gaspard QUENTIN, Robin MARLIN-LEFEBVRE'
__email__ = 'gaspard.quentin1905@gmail.com'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

MOCK_GRAPHAB_QGIS_PROJECT = None

class GraphabPlugin:
    """QGIS Plugin Implementation."""

    PROBA_DIST = ['0.5', '0.05']
    GMETRICS = ['EC', 'PC','IIC']
    LMETRICS = ['F', 'IF', 'BC']
    UNITS = []

    graphabProvider = None
    java = True

    projects = {}

    def __init__(self, iface):
        """Constructor.
        
        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        self.test_layer = None

        # initialize locale if we are not in a container
        locale = QSettings().value('locale/userLocale')
        if locale:
            locale = locale[0:2]
            locale_path = os.path.join(
                self.plugin_dir,
                'i18n',
                '{}.qm'.format(locale))
        else:
            locale_path = ""

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        
        # all available styles for Graph circles
        self.stylesTabUnabled = ["Red","Blue"]
        
        # csv prefix that is important to know because it's in imported fieldnames
        self.prefix = '_'

        self.UNITS = [self.translate('py', 'Meter'), self.translate('py', 'Cost')]

        # All necessary objects
        self.GraphabStyle = GraphabStyle(self)
        self.CreateProjectDialog = CreateProjectDialog(self)
        self.GraphSymbologyDialog = GraphSymbologyDialog(self)
        self.CreateGraphDialog = CreateGraphDialog(self)
        self.CreateLinksetDialog = CreateLinksetDialog(self)
        self.CalculateMetricDialogGlobal = CalculateMetricDialog(self.GMETRICS, 0, self)
        self.CalculateMetricDialogLocal = CalculateMetricDialog(self.LMETRICS, 1, self)
        self.CorridorDialog = CorridorDialog(self)
        self.CreateHabitatDialog = CreateHabitatDialog(self)

        # Initialization of OsRaster
        self.OsRaster = OsRaster(self)

        self.graphabProvider = Graphab3Provider(self)
        QgsApplication.processingRegistry().addProvider(self.graphabProvider)

        javaExec = self.graphabProvider.get_java_command()
        try:
            ret = subprocess.run([javaExec, '-version'], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            self.java = ret.returncode == 0
        except FileNotFoundError:
            self.java = False

        if not self.java:
            QgsMessageLog.logMessage("Java not found for Graphab plugin.\n" + javaExec + "\n", 'Extensions', Qgis.Warning)

    #--------------------------------------------------------------------------
    @staticmethod
    def translate(context, string: str) -> str:
        return QCoreApplication.translate(context, string)

    #--------------------------------------------------------------------------

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip:
            action.setStatusTip(status_tip)

        if whats_this:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    #--------------------------------------------------------------------------

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        self.menu = u'&Graphab3'

        self.toolbar = self.iface.addToolBar(u'Graphab3')
        self.toolbar.setObjectName(u'Graphab3')
        self.actions = []

        # Load Graphab project button
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_open_project.png',
            text=self.translate('py', u'Load a project'),
            callback=self.openProject,
            parent=self.iface.mainWindow())

        # OsRaster button for rasterize vector and/or merge rasters file
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_osraster.png',
            text=self.translate('py', u'Create landscape map'),
            callback=self.OsRaster.showDialog,
            parent=self.iface.mainWindow())

        # Create Graphab Project button
        self.add_action(
                ':/plugins/graphab3-for-qgis/icons/icon_project.png',
                text=self.translate('py', u'Create Project'),
                callback=self.CreateProjectDialog.showDialog,
                parent=self.iface.mainWindow())
        # Create Habitat button
        self.add_action(
                ':/plugins/graphab3-for-qgis/icons/icon_habitat.png',
                text=self.translate('py', u'Create Habitat'),
                callback=self.CreateHabitatDialog.showDialog,
                parent=self.iface.mainWindow())
        # Remove Habitat button
        self.add_action( 
            ':/plugins/graphab3-for-qgis/icons/icon_habitat_rem.png',
            text=self.translate('py', u'Remove Habitat'),
            callback=self.removeHabitat,
            parent = self.iface.mainWindow())
        # Create Linkset button
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_linkset.png',
            text=self.translate('py', u'Create Linkset'),
            callback=self.CreateLinksetDialog.showDialog,
            parent=self.iface.mainWindow())
        # Remove Linkset button
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_linkset_rem.png',
            text=self.translate('py', u'Remove Linkset'),
            callback=self.removeLinkset,
            parent=self.iface.mainWindow())
        # Create Graph button
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_graph.png',
            text=self.translate('py', u'Create Graph'),
            callback=self.CreateGraphDialog.showDialog,
            parent=self.iface.mainWindow())
        # Remove Graph button
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_graph_rem.png',
            text=self.translate('py', u'Remove Graph'),
            callback=self.removeGraph,
            parent=self.iface.mainWindow())
        # Calculate corridors
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_corridor.png',
            text=self.translate('py', u'Calculate Corridors'),
            callback=self.CorridorDialog.showDialog,
            parent=self.iface.mainWindow())
        # Calculate Local Metrics
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_mlocal.png',
            text=self.translate('py', u'Calculate Local Metric'),
            callback=self.CalculateMetricDialogLocal.showDialog,
            parent=self.iface.mainWindow())
        # Calculate Global Metrics
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_mglobal.png',
            text=self.translate('py', u'Calculate Global Metric'),
            callback=self.CalculateMetricDialogGlobal.showDialog,
            parent=self.iface.mainWindow())

        # Change symbology button
        self.add_action(
            ':/plugins/graphab3-for-qgis/icons/icon_symbo.png',
            text=self.translate('py', u'Display metric'),
            callback=self.GraphSymbologyDialog.showDialog,
            parent=self.iface.mainWindow())

    # --------------------------------------------------------------------------

    def showJavaNotFound(self):
        QMessageBox.warning(self.iface.mainWindow(), "Graphab plugin",
                            "Java executable not found, you can only display Graphab project. "
                            "Please install Java from https://adoptopenjdk.net to access all features of the plugin")

    # --------------------------------------------------------------------------

    def openProject(self):
        # open a window to select a project xlm of Graphab a
        filename, _filter = QFileDialog.getOpenFileName(None, "Select Graphab project", "", '*.xml')

        # when you close the window but you haven't select a file
        if filename is None or filename == '':
            return

        self.loadProject(filename)


    # --------------------------------------------------------------------------

    def loadProject(self, filename):
        filename = os.path.normcase(filename)
        self.updateLoadedProject()
        # if the file is already loaded the plugin stop
        if filename in self.projects.keys():
            return

        self.projects[filename] = GraphabProject(self, filename)
    # --------------------------------------------------------------------------
    def removeHabitat(self):
        if not self.java:
            self.showJavaNotFound()
            return

        habitatNames = []
        for prj in self.getProjects():
            for habitat in prj.project.habitats:
                habitatNames.append(prj.project.name + ' -> ' + habitat.name)

        res, ok = QInputDialog().getItem(self.iface.mainWindow(), self.translate('py', 'Remove habitat'), 
                                         self.translate('py', 'Select the habitat to remove'), habitatNames, 0, False)
        if ok and res:
            project, habitat = res.split(' -> ')
            prj = self.getProject(project)
            if not prj.remove_habitat(habitat):
                QMessageBox.warning(self.iface.mainWindow(), self.translate('py', 'Remove Habitat'), 'Impossible to remove habitat ' + habitat)

    # --------------------------------------------------------------------------

    def removeLinkset(self):
        """Removes a linkset loaded in QGIS from it's project and QGIS."""
        if not self.java:
            self.showJavaNotFound()
            return

        linksetNames = []
        for prj in self.getProjects():
            for linkset in prj.project.linksets:
                linksetNames.append(prj.project.name + ' -> ' + linkset.name)

        res, ok = QInputDialog().getItem(self.iface.mainWindow(), self.translate('py', "Remove linkset"),
                                         self.translate('py', "Select the linkset to remove :"), linksetNames, 0, False)
        if ok and res:
            project, linkset = res.split(' -> ')
            prj = self.getProject(project)
            if not prj.remove_linkset(linkset):
                QMessageBox.warning(self.iface.mainWindow(), self.translate('py', "Remove linkset"), "Impossible to remove linkset " + linkset)

    # --------------------------------------------------------------------------

    def removeGraph(self):
        """Removes a graph loaded in QGIS from it's project and QGIS."""
        if not self.java:
            self.showJavaNotFound()
            return

        graphNames = []
        for prj in self.getProjects():
            for graph in prj.project.graphs:
                graphNames.append(prj.project.name + ' -> ' + graph.name)

        res, ok = QInputDialog().getItem(self.iface.mainWindow(), self.translate('py', "Remove graph"),
                                         self.translate('py', "Select the graph to remove :"), graphNames, 0, False)
        if ok and res:
            project, graph = res.split(' -> ')
            prj = self.getProject(project)
            if not prj.remove_graph(graph):
                QMessageBox.warning(self.iface.mainWindow(), self.translate('py', "Remove graph"), "Impossible to remove graph " + graph)

    # --------------------------------------------------------------------------

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        if self.graphabProvider:
            QgsApplication.processingRegistry().removeProvider(self.graphabProvider)

        for action in self.actions:
            self.iface.removePluginMenu(
                self.menu,
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    # --------------------------------------------------------------------------

    def getProject(self, projectName: str) -> GraphabProject:
        """
        This method gives the project object of name projectName.

        :param projectName: the name of the project we want to get 
        :type projectName: str

        :returns: the project named projectName or None if not found
        :rtype: GraphabProject or None 
        """
        for project in self.projects.values():
            if project.project.name == projectName:
                return project

        return None

    # --------------------------------------------------------------------------
    
    def getProjectFromPath(self, projectFilePath: str) -> GraphabProject:
        """
        This method gives the project object of project file 
        path projectFilePath

        :param projectFilePath: the file path of the file
        representing the project we want to get 
        :type projectFilePath: str

        :returns: the project or None if not found
        :rtype: GraphabProject or None 
        """
        try:
            return self.projects[projectFilePath]
        except KeyError:
            return None


    # --------------------------------------------------------------------------

    def updateLoadedProject(self):
        """This method updates all the projects whose project file has been changed"""
        projects = {}
        root = QgsProject.instance().layerTreeRoot()
        for group in root.findGroups():
            if "Graphab3Project" in group.customProperties():
                prjFile = group.customProperty("Graphab3Project")
                if os.path.isfile(prjFile):
                    if prjFile in self.projects.keys():
                        projects[prjFile] = self.projects[prjFile]
                        projects[prjFile].upgrade_loaded_project()
                    else:
                        projects[prjFile] = GraphabProject(self, prjFile, group)
                else:
                    group.removeCustomProperty("Graphab3Project")


        self.projects = projects

    # --------------------------------------------------------------------------

    def getProjects(self) -> list[GraphabProject]:
        self.updateLoadedProject()
        return list(self.projects.values())


    # --------------------------------------------------------------------------
    def get_project_from_graph(self, graph_name: str) -> GraphabProject|None:
        """Returns the first project that has a graph named graph_name.

        :param graph_name: The graph's name
        :type graph_name: str

        :return: The first project containing a graph named graph_name or None if the graph
        was found nowhere.
        :rtype: GraphabProject or None
        """
        for project in self.getProjects():
            for graph in project.project.graphs:
                if graph.name == graph_name:
                    return project
        return None

    # --------------------------------------------------------------------------
    @staticmethod
    def instance(iface=None):
        """Returns the running instance of the plugin or create one with 
        a mock qgis interface when no QGIS GUI is running 
        (usefull for unit testing). 

        :param iface: An interface instance which provides the hook by which you can
        manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        try: 
            return Qgis.utils.plugins['graphab34qgis']
        except Exception as e:
            global MOCK_GRAPHAB_QGIS_PROJECT
            if MOCK_GRAPHAB_QGIS_PROJECT is not None:
                return MOCK_GRAPHAB_QGIS_PROJECT
            from .tests.utilities import get_qgis_app
            print("[Graphab34Qgis] create Mock project")
            if iface is None:
                qgis_app = get_qgis_app()
                iface = qgis_app[2]
            plugin = GraphabPlugin(iface)
            if plugin.graphabProvider is None:
                plugin.graphabProvider = Graphab3Provider(plugin)
            plugin.graphabProvider.loadAlgorithms()
            dir = os.path.dirname(os.path.abspath(__file__))
            plugin.loadProject(os.path.join(dir, 'tests/computed_data/Graphab3prj/Graphab3prj.xml'))
            vector_layer = QgsVectorLayer(os.path.join(dir, 'tests/computed_data/Graphab3prj/habitat1/patches.gpkg'))
            if not vector_layer.isValid():
                raise Exception("Mock layer couldn't be instanciated")
            plugin.test_layer = vector_layer
            if plugin.iface:
                plugin.iface.addLayers([plugin.test_layer])
            plugin.initGui()
            MOCK_GRAPHAB_QGIS_PROJECT = plugin
            return MOCK_GRAPHAB_QGIS_PROJECT
