# -*- coding: utf-8 -*-
"""
 xmlToPython
 This module convert a specific Gaphab XML file into a Python object. It use declxml.
"""

from .declxml import *
import re

__author__ = 'Gaspard QUENTIN, Robin MARLIN-LEFEBVRE'
__email__ = 'gaspard.quentin1905@gmail.com'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

class Project:

    """
    Contains all the project's data that have been parsed from the Project.xml file.
    """
    def __init__(self):
        self.name = None
        self.graphabVersion = None
        self.codes = []
        self.noData = None
        self.capacityParams = []
        self.resolution = None
        self.grid2space = []
        self.space2grid = []
        
        self.habitats: list[AbstractHabitat] = []
        self.monoHabitats: list[MonoHabitat] = []
        self.monoVectorHabitats: list[MonoVectorHabitat] = []
        self.vectorDatasets: list[VectorDataset] = []
        
        self.linksets: list[AbstractLinkset] = []
        self.euclideLinksets: list[EuclideLinkset] = []
        self.costLinksets: list[CostLinkset] = []
        self.circuitLinksets: list[CircuitLinkset] = []

        self.metrics: list[AbstractMetricResult] = []
        self.localMetrics: list[MetricResult] = []
        self.multigraphs: list[MultiGraph] = []
        self.graphs: list[AbstractGraph] = []

        self.zone = []
        self.demFile = None
        self.wktCRS = None

    def __repr__(self):
        return '\nProject(name={}, graphabVersion={}, codes={}, noData={}, ' \
               '\ncapacityParams={}, resolution={}, \ngrid2space={}, \nspace2grid={}, \nhabitats={},' \
                '\nlinksets={}, \neuclideLinksets={}, \ncostLinksets={}, \ncircuitLinksets={},' \
                '\nmultrigraphs={}, ' \
               '\ngraphs={}, ' \
               '\nzone={}, ' \
               '\ndemFile={}' \
               '\nwktCRS={})'.format(self.name, self.graphabVersion, self.codes, self.noData,
                                     self.capacityParams, self.resolution, self.grid2space,
                                     self.space2grid, self.habitats, self.linksets, self.euclideLinksets,
                                     self.costLinksets, self.circuitLinksets, self.multigraphs, self.graphs, self.zone, self.demFile, self.wktCRS)


class AbstractHabitat:
    """ 
    Abstract class, no instances should be made.
    """
    def __init__(self) -> None:
        self.id: int|None = None
        self.name: str|None = None
        self.entryIndex = 0

    def __repr__(self) -> str:
        return 'Habitat(idHab={}, name={}'.format(self.id, self.name)

class MonoHabitat(AbstractHabitat):
    def __init__(self):
        super().__init__()
        self.patchCodes = []
        self.minArea: float|None = None 
        self.maxSize: float|None = None
        self.capacityParams = []
        self.con8: bool|None = None 

    def __repr__(self) -> str:
        return "\nMono" + super().__repr__() + 'patchCodes={}, minArea={}, maxSize={}, ' \
                                            'capacityParams={}, con8={})'.format(self.patchCodes,
                                                     self.minArea, self.maxSize,
                                                     self.capacityParams, self.con8)
class MonoVectorHabitat(AbstractHabitat):
    def __init__(self):
        super().__init__()
    
    def __repr__(self) -> str:
        return "\nMonoVector" + super().__repr__() + ")"

class VectorDataset(AbstractHabitat):
    def __init__(self):
        super().__init__()

    def __repr__(self) -> str:
        return "\nVectorDataset" + super().__repr__()

class CapacityParam:
    def __init__(self):
        self.calcArea = None
        self.exp = 0.0
        self.maxCost = None
        self.weightCost = None

    def __repr__(self) -> str:
        return '\nCapacityParam(calcArea={}, exp={}, maxCost={}, weightCost={})'.format(self.calcArea, self.exp,
                                                                                self.maxCost, self.weightCost)


class Grid2space:
    def __init__(self):
        self.m00 = None
        self.m01 = None
        self.m02 = None
        self.m10 = None
        self.m11 = None
        self.m12 = None

    def __repr__(self) -> str:
        return '\nGrid2space(m00={}, m01={}, m02={}, m10={}, m11={}, m12={})'.format(self.m00, self.m01, self.m02,
                                                                                     self.m10, self.m11, self.m12)


class Space2grid:
    def __init__(self):
        self.m00 = None
        self.m01 = None
        self.m02 = None
        self.m10 = None
        self.m11 = None
        self.m12 = None

    def __repr__(self) -> str:
        return '\nSpace2grid(m00={}, m01={}, m02={}, m10={}, m11={}, m12={})'.format(self.m00, self.m01, self.m02,
                                                                                     self.m10, self.m11, self.m12)




class AbstractLinkset:
    """ 
    Abstract class, no instances should be made.
    """
    def __init__(self):
        self.entryIndex = 0
        self.name: str = ""
        self.typeDist = None
        
        self.inter: bool = False
        self.habitats: list[str]|list[AbstractHabitat] = []
        self.datasets: list[str] = []
        self.monoVectorHabitats: list[str] = []
        # self.habitat will first be the reference and then the object itself
        self.habitat: str|AbstractHabitat|None = None 
        self.classHabitat: str|None = None
        self.habitatName: str = ""
        
        
        self.topology: str|None = None
        self.costs = []
        self.realPaths = None
        self.removeCrossPatch = None
        self.coefSlope = None
        self.distMax = None
        self.extCostFile = None
        self.optimCirc = None

    def __repr__(self) -> str:
        return 'Linkset(name={}, typeDist={}, inter={}, habitat={}, habitats={}, habitat(s)Name={}, classHabitat={}, topology={}, costs={}, realPaths={}, ' \
               'removeCrossPatch={}, coefSlope={}, distMax={}, extCostFile={}, ' \
               'optimCirc={})'.format(self.name, self.typeDist, self.inter, self.habitat,
                                      self.habitats, self.habitatName, self.classHabitat, self.topology, self.costs,self.realPaths,
                                      self.removeCrossPatch, self.coefSlope, self.distMax,self.extCostFile, self.optimCirc)
    def updateHabitatName(self) -> None:
        """
        This method updates the name of the habitat in order to be able to easily get the filepath of it.
        It is usefull only for interlinksets.
        the name will be in this case like this : habitat1-habitat2-...-habitatn
        """
        if not self.inter:
            if isinstance(self.habitat, AbstractHabitat):
                self.habitatName = str(self.habitat.name)
            return 
        s = ''
        for i in range(len(self.habitats)):
            hab = self.habitats[i]
            if isinstance(hab, AbstractHabitat):
                s += str(hab.name)
            if i != len(self.habitats) - 1:
                s += '-'
        self.habitatName = s

class EuclideLinkset(AbstractLinkset):
    def __init__(self):
        super().__init__()
        self.typeDist = 'EUCLID'

    def __repr__(self) -> str:
        return "\nEuclide" + super().__repr__()

class CostLinkset(AbstractLinkset):
    def __init__(self):
        super().__init__()
        self.typeDist = 'COST'

    def __repr__(self):
        return "\nCost" + super().__repr__()

class CircuitLinkset(AbstractLinkset):
    def __init__(self):
        super().__init__()
        self.typeDist = 'CIRCUIT'

    def __repr__(self):
        return "\nCircuit" + super().__repr__()




class AlphaParam:
    def __init__(self) -> None:
        self.hasBeta: bool|None = None
        self.alpha: float|None
        self.d: float|None = None
        self.p: float|None = None
        self.beta: float|None = None
        self.maxD: str = ""
        self.minP: float|None = None

    def __repr__(self) -> str:
        return "\nAlphaParam(hasBeta={}, alpha={}, d={}, p={}, beta={}, maxD={}, minP={}),".format(self.hasBeta, 
                                                         self.alpha, self.d, self.p, self.beta, self.maxD, self.minP)

class Metric:
    def __init__(self) -> None:
        self.alphaParam: AlphaParam|None = None
        # if the MetricResult is Local then those will stay None
        self.result: float|None = None
        self.cum: float|None = None
        self.index: list[int] = []

    def __repr__(self) -> str:
        if self.result or self.cum or len(self.index) > 0:
            return "\nMetric(alphaParam= {}, result={}, cum={}, index={}),".format(self.alphaParam, self.result, 
                                                                                         self.cum, self.index)
        return "\nMetric(alphaParam= {}),".format(self.alphaParam)

class AbstractMetricResult:
    """ 
    Abstract class, no instances should be made.
    """
    def __init__(self) -> None:
        self.name: str = ""
        self.metricKind: str = ""
        self.metric: Metric|None = None
        self.method: str = ""
        self.graph: str|AbstractGraph = "" 

    def __repr__(self) -> str:
        return "MetricResult(name={}, metricKind={}, alphaParam={}, method={}, graph=[...]".format(self.name,
                                                self.metricKind, self.metric, self.method)#, self.graph)

class MetricResult(AbstractMetricResult):
    def __init__(self) -> None:
        super().__init__()
        self.attrName: str = ""

    def __repr__(self) -> str:
        return "\nLocal" + super().__repr__() +", attrName={}),".format(self.attrName)

#TODO: check if there is multiple results or only one (result a float or array ?)
class GlobalMetricResult(AbstractMetricResult):
    def __init__(self) -> None:
        super().__init__()
        self.result: float|str = ""

    def __repr__(self) -> str:
        return "\nGlobal" + super().__repr__() + ", result={}),".format(self.result)



class AbstractGraph:
    """ 
    Abstract class, no instances should be made.
    """
    def __init__(self) -> None:
        self.name: str = ""
        self.intraPatchDist: bool | None = None
        self.metrics = []
        self.globalMetrics: list[MetricResult] = []
        self.isMulti: bool = False

    def __repr__(self) -> str:
        return "Graph(name={}, intraPatchDist={}, metrics={}, globalMetrics={})".format(self.name,
                                                        self.intraPatchDist, self.metrics, self.globalMetrics)

class Graph(AbstractGraph):
    def __init__(self):
        super().__init__()
        self.linkset: AbstractLinkset | None = None
        self.threshold = None

    def __repr__(self):
        return "\n" + super().__repr__() + ", linkset={}, threshold={}".format(self.linkset, self.threshold)

class MultiGraph(AbstractGraph):
    def __init__(self) -> None:
        super().__init__()
        self.graphs: list[AbstractGraph]|list[str] = []
        self.isMulti = True
        self.habitats: list[AbstractHabitat] = []
        
    def __repr__(self) -> str:
        return "\nMulti" + super().__repr__() + ", graphs={})".format(self.graphs)
        
    def updateHabitats(self) -> None:
        """
        This method adds every habitat from each linkset of the graph to the habitats list, 
        which permits an easier access to those afterwards.
        """
        for graph in self.graphs:

            # needed for static code analysis 
            if not isinstance(graph, Graph) or not isinstance(graph.linkset, AbstractLinkset):
                continue

            if graph.linkset.inter:
                for habitat in graph.linkset.habitats:
                    if isinstance(habitat, AbstractHabitat):
                        self.habitats.append(habitat)
            else:
                if isinstance(graph.linkset.habitat, AbstractHabitat):
                    self.habitats.append(graph.linkset.habitat)


class Zone:
    def __init__(self):
        self.x = None
        self.y = None
        self.width = None
        self.height = None

    def __repr__(self):
        return '\nZone(x={}, y={}, width={}, height={})'.format(self.x, self.y, self.width, self.height)


class GraphabVersionTooLowException(Exception):
    def __init__(self):
        super().__init__("The project loaded is from a version of Graphab < 3.\nPlease upgrade it or use the appropriate plugin for earlier versions")


def linkHabitatToLinkset(p: Project) -> None:
    """Link every linkset to it's habitat.

    :param p: the project that was parsed
    :type p: Project

    :raise: Exception if there were a problem while linking
    """
    for linkset in p.linksets:
        if linkset.inter:
            linkHabitatsToInterLinkset(linkset, p)
            continue
        regex = re.compile(r'habitats/entry\[?(\d+)?\]?/(?:org\.thema\.graphab\.(habitat\.MonoVectorHabitat|dataset\.VectorDataset)|Habitat)')
        result = regex.search(str(linkset.habitat))
        if not result or not (indexHabitat := result.group(1)) :
            if len(p.habitats) > 0:
                linkset.habitat = p.habitats[0]
                continue
            else:
                raise Exception("Problem while linking Linksets to Habitat.")
        linkset.habitat = p.habitats[int(indexHabitat) - 1]


def linkHabitatsToInterLinkset(linkset: AbstractLinkset, p: Project) -> None:
    """Link the inter linkset to all it's habitat

    :param linkset: the linkset that will be linked with the habitats
    :type linkset: AbstractLinkset

    :param p: the project that was parsed
    :type p: Project

    :raise: Exception if there were a problem while linking
    """
    regex = re.compile(r'habitats/entry\[?(\d+)?\]?/(?:org\.thema\.graphab\.(habitat\.MonoVectorHabitat|dataset\.VectorDataset)|Habitat)')
    linkset_habitats = []
    for habitat in linkset.habitats:
        match = regex.search(str(habitat))
        if not match or not (indexHabitat := match.group(1)):
            if len(p.habitats) > 0:
                linkset_habitats.append(p.habitats[0])
                continue
            else:
                raise Exception("Problem while linking Linksets to Habitat.")

        linkset_habitats.append(p.habitats[int(indexHabitat) - 1])
    linkset.habitats = linkset_habitats



def linkLinksetToGraph(p: Project) -> None:
    """Link a Linkset to a Graph by using the Graph xPath

    :param p: the project that was parsed
    :type p: Project

    :raise: Exception if there were a problem while linking
    """
    for graph in p.graphs:
        regex = re.compile(r'linksets/entry\[?(\d+)?\]?/(Euclide|Cuircuit|Cost)Linkset')
        if not isinstance(graph, Graph) or not isinstance(graph.linkset, str):
            continue
        match = regex.search(graph.linkset)
        if not match or not (indexLinkset := match.group(1)):
            if len(p.linksets) > 0:
                graph.linkset = p.linksets[0]
                continue
            else:
                raise Exception("Problem while loading linksets.")
        graph.linkset = p.linksets[int(indexLinkset) - 1]

def linkGraphsToMultiGraph(p: Project) -> None:
    """Link all the MultiGraphs with their graphs

    :param p: the project that was parsed
    :type p: Project

    :raise: Exception if there were a problem while linking
    """
    for multi in p.multigraphs:
        multi_graphs: list[AbstractGraph] = []
        for i in range(len(multi.graphs)):
            graph = multi.graphs[i]
            regex = re.compile(r'entry\[?(\d+)?\]?/Graph')
            match = regex.search(str(graph))
            if not match or not (indexGraph := match.group(1)):
                if len(p.graphs) > 0:
                    multi_graphs.append(p.graphs[0])
                    continue
                else:
                    raise Exception("Problem while linking graphs to MultiGraph")
            multi_graphs.append(p.graphs[int(indexGraph) - 1])
        multi.graphs = multi_graphs

def linkGraphToMetric(p: Project) -> None:
    """Link all metrics with it's graph
 
    :param p: the project that was parsed
    :type p: Project

    :raise: Exception if there were a problem while linking
    """
    for graph in p.graphs:
        for metric in graph.metrics:
            metric.graph = graph


def reorderProjectHabitats(p: Project, filepath: str) -> None:
    """This procedure sorts the habitats parsed in order of their apparitioin in the xml
    this is needed to ensure that every linkset will be linked to the good habitat(s).

    :param p: the parsed project
    :type p: Project

    :param filepath: the path of the project XML file
    :type filepath: str
    """
    habitatOrder = {}
    with open(filepath, 'r') as f:
        isInHabitatSection = False
        countHabitat = 0
        for line in f.readlines():
            if "<habitats>" in line:
                isInHabitatSection = True
            if "</habitats>" in line:
                isInHabitatSection = False
            if isInHabitatSection and "<name>" in line:
                habitatOrder[line.split(">")[1].split("<")[0]] = countHabitat
                countHabitat += 1 
    for habitat in p.habitats:
        habitat.entryIndex = habitatOrder[habitat.name]
    p.habitats.sort(key= lambda habitat: habitat.entryIndex)


def reorderProjectLinksets(p: Project, filepath: str) -> None:
    """This procedure sorts the linksets parsed in order of their apparition in the xml
    this is needed to ensure that every graph will be linked to the good linkset

    :param p: the parsed project
    :type p: Project

    :param filepath: the path of the project XML file.
    :type filepath: str
    """
    linksetOrder = {}
    with open(filepath, 'r') as f:
        isInLinksetSection = False
        countLinkset = 0
        for line in f.readlines():
            if "<linksets>" in line:
                isInLinksetSection = True
            if "</linksets>" in line:
                isInLinksetSection = False
            if isInLinksetSection and "<name>" in line:
                linksetOrder[line.split(">")[1].split("<")[0]] = countLinkset
                countLinkset += 1
    for linkset in p.linksets:
        linkset.entryIndex = linksetOrder[linkset.name]
    p.linksets.sort(key= lambda linkset: linkset.entryIndex)


def parseXmlFile(xml_string: str) -> Project:
    """ Parses the project xml file and returns an instance of Project.
    It uses the user_project parser from the declxml library.

    :param xml_string: the xml file in a string
    :type xml_string: str

    :return: The QGIS project object, without elements linking and reordering
    :rtype: Project
    """
    mono_habitat_proc = user_object('Habitat', MonoHabitat, [
        integer('idHab', alias='id'),
        string('name', alias='name'),
        array(integer('int'), alias='patchCodes', nested='patchCodes'),
        floating_point('minArea', alias='minArea', required=False),
        floating_point('maxSize', alias='maxSize', required=False),
        user_object('capacityParams', CapacityParam, [
            boolean('calcArea', alias='calcArea'),
            floating_point('maxCost', alias='maxCost'),
            floating_point('exp', alias='exp'),
            boolean('weightCost', alias='weightCost')
        ], alias='capacityParams'),
        boolean('con8', alias='con8'),
    ], required=False)

    vector_habitat_proc = user_object('org.thema.graphab.habitat.MonoVectorHabitat', MonoVectorHabitat, [
        integer('idHab', alias='id'),
        string('name', alias='name'),
        floating_point('minArea', alias='minArea', required=False),
    ], required=False)

    vector_datasets_proc = user_object('org.thema.graphab.dataset.VectorDataset', VectorDataset, [
        integer('idHab', alias='id'),
        string('name', alias='name'),
        floating_point('minArea', alias='minArea', required=False),
    ], required=False)

    euclide_linkset_proc = user_object('EuclideLinkset', EuclideLinkset, [ 
        string('name', alias='name'),
        string('habitat', attribute='reference', alias='habitat', required=False),
        string('habitat', attribute='class', alias='classHabitat'),
        array(
            string('org.thema.graphab.dataset.VectorDataset', required=False, attribute='reference'),
            alias='datasets', nested='habitat/habitats'),
        array(
            string('org.thema.graphab.habitat.MonoVectorHabitat', required=False, attribute='reference'),
            alias='monoVectorHabitats', nested='habitat/habitats'),
        array(
            string('Habitat', required=False, attribute='reference')
            ,alias='habitats', nested='habitat/habitats'),
        boolean('inter', alias='inter'),
        string('topology', alias='topology'),
        array(floating_point('double', required=False), alias='costs', nested='costs'),
        boolean('realPaths', alias='realPaths'),
        floating_point('coefSlope', alias='coefSlope', required=False),
        floating_point('distMax', alias='distMax'),
        string('extCostFile', alias='extCostFile', required=False),
        boolean('optimCirc', alias='optimCirc', required=False)
    ], required=False)

    cost_linkset_proc = user_object('CostLinkset', CostLinkset, [ 
        string('name', alias='name'),
        string('habitat', attribute='reference', alias='habitat', required=False),
        string('habitat', attribute='class', alias='classHabitat'), 
        array(
            string('org.thema.graphab.dataset.VectorDataset', required=False, attribute='reference'),
            alias='datasets', nested='habitat/habitats'),
        array(
            string('org.thema.graphab.habitat.MonoVectorHabitat', required=False, attribute='reference'),
            alias='monoVectorHabitats', nested='habitat/habitats'),
        array(
            string('Habitat', required=False, attribute='reference')
            ,alias='habitats', nested='habitat/habitats'),
        boolean('inter', alias='inter'),
        string('topology', alias='topology'),
        array(floating_point('double', required=False), alias='costs', nested='costs'),
        boolean('realPaths', alias='realPaths'),
        boolean('removeCrossPatch', alias='removeCrossPatch'),
        floating_point('coefSlope', alias='coefSlope', required=False),
        floating_point('distMax', alias='distMax'),
        string('extCostFile', alias='extCostFile', required=False),
        boolean('optimCirc', alias='optimCirc', required=False)
    ], required=False)

    circuit_linkset_proc = user_object('CircuitLinkset', CircuitLinkset, [ 
        string('name', alias='name'),
        string('habitat', attribute='reference', alias='habitat', required=False),
        string('habitat', attribute='class', alias='classHabitat'), 
        array(
            string('org.thema.graphab.dataset.VectorDataset', required=False, attribute='reference'),
            alias='datasets', nested='habitat/habitats'),
        array(
            string('org.thema.graphab.habitat.MonoVectorHabitat', required=False, attribute='reference'),
            alias='monoVectorHabitats', nested='habitat/habitats'),
        array(
            string('Habitat', required=False, attribute='reference')
            ,alias='habitats', nested='habitat/habitats'),
        boolean('inter', alias='inter'),
        string('topology', alias='topology'),
        array(floating_point('double', required=False), alias='costs', nested='costs'),
        boolean('realPaths', alias='realPaths'),
        floating_point('coefSlope', alias='coefSlope', required=False),
        floating_point('distMax', alias='distMax'),
        string('extCostFile', alias='extCostFile', required=False),
        boolean('optimCirc', alias='optimCirc', required=False)
    ], required=False)


    alpha_param_proc = user_object('alphaParam', AlphaParam, [ 
        boolean('hasBeta', alias='hasBeta'),
        floating_point('alpha', alias='alpha'),
        floating_point('d', alias='d'),
        floating_point('p', alias='p'),
        floating_point('beta', alias='beta', required=False),
        string('maxD', alias='maxD'),
        floating_point('minP', alias='minP')
    ], required=False)

    local_metric_proc = user_object('org.thema.graphab.metric.local.LocalMetricResult', MetricResult, [ 
        string('name', alias='name'),
        string('metric', alias='metricKind', attribute='class'),
        user_object('metric', Metric, [ 
            alpha_param_proc
        ], alias='metric'),
        string('graph', attribute='reference'),
        string('method', alias='method'),
        dictionary('attrName', [string('string')], required=False)
    ],required=False)

    global_metric_proc = user_object('org.thema.graphab.metric.global.GlobalMetricResult', GlobalMetricResult, [ 
        string('name', alias='name'),
        string('metric', alias='metricKind', attribute='class'),
        user_object('metric', Metric, [ 
            alpha_param_proc,
            dictionary('result', [floating_point('double')], required=False),
            dictionary('cum', [floating_point('double')], required=False),
            dictionary('index', [dictionary('int-array', [array(integer('int', required=False))])], required=False), #
        ], alias='metric'),
        string('graph', attribute='reference', required=False),
        string('method', alias='method'),
        string('result', alias='result', attribute='reference', required=False),
        dictionary('result', [floating_point('double', required=False)], required=False),
    ],required=False)


    graph_proc = user_object('Graph', Graph, [
        string('name', alias='name'),
        array(local_metric_proc, alias="metrics", nested="metrics/entry"),
        array(global_metric_proc, alias="globalMetrics", nested="metrics/entry"),
        string('linkset', attribute='reference', alias='linkset'),
        floating_point('threshold', alias='threshold'),
        boolean('intraPatchDist', alias='intraPatchDist'),
    ], required=False)

    multigraph_proc = user_object('MultiGraph', MultiGraph, [ 
            string('name', alias='name'), 
            array(local_metric_proc, alias="metrics", nested="metrics/entry"),
            array(global_metric_proc, alias="globalMetrics", nested="metrics/entry"),
            boolean('intraPatchDist', alias='intraPatchDist'),
            array(string('Graph', attribute='reference', required=False), alias='graphs', nested='graphs'),
        ], required=False)

    zone_proc = user_object('zone', Zone, [
            floating_point('x', alias='x'),
            floating_point('y', alias='y'),
            floating_point('width', alias='width'),
            floating_point('height', alias='height')
        ])

    project_proc = user_object('Project', Project, [
        string('name', alias='name'),
        string('version', alias='graphabVersion'),
        array(integer('int'), alias='codes', nested='codes'),
        floating_point('noData', alias='noData'),
        floating_point('resolution', alias='resolution'),
        user_object('grid2space', Grid2space, [
            floating_point('m00', alias='m00'),
            floating_point('m01', alias='m01'),
            floating_point('m02', alias='m02'),
            floating_point('m10', alias='m10'),
            floating_point('m11', alias='m11'),
            floating_point('m12', alias='m12')
        ], alias='grid2space'),
        user_object('space2grid', Space2grid, [
            floating_point('m00', alias='m00'),
            floating_point('m01', alias='m01'),
            floating_point('m02', alias='m02'),
            floating_point('m10', alias='m10'),
            floating_point('m11', alias='m11'),
            floating_point('m12', alias='m12')
        ], alias='space2grid'),
        array(mono_habitat_proc, alias='monoHabitats', nested='habitats/entry'),
        array(vector_habitat_proc, alias='monoVectorHabitats', nested='habitats/entry'),
        array(vector_datasets_proc, alias="vectorDatasets", nested='habitats/entry'),
        array(euclide_linkset_proc, alias='euclideLinksets', nested='linksets/entry'),
        array(cost_linkset_proc, alias='costLinksets', nested='linksets/entry'),
        array(circuit_linkset_proc, alias='circuitLinksets', nested='linksets/entry'),
        array(graph_proc, alias='graphs', nested="graphs/entry"),
        array(multigraph_proc, alias='multigraphs', nested='graphs/entry'),
        array(zone_proc, alias='zone'),
        string('demFile', alias='demFile', required=False),
        string('wktCRS', alias='wktCRS', required=False)
    ])

    return parse_from_string(project_proc, xml_string)


def convertXmlToPy(filepath: str) -> Project:
    """Main function of this xml_to_python module to use,
    it translates an XML Graphab3+ project file into a graphab4qgis project

    this function first parses the xml file and instanciate a Project, 
    reorders every elements depending on their position in the file
    then links each one of them that are in relationship with others together

    :param filepath: the project file path on the user's drive.

    :raises GraphabVersionTooLowException: raised if the XML file comes from a version
        of Graphab < 3, which is not compatible with this version of the plugin.

    :return: the graphab4qgis project object translated from the XML Graphab3+ project file.
    :rtype: Project
    """
    file = open(filepath)
    xml_string = file.read()
    file.close()

    version_verif = parse_from_string(dictionary('Project', [string('version', required=False)]), xml_string)
    if version_verif['version'] == '':
        raise GraphabVersionTooLowException()

    p = parseXmlFile(xml_string)

    # regroup all the linksets in one single list
    p.linksets += p.euclideLinksets + p.costLinksets + p.circuitLinksets

    reorderProjectLinksets(p, filepath)
    for linkset in p.linksets:
        linkset.habitats += linkset.datasets + linkset.monoVectorHabitats
    # regroup all the habitats in one single list
    p.habitats += p.monoHabitats + p.monoVectorHabitats + p.vectorDatasets
    reorderProjectHabitats(p, filepath)

    # link every elements of the project that have a relation with another object to it
    linkHabitatToLinkset(p)
    linkLinksetToGraph(p) 
    linkGraphsToMultiGraph(p)

    # update the name of the linkset's habitat
    for linkset in p.linksets:
        linkset.updateHabitatName()

    # regroup the graphs with the multigraphs together
    p.graphs += p.multigraphs 

    linkGraphToMetric(p)

    p.localMetrics = [metric for graph in p.graphs for metric in graph.metrics]
    p.metrics = p.localMetrics + [metric for graph in p.graphs for metric in graph.globalMetrics]

    for multigraph in p.multigraphs:
        multigraph.updateHabitats()

    return p


