# -*- coding: utf-8 -*-
"""
 GraphabStyle
"""
import os
import warnings # This is is needed in the pyqgis console also

from qgis.core import QgsFillSymbol, QgsLineSymbol, QgsMapLayer, QgsGradientStop, NULL
from PyQt5.QtGui import QColor
# Import to create tempory files
import tempfile

from .utils import getLayerMinMaxValue

__author__ = 'Gaspard QUENTIN, Robin MARLIN-LEFEBVRE'
__email__ = 'gaspard.quentin1905@gmail.com'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

class GraphabStyleLoadingFailed(Exception):
    pass

class GraphabStyle:
    """
    This class is used to apply styles to the layers of the project.
    """

    def __init__(self, plugin):
        # Link the object to the main object (GraphabPlugin) for shared variables
        self.plugin = plugin
        self.habitatColors = ["#65A252", "#556450", "#aac8a1", "#809679", "#d5fac9",
                              "#6ec852", "#376429", "#8afa67", "#53963e",
                              "#50c82b", "#286416", "#64fa36", "#3c9620"]
    #--------------------------------------------------------------------------

    def prepare_style(self, fieldname: str, layer: QgsMapLayer, style_name: str, min_scale: int = 1, max_scale: int = 15, size_capacity: bool = True, min_value: int = -1, max_value: int = -1, habitat_id: int = -1):
        """Edit pre-created symbology files with parameters choose by the user to change the 
        symbology of a choosen Graph. The symbology file created is not saved because it use temporaly 
        files."""

        # get path of style file
        _input = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'styles', style_name + '.qml')

        #read input file
        with open(_input, "rt") as fin:
            #read file contents to string
            data = fin.read()
            # get the field of a layer with his name

            if min_value == 1 and max_value == -1:
                min_value, max_value = getLayerMinMaxValue([layer], fieldname)

            if min_value == NULL or max_value == NULL:
                raise GraphabStyleLoadingFailed()

            # avoid having a display problem because the min and max are identical
            if min_value >= max_value:
                min_value = 0

            if max_scale <= 1:
                max_scale = 15
            if min_scale >= max_scale or min_scale < 0:
                min_scale = 1

            #if the checkbox is checked : the size of centroid is based on the Capacity
            #else it's based on the current fieldname choose by the user
            size_fieldname: str  = 'capacity' if size_capacity else fieldname
            (size_value_min, size_value_max) = (min_value, max_value) if size_capacity else getLayerMinMaxValue([layer], 'capacity')

            # avoid having a display problem because the min and max are identical
            if size_value_min >= size_value_max:
                size_value_min = 0

            #replace all occurrences of the required string
            for i in range(0,7):
                data = data.replace('@VAL'+str(i)+'@', str(min_value+i*(max_value-min_value)/6))

            data = data.replace('@MINIMUM2@', str(size_value_min))
            data = data.replace('@MAXIMUM2@', str(size_value_max))
            data = data.replace('@FIELDNAME1@', fieldname)
            data = data.replace('@FIELDNAME2@', size_fieldname)
            data = data.replace('@MINSCALE@', str(min_scale))
            data = data.replace('@MAXSCALE@', str(max_scale))
            #close the input file
            fin.close()
            #open the a temporary output file in write and read mode
            _output = tempfile.NamedTemporaryFile(mode = 'w+', dir = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'styles'), prefix = 'temp', suffix = '.qml', delete = False)
            #overrite the output file with the resulting data
            _output.write(data)
            #close the file
            _output.close()

            #load the syle file
            self.loadStyle(layer, _output.name)

            #remove temporary file
            os.remove(_output.name)
        if habitat_id != -1: # case we want to color the graph in the color of it's habitat
            self.uniColorStyle(QColor(self.getHabitatColors(habitat_id)), layer)
            layer.triggerRepaint()
            self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())

    #--------------------------------------------------------------------------

    def loadStyle(self, layer, styleFile):
        """Change the symbology of a layer with a pre-created symbology file"""

        layer.loadNamedStyle(os.path.realpath(styleFile))

        # refresh
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())

    #--------------------------------------------------------------------------

    def colorizeLine(self, layer, color):
        """Change color of the layer with the color asked. 
        This function works only with layer which represents lines."""

        # create a new symbol
        symbol = QgsLineSymbol.createSimple({'line_style': 'solid', 'color': color})

        # apply symbol to layer renderer
        layer.renderer().setSymbol(symbol)

        # repaint the layer
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())

    #--------------------------------------------------------------------------

    def colorizeFill(self, layer, color):
        """Change color of the layer with the color asked.
        This function works only with layer which represents things like circle or square"""

        # create a new symbol
        symbol = QgsFillSymbol.createSimple({'line_style': 'solid', 'color': color})

        # apply symbol to layer renderer
        layer.renderer().setSymbol(symbol)

        # repaint the layer
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())


    #--------------------------------------------------------------------------
    def getHabitatColors(self, index: int):
        i = index if index < len(self.habitatColors) else index%len(self.habitatColors)
        return self.habitatColors[i]

    #--------------------------------------------------------------------------
    def colorizeHabitatLayer(self, habitatLayer: QgsMapLayer, index: int):
        self.colorizeFill(habitatLayer, self.getHabitatColors(index))

    #--------------------------------------------------------------------------
    def uniColorStyle(self, color: QColor, layer: QgsMapLayer) -> None:
        """ 
            Colorize the symbology of the layer with only one color.
            the layer's style should already have been prepared with .prepare_style()
        """
        layer.renderer().sourceColorRamp().setColor1(color)
        layer.renderer().sourceColorRamp().setColor2(color)
        stops = [QgsGradientStop(1, color) for _ in range(len(layer.renderer().sourceColorRamp().stops()))]
        layer.renderer().sourceColorRamp().setStops(stops)
        layer.renderer().updateColorRamp()
