# -*- coding: utf-8 -*-

__author__ = 'gaspard.quentin1905@gmail.com'
__date__ = '2024-05-06'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

import os
import processing


from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QFileDialog
from PyQt5.QtWidgets import QGridLayout, QHeaderView, QTableWidgetItem, QMessageBox
from PyQt5.QtCore import Qt, QCoreApplication

from qgis.core import QgsProcessing, QgsMapLayer, QgsRasterLayer, QgsVectorLayer

from ..utils import NoFeedback

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'ui_files', 'create_habitat_dialog.ui'))

class CreateHabitatDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, plugin) -> None:
        super(CreateHabitatDialog, self).__init__(None) 
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        self.plugin = plugin
        
        # EXTREMELY IMPORTANT VARIABLE IF windowProcess NOT STOCK HE CAN'T USE POSTPROCESSALGORITHM()
        self.windowProcess = None

        self.projectComboBox.currentIndexChanged.connect(self.select_project)
        self.rasterRadio.clicked.connect(lambda : self.hideTab(1))
        self.vectorRadio.clicked.connect(lambda : self.hideTab(0))
        self.fromFileRadio.clicked.connect(lambda : self.disableSourceWidget(False))
        self.fromHabitatRadio.clicked.connect(lambda : self.disableSourceWidget(True))


        self.vectorTabWidgets = [
                self.vectorMiniCapaSpinBox,
                self.vectorCapacityComboBox]

        self.vectorMiniCapaCheckBox.clicked.connect(lambda: self.vectorMiniCapaSpinBox.setEnabled(self.vectorMiniCapaCheckBox.isChecked()))
        self.vectorCapaFieldCheckBox.clicked.connect(lambda : self.vectorCapacityComboBox.setEnabled(self.vectorCapaFieldCheckBox.isChecked()))


        self.maxUniquesCodePossible = 1000

        self.rasterMinAreaCB.clicked.connect(lambda: self.rasterMinAreaSB.setEnabled(self.rasterMinAreaCB.isChecked()))
        self.rasterMaxSizeCB.clicked.connect(lambda: self.rasterMaxSizeSB.setEnabled(self.rasterMaxSizeCB.isChecked()))
        self.habitatComboBox.currentIndexChanged.connect(lambda : self.populateCapacityComboBox(self.habitatComboBox.currentText()))
        self.habitatFileWidget.fileChanged.connect(lambda: self.populateCapacityComboBox(self.habitatFileWidget.filePath()))
        self.codesTable.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)

        self.vectorLayout = QGridLayout(self.vector_tab)
    
    def disableSourceWidget(self, disable_file: bool):
        self.habitatComboBox.setEnabled(disable_file)
        self.habitatFileWidget.setEnabled(not disable_file)
        self.fileLabel.setEnabled(not disable_file)
        self.habitatLabel.setEnabled(disable_file)
        if disable_file:
            project = self.plugin.getProject(self.projectComboBox.currentText())
            self.habitatComboBox.addItems([habitat.name for habitat in project.project.habitats])
        self.vectorMiniCapaCheckBox.setEnabled(disable_file)
        self.vectorCapaFieldCheckBox.setEnabled(True)
        

    # --------------------------------------------------------------------------
    def populateCapacityComboBox(self, habitatOrFile: str) -> None:

        while (count := self.vectorCapacityComboBox.count()) > 1:
            self.vectorCapacityComboBox.removeItem(count-1)

        if habitatOrFile == '':
            return

        if self.fromHabitatRadio.isChecked():
            # populate the combobox from habitat name
            layer: QgsMapLayer = self.projects[self.projectComboBox.currentIndex()].getHabitatLayer(habitatOrFile)
            if layer is None:
                return
            if isinstance(layer, QgsRasterLayer):
                return
            self.vectorCapacityComboBox.addItems([field.name() for field in layer.fields()])
        else:
            if not os.path.isfile(habitatOrFile):
                return
            layer: QgsVectorLayer = QgsVectorLayer(habitatOrFile)
            if layer is None:
                return
            self.vectorCapacityComboBox.addItems([field.name() for field in layer.fields()])


    # --------------------------------------------------------------------------
    def showDialog(self):
        """Initialize basic window and waiting for any action of the user"""
        if not self.plugin.java:
            self.plugin.showJavaNotFound()
            return
        # upgrade the list of loaded project
        self.projects = self.plugin.getProjects()
        if len(self.projects) == 0:
            QMessageBox.information( 
                None,
                self.translate('no_project', 'ERROR: no project opened'),
                self.translate('no_project', 'please open or create a project')
            )
            return

        self.rasterRadio.setChecked(True)
        self.hideTab(1)

        self.projectComboBox.clear()
        self.projectComboBox.addItems([prj.project.name for prj in self.projects])
        self.adjustSize()
        self.show()
        
        self.habitatComboBox.setEnabled(False)
        self.habitatFileWidget.setEnabled(False)
        self.fileLabel.setEnabled(False)
        self.habitatLabel.setEnabled(False)
#
        self.vectorMiniCapaCheckBox.setEnabled(False)
        self.vectorCapaFieldCheckBox.setEnabled(False)
        self.vectorMiniCapaSpinBox.setEnabled(False)
        self.vectorCapacityComboBox.setEnabled(False)
        self.rasterMinAreaSB.setEnabled(False)
        self.rasterMaxSizeSB.setEnabled(False)
        for widget in self.vectorTabWidgets:
            widget.setEnabled(False)

        self.select_project(self.projectComboBox.currentIndex())
        self.adjustSize()


        while self.exec_() and not self.createHabitat():
            pass

    # --------------------------------------------------------------------------
    def select_project(self, index):
        """Open a window to select a project if there is no project selected yet and
        initialize some items with project values"""
        if not self.isVisible():
            return
        project = self.projects[index].project

        # find a non-use linkset name
        basicName = "newHabitat"
        counter = 1
        while (basicName + str(counter)) in [habitat.name for habitat in project.habitats]:
            counter += 1

        # set the default graph name found
        self.nameLineEdit.setText(basicName + str(counter))
        self.populateCodesTable(self.projects[index].getSource().source())

    # --------------------------------------------------------------------------
    def getUniquesValuesFromRaster(self, raster):
        params = {
            "INPUT": raster,
            "BAND": 1,
            "OUTPUT_TABLE": QgsProcessing.TEMPORARY_OUTPUT
        }

       
        vectorLayer = processing.run("native:rasterlayeruniquevaluesreport", params, feedback=NoFeedback())["OUTPUT_TABLE"]

        params = {
            "INPUT": vectorLayer,
            "FIELDS": 'value',
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT
        }
        stringValues = processing.run("qgis:listuniquevalues", params, feedback=NoFeedback())["UNIQUE_VALUES"]
        tabValues = stringValues.split(';')
        if tabValues.count("\"") == 0: #for a strange reason, when there is no result, tabValues == ["\""]
            return sorted(tabValues, key=lambda x: int(float(x)))
        return []

    # --------------------------------------------------------------------------
    def hideTab(self, to_disable: int) -> None:
        if to_disable > 1 or to_disable < 0:
            return
        to_enable = 0 if to_disable == 1 else 1
        if self.tabWidget.isTabEnabled(to_enable):
            if to_enable == 1:
                self.vectorRadio.setChecked(True)
            else:
                self.rasterRadio.setChecked(True)
        self.tabWidget.setTabEnabled(to_enable, True)
        self.tabWidget.setTabEnabled(to_disable, False)

    # --------------------------------------------------------------------------
    def createHabitat(self) -> bool:
        """Get all parameters set by the user and send them to the processing to create an Habitat"""
        alg = ''
        input = self.projects[self.projectComboBox.currentIndex()].projectFile
        params = {
            'INPUT': input,
            'HABITATNAME': self.nameLineEdit.text(),
        }
        if self.rasterRadio.isChecked():
            alg = 'graphab3:create_raster_habitat'
            if len(self.codesTable.selectedItems()) == 0:
                QMessageBox.critical(
                    None,
                    "",
                    self.translate('create_habitat_dialog', "ERROR: please select habitat codes"),
                )
                return False
            params['CODES'] = ''.join(item.text() + ',' for item in self.codesTable.selectedItems())[:-1]
            if self.rasterMinAreaCB.isChecked():
                params['MINAREA'] = self.rasterMinAreaSB.value()
            if self.rasterMaxSizeCB.isChecked():
                params['MAXSIZE'] = self.rasterMaxSizeSB.value()
        else:
            if self.fromFileRadio.isChecked():
                alg = 'graphab3:create_vector_habitat_from_file'
                params['FILE'] = self.habitatFileWidget.lineEdit().text()
            else:
                alg = 'graphab3:create_vector_habitat_from_habitat'
                params['HABITATSRC'] = self.habitatComboBox.currentText()
            if self.vectorMiniCapaCheckBox.isChecked():
                params['MINCAPA'] = self.vectorMiniCapaSpinBox.value()
            if self.vectorCapaFieldCheckBox.isChecked() and (text := self.vectorCapacityComboBox.currentText()) != '(none)':
                params['CAPA'] = text

        self.windowProcess = processing.createAlgorithmDialog(alg, params)
        self.windowProcess.show()
        self.windowProcess.runAlgorithm()
        return True

    # --------------------------------------------------------------------------
    def populateCodesTable(self, raster=None):
        """Initialize the encode tab with the selected raster, if raster is None it clears the tab"""
        #self.codesTable.setRowCount(0)

        if raster is None:
            return

        unique_values = self.getUniquesValuesFromRaster(raster)
        if len(unique_values) > self.maxUniquesCodePossible:
            QMessageBox.information(
                None,
                self.translate('osraster_raster', "ERROR : Raster unique codes"),
                self.translate('osraster_raster', "This raster has been ignored because there is too much unique code values. (MAX=%d)") % self.maxUniquesCodePossible
            )
            self.resetNoFile()
            return

        # Add a number of row depending of the number of unique values
        self.codesTable.setRowCount(len(unique_values))
        self.codesTable.setColumnCount(1)

        # Fill lines
        counter = 0
        for code in unique_values:
            intCode = int(float(code))
            item = QTableWidgetItem(str(intCode))
            item.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
            
            self.codesTable.setItem(counter, 0, item)
            counter += 1

    # --------------------------------------------------------------------------
    def translate(self, context, string):
        return QCoreApplication.translate(context, string)

