# -*- coding: utf-8 -*-
"""
 GraphabStyle
"""
import os # This is is needed in the pyqgis console also

from qgis.core import QgsFillSymbol, QgsLineSymbol, QgsMapLayer, QgsGradientStop
from PyQt5.QtGui import QColor
# Import to create tempory files
import tempfile

__author__ = 'Gaspard QUENTIN, Robin MARLIN-LEFEBVRE'
__email__ = 'gaspard.quentin1905@gmail.com'
__copyright__ = 'Copyright 2024, Laboratoire ThéMA'

class GraphabStyle:
    """
    This class is used to apply styles to the layers of the project.
    """

    def __init__(self, plugin):
        # Link the object to the main object (GraphabPlugin) for shared variables
        self.plugin = plugin
        self.habitatColors = ["#65A252", "#556450", "#aac8a1", "#809679", "#d5fac9",
                            "#6ec852", "#376429", "#8afa67", "#53963e",
        "#50c82b", "#286416", "#64fa36", "#3c9620"]
    #--------------------------------------------------------------------------
    
    def prepareStyle(self, fieldname, layer, styleName, minScale = None, maxScale = None, sizeCapacity = True, minValue=None, maxValue=None, habitatID = -1):
        """Edit pre-created symbology files with parameters choose by the user to change the 
        symbology of a choosen Graph. The symbology file created is not saved because it use temporaly 
        files."""



        # get path of style file
        _input = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'styles', styleName + '.qml')

        #read input file
        with open(_input, "rt") as fin:
            #read file contents to string
            data = fin.read()
            # get the field of a layer with his name
            min = -1
            max = -1
            if not minValue and not maxValue:
                idx = layer.fields().indexFromName(fieldname)
                if idx == -1:
                    raise Exception(f'field {fieldname} not found' + "\n" + str([e.name() for e in layer.fields()]))
                # get min and max of the field wanted
                min = layer.minimumValue(idx)
                max = layer.maximumValue(idx)
            else:
                min = minValue
                max = maxValue


            # avoid having a display problem because the min and max are identical
            if min >= max:
                min = 0
        
            if maxScale == None or int(maxScale) <= 1:
                maxScale = 15
                
            if minScale == None or int(minScale) >= int(maxScale) or int(minScale) < 0:
                minScale = 1
                
            #if the checkbox is checked : the size of centroid is based on the Capacity
            #else it's based on the current fieldname choose by the user

            sizeValueMin = -1
            sizeValueMax = -1
            sizeFieldname = ''
            if sizeCapacity:
                sizeFieldname = 'capacity'
                # get the field of a layer with his name
                idx = layer.fields().indexFromName(sizeFieldname)
                if idx == -1:
                    raise Exception('field capacity not found')
                # get min and max of the field wanted
                sizeValueMin = layer.minimumValue(idx)
                sizeValueMax = layer.maximumValue(idx)
            else:
                sizeFieldname = fieldname
                sizeValueMin = min
                sizeValueMax = max

                        # avoid having a display problem because the min and max are identical
            if sizeValueMin >= sizeValueMax:
                sizeValueMin = 0
            
            #replace all occurrences of the required string
            for i in range(0,7):
                data = data.replace('@VAL'+str(i)+'@', str(min+i*(max-min)/6))

            data = data.replace('@MINIMUM2@', str(sizeValueMin))
            data = data.replace('@MAXIMUM2@', str(sizeValueMax))
            data = data.replace('@FIELDNAME1@', fieldname)
            data = data.replace('@FIELDNAME2@', sizeFieldname)
            data = data.replace('@MINSCALE@', str(minScale))
            data = data.replace('@MAXSCALE@', str(maxScale))
            #close the input file
            fin.close()
            #open the a temporary output file in write and read mode
            _output = tempfile.NamedTemporaryFile(mode = 'w+', dir = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'styles'), prefix = 'temp', suffix = '.qml', delete = False)
            #overrite the output file with the resulting data
            _output.write(data)
            #close the file
            _output.close()
            
            #load the syle file
            self.loadStyle(layer, _output.name)
            
            #remove temporary file
            os.remove(_output.name)
        if habitatID != -1: # case we want to color the graph in the color of it's habitat
            self.uniColorStyle(QColor(self.getHabitatColors(habitatID)), layer)
            layer.triggerRepaint()
            self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())
    
    #--------------------------------------------------------------------------
    
    def loadStyle(self, layer, styleFile):
        """Change the symbology of a layer with a pre-created symbology file"""

        layer.loadNamedStyle(os.path.realpath(styleFile))
            
        # refresh
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())
    
    #--------------------------------------------------------------------------
    
    def colorizeLine(self, layer, color):
        """Change color of the layer with the color asked. 
        This function works only with layer which represents lines."""
        
        # create a new symbol
        symbol = QgsLineSymbol.createSimple({'line_style': 'solid', 'color': color})
         
        # apply symbol to layer renderer
        layer.renderer().setSymbol(symbol)
         
        # repaint the layer
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())
    
    #--------------------------------------------------------------------------
    
    def colorizeFill(self, layer, color):
        """Change color of the layer with the color asked.
        This function works only with layer which represents things like circle or square"""
        
        # create a new symbol
        symbol = QgsFillSymbol.createSimple({'line_style': 'solid', 'color': color})
         
        # apply symbol to layer renderer
        layer.renderer().setSymbol(symbol)
         
        # repaint the layer
        layer.triggerRepaint()
        self.plugin.iface.layerTreeView().refreshLayerSymbology(layer.id())


    #--------------------------------------------------------------------------
    def getHabitatColors(self, index: int):
        i = index if index < len(self.habitatColors) else index%len(self.habitatColors)
        return self.habitatColors[i]

    #--------------------------------------------------------------------------
    def colorizeHabitatLayer(self, habitatLayer: QgsMapLayer, index: int):
        self.colorizeFill(habitatLayer, self.getHabitatColors(index))
    
    #--------------------------------------------------------------------------
    def uniColorStyle(self, color: QColor, layer: QgsMapLayer) -> None:
        """ 
            Colorize the symbology of the layer with only one color.
            the layer's style should already have been prepared with .prepareStyle()
        """
        layer.renderer().sourceColorRamp().setColor1(color)
        layer.renderer().sourceColorRamp().setColor2(color)
        stops = [QgsGradientStop(1, color) for _ in range(len(layer.renderer().sourceColorRamp().stops()))]
        layer.renderer().sourceColorRamp().setStops(stops)
        layer.renderer().updateColorRamp()
