# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeoportalRlpMetadataSearch
                                 A QGIS plugin
 GeoPortal.rlp metadata search module
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-02-09
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Armin Retterath
        email                : armin.retterath@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import Qt, QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon, QPixmap
#from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtWidgets import *
# some things for doing http requests
from qgis.PyQt.QtCore import QUrl, QEventLoop, QUrlQuery
from qgis.PyQt.QtNetwork import QNetworkRequest
from qgis.core import Qgis, QgsNetworkAccessManager, QgsMessageLog, QgsSettings, QgsProviderRegistry, QgsBrowserModel
from qgis.core import QgsProject, QgsRasterLayer
from qgis.gui import QgsMessageBar, QgsGui, QgsMessageViewer
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .gprlp_metadata_search_dialog import GeoportalRlpMetadataSearchDialog
import os.path
import urllib
import requests
# some libs for json parsing under python3
import json
import re
import math
from types import SimpleNamespace


class GeoportalRlpMetadataSearch:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'gprlp_metadata_search_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&GeoPortal.rlp Metadata Search')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # read all settings from qgis settings
        self.settings = QgsSettings()

        # read actual browser model
        self.browser_model = QgsBrowserModel()

        # definition of searchResources
        self.search_resources = [
            "wmc",
            "wms"
        ]

        # definition of searchResources new
        self.search_resources_list = [
            {
                "title": self.tr("Dataset"),
                "value": "dataset",
                },
            {
                "title": self.tr("Map Layer"),
                "value": "wms",
                },
            {
                "title": self.tr("OWS Context"),
                "value": "wmc",
                },
            {
                "title": self.tr("Remote CSW"),
                "value": "csw",
                },
            {
                "title": self.tr("MrMap Context"),
                "value": "owscontext",
                },
        ]
        # TODO - remote CWS only for test purposes
        # definition of catalogue uri
        self.search_catalogues = [
            {
                "title": self.tr("Rhineland-Palatinate"),
                "value": "https://www.geoportal.rlp.de",
                },
            {
                "title": self.tr("Hesse"),
                "value": "https://www.geoportal.hessen.de",
                },
            {
                "title": self.tr("Saarland"),
                "value": "https://geoportal.saarland.de",
                },
        ]

        #network access
        self.na_manager = QgsNetworkAccessManager.instance()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GeoportalRlpMetadataSearch', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(os.path.dirname(__file__), "mb.png")
        # icon_path = ':/plugins/gprlp_metadata_search/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'GeoPortal.rlp metadata search'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(
                self.tr(u'&GeoPortal.rlp Metadata Search'),
                action)
            self.iface.removeToolBarIcon(action)


    def clean_ows_url(self, url):
        """clean an OWS URL of added basic service parameters"""
        url = QUrl(url)
        query_string = url.query()
        if query_string:
            query_string = QUrlQuery(query_string)
            query_string.removeQueryItem('service')
            query_string.removeQueryItem('SERVICE')
            query_string.removeQueryItem('request')
            query_string.removeQueryItem('REQUEST')
            url.setQuery(query_string)
        return url.toString()


    def add_ows(self, data_url, given_service_type="wms"):

        # from MetaSearch Plugin - thx to tom kralidis
        # https://github.com/qgis/QGIS/blob/master/python/plugins/MetaSearch/dialogs/maindialog.py#L708
        #
        # from https://github.com/qgis/QGIS/blob/4ff41e71ad7a397ca05af4186ea0339af81752cc/python/plugins/MetaSearch/util.py
        # QgsMessageLog.logMessage("Url to clean: " + data_url, 'GeoPortal.rlp search', level=Qgis.Info)
        """
        problem: qgis 3.30 implements a new QgsSettingsTree - registrating a source works not as it works before !

        """

        QgsMessageLog.logMessage("Load WMS cleaned url: " + self.clean_ows_url(data_url), 'GeoPortal.rlp search', level=Qgis.Info)

        qgis_version = Qgis.versionInt()

        if qgis_version >= 33000:
            # QgsMessageLog.logMessage("QGIS version is greater than 3.3 : " + Qgis.version() + " registrating resources in browser won't work as expected!", 'GeoPortal.rlp search', level=Qgis.Info)
            from qgis.core import QgsSettingsTree
            if given_service_type == "wms":
                service_type = 'OGC:WMS/OGC:WMTS'
                sname = 'WMS'
                dyn_param = ['wms']
                provider_name = 'wms'
                setting_node = QgsSettingsTree.node('connections').childNode('ows').childNode('connections')
            if given_service_type == "wfs":
                service_type = 'OGC:WFS'
                sname = 'WFS'
                dyn_param = ['wfs']
                provider_name = 'WFS'
                setting_node = QgsSettingsTree.node('connections').childNode('ows').childNode('connections')

            service_name = 'Result from GeoPortal.rlp search plugin' # % service_type[1]
            conn_name_matches = []
            dyn_param.append(service_name)
            setting_node.childSetting('url').setValue(self.clean_ows_url(data_url), dyn_param)
            browser_model = self.iface.browserModel()
            #browser_model.initialize()
            browser_model.reload()

        if qgis_version < 33000:
            if given_service_type == "wms":
                service_type = ['OGC:WMS/OGC:WMTS', 'wms', 'wms']
                provider_name = 'WMS'
            if given_service_type == "wfs":
                service_type = ['OGC:WFS', 'wfs', 'WFS']
                provider_name = 'WFS'
            service_name = 'Result from GeoPortal.rlp search plugin' # % service_type[1]
            conn_name_matches = []
            # store connection in browser
            # check if there is a connection with same name
            self.settings.beginGroup('/qgis/connections-%s' % service_type[1])
            keys = self.settings.childGroups()
            self.settings.endGroup()
            for key in keys:
                #QgsMessageLog.logMessage("key of childGroups: " + key)
                if key.startswith(service_name):
                    QgsMessageLog.logMessage("found key in childGroups: " + key)
                    conn_name_matches.append(key)
            if conn_name_matches:
                service_name = conn_name_matches[-1]
            QgsMessageLog.logMessage("Service name to register: " + service_name)
            # check for duplicates
            """
            if service_name in keys:  # duplicate found
                msg = self.tr('Connection {0} exists. Overwrite?').format(service_name)
                res = QMessageBox.warning(
                    None, self.tr('Saving server'), msg,
                    QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel)
                if res == QMessageBox.No:  # assign new name with serial
                    service_name = serialize_string(service_name)
                elif res == QMessageBox.Cancel:
                    return
            """
            # no duplicates detected or overwrite is allowed
            self.settings.beginGroup('/qgis/connections-%s' % service_type[1])
            self.settings.setValue('/%s/url' % service_name + 'test', self.clean_ows_url(data_url))
            self.settings.endGroup()
            #refresh browser content - but howto reload tree?
            # only refresh the view not the model!
            # self.browser_model.initialize()

            browser_model = self.iface.browserModel()
            #browser_model.initialize()
            browser_model.reload()
            browser_dock_widget = self.iface.mainWindow().findChildren(QWidget, 'Browser')[0]
            tree = browser_dock_widget.findChildren(QTreeView)
            tree_0 = tree[0]
            proxy = tree_0.model()
            # print(proxy.rowCount()) - 16
            # only for the first level - column = 0
            """
            for row in range(proxy.rowCount()):
                index = proxy.index(row, 0)
                tree_0.expand(index)
            """
            # simply add wms layer
            # rlayer = QgsRasterLayer(clean_ows_url(data_url), 'test', 'wms')
            # QgsProject.instance().addMapLayer(rlayer)
            # tree_0.expand(10)
            # self.iface.mainWindow().findChildren(QWidget, 'Browser')[0].refresh()
            # browser_model = self.iface.browserModel()
            # browser_model.reload()
            # browser_widget = self.iface.mainWindow().findChildren(QWidget, 'Browser')[0]
            # get child with id of wms ... and setExpanded https://doc.qt.io/qt-5/qtreeview.html#setExpanded
            # QTreeView::setTreePosition(int index)
            # open wms part?
            # self.browser_model.initialize()
            # QgsGui.QgsBrowserDockWidget.initialize()
            # QgsGui.
        output = QgsMessageViewer()
        output.setFixedSize(500,200)
        output.setTitle(self.tr("Resource registration"))
        output_text = "%s Resource added to QGIS Browser under 'Result from GeoPortal.rlp search plugin'. You may open the Browser and add it to your Layers." % provider_name
        output.setMessageAsPlainText(output_text)
        output.showMessage()
        return

        # wms_dlg = QgsProviderRegistry.instance()
        # wms_dlg.show()
        # provider_list = QgsGui.providerGuiRegistry.providerList()
        # for provider in provider_list:
        #    QgsMessageLog.logMessage("Provider: " + provider, 'GeoPortal.rlp search', level=Qgis.Info)
        """
        ows_provider = QgsGui.sourceSelectProviderRegistry().\
            createSelectionWidget(
                service_type[2], self, Qt.Widget,
                QgsProviderRegistry.WidgetMode.Embedded)
        
        ows_provider.addRasterLayer.connect(self.iface.addRasterLayer)
        conn_cmb = ows_provider.findChild(QWidget, 'cmbConnections')
        connect = 'btnConnect_clicked'
        ows_provider.setModal(False)
        ows_provider.show()

        # open provider dialogue against added OWS
        index = conn_cmb.findText(service_name)
        if index > -1:
            conn_cmb.setCurrentIndex(index)
        getattr(ows_provider, connect)()
        """


    def on_clicked_context_layer(self, item):
        """
        /***************************************************************************
        function which will be invoked, if a layer of an ows context layertree is selected
        /***************************************************************************
        """
        self.reset_resource_view()
        selected_feature = item.data(1, 0)
        # QgsMessageLog.logMessage("Title of clicked layer: " + selected_feature.properties.title, 'GeoPortal.rlp search', level=Qgis.Info)
        # add layer title to resource view
        self.dlg.labelResourceType.setText(self.tr("Context layer"))
        try:
            context_layer_title = selected_feature.properties.title
        except:
            QgsMessageLog.logMessage("Context layer has no title!", 'GeoPortal.rlp search', level=Qgis.Info)
            return
        self.dlg.textBrowserResourceAbstract.append(context_layer_title)
        # get getmap uri from offering
        try:
            offerings = selected_feature.offerings
        except:
            QgsMessageLog.logMessage("Context layer has no offering!", 'GeoPortal.rlp search', level=Qgis.Info)
            return
        # QgsMessageLog.logMessage("Offerings: " + str(test), 'GeoPortal.rlp search', level=Qgis.Info)
        wms_capabilities = False
        for offering in offerings:
            if offering.code == "http://www.opengis.net/spec/owc-atom/1.0/req/wms":
                QgsMessageLog.logMessage("Offering with wms code found", 'GeoPortal.rlp search', level=Qgis.Info)
                for operation in offering.operations:
                    if operation.code == "GetCapabilities":
                        wms_capabilities = operation.href
                        try:
                            capabilities_url = str(operation.href)
                            self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
                            self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                            # add function to load button
                            self.dlg.pushButtonLoad.setEnabled(True)
                            self.dlg.pushButtonLoad.disconnect()
                            self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url))
                        except:
                            QgsMessageLog.logMessage("No attribute getCapabilitiesUrl for this resource",
                                                     'GeoPortal.rlp search',
                                                     level=Qgis.Info)
                    if operation.code == "GetMap":
                        QgsMessageLog.logMessage("GetMap found", 'GeoPortal.rlp search', level=Qgis.Info)
                        if wms_capabilities:
                            self.dlg.pushButtonLoad.setEnabled(True)
                            self.dlg.pushButtonLoad.disconnect()
                            self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(wms_capabilities))
                        else:
                            self.dlg.pushButtonLoad.disconnect()
                            self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(operation.href))
                        return
                    else:
                        pass
                        #self.dlg.resource_uri.setText("No WMS URI found!")


    def on_clicked_resource(self, item, type):
        """
        Function that may be usefull, if we may handle the resources in a common way. Not used now!
        """
        QgsMessageLog.logMessage("type found: " + type, 'GeoPortal.rlp search', level=Qgis.Info)
        return
        type = "wmc"
        # check for type
        if type == "wmc":
            pass
        if type == "layer":
            pass
        if type == "context":
            pass
        pass


    def on_clicked_wmc(self, item):
        """Show detailed information about the ows context in the detail form"""
        self.reset_resource_view()
        try:
            resource_id = item.data(1, 0).id
        except:
            QgsMessageLog.logMessage("No attribute id for this resource - no detailed info available", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            resource_id = 0
            return
        self.dlg.labelResourceId.setText(str(resource_id))
        self.dlg.labelResourceType.setText(self.tr("OWS Context"))
        self.dlg.pushButtonLoad.setEnabled(False)
        self.dlg.textBrowserResourceAbstract.clear()
        self.dlg.treeWidgetResourceDetail.clear()
         # check if mrmap context is selected
        resource_type = str(self.dlg.comboBoxSearchResources.currentData())
        if resource_type == "owscontext":
            search_domain = "https://mrmap.geospatial-interoperability-solutions.eu"
            search_path = item.data(1, 0).attributes.ogcMapcontextUrl
            request_url = search_domain + search_path
            self.dlg.labelAccessUrl.setText('<a href="' + request_url + '">OWS Context json</a>')
            self.dlg.labelAccessUrl.setOpenExternalLinks(True)
            self.dlg.pushButtonLoad.setEnabled(True)
            self.dlg.pushButtonLoad.clicked.disconnect()
            self.dlg.pushButtonLoad.clicked.connect(self.load_ows_context)
            return

        # load the preview image from registry
        try:
            preview_url = item.data(1, 0).previewURL
            result_content = self.open_remote(preview_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelPreview.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelPreview.setText(self.tr("No preview"))
        try:
            load_count = str(item.data(1, 0).loadCount)
            self.dlg.labelLoadCount.setText(load_count)
        except:
            QgsMessageLog.logMessage("No attribute loadCount for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # add further information
        # select domain from comboBox
        search_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        # search_domain = "https://www.geoportal.rlp.de"
        search_path = "/mapbender/php/mod_exportWmc.php" #?wmcId=14971&outputFormat=json"
        load_parameters = {
            "outputFormat": "json",
            "wmcId": str(resource_id),
        }
        # add parameter
        request_url = search_domain + search_path + "?" + urllib.parse.urlencode(load_parameters)
        #"https://www.geoportal.rlp.de/mapbender/php/mod_exportWmc?wmcId=666&outputFormat=json"
        self.dlg.labelAccessUrl.setText('<a href="' + request_url + '">OWS Context json</a>')
        self.dlg.labelAccessUrl.setOpenExternalLinks(True)
        # respOrg
        self.dlg.labelOrga.setText(str(item.data(1, 0).respOrg))
        # date
        self.dlg.labelDate.setText(str(item.data(1, 0).date))
        # title
        # abstract
        self.dlg.textBrowserResourceAbstract.append(str(item.data(1, 0).title))
        self.dlg.textBrowserResourceAbstract.append(str(item.data(1, 0).abstract))
        # mdLink
        self.dlg.labelMetadata.setText('<a href="' + str(item.data(1, 0).mdLink) + '">Online Metadata</a>')
        self.dlg.labelMetadata.setOpenExternalLinks(True)
        # onlineUsage
        # read search domain from comboBox
        resolve_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        #search_domain = "https://www.geoportal.rlp.de"
        resolve_path = "/map"
        resolve_parameters = {
            "WMC": item.data(1, 0).id
        }
        # add parameters
        request_url = resolve_domain + resolve_path + "?" + urllib.parse.urlencode(resolve_parameters)
        self.dlg.labelOnlineUsage.setText('<a href="' + request_url + '">Open in Browser</a>')
        self.dlg.labelOnlineUsage.setOpenExternalLinks(True)

        self.dlg.pushButtonLoad.setEnabled(True)
        self.dlg.pushButtonLoad.clicked.disconnect()
        self.dlg.pushButtonLoad.clicked.connect(self.load_ows_context)


    def on_clicked_layer(self, item):
        """Show detailed information about layer from layer search"""
        self.reset_resource_view()
        self.dlg.labelResourceType.setText(self.tr("Map layer"))
        try:
            resource_id = item.data(1, 0).id
        except:
            QgsMessageLog.logMessage("No attribute id for this resource - no detailed info available", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            resource_id = 0
            return
        #self.dlg.labelResourceId.setText("")
        self.dlg.pushButtonLoad.setEnabled(False)
        #self.dlg.textBrowserResourceAbstract.clear()
        self.dlg.treeWidgetResourceDetail.clear()
        # TODO alter this for clicked item in tree widget
        self.dlg.textBrowserResourceAbstract.append(item.text(0))
        try:
            abstract = str(item.data(1, 0).abstract)
            self.dlg.textBrowserResourceAbstract.append(abstract)
        except:
            QgsMessageLog.logMessage("No attribute abstract for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # load preview image
        try:
            preview_url = item.data(1, 0).previewURL
            result_content = self.open_remote(preview_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelPreview.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + preview_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelPreview.setText(self.tr("No preview"))
        # load extent - remote image is to slow!
        try:
            # get extent from json
            bbox = str(item.data(1, 0).bbox)
            self.dlg.labelExtent.setText(bbox)
        except:
            QgsMessageLog.logMessage("No attribute bbox for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # load extent image
        """
        try:
            # get extent from json
            bbox = item.data(1, 0).bbox
            bbox_list = str(bbox).split(',')
            extent_url = "https://www.geoportal.rlp.de/cgi-bin/mapserv?map=/data/mapbender/tools/wms_extent/extents.map&VERSION=1.1.1&REQUEST=GetMap&SERVICE=WMS&LAYERS=demis,ows_layer_target,extent,metadata_polygon&STYLES=&SRS=EPSG:4326&BBOX=4,45,9,56&WIDTH=120&HEIGHT=120&FORMAT=image/png&BGCOLOR=0xffffff&TRANSPARENT=TRUE&EXCEPTIONS=application/vnd.ogc.se_inimage&minx=" + bbox_list[0] + "&miny=" + bbox_list[1] + "&maxx=" + bbox_list[2] + "&maxy=" + bbox_list[3]
            result_content = self.open_remote(extent_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelExtent.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + extent_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelExtent.setText("No Extent")
        """
        # add load count
        try:
            load_count = item.data(1, 0).loadCount
            self.dlg.labelLoadCount.setText(str(item.data(1, 0).loadCount))
        except:
            QgsMessageLog.logMessage("No attribute loadCount for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # set access url and activate load button, if the url ia available
        self.dlg.labelResourceId.setText(str(resource_id))
        try:
            capabilities_url = str(item.data(1, 0).getCapabilitiesUrl)
            self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
            self.dlg.labelAccessUrl.setOpenExternalLinks(True)
            # add function to load button
            self.dlg.pushButtonLoad.setEnabled(True)
            self.dlg.pushButtonLoad.disconnect()
            self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url))
        except:
            QgsMessageLog.logMessage("No attribute getCapabilitiesUrl for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        try:
            metadata_url = str(item.data(1, 0).mdLink)
            self.dlg.labelMetadata.setText('<a href="' + metadata_url + '">Online Metadata</a>')
            self.dlg.labelMetadata.setOpenExternalLinks(True)
        except:
            QgsMessageLog.logMessage("No attribute getCapabilitiesUrl for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # online usage
        # read search domain from comboBox
        resolve_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        #search_domain = "https://www.geoportal.rlp.de"
        resolve_path = "/map"
        resolve_parameters = {
            "LAYER[id]": item.data(1, 0).id,
            "LAYER[visible]": 1,
            "LAYER[zoom]": 1,
        }
        # add parameters
        request_url = resolve_domain + resolve_path + "?" + urllib.parse.urlencode(resolve_parameters)
        self.dlg.labelOnlineUsage.setText('<a href="' + request_url + '">' + self.tr("Open in Browser") + '</a>')
        self.dlg.labelOnlineUsage.setOpenExternalLinks(True)


        # pull top level information - this is the information of the service itself
        top_level_item = item
        while top_level_item.parent():
            try:
                has_org_tag = top_level_item.data(1, 0).respOrg
            except:
                top_level_item = top_level_item.parent()
            else:
                break

        try:
            resp_org = str(top_level_item.data(1, 0).respOrg)
            self.dlg.labelOrga.setText(resp_org)
        except:
            QgsMessageLog.logMessage("No attribute respOrg for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            self.dlg.labelOrga.setText("No organisation found")
        try:
            date = str(top_level_item.data(1, 0).date)
            # TODO - use iso format
            self.dlg.labelDate.setText(date)
        except:
            QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # license / restrictions
        # hasConstraints, isopen, symbolLink, license_id
        # first: generate link to use conditions of portal (srv.id)
        resolve_path = "/mapbender/php/mod_getServiceDisclaimer.php"
        resolve_parameters = {
            "id": top_level_item.data(1, 0).id,
            "type": "wms",
            "languageCode": "en",
            "withHeader": "true",
        }
        # add parameters
        request_url = resolve_domain + resolve_path + "?" + urllib.parse.urlencode(resolve_parameters)
        self.dlg.labelLicence.setText('<a href="' + request_url + '">' + self.tr("Open in Browser") + '</a>')
        self.dlg.labelLicence.setOpenExternalLinks(True)

        # license logo
        try:
            license_logo = str(top_level_item.data(1, 0).symbolLink)
            if license_logo != "":
                result_content = self.open_remote(license_logo)
                if result_content:
                    # build
                    pixmap = QPixmap()
                    pixmap.loadFromData(result_content)
                    # draw preview
                    self.dlg.labelLicenceLogo.setPixmap(pixmap)
                else:
                    QgsMessageLog.logMessage("An error occured while try to open url: " + license_logo,
                                             'GeoPortal.rlp search',
                                             level=Qgis.Critical)
        except:
            pass
        # open data
        try:
            if str(top_level_item.data(1, 0).isopen) == "1":
                result_content = self.open_remote("https://www.geoportal.rlp.de/static/searchCatalogue/images/open-data.jpg")
                if result_content:
                    # build
                    pixmap = QPixmap()
                    pixmap.loadFromData(result_content)
                    # draw preview
                    self.dlg.labelLicenceOpen.setPixmap(pixmap)
        except:
            pass


        try:
            date = str(top_level_item.data(1, 0).symbolLink)
        except:
            QgsMessageLog.logMessage("No attribute license_symbol for this resource", 'GeoPortal.rlp search', level=Qgis.Info)


    def on_clicked_dataset(self, item):
        """Show detailed information about the dataset which was found"""
        self.reset_resource_view()
        self.dlg.labelResourceType.setText(self.tr("Dataset"))
        try:
            resource_id = item.data(1, 0).id
        except:
            QgsMessageLog.logMessage("No attribute id for this resource - no detailed info available", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            resource_id = 0
            return
        self.dlg.pushButtonLoad.setEnabled(False)
        self.dlg.treeWidgetResourceDetail.clear()
        # TODO alter this for clicked item in tree widget
        self.dlg.textBrowserResourceAbstract.append(item.text(0))
        try:
            abstract = str(item.data(1, 0).abstract)
            self.dlg.textBrowserResourceAbstract.append(abstract)
        except:
            QgsMessageLog.logMessage("No attribute abstract for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # load preview image
        try:
            preview_url = item.data(1, 0).previewURL
            result_content = self.open_remote(preview_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelPreview.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + preview_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelPreview.setText(self.tr("No preview"))
        # load extent - remote image is to slow!
        try:
            # get extent from json
            bbox = str(item.data(1, 0).bbox)
            self.dlg.labelExtent.setText(bbox)
        except:
            QgsMessageLog.logMessage("No attribute bbox for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        # load extent image
        """
        try:
            # get extent from json
            bbox = item.data(1, 0).bbox
            bbox_list = str(bbox).split(',')
            extent_url = "https://www.geoportal.rlp.de/cgi-bin/mapserv?map=/data/mapbender/tools/wms_extent/extents.map&VERSION=1.1.1&REQUEST=GetMap&SERVICE=WMS&LAYERS=demis,ows_layer_target,extent,metadata_polygon&STYLES=&SRS=EPSG:4326&BBOX=4,45,9,56&WIDTH=120&HEIGHT=120&FORMAT=image/png&BGCOLOR=0xffffff&TRANSPARENT=TRUE&EXCEPTIONS=application/vnd.ogc.se_inimage&minx=" + bbox_list[0] + "&miny=" + bbox_list[1] + "&maxx=" + bbox_list[2] + "&maxy=" + bbox_list[3]
            result_content = self.open_remote(extent_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelExtent.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + extent_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelExtent.setText("No Extent")
        """
        # add load count
        try:
            load_count = item.data(1, 0).loadCount
            self.dlg.labelLoadCount.setText(str(item.data(1, 0).loadCount))
        except:
            QgsMessageLog.logMessage("No attribute loadCount for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # set access url and activate load button, if the url is available
        self.dlg.labelResourceId.setText(str(resource_id))
        """
        try:
            capabilities_url = str(item.data(1, 0).getCapabilitiesUrl)
            self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
            self.dlg.labelAccessUrl.setOpenExternalLinks(True)
            # add function to load button
            self.dlg.pushButtonLoad.setEnabled(True)
            self.dlg.pushButtonLoad.disconnect()
            self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url))
        except:
            QgsMessageLog.logMessage("No attribute getCapabilitiesUrl for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        """
        try:
            metadata_url = str(item.data(1, 0).mdLink)
            self.dlg.labelMetadata.setText('<a href="' + metadata_url + '">' + self.tr("Online Metadata") + '</a>')
            self.dlg.labelMetadata.setOpenExternalLinks(True)
        except:
            QgsMessageLog.logMessage("No attribute mdLink for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        """
        # pull top level information - this is the information of the service itself
        top_level_item = item
        i = 0
        while top_level_item.parent():
            i = i + 1
            top_level_item = top_level_item.parent()
        top_level_item = top_level_item.child(0) # cause the top level ist not the service, but the header
        try:
            resp_org = str(top_level_item.data(1, 0).respOrg)
            self.dlg.labelOrga.setText(resp_org)
        except:
            QgsMessageLog.logMessage("No attribute respOrg for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        """
        try:
            date = str(item.data(1, 0).date)
            # TODO - use iso format
            self.dlg.labelDate.setText(date)
        except:
            QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        try:
            resp_orga = str(item.data(1, 0).respOrg)
            # TODO - use iso format
            self.dlg.labelOrga.setText(resp_orga)
        except:
            QgsMessageLog.logMessage("No attribute respOrg for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # license / restrictions
        # hasConstraints, isopen, symbolLink, license_id
        # first: generate link to use conditions of portal (srv.id)
        resolve_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        resolve_path = "/mapbender/php/mod_getServiceDisclaimer.php"
        resolve_parameters = {
            "id": item.data(1, 0).id,
            "type": "metadata",
            "languageCode": "en",
            "withHeader": "true",
        }
        # add parameters
        request_url = resolve_domain + resolve_path + "?" + urllib.parse.urlencode(resolve_parameters)
        self.dlg.labelLicence.setText('<a href="' + request_url + '">' + self.tr("Open in Browser") + '</a>')
        self.dlg.labelLicence.setOpenExternalLinks(True)

        # license logo
        try:
            license_logo = str(item.data(1, 0).symbolLink)
            if license_logo != "":
                result_content = self.open_remote(license_logo)
                if result_content:
                    # build
                    pixmap = QPixmap()
                    pixmap.loadFromData(result_content)
                    # draw preview
                    self.dlg.labelLicenceLogo.setPixmap(pixmap)
                else:
                    QgsMessageLog.logMessage("An error occured while try to open url: " + license_logo,
                                             'GeoPortal.rlp search',
                                             level=Qgis.Critical)
        except:
            pass
        # open data
        try:
            if str(item.data(1, 0).isopen) == "1":
                result_content = self.open_remote("https://www.geoportal.rlp.de/static/searchCatalogue/images/open-data.jpg")
                if result_content:
                    # build
                    pixmap = QPixmap()
                    pixmap.loadFromData(result_content)
                    # draw preview
                    self.dlg.labelLicenceOpen.setPixmap(pixmap)
        except:
            pass
        self.load_access_options(item)


    def show_loader_img(self):
        """
        Should be used to show the loading of an external resource - does not work at the moment
        """
        self.dlg.labelSearchAnimation.setText(self.tr("Searching..."))
        pass


    def hide_loader_img(self):
        """
        Should be used to hide the loading of an external resource - does not work at the moment
        """
        self.dlg.labelSearchAnimation.setText(self.tr("Ready"))
        pass


    def on_clicked_remote_dataset(self, item):
        """Show detailed information about the dataset which was found by remote CSW search"""
        self.reset_resource_view()
        self.dlg.labelResourceType.setText(self.tr("Dataset"))
        try:
            resource_id = item.data(1, 0).datasetId
        except:
            QgsMessageLog.logMessage("No attribute datasetId for this resource - no detailed info available", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            resource_id = 0
            return
        self.dlg.pushButtonLoad.setEnabled(False)
        self.dlg.treeWidgetResourceDetail.clear()
        # TODO alter this for clicked item in tree widget
        self.dlg.textBrowserResourceAbstract.append(item.text(0))
        try:
            abstract = str(item.data(1, 0).abstract)
            self.dlg.textBrowserResourceAbstract.append(abstract)
        except:
            QgsMessageLog.logMessage("No attribute abstract for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # load preview image
        try:
            preview_url = item.data(1, 0).previewURL
            result_content = self.open_remote(preview_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelPreview.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + preview_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelPreview.setText(self.tr("No preview"))

        # load extent - remote image is to slow!
        try:
            # get extent from json
            bbox = str(item.data(1, 0).bbox)
            self.dlg.labelExtent.setText(bbox)
        except:
            QgsMessageLog.logMessage("No attribute bbox for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        # load extent image
        """
        try:
            # get extent from json
            bbox = item.data(1, 0).bbox
            bbox_list = str(bbox).split(',')
            extent_url = "https://www.geoportal.rlp.de/cgi-bin/mapserv?map=/data/mapbender/tools/wms_extent/extents.map&VERSION=1.1.1&REQUEST=GetMap&SERVICE=WMS&LAYERS=demis,ows_layer_target,extent,metadata_polygon&STYLES=&SRS=EPSG:4326&BBOX=4,45,9,56&WIDTH=120&HEIGHT=120&FORMAT=image/png&BGCOLOR=0xffffff&TRANSPARENT=TRUE&EXCEPTIONS=application/vnd.ogc.se_inimage&minx=" + bbox_list[0] + "&miny=" + bbox_list[1] + "&maxx=" + bbox_list[2] + "&maxy=" + bbox_list[3]
            result_content = self.open_remote(extent_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelExtent.setPixmap(pixmap)
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + extent_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        except:
            self.dlg.labelExtent.setText("No Extent")
        """
        # add load count
        try:
            load_count = item.data(1, 0).loadCount
            self.dlg.labelLoadCount.setText(str(item.data(1, 0).loadCount))
        except:
            QgsMessageLog.logMessage("No attribute loadCount for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        # set access url and activate load button, if the url is available
        self.dlg.labelResourceId.setText(str(resource_id))
        """
        try:
            capabilities_url = str(item.data(1, 0).getCapabilitiesUrl)
            self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
            self.dlg.labelAccessUrl.setOpenExternalLinks(True)
            # add function to load button
            self.dlg.pushButtonLoad.setEnabled(True)
            self.dlg.pushButtonLoad.disconnect()
            self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url))
        except:
            QgsMessageLog.logMessage("No attribute getCapabilitiesUrl for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        """
        try:
            metadata_url = str(item.data(1, 0).htmlLink)
            self.dlg.labelMetadata.setText('<a href="' + metadata_url + '">' + self.tr("Online Metadata") + '</a>')
            self.dlg.labelMetadata.setOpenExternalLinks(True)
        except:
            QgsMessageLog.logMessage("No attribute mdLink for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        """
        # pull top level information - this is the information of the service itself
        top_level_item = item
        i = 0
        while top_level_item.parent():
            i = i + 1
            top_level_item = top_level_item.parent()
        top_level_item = top_level_item.child(0) # cause the top level ist not the service, but the header
        try:
            resp_org = str(top_level_item.data(1, 0).respOrg)
            self.dlg.labelOrga.setText(resp_org)
        except:
            QgsMessageLog.logMessage("No attribute respOrg for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        """
        try:
            date = str(item.data(1, 0).date)
            # TODO - use iso format
            self.dlg.labelDate.setText(date)
        except:
            QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)

        try:
            resp_orga = str(item.data(1, 0).respOrg)
            # TODO - use iso format
            self.dlg.labelOrga.setText(resp_orga)
        except:
            QgsMessageLog.logMessage("No attribute respOrg for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        self.load_remote_access_options(item)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = GeoportalRlpMetadataSearchDialog()
            for resource in self.search_resources_list:
                self.dlg.comboBoxSearchResources.addItem(resource["title"], resource["value"])
            # search_catalogues
            for resource in self.search_catalogues:
                self.dlg.comboBoxSearchCatalogue.addItem(resource["title"], resource["value"])
            # important - events should only be added once - otherwise we will go into trouble!
            self.dlg.pushButton.clicked.connect(lambda: self.start_search(page=1, resource_type=self.dlg.comboBoxSearchResources.currentData()))
            self.dlg.pushButtonLoad.clicked.connect(self.load_ows_context)
            # if selected a remote search - the available CSW should be listed in the combo box below
            self.dlg.comboBoxSearchResources.currentIndexChanged.connect(self.on_select_search_resource)
            self.dlg.comboBoxSearchCatalogue.currentIndexChanged.connect(self.on_select_search_catalogue)
            # add logo
            logo_url = "https://www.geoportal.rlp.de/static/useroperations/images/logo-geoportal.png"
            result_content = self.open_remote(logo_url)
            if result_content:
                # build
                pixmap = QPixmap()
                pixmap.loadFromData(result_content)
                # draw preview
                self.dlg.labelLogo.setPixmap(pixmap.scaled(self.dlg.labelLogo.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation))
            else:
                QgsMessageLog.logMessage("An error occured while try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
            # add link to github for help
            help_icon_path = os.path.join(os.path.dirname(__file__), "questionmark.png")
            pixmap_help = QPixmap()
            pixmap.load(help_icon_path)
            self.dlg.labelHelp.setPixmap(pixmap.scaled(self.dlg.labelHelp.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation))
            self.dlg.labelHelp.setText('<a href="https://github.com/mrmap-community/gprlp_metadata_search">' +
                                       self.tr("Help") + '</a>')
            self.dlg.labelHelp.setOpenExternalLinks(True)
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass


    def find_layer_by_id(self, layer, layer_id):
        """Function to find layer by id in hierarchical layertree which is given by the search module"""
        for sub_layer in layer:
            # print(str(sub_layer['title']))
            # print(str(sub_layer['id']))
            if int(sub_layer.id) == int(layer_id):
                # QgsMessageLog.logMessage("layer found with id: " + str(sub_layer.id) + " and title: " + str(sub_layer.title), 'GeoPortal.rlp search', level=Qgis.Info)
                return sub_layer
            # print("after compare")
            if hasattr(sub_layer, "layer"):
                # very important - return before invoking recursive function !!!
                return self.find_layer_by_id(sub_layer.layer, layer_id)


    def load_access_options(self, item):
        """Function to load the access options from the search results. They are in the coupledResources object."""
        # build
        parent_node = QTreeWidgetItem()
        parent_node.setText(0, self.tr("Access options"))
        try:
            coupled_resources = item.data(1, 0).coupledResources
        except:
            QgsMessageLog.logMessage("An error occured while reading coupledResources", 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
            return
        try:
            download_count = str(len(coupled_resources.inspireAtomFeeds))
        except:
            download_count = "0"
        try:
            view_count = str(len(coupled_resources.layer))
        except:
            view_count = "0"
        download_node = QTreeWidgetItem()
        download_node.setText(0, self.tr("Download options") + " (" + download_count + ")")
        if download_count != "0":
            for download_option in coupled_resources.inspireAtomFeeds:
                download_option.service_type = "download"
                download_option_node = QTreeWidgetItem()
                download_option_node.setText(0, download_option.serviceTitle)
                download_option_node.setToolTip(0, download_option.serviceTitle)
                download_option_node.setData(1, 0, download_option)
                download_node.addChild(download_option_node)
        view_node = QTreeWidgetItem()
        view_node.setText(0, self.tr("View options") + " (" + view_count + ")")
        if view_count != "0":
            for view_option in coupled_resources.layer:
                view_option_node = QTreeWidgetItem()
                layer_id = view_option.id
                # TODO  test if layer is not already set at top level!
                service_information = {}
                service_information["serviceId"] = view_option.srv.id
                # TODO: add this info to coupled_layer below!
                coupled_layer = self.find_layer_by_id(view_option.srv.layer, layer_id)

                QgsMessageLog.logMessage("Coupled layer title: " + coupled_layer.title, 'GeoPortal.rlp search',
                                         level=Qgis.Info)
                if coupled_layer:
                    coupled_layer.service_type = "view"
                    coupled_layer.serviceId = view_option.srv.id
                    coupled_layer.respOrg = view_option.srv.respOrg
                    coupled_layer.symbolLink = view_option.srv.symbolLink
                    coupled_layer.isopen = view_option.srv.isopen
                    coupled_layer.date = view_option.srv.date
                try:
                    view_option_node.setText(0, coupled_layer.title + " (" + str(layer_id) + ")")
                    view_option_node.setToolTip(0, coupled_layer.title + " (" + str(layer_id) + ")")
                    view_option_node.setData(1, 0, coupled_layer)
                except:
                    view_option_node.setText(0, str(layer_id))
                view_node.addChild(view_option_node)
        """
        for view_layer in coupled_resources.layer:
            pass
        """
        self.dlg.treeWidgetResourceDetail.clear()
        parent_node.addChild(view_node)
        parent_node.addChild(download_node)
        # QgsMessageLog.logMessage("Widget cleared", 'GeoPortal.rlp search', level=Qgis.Info)
        #self.build_tree_recursive(result_object.features, parent_node, "")
        self.dlg.treeWidgetResourceDetail.addTopLevelItem(parent_node)
        #self.dlg.treeWidgetResourceDetail.disconnect()
        self.dlg.treeWidgetResourceDetail.expandAll()
        try:
            self.dlg.treeWidgetResourceDetail.itemClicked.disconnect()
        except:
            pass
        self.dlg.treeWidgetResourceDetail.itemClicked.connect(self.on_clicked_service_option)


    def load_remote_access_options(self, item):
        """Function to load the remote access options by using the GeoPortal.rlp coupled resource resolver."""
        # build
        parent_node = QTreeWidgetItem()
        parent_node.setText(0, self.tr("Access options"))
        # resolve coupled resources by csw request
        """
        Build initial resolving  request
        """
        # read search domain from comboBox
        resolve_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        #search_domain = "https://www.geoportal.rlp.de"
        resolve_path = "/mapbender/php/mod_getCoupledResourcesForDataset.php"
        resolve_parameters = {
            "getRecordByIdUrl": item.data(1, 0).mdLink
        }
        # add parameters
        request_url = resolve_domain + resolve_path + "?" + urllib.parse.urlencode(resolve_parameters)
        QgsMessageLog.logMessage("Try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        result_content = self.open_remote(request_url)
        result_object = False
        if result_content:
            result_object = json.loads(bytes(result_content).decode(), object_hook=lambda d: SimpleNamespace(**d))
        else:
            QgsMessageLog.logMessage("An error occured while try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        if result_object and result_object.success == True:
            QgsMessageLog.logMessage("Resolved coupled services successfully", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            count_dls = 0
            count_view = 0
            download_node = QTreeWidgetItem()
            view_node = QTreeWidgetItem()
            if result_object.result.service:
                for service in result_object.result.service:
                    if service.serviceType == "download":
                        count_dls = count_dls + 1
                        download_option_node = QTreeWidgetItem()
                        download_option_node.setText(0, service.serviceTitle)
                        download_option_node.setToolTip(0, service.serviceTitle)
                        download_option_node.setData(1, 0, service)
                        download_node.addChild(download_option_node)
                    if service.serviceType == "view":
                        count_view = count_view + 1
                        view_option_node = QTreeWidgetItem()
                        view_option_node.setText(0, service.serviceTitle)
                        view_option_node.setToolTip(0, service.serviceTitle)
                        view_option_node.setData(1, 0, service)
                        view_node.addChild(view_option_node)
            download_node.setText(0, self.tr("Download options") + " (" + str(count_dls) + ")")
            view_node.setText(0, self.tr("View options") + " (" + str(count_view) + ")")
            self.dlg.treeWidgetResourceDetail.clear()
            parent_node.addChild(view_node)
            parent_node.addChild(download_node)
            self.dlg.treeWidgetResourceDetail.addTopLevelItem(parent_node)
            self.dlg.treeWidgetResourceDetail.expandAll()
            try:
                self.dlg.treeWidgetResourceDetail.itemClicked.disconnect()
            except:
                pass
            # add on click signal
            self.dlg.treeWidgetResourceDetail.itemClicked.connect(self.on_clicked_remote_service_option)
            return
        else:
            QgsMessageLog.logMessage("An error occured while resolving coupledResources", 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
            return


    def on_clicked_remote_service_option(self, item):
        """

        Show detailed information about service resource

        """
        self.reset_resource_view()
        self.dlg.pushButtonLoad.setEnabled(False)
        if item.data(1, 0):
            if item.data(1, 0).serviceType == "view":
                self.dlg.labelResourceType.setText("WMS")
                try:
                    capabilities_url = str(item.data(1, 0).accessUrl)
                    self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
                    self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                    # add function to load button
                    self.dlg.pushButtonLoad.setEnabled(True)
                    self.dlg.pushButtonLoad.disconnect()
                    self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url))
                except:
                    QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
            if item.data(1, 0).serviceType == "download":
                if item.data(1, 0).serviceSubType == "ATOM":
                    self.dlg.labelResourceType.setText("ATOM Feed")
                    try:
                        access_url = str(item.data(1, 0).accessClient)
                        self.dlg.labelAccessUrl.setText('<a href="' + access_url + '">Access URL</a>')
                        self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                    except:
                        QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                                 level=Qgis.Info)
                if item.data(1, 0).serviceSubType == "WFS":
                    self.dlg.labelResourceType.setText("WFS")
                    try:
                        capabilities_url = str(item.data(1, 0).accessUrl)
                        self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
                        self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                        # add function to load button
                        self.dlg.pushButtonLoad.setEnabled(True)
                        self.dlg.pushButtonLoad.disconnect()
                        self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url, "wfs"))
                    except:
                        QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                                 level=Qgis.Info)
            # TODO alter this for clicked item in tree widget
            self.dlg.textBrowserResourceAbstract.append(item.text(0))
            try:
                metadata_url = str(item.data(1, 0).htmlLink)
                self.dlg.labelMetadata.setText('<a href="' + metadata_url + '">' + self.tr("Online Metadata") + '</a>')
                self.dlg.labelMetadata.setOpenExternalLinks(True)
            except:
                QgsMessageLog.logMessage("No attribute mdLink for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
            try:
                resource_id = item.data(1, 0).id
            except:
                QgsMessageLog.logMessage("No attribute id for this resource - no detailed info available",
                                         'GeoPortal.rlp search',
                                         level=Qgis.Info)
                resource_id = 0


    def on_clicked_service_option(self, item, service_type):
        """

        Invoked when clicked on a coupled service information from geoportal dataset search

        """
        self.reset_resource_view()
        self.dlg.pushButtonLoad.setEnabled(False)
        if item.data(1, 0):
            if item.data(1, 0).service_type == "view":
                self.dlg.labelResourceType.setText(self.tr("WMS Layer"))
                try:
                    capabilities_url = str(item.data(1, 0).getCapabilitiesUrl)
                    self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
                    self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                    # add function to load button
                    self.dlg.pushButtonLoad.setEnabled(True)
                    self.dlg.pushButtonLoad.disconnect()
                    self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url))
                except:
                    QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
                try:
                    resp_org = str(item.data(1, 0).respOrg)
                    self.dlg.labelOrga.setText(resp_org)
                except:
                    QgsMessageLog.logMessage("No attribute respOrg for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
                    self.dlg.labelOrga.setText(self.tr("No organisation found"))
                try:
                    date = str(item.data(1, 0).date)
                    # TODO - use iso format
                    self.dlg.labelDate.setText(date)
                except:
                    QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
                try:
                    load_count = str(item.data(1, 0).loadCount)
                    # TODO - use iso format
                    self.dlg.labelLoadCount.setText(load_count)
                except:
                    QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
                try:
                    id = str(item.data(1, 0).id)
                    # TODO - use iso format
                    self.dlg.labelResourceId.setText(id)
                except:
                    QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
                try:
                    bbox = str(item.data(1, 0).bbox)
                    # TODO - use iso format
                    self.dlg.labelExtent.setText(bbox)
                except:
                    QgsMessageLog.logMessage("No attribute date for this resource", 'GeoPortal.rlp search',
                                             level=Qgis.Info)
                try:
                    preview_url = item.data(1, 0).previewURL
                    result_content = self.open_remote(preview_url)
                    if result_content:
                        # build
                        pixmap = QPixmap()
                        pixmap.loadFromData(result_content)
                        # draw preview
                        self.dlg.labelPreview.setPixmap(pixmap)
                    else:
                        QgsMessageLog.logMessage("An error occured while try to open url: " + request_url,
                                                 'GeoPortal.rlp search',
                                                 level=Qgis.Critical)
                except:
                    self.dlg.labelPreview.setText(self.tr("No preview"))


                # license / restrictions
                # hasConstraints, isopen, symbolLink, license_id
                # first: generate link to use conditions of portal (srv.id)
                try:
                    service_id = item.data(1, 0).serviceId
                    if service_id:
                        resolve_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
                        resolve_path = "/mapbender/php/mod_getServiceDisclaimer.php"
                        resolve_parameters = {
                            "id": str(service_id),
                            "type": "wms",
                            "languageCode": "en",
                            "withHeader": "true",
                        }
                        # add parameters
                        request_url = resolve_domain + resolve_path + "?" + urllib.parse.urlencode(resolve_parameters)
                        self.dlg.labelLicence.setText('<a href="' + request_url + '">' + self.tr("Open in Browser") + '</a>')
                        self.dlg.labelLicence.setOpenExternalLinks(True)

                        # license logo
                        try:
                            license_logo = str(item.data(1, 0).symbolLink)
                            if license_logo != "":
                                result_content = self.open_remote(license_logo)
                                if result_content:
                                    # build
                                    pixmap = QPixmap()
                                    pixmap.loadFromData(result_content)
                                    # draw preview
                                    self.dlg.labelLicenceLogo.setPixmap(pixmap)
                                else:
                                    QgsMessageLog.logMessage("An error occured while try to open url: " + license_logo,
                                                             'GeoPortal.rlp search',
                                                             level=Qgis.Critical)
                        except:
                            pass
                        # open data
                        try:
                            if str(top_level_item.isopen) == "1":
                                result_content = self.open_remote(
                                    "https://www.geoportal.rlp.de/static/searchCatalogue/images/open-data.jpg")
                                if result_content:
                                    # build
                                    pixmap = QPixmap()
                                    pixmap.loadFromData(result_content)
                                    # draw preview
                                    self.dlg.labelLicenceOpen.setPixmap(pixmap)
                        except:
                            pass
                except:
                    pass

            if item.data(1, 0).service_type == "download":
                if item.data(1, 0).serviceSubType == "ATOM":
                    self.dlg.labelResourceType.setText("ATOM Feed")
                    try:
                        access_url = str(item.data(1, 0).accessClient)
                        self.dlg.labelAccessUrl.setText('<a href="' + access_url + '">Access URL</a>')
                        self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                    except:
                        QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                                 level=Qgis.Info)
                if item.data(1, 0).serviceSubType == "REST":
                    self.dlg.labelResourceType.setText("OGC API Features")
                    try:
                        capabilities_url = str(item.data(1, 0).accessUrl).rstrip('/api')
                        self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">API</a>')
                        self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                        # add function to load button
                        self.dlg.pushButtonLoad.setEnabled(True)
                        self.dlg.pushButtonLoad.disconnect()
                        self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url, "wfs"))
                    except:
                        QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                                 level=Qgis.Info)
                if item.data(1, 0).serviceSubType == "DIRECTWFS":
                    self.dlg.labelResourceType.setText("WFS")
                    try:
                        capabilities_url = str(item.data(1, 0).accessUrl).rstrip('/api')
                        self.dlg.labelAccessUrl.setText('<a href="' + capabilities_url + '">GetCapabilities</a>')
                        self.dlg.labelAccessUrl.setOpenExternalLinks(True)
                        # add function to load button
                        self.dlg.pushButtonLoad.setEnabled(True)
                        self.dlg.pushButtonLoad.disconnect()
                        self.dlg.pushButtonLoad.clicked.connect(lambda: self.add_ows(capabilities_url, "wfs"))
                    except:
                        QgsMessageLog.logMessage("No attribute accessUrl for this resource", 'GeoPortal.rlp search',
                                                 level=Qgis.Info)
            # TODO alter this for clicked item in tree widget
            self.dlg.textBrowserResourceAbstract.append(item.text(0))
            try:
                metadata_url = str(item.data(1, 0).htmlLink)
                self.dlg.labelMetadata.setText('<a href="' + metadata_url + '">' + self.tr("Online Metadata") + '</a>')
                self.dlg.labelMetadata.setOpenExternalLinks(True)
            except:
                QgsMessageLog.logMessage("No attribute mdLink for this resource", 'GeoPortal.rlp search',
                                         level=Qgis.Info)
            try:
                resource_id = item.data(1, 0).id
            except:
                QgsMessageLog.logMessage("No attribute id for this resource - no detailed info available",
                                         'GeoPortal.rlp search',
                                         level=Qgis.Info)
                resource_id = 0


    def on_select_search_catalogue(self):
        """Function that is invoked, when the user selects the search resource type"""
        # switch self.dlg.comboBoxSearchResources to default value
        # search_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        index = self.dlg.comboBoxSearchResources.findData("dataset")
        self.dlg.comboBoxSearchResources.setCurrentIndex(index)
        # delete entries for remote catalogues
        self.dlg.comboBoxRemoteCsw.clear()
        self.dlg.comboBoxRemoteCsw.setEnabled(False)
        pass


    def on_select_search_resource(self):
        """Function that is invoked, when the user selects the search resource type for remote search via csw"""
        if self.dlg.comboBoxSearchResources.currentData() == "csw":
            list_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
            list_path = "/mapbender/php/mod_showCswList.php"
            request_url = list_domain + list_path
            result_content = self.open_remote(request_url)
            if result_content:
                result_object = json.loads(bytes(result_content).decode(), object_hook=lambda d: SimpleNamespace(**d))
                # set options
                for catalogue in result_object.catalogues:
                    # QgsMessageLog.logMessage("Catalogues: " + catalogue.title + " - " + catalogue.id, 'GeoPortal.rlp search', level=Qgis.Info)
                    self.dlg.comboBoxRemoteCsw.addItem(catalogue.title, catalogue.id)
                    self.dlg.comboBoxRemoteCsw.setEnabled(True)
            else:
                QgsMessageLog.logMessage("An error occured while try to list available CSW interfaces: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Info)
                # delete options if they are set
                self.dlg.comboBoxRemoteCsw.clear()
                # deactivate widget
                self.dlg.comboBoxRemoteCsw.setEnabled(False)
        else:
            self.dlg.comboBoxRemoteCsw.clear()
            self.dlg.comboBoxRemoteCsw.setEnabled(False)


    def load_ows_context(self):
        """Function that is invoked, if an ows context resource should be loaded.
        See: https://portal.ogc.org/files/?artifact_id=68826
        """
        # get ows context id from label_field
        context_id = int(self.dlg.labelResourceId.text())

        resource_type = str(self.dlg.comboBoxSearchResources.currentData())
        if resource_type == "owscontext":
            search_domain = "https://mrmap.geospatial-interoperability-solutions.eu"
            search_path = "/mrmap-proxy/ows/" + str(context_id)
            request_url = search_domain + search_path
        else:
            search_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
            search_path = "/mapbender/php/mod_exportWmc.php" #?wmcId=14971&outputFormat=json"
            load_parameters = {
                "outputFormat": "json",
                "wmcId": context_id,
            }
            # add parameter
            request_url = search_domain + search_path + "?" + urllib.parse.urlencode(load_parameters)
        QgsMessageLog.logMessage("Try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        result_content = self.open_remote(request_url)
        if result_content:
            result_object = json.loads(bytes(result_content).decode(), object_hook=lambda d: SimpleNamespace(**d))
        else:
            QgsMessageLog.logMessage("An error occured while try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        for feature in result_object.features:

            try:
                for offering in feature.offerings:
                    if offering.code == "http://www.opengis.net/spec/owc-atom/1.0/req/wms":
                        for operation in offering.operations:
                            if operation.code == "GetMap":
                                # add wms layer to qgis
                                # feature.titles
                                # self.dlg.textBrowser.append(feature.properties.title)
                                # layer name
                                # getmap
                                # ...scale hints
                                pass
            except:
                QgsMessageLog.logMessage("Feature has no offering!", 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        # build tree
        # tree = QTreeWidget()
        # last_folder = ""
        parent_node = QTreeWidgetItem()
        # parent_node.expandAll()
        # start_node = parent_node
        parent_node.setText(0, result_object.properties.title)
        self.dlg.treeWidgetResourceDetail.clear()
        # QgsMessageLog.logMessage("Widget cleared", 'GeoPortal.rlp search', level=Qgis.Info)
        self.build_tree_recursive(result_object.features, parent_node, "")
        self.dlg.treeWidgetResourceDetail.addTopLevelItem(parent_node)
        #self.dlg.treeWidgetResourceDetail.disconnect()
        try:
            self.dlg.treeWidgetResourceDetail.itemClicked.disconnect()
        except:
            pass
        self.dlg.treeWidgetResourceDetail.itemClicked.connect(self.on_clicked_context_layer)


    def build_layer_tree_recursive(self, srv, parent):
        """Funtion to build up the layertree from layer search results."""
        QgsMessageLog.logMessage("srv: " + srv.title + " - type: " + str(type(srv)), 'GeoPortal.rlp search', level=Qgis.Info)
        try:
            iter(srv.layer)
            for layer in srv.layer:
                newTreeWidgetItem = QTreeWidgetItem()
                newTreeWidgetItem.setText(0, layer.title)
                newTreeWidgetItem.setToolTip(0, layer.title)
                newTreeWidgetItem.setData(1, 0, layer)
                parent.addChild(newTreeWidgetItem)
                self.build_layer_tree_recursive(layer, newTreeWidgetItem)
        except AttributeError:
            pass
                #features = list(filter(lambda num: num.properties.folder != feature.properties.folder, features))
                #self.build_layer_tree_recursive(layer, newTreeWidgetItem)


    def build_tree_recursive(self, features, parent, parent_path):
        """Funtion to build up the layertree from an ows context document."""
        # leaf consists of title, folder and feature object
        # "leaf 1", "/0", feature 1
        # "leaf 2", "/0/1", feature 2
        # "leaf 3", "/1", feature 3
        # and so on ...
        # we begin with parent object - ows context object - maybe it's title should be used
        # QgsMessageLog.logMessage("Actual parent_path to search for (top of function): " + parent_path, 'GeoPortal.rlp search', level=Qgis.Info)
        for feature in features:
            match = re.match("^" + parent_path + "/[0-9]+$", feature.properties.folder)
            if match:
                # QgsMessageLog.logMessage("Found feature with path: " + feature.properties.folder, 'GeoPortal.rlp search', level=Qgis.Info)
                # generate new QTreeWidgetItem
                newTreeWidgetItem = QTreeWidgetItem()
                # add information to item - first - 0 column will be used for title
                newTreeWidgetItem.setText(0, feature.properties.title)
                # https://doc.qt.io/qt-5/qtreewidgetitem.html#setData
                newTreeWidgetItem.setData(1, 0, feature)
                parent.addChild(newTreeWidgetItem)
                # delete feature with this folder from features array - cause it already exists
                features = list(filter(lambda num: num.properties.folder != feature.properties.folder, features))
                # QgsMessageLog.logMessage("Invoke recursive with path: " + feature.properties.folder, 'GeoPortal.rlp search', level=Qgis.Info)
                # invoke recursive with new folder string e.g. "/0"
                self.build_tree_recursive(features, newTreeWidgetItem, feature.properties.folder)


    def open_remote(self, url, timeout=4000, content_type=False):
        """
        Howto to do it the QGIS way
        https://python.hotexamples.com/de/examples/qgis.core/QgsNetworkAccessManager/-/python-qgsnetworkaccessmanager-class-examples.html
        """
        #self.show_loader_img()
        request = QNetworkRequest()
        request.setUrl(QUrl(url))
        if content_type:
            request.setHeader(QNetworkRequest.ContentTypeHeader, content_type)
        request.setHeader(QNetworkRequest.UserAgentHeader, "QGIS 3.X - Metadata search")
        # manager = QgsNetworkAccessManager.instance()
        # blockingGet
        # result = manager.blockingGet(request, forceRefresh=True)
        self.na_manager.setTimeout(timeout)
        result = self.na_manager.blockingGet(request)
        if result.errorString() != "":
            QgsMessageLog.logMessage("An error occured while trying to make a network request - Error: " + result.errorString(), 'GeoPortal.rlp search',
                                     level=Qgis.Critical)
        # https://qgis.org/api/classQgsNetworkReplyContent.html
        # byte_array
        #self.hide_loader_img()
        return result.content()


    def reset_form(self, reset_input=False):
        if reset_input:
            self.dlg.textEditSearchText.clear()
        self.dlg.numberOfAllPagesLabel.setText('0')
        self.dlg.numberOfPageLabel.setText('0')
        self.dlg.numberOfAllResultsLabel.setText('0')
        self.dlg.treeWidgetResource.clear()
        self.dlg.textBrowserResourceAbstract.clear()
        self.dlg.treeWidgetResourceDetail.clear()
        self.dlg.pushButtonLoad.setEnabled(False)
        self.dlg.pushButtonPageBack.setEnabled(False)
        self.dlg.pushButtonPageForward.setEnabled(False)
        # very important thing:
        try:
            self.dlg.treeWidgetResource.itemClicked.disconnect()
        except:
            pass


    def reset_resource_view(self, reset_input=False):
        if reset_input:
            self.dlg.textEditSearchText.clear()
        self.dlg.labelResourceType.setText('')
        self.dlg.labelResourceId.setText('')
        self.dlg.labelPreview.setText(self.tr('Preview...'))
        self.dlg.labelExtent.setText('...')
        self.dlg.labelLoadCount.setText('')
        self.dlg.labelOrga.setText('')
        self.dlg.labelDate.setText('')
        self.dlg.labelLicence.setText('')
        self.dlg.labelRestrictions.setText('')
        self.dlg.labelAccessUrl.setText('')
        self.dlg.labelMetadata.setText('')
        self.dlg.labelOnlineUsage.setText('')
        self.dlg.textBrowserResourceAbstract.clear()
        self.dlg.labelLicenceLogo.clear()
        self.dlg.labelLicenceOpen.clear()


    def alter_pager(self, actual_page=1):
            self.dlg.pushButtonPageForward.disconnect()
            self.dlg.pushButtonPageBack.disconnect()
            self.dlg.pushButtonPageForward.clicked.connect(lambda: self.start_search(page=(actual_page + 1), resource_type=self.dlg.comboBoxSearchResources.currentData()))
            self.dlg.pushButtonPageBack.clicked.connect(lambda: self.start_search(page=(actual_page - 1), resource_type=self.dlg.comboBoxSearchResources.currentData()))


    def start_mrmap_search(self, page=1, resource_type='owscontext'):
        results_per_page = 10
        self.reset_form(False)
        self.reset_resource_view()
        search_domain = "https://mrmap.geospatial-interoperability-solutions.eu"
        search_path = "/api/v1/registry/mapcontexts/"
        search_text_string = self.dlg.textEditSearchText.toPlainText()
        if search_text_string == "":
            search_text = ""
        else:
            search_text = search_text_string
        search_parameters = {
            "filter[search]": search_text,
            "page[number]": page,
            "page[size]": results_per_page,
            "fields[MapContext]": "title,ogc_mapcontext_url",
        }
        request_url = search_domain + search_path + "?" + urllib.parse.urlencode(search_parameters)
        QgsMessageLog.logMessage("Try to open url: " + request_url, 'MrMap search',
                                         level=Qgis.Info)
        self.show_loader_img()
        result_content = self.open_remote(request_url, timeout=5000, content_type="application/vnd.api+json")
        if result_content:
            result_object = json.loads(bytes(result_content).decode(), object_hook=lambda d: SimpleNamespace(**d))
        else:
            QgsMessageLog.logMessage("An error occured while try to open url: " + request_url, 'MrMap search',
                                         level=Qgis.Critical)
        self.hide_loader_img()
        """ After the results came back from the metadata server"""
        self.dlg.treeWidgetResource.clear()
        self.dlg.pushButtonLoad.setEnabled(False)
        # read result from json
        if resource_type == 'owscontext':
            search_result = result_object.data
        # paging and count
        number_of_results = result_object.meta.pagination.count
        actual_page = result_object.meta.pagination.page
        number_of_all_pages = result_object.meta.pagination.pages
        self.dlg.treeWidgetResource.clear()
        parent_node = QTreeWidgetItem()
        parent_node.setText(0, self.tr("Found OWS Context documents"))
        for resource in search_result:
            resource_node = QTreeWidgetItem()
            resource_node.setText(0, resource.attributes.title)
            resource_node.setToolTip(0, resource.attributes.title)
            resource_node.setData(1, 0, resource)
            parent_node.addChild(resource_node)
        self.dlg.treeWidgetResource.addTopLevelItem(parent_node)
        self.dlg.treeWidgetResource.expandAll()
        self.dlg.treeWidgetResource.itemClicked.connect(self.on_clicked_wmc)


    def start_search(self, page=1, resource_type='wmc'):
        """
        Build initial search request
        https://documents.geoportal.rlp.de/mediawiki/index.php/SearchInterface
        """
        if resource_type=="owscontext":
            self.start_mrmap_search(page, resource_type)
            return
        self.reset_form(False)
        self.reset_resource_view()
        # read search domain from comboBox
        search_domain = str(self.dlg.comboBoxSearchCatalogue.currentData())
        remote_search = False
        #search_domain = "https://www.geoportal.rlp.de"
        if resource_type == "csw":
            search_path = "/mapbender/php/mod_callCswMetadata.php"
            resource_type = "dataset"
            remote_search = True
        else:
            search_path = "/mapbender/php/mod_callMetadata.php"
        # get string from input field
        search_text_string = self.dlg.textEditSearchText.toPlainText()
        if search_text_string == "":
            search_text = "*"
        else:
            search_text = search_text_string
        search_parameters = {
            "searchResources": resource_type,
            "searchPages": page,
            "maxResults": 10,
            "resultTarget": "webclient",
            "searchEPSG": "EPSG:4326",
            "languageCode": "en",
            "searchText": search_text,
        }
        # add parameters
        request_url = search_domain + search_path + "?" + urllib.parse.urlencode(search_parameters)
        if resource_type == "dataset":
            request_url = request_url + "&resolveCoupledResources=true"
        if remote_search:
            request_url = request_url + "&catalogueId=" + str(self.dlg.comboBoxRemoteCsw.currentData())

        QgsMessageLog.logMessage("Try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        self.show_loader_img()
        result_content = self.open_remote(request_url, timeout=5000)

        if result_content:
            result_object = json.loads(bytes(result_content).decode(), object_hook=lambda d: SimpleNamespace(**d))
            #result_object = json.loads(bytes(result_content).decode())
        else:
            QgsMessageLog.logMessage("An error occured while try to open url: " + request_url, 'GeoPortal.rlp search',
                                         level=Qgis.Critical)
        self.hide_loader_img()
        """ After the results came back from the metadata server"""
        self.dlg.treeWidgetResource.clear()
        self.dlg.pushButtonLoad.setEnabled(False)
        # read result from json
        if resource_type == 'wmc':
            search_result = result_object.wmc.wmc
        if resource_type == 'wms':
            search_result = result_object.wms.wms
        if resource_type == 'dataset' and remote_search:
            search_result = result_object.dataset
        if resource_type == 'dataset' and remote_search is False:
            search_result = result_object.dataset.dataset
        number_of_results = search_result.md.nresults
        actual_page = int(search_result.md.p)
        results_per_page = int(search_result.md.rpp)
        search_time = search_result.md.genTime
        # Add info into form
        self.dlg.numberOfAllResultsLabel.setText(str(number_of_results))
        self.dlg.numberOfPageLabel.setText(str(actual_page))
        # Calculate number of all pages
        number_of_all_pages = math.ceil(int(number_of_results) / results_per_page)
        if actual_page < number_of_all_pages:
            self.dlg.pushButtonPageForward.setEnabled(True)
        if actual_page > 1:
            self.dlg.pushButtonPageBack.setEnabled(True)
        self.dlg.numberOfAllPagesLabel.setText(str(number_of_all_pages))
        QgsMessageLog.logMessage("Used seconds: " + str(search_time), 'GeoPortal.rlp search',
                                         level=Qgis.Info)
        self.dlg.labelSearchAnimation.setText("(" + str(round(search_time, 3)) + " " + self.tr("seconds") + ")")
        # clear last results
        self.dlg.treeWidgetResource.clear()
        parent_node = QTreeWidgetItem()
        if resource_type == 'wmc':
            parent_node.setText(0, self.tr("Found OWS Context documents"))
        if resource_type == 'wms':
            parent_node.setText(0, self.tr("Found map layers (WMS)"))
        if resource_type == 'dataset':
            parent_node.setText(0, self.tr("Found datasets"))
        for resource_type_name in search_result.srv:
            resource_type_node = QTreeWidgetItem()
            resource_type_node.setText(0, resource_type_name.title)
            resource_type_node.setToolTip(0, resource_type_name.title)
            """
            if resource_type == 'dataset' and remote_search is False:
                resource_type_name.respOrg = search_result.srv.respOrg
                resource_type_name.date = search_result.srv.date
                resource_type_name.symbolLink = search_result.srv.symbolLink
                resource_type_name.isopen = search_result.srv.isopen
                resource_type_name.serviceId = search_result.srv.id
            """
            resource_type_node.setData(1, 0, resource_type_name)
            """
            Create layer items recursively
            """
            if resource_type == "wms":
                self.build_layer_tree_recursive(resource_type_name, resource_type_node)
            parent_node.addChild(resource_type_node)
        QgsMessageLog.logMessage("Found " + str(len(search_result.srv)) + " srv objects" , 'GeoPortal.rlp search', level=Qgis.Info)
        self.dlg.treeWidgetResource.addTopLevelItem(parent_node)
        self.dlg.treeWidgetResource.expandAll()
        if resource_type == "wmc":
            self.dlg.treeWidgetResource.itemClicked.connect(self.on_clicked_wmc)
        if resource_type == "wms":
            self.dlg.treeWidgetResource.itemClicked.connect(self.on_clicked_layer)
        if resource_type == "dataset" and remote_search is False:
            self.dlg.treeWidgetResource.itemClicked.connect(self.on_clicked_dataset)
        if resource_type == 'dataset' and remote_search:
            self.dlg.treeWidgetResource.itemClicked.connect(self.on_clicked_remote_dataset)
        # https://www.geoportal.rlp.de/mapbender/php/mod_getCoupledResourcesForDataset.php
        #self.dlg.treeWidgetResource.itemClicked.connect(lambda: self.on_clicked_resource(self, resource_type))
        self.alter_pager(actual_page)

