# -*- coding: utf-8 -*-
"""
/***************************************************************************
 gpapExtract
                                 A QGIS plugin
 Extract data from Geopaparazzi-Smash database
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-11-03
        git sha              : https://github.com/geociencio/gpapext
        copyright            : (C) 2020 by Juan M Bernales
        email                : juanbernales@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import sqlite3
import json
import configparser
from typing import Any, List
from qgis.utils import iface, QgsMessageBar
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QUrl
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import (QgsCoordinateReferenceSystem,
                       QgsCoordinateTransform,
                       QgsProject,
                       QgsPointXY)

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .gpapext_dialog import gpapExtractDialog
import os.path


class gpapExtract:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(self.plugin_dir, 'i18n',
                                   'gpapExtract_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&gpap Data Extract')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('gpapExtract', message)

    def add_action(self, icon_path, text, callback, enabled_flag=True,
                   add_to_menu=True,
                   add_to_toolbar=True,
                   status_tip=None,
                   whats_this=None,
                   parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToDatabaseMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/gpapext/icon.png'
        self.add_action(icon_path,
                        text=self.tr(u'gpap data extractor'),
                        callback=self.run,
                        parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginDatabaseMenu(
                self.tr(u'&gpap Data Extract'),
                action)
            self.iface.removeToolBarIcon(action)

    def show_help(self):
        """Display application help to the user."""
        help_file = 'file:///%s/help/index.html' % self.plugin_dir
        QDesktopServices.openUrl(QUrl(help_file))

    def gpapinput(self):
        """
        Load gpap database and collect field notes
        """
        options = QFileDialog.Options()
        options |= QFileDialog.DontUseNativeDialog
        gpapfile, _filter = QFileDialog.getOpenFileName(self.dlg,
                                                        "Select database", "",
                                                        "Geopaparazzi-Smash databases (*.gpap);;gpap Files (*.gpap)",
                                                        options=options)
        self.dlg.gpapinput.setText(gpapfile)
        con = sqlite3.connect(str(gpapfile))
        sel = 'SELECT * FROM' + ' ' + 'notes'
        notes = []
        with con:
            cur = con.cursor()
            cur.execute(sel)
            rows = cur.fetchall()
            for row in rows:
                jsnst = row[7]
                topy = json.loads(jsnst)
                notes.append(topy['sectionname'])
        frecuenciapalab = [notes.count(p) for p in notes]
        listanotas = dict(zip(notes, frecuenciapalab))
        aux = [(listanotas[key], key) for key in listanotas]
        aux.sort()
        aux.reverse()
        for dat in aux:
            self.dlg.formatosjson.addItem(dat[1])

    def folddest(self):
        """select output directory"""
        carpet = QFileDialog.getExistingDirectory(self.dlg,
                                                  str(
                                                      QtCore.QCoreApplication.translate(
                                                          "dialog",
                                                          "Select Output Directory")),
                                                  str(),
                                                  QFileDialog.ShowDirsOnly)
        self.dlg.fldout.setText(carpet)

    def crsproject(self):
        prj = iface.mapCanvas().mapSettings()
        prj = prj.destinationCrs()
        self.dlg.crsproj4string.setText(prj.toProj4())
        self.dlg.cdepsg.setText(prj.authid())
        self.dlg.qcrsid.setText(str(prj.srsid()))

    def crsactlayer(self):
        layer = iface.activeLayer()
        self.dlg.crsproj4string.clear()
        if layer:
            self.dlg.crsproj4string.setText(str(layer.crs().toProj4()))
            self.dlg.cdepsg.setText(layer.crs().authid())
            self.dlg.qcrsid.setText(str(layer.crs().srsid()))
        else:
            QgsMessageBar.information(self.iface.mainWindow(),
                                      "Get active layer CRS",
                                      "No active layer in TOC.")

    def crsselman(self):
        self.dlg.crsproj4string.clear()
        crsman = self.dlg.crsinput.crs()
        self.dlg.crsproj4string.setText(crsman.toProj4())
        self.dlg.cdepsg.setText(crsman.authid())
        self.dlg.qcrsid.setText(str(crsman.srsid()))

    def processdata(self):
        try:
            arch = str(self.dlg.gpapinput.text())
            con = sqlite3.connect(arch)
            with con:
                arcsv = open(self.dlg.fldout.text() + '/' +
                             self.dlg.formatosjson.currentText() + '.csv', 'w')
                dtstabla: List[Any] = []
                cur = con.cursor()
                cur.execute('SELECT * FROM notes')
                rows = cur.fetchall()
                for dat in rows:
                    topy = json.loads(dat[7])
                    if topy['sectionname'] == str(
                            self.dlg.formatosjson.currentText()):
                        dtstabla = [dts['formitems'][0]['key']
                                    for dts in topy['forms']]
                frequencies = [dtstabla.count(p) for p in dtstabla]
                listable = dict(zip(dtstabla, frequencies))

                for item in listable:
                    arcsv.write(str(item) + ';')

                if self.dlg.gtprjbox.isChecked():
                    project = QgsProject.instance()
                    arcsv.write('lon;lat;xrep;yrep' + '\n')
                    # create coordinate transformation
                    outreftx = self.dlg.cdepsg.toPlainText()
                    insref = QgsCoordinateReferenceSystem(4326)
                    outsref = QgsCoordinateReferenceSystem(int(outreftx[5:]))
                    coordtrf = QgsCoordinateTransform(insref, outsref, project)
                else:
                    arcsv.write('lon;lat' + '\n')

                for dat in rows:
                    lon = dat[1]
                    lat = dat[2]
                    jsndata = json.loads(dat[7])
                    if jsndata['sectionname'] == str(
                            self.dlg.formatosjson.currentText()):
                        datsa = [dts['formitems'][0]['value'] for dts in
                                 jsndata['forms']]
                        if self.dlg.gtprjbox.isChecked():
                            point = coordtrf.transform(
                                QgsPointXY(float(lon), float(lat)))
                            for dts2 in datsa:
                                arcsv.write(dts2 + ';')
                            arcsv.write(str(lon) + ';' + str(lat) + ';' +
                                        str("{0:8.2f}".format(point[0])) + ';' +
                                        str("{0:8.2f}".format(point[1])) + '\n')
                        else:
                            for dts1 in datsa:
                                arcsv.write(dts1 + ';')
                            arcsv.write(str(lon) + ';' + str(lat) + '\n')
                        datsa.clear()
            self.dlg.datos.clear()
            self.dlg.datos.append(self.dlg.gpapinput.text())
            self.dlg.datos.append(str(self.dlg.formatosjson.currentText()))
            self.dlg.datos.append(str(self.dlg.fldout.text() + '\\' +
                                      self.dlg.formatosjson.currentText() +
                                      '.csv'))
            arcsv.close()
            con.close()
        except IOError as e:
            self.messageBar.pushMessage(e, QgsMessageBar.WARNING, 12)

    def reset(self):
        self.dlg.gpapinput.clear()
        self.dlg.fldout.clear()
        self.dlg.formatosjson.clear()
        self.dlg.datos.clear()
        self.dlg.formatosjson.clear()
        self.dlg.cdepsg.clear()
        self.dlg.qcrsid.clear()
        self.dlg.crsproj4string.clear()

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback
        if self.first_start == True:
            self.first_start = False
            self.dlg = gpapExtractDialog()
        # get version
        cfg = configparser.ConfigParser()
        cfg.read(os.path.join(self.plugin_dir, 'metadata.txt'))
        version = cfg.get('general', 'version')
        self.dlg.setWindowTitle('gpap Data Extract - {}'.format(version))
        self.dlg.resetdat.clicked.connect(self.reset)
        self.dlg.hlp.clicked.connect(self.show_help)
        self.dlg.crsinput.setCrs(QgsCoordinateReferenceSystem(4326))
        self.dlg.gpapbrw.clicked.connect(self.gpapinput)
        self.dlg.fldbrw.clicked.connect(self.folddest)
        self.dlg.crsproject.clicked.connect(self.crsproject)
        self.dlg.crslayer.clicked.connect(self.crsactlayer)
        self.dlg.crsinput.crsChanged.connect(self.crsselman)
        self.dlg.process.clicked.connect(self.processdata)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            self.dlg.gpapinput.clear()
            self.dlg.fldout.clear()
            self.dlg.formatosjson.clear()
            self.dlg.datos.clear()
            pass
