# -*- coding: utf-8 -*-

"""
/***************************************************************************
 GoogleStreetViewLayer
                                 A QGIS plugin
 A QGIS plugin that uses your road layer, and creates a layer or table with the most recent Google Street View capture
 data. Google API Key required!
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-04-04
        copyright            : (C) 2022 by Nathan Saylor
        email                : gisn8@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Nathan Saylor'
__date__ = '2022-04-04'
__copyright__ = '(C) 2022 by Nathan Saylor'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import json
import os
import requests

from datetime import datetime

from PyQt5.QtCore import QVariant
from PyQt5.QtWidgets import QMessageBox
from osgeo import ogr

from qgis import processing
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis._core import QgsProperty, QgsProcessingParameterEnum, QgsCoordinateReferenceSystem, QgsField, \
    QgsProject, QgsFeatureRequest, QgsVectorLayer, QgsGeometry, QgsPointXY
from qgis.core import (QgsApplication,
                       QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingParameterField,
                       QgsProcessingParameterString,
                       QgsProcessingParameterBoolean,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink
                       )

# Pycharm thinks not used; it is.
from . import resources

class GoogleStreetViewLayerAlgorithm(QgsProcessingAlgorithm):
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    # Testing mode set to 0 will let the plugin operate as expected.
    # Testing mode set to 1 will:
    # - Print to the console the process functions being accessed, and important variable values along the way.
    # - Unless a valid API key is entered, will assume anything entered under 39 characters is initially valid; if
    # key given is invalid, the status on the calls will always be REQUEST DENIED. A valid key may be used during
    # testing and will make legitimate API calls that could result in charges from Google; a very small sample layer is
    # strongly suggested; USE WITH CAUTION!
    testing = 0

    # Get QGIS User directory for log files and intermediate layer storage.
    # This needs called now as the help panel needs to reference this location before the user begins processing.
    user_path = QgsApplication.qgisSettingsDirPath()
    logs_dir = f'{user_path}gsvl_logs'

    # These timestamped variables will be populated once processing begins.
    now = ''
    log_dir = ''
    log_file = ''
    output_gpkg = ''

    # User input variables
    INPUT = 'INPUT'
    OUTPUT_TYPE = 'OUTPUT_TYPE'
    FIELD_NAME = 'FIELD_NAME'
    API_KEY = 'API_KEY'
    DISCLAIMER = 'DISCLAIMER'
    OUTPUT = 'OUTPUT'

    ##############################
    # PROCESSING TOOLS LIST ICON #
    ##############################
    def icon(self):
        # https://gis.stackexchange.com/questions/382688/changing-the-processing-plugin-icon-qgis
        return QIcon(':/plugins/google_street_view_layer/icon.png')

    #################################
    # REQUIRED PROCESSING FUNCTIONS #
    #################################
    # Afaik, these function names are specifically looked for by QGIS to run. The first three here we tweak,
    # the other six are templated.

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Limiting input to line geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Road layer (line geometry)'),
                [QgsProcessing.TypeVectorLine]
            )
        )

        #############
        output_type = QgsProcessingParameterEnum(
                self.OUTPUT_TYPE,
                self.tr('Choose output preference:'),
                options=[
                    'Duplicate road layer with amended attributes',
                    'Joinable point layer adjusted to Street View location',
                    'Joinable table (requires table-ready output format)'
                ],
                allowMultiple=False, defaultValue=None
            )
        self.addParameter(output_type)

        field_name = QgsProcessingParameterField(
                self.FIELD_NAME,
                'Choose a field with unique values.',
                parentLayerParameterName=self.INPUT
            )
        self.addParameter(field_name)

        api_key = QgsProcessingParameterString(
                self.API_KEY,
                self.tr('Google API Key'),
                ' '
            )
        self.addParameter(api_key)

        disclaimer = QgsProcessingParameterBoolean(
                self.DISCLAIMER,
                # Formatted like this to avoid the text running off the form window when narrowed.
                self.tr("""DISCLAIMER: 
An API call will be used to validate the API key 
and then for every feature or selected feature in 
the chosen layer. You may incur use charges from 
Google. See your Google API agreement for 
details.
The developer of this tool bears no responsibility 
for such charges. Check here if you understand 
and can legally consent to Google's charges to 
continue."""), 0
        )
        self.addParameter(disclaimer)
        ##################

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # Process is set up and outputted here, but the processing itself is conducted in the run_process function.

        self.feedback = feedback

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)

        output_type = self.parameterAsEnum(parameters, self.OUTPUT_TYPE, context)
        field_name = self.parameterAsString(parameters, self.FIELD_NAME, context)
        api_key = self.parameterAsString(parameters, self.API_KEY, context).strip()
        disclaimer = self.parameterAsBoolean(parameters, self.DISCLAIMER, context)

        # Create layer from source so processes can read it.
        layer = source.materialize(QgsFeatureRequest())

        self.init_logfiles()

        self.print(f"""Inputs: 
        source (obj): {source}
        output_type: {output_type}
        field_name: {field_name}
        api_key: {api_key}
        disclaimer: {disclaimer}
        layer (obj): {layer}
        """)

        source = self.run_process(layer, output_type, field_name, api_key, disclaimer, feedback)

        # This is to catch invalid CRS on table outputs. Layer CRS outputs should end up being the same otherwise.
        if source.sourceCrs() != layer.crs():
            crs = QgsCoordinateReferenceSystem("EPSG:4326")
        else:
            crs = source.sourceCrs()

        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                                               context, source.fields(), source.wkbType(), crs)

        # Normally, the progress bar is updated at this stage, but since the majority of processing takes places in the
        # API calls and filling in the fields, moved it to self.populate_gsv_fields; everything else including this goes
        # quite fast.
        self.print(f'Output featureCount: {source.featureCount()}')
        features = source.getFeatures()

        for feature in features:
            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

        self.print('Sending to selected destination.')
        return {self.OUTPUT: dest_id}

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        # The Help panel
        return self.tr("Need a Google Maps API Key? <a href=\"https://developers.google.com/maps/documentation/"
                       "javascript/get-api-key\">Click here!</a>\n"
                       "For best results, it is recommended that the road layer you use be broken down block by block;"
                       " the Vector overlay > Split with lines processing tool in QGIS can be used for this, but be"
                       " sure to QA/QC the results and be familiar with the data before continuing.\n"
                       "\n"
                       f"Should the process fail, review the log files found in"
                       f" <a href=\"file:///{self.logs_dir}\">{self.logs_dir}</a>."
                       f" A Geopackage with the processed layers is included and may save you from unnecessary API"
                       f" calls."
                       )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Google Street View Layer'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Google API Tools'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return GoogleStreetViewLayerAlgorithm()

    ##################
    # FEEDBACK TOOLS #
    ##################
    def ts(self):
        # Timestamp function
        return datetime.now().strftime("%H:%M:%S")

    def print(self, msg):
        f = open(self.log_file, "a")
        f.write(f'\n{self.ts()} - {msg}')
        f.close()

        if self.testing == 1:
            print(f'{self.ts()} - {msg}')

        self.feedback.pushInfo(f'{self.ts()} - {msg}')

    def warning(self, msg):
        # This is causing Windows instances to crash. Struggling to find a messagebar solution.
        # self.msgbox(title='ATTENTION', icon=QMessageBox.Warning, text=msg)
        self.print(f'ATTENTION: {msg}')
        self.feedback.pushWarning(f'ATTENTION: {msg}')

    def msgbox(self, title=None, icon=None, text=None, info=None):
        msg = QMessageBox()
        if title:
            msg.setWindowTitle(title)
        if icon:
            # Options: QMessageBox.Question, Information, Warning, Critical
            msg.setIcon(icon)
        if text:
            msg.setText(text)
        if info:
            msg.setInformativeText(info)
        msg.setStandardButtons(QMessageBox.Ok)
        msg.exec_()

    ################
    # SHARED TOOLS #
    ################
    def get_field(self, input_layer, field_name):
        field_idx = input_layer.fields().indexFromName(field_name)
        field = input_layer.fields()[field_idx]

        return field

    def build_url(self, api_key, y, x):
        # Will be easier to find and change in the event the URL format changes.
        url = f'https://maps.googleapis.com/maps/api/streetview/metadata?location={y},{x}&key={api_key}'
        self.print(f'URL call: {url}')
        return url

    def add_intermediate_layer_to_gpkg(self, input_layer):
        self.print(f'Sending {input_layer.name()} to {self.output_gpkg}.')

        processing.run("native:package", {
            'LAYERS': [input_layer],
            'OUTPUT': self.output_gpkg,
            'OVERWRITE': False,
            'SAVE_STYLES': False,
            'SAVE_METADATA': False,
            'SELECTED_FEATURES_ONLY': False})

    ##############
    # PROCESSING #
    ##############
    def init_logfiles(self):
        self.now = datetime.now().strftime("%Y-%m-%d_T%H_%M_%S")

        # Create directory for log files. We don't want in plugin folder in the even it is uninstalled or reinstalled.
        if not os.path.exists(self.logs_dir):
            os.makedirs(self.logs_dir)

        # Create logging directory for this session.
        self.log_dir = f'{self.logs_dir}/{self.now}'
        if not os.path.exists(self.log_dir):
            os.makedirs(self.log_dir)

        self.log_file = f'{self.log_dir}/log.txt'

        self.output_gpkg = f'{self.log_dir}/outputs.gpkg'

        # Creates the log_file
        f = open(self.log_file, "x")
        f.write(f'Began processing at {self.now}')
        f.close()

    def run_process(self, layer, output_type, field_name, api_key, disclaimer, feedback):
        self.print(f"""Variable checks:
        source: {layer}
        source_name: {layer.sourceName()}
        feature count: {layer.featureCount()}
        output_type: {output_type}
        field: {field_name}
        api_key: {api_key}
        len(api_key): {len(api_key)}
        disclaimer: {disclaimer}
        """)

        resume = True

        # Want to make sure Disclaimer is agreed to (simple check) and field validates (process check) before pinging
        # their API for validation.
        if disclaimer != 1:
            self.warning('You must acknowledge and consent to the disclaimer to continue!')
            resume = False

        if resume:
            field_validated = self.check_for_unique_values(layer, field_name)
            self.print(f'field_validated = {field_validated}')

            if field_validated is False:
                self.warning('Selected field does not contain unique values.')
                resume = False

        if resume:
            valid_api_key = self.validate_api_key(api_key)

            if valid_api_key != 1:
                self.warning('Invalid API key! Please provide a valid API key.')
                resume = False

        # Main processing
        if resume:
            # If not creating a duplicate layer, strip the fields down to selected join field
            # 0     'Duplicate road layer with amended attributes',
            # 1     'Joinable point layer'
            # 2     'Joinable table'
            if output_type > 0:
                reduced_fields_layer = self.reduce_fields(layer, field_name)
                local_input = reduced_fields_layer
            else:
                local_input = layer

            # Create midpoints layer from local_input
            midpoints_layer = self.create_midpoints_layer(local_input)

            # Reproject layer to google readable projection
            reprojected_layer = self.reproject_layer(midpoints_layer)

            # Add geometry data. The next few steps is editing this layer.
            geom_added_layer = self.add_geom_attributes(reprojected_layer)

            # Add Google Street View fields
            gsv_layer = self.add_gsv_fields(geom_added_layer)

            # Changes the name of the layer for the TOC
            gsv_layer.setName('gsv_layer')

            # Add GSV data
            self.populate_gsv_fields(gsv_layer, api_key, field_name, feedback)

            if output_type == 0:  # Duplicated roads layer with added GSV data
                combined_layer = self.join_to_duplicate_source(source_layer=layer, input_layer=gsv_layer, joining_field_name=field_name)
                final_output = combined_layer
            if output_type == 1:  # GSV point layer
                final_output = gsv_layer
            if output_type == 2:  # GSV table only
                table = self.export_table(gsv_layer)
                final_output = table

            # Adds stored outputs to map in testing mode.
            self.add_gpkg_layers()

            return final_output

        else:
            return None

    ########################
    # PROCESSING FUNCTIONS #
    ########################
    def check_for_unique_values(self, input_layer, field_name):
        self.print('Validating selected field')

        field_idx = input_layer.fields().indexFromName(field_name)
        unique = len(input_layer.uniqueValues(field_idx))

        self.print(f'Count checks: \n features = {input_layer.featureCount()} \n unique values = {unique}')
        if unique == input_layer.featureCount():
            return True
        else:
            return False

    def validate_api_key(self, api_key):
        # This uses an Street View API call at 0,0 to either validate the key and continue the process, or save the
        # trouble and stop the process. If testing mode is on, this will return any key value entered is valid, but if
        # invalid, Google will deny the calls.
        # Derived from https://www.askpython.com/python/examples/pull-data-from-an-api
        # Two possible outcomes:
        # Valid: "status" : "ZERO_RESULTS"
        # Invalid: "status" : "REQUEST_DENIED"

        if len(api_key) < 39 and self.testing == 1:
            return 1
        else:
            req = self.build_url(y=0, x=0, api_key=api_key)

            json_returned = self.get_gsv_json(req, fid=-1)  # fid not necessary for this, but for future processing
            status = json_returned['status']

            self.print(status)

            if status == 'REQUEST_DENIED':
                return 0
            else:
                return 1

    def reduce_fields(self, input_layer, field_name):
        self.print('Reducing fields')

        field = self.get_field(input_layer, field_name)
        self.print(f'input_layer obj: {input_layer}')
        self.print(f'field obj: {field}')
        reduced = processing.run("native:refactorfields", {
            'INPUT': input_layer,
            # 'FIELDS_MAPPING': [{'expression': '"segid"', 'length': -1, 'name': 'segid', 'precision': 0, 'type': 6}],
            'FIELDS_MAPPING': [{
                'expression': f'"{field.name()}"',
                'length': {field.length()},
                'name': f'{field.name()}',
                'precision': field.precision(),
                'type': field.type()
            }],
            'OUTPUT': 'TEMPORARY_OUTPUT'
        })['OUTPUT']

        self.add_intermediate_layer_to_gpkg(reduced)

        return reduced

    def create_midpoints_layer(self, input_layer):
        self.print('Creating midpoints layer')
        # Using $length/2 takes the length of the feature in meters, then plots the mid point in feet.
        # For example, if the segment is 1000 meters, it's plotting the point at 500 feet down line. Have to compensate
        # by multiplying the result by 3.28084 (ft to m).
        # length($geometry)/2 uses the proper unit throughout.

        midpoints_layer = processing.run("native:interpolatepoint", {
            'INPUT': input_layer,
            'DISTANCE': QgsProperty.fromExpression('length($geometry)/2'),
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT})['OUTPUT']

        self.add_intermediate_layer_to_gpkg(midpoints_layer)

        return midpoints_layer

    def reproject_layer(self, input_layer):
        self.print('Reprojecting layer')

        reprojected_layer = processing.run("native:reprojectlayer", {
            'INPUT': input_layer,
            'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:4326'),
            'OUTPUT': 'TEMPORARY_OUTPUT'})['OUTPUT']

        self.add_intermediate_layer_to_gpkg(reprojected_layer)

        return reprojected_layer

    def add_geom_attributes(self, input_layer):
        self.print('Adding geometry attributes')

        geom_added_layer = processing.run("qgis:exportaddgeometrycolumns", {
            'INPUT': input_layer,
            'CALC_METHOD': 0, 'OUTPUT': 'TEMPORARY_OUTPUT'})['OUTPUT']

        self.add_intermediate_layer_to_gpkg(geom_added_layer)

        return geom_added_layer

    def add_gsv_fields(self, input_layer):
        self.print(f'Adding gsv fields.')
        layer_dp = input_layer.dataProvider()
        layer_dp.addAttributes([
            QgsField("gsv_lat", QVariant.Double),
            QgsField("gsv_lng", QVariant.Double),
            QgsField("gsv_url", QVariant.String),
            QgsField("gsv_status", QVariant.String),
            QgsField("gsv_date", QVariant.String),
            QgsField("pano_id", QVariant.String),
            QgsField("gsv_json", QVariant.String)
        ])
        input_layer.updateFields()

        # This function operating on existing and not outputting a new layer.
        # self.add_intermediate_layer_to_gpkg(input_layer)

        return input_layer

    def populate_gsv_fields(self, layer, api_key, field_name, feedback):
        self.print('Populating GSV fields')

        layer.startEditing()

        # Compute the number of steps to display within the progress bar and
        # get features from source
        self.print(f"FeatureCount to populate GSV fields: {layer.featureCount()}")
        total = 100.0 / layer.featureCount() if layer.featureCount() else 0

        features = layer.getFeatures('1=1')
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            api_url = self.build_url(api_key, y=feature['ycoord'], x=feature['xcoord'])

            json_returned = self.get_gsv_json(api_url, fid=feature[f'{field_name}'])
            feature['gsv_json'] = f'{json_returned}'

            feature['gsv_status'] = json_returned['status']

            if json_returned['status'] == 'OK':
                feature['gsv_lat'] = json_returned['location']['lat']
                feature['gsv_lng'] = json_returned['location']['lng']
                feature['gsv_date'] = json_returned['date']
                # Not prepending gsv for SHP headers.
                feature['pano_id'] = json_returned['pano_id']
                feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(json_returned['location']['lng'], json_returned['location']['lat'])))

                # We DON'T want the api_url being in the data to accidentally be putting strikes on the API Key if used.
                # Instead, we'll use the streetview url! Invalid locations only return a black image.
                # Alternatively, instead of viewpoint, could use the pano_id= parameter, but this template will be more
                # useful whether or not we have the proper pano_id.
                # Example: https://www.google.com/maps/@?api=1&map_action=pano&viewpoint=40.64769463488166,-83.60925155578865
                feature['gsv_url'] = f"https://www.google.com/maps/@?api=1&map_action=pano&viewpoint={feature['ycoord']},{feature['xcoord']}"

            # Update the progress bar
            feedback.setProgress(int(current * total))
            self.print(f'Export progress: {int(current * total)}%')

            layer.updateFeature(feature)

        layer.commitChanges()

    def get_gsv_json(self, url, fid):
        self.print('Getting JSON')

        # The first piece of info we want is the "status". There are three returns possible:
        # GSV found: "status" : "OK"
        # GSV not found: "status" : "ZERO_RESULTS"
        # Bad or no API Key provided: "status" : "REQUEST_DENIED" (An "error_message" is also returned with details.)

        response_api = requests.get(url)

        data = response_api.text

        # parse_json carries the dictionary for each call
        json_returned = json.loads(data)

        self.print(f'JSON returned for feature {fid}: {json_returned}')

        return json_returned

    def join_to_duplicate_source(self, source_layer, input_layer, joining_field_name):
        self.print('Joining output to duplicate source')

        source_fields = []
        for field in source_layer.fields():
            source_fields.append(field.name())

        fields_to_join = []

        for field in input_layer.fields():
            if field.name() not in source_fields:
                fields_to_join.append(field.name())

        joined_layer = processing.run("native:joinattributestable", {
            'INPUT': source_layer,
            'FIELD': joining_field_name,
            'INPUT_2': input_layer,
            'FIELD_2': joining_field_name,
            'FIELDS_TO_COPY': fields_to_join,
            'METHOD': 1, 'DISCARD_NONMATCHING': False, 'PREFIX': '', 'OUTPUT': 'TEMPORARY_OUTPUT'})['OUTPUT']

        self.add_intermediate_layer_to_gpkg(joined_layer)

        return joined_layer

    def export_table(self, input_layer):
        self.print('Exporting to table.')

        table_layer = QgsVectorLayer("NoGeometry", "gsv_table", "memory")

        feats = [feat for feat in input_layer.getFeatures()]

        table_layer_data = table_layer.dataProvider()
        attr = input_layer.dataProvider().fields().toList()
        table_layer_data.addAttributes(attr)
        table_layer.updateFields()
        table_layer_data.addFeatures(feats)

        self.add_intermediate_layer_to_gpkg(table_layer)

        return table_layer

    def add_gpkg_layers(self):
        if self.testing == 1:
            layer_names = [l.GetName() for l in ogr.Open(self.output_gpkg)]

            for layer_name in layer_names:
                self.print(f'Adding {layer_name} to map')
                layer = QgsVectorLayer(self.output_gpkg + f'|layername={layer_name}', layer_name, 'ogr')
                QgsProject.instance().addMapLayer(layer)
