# -*- coding: utf-8 -*-
#./gis_auditor_report_dialog.py
"""
/***************************************************************************
 GISAuditorReportDialog
                                 A QGIS plugin
 It allows users to define and run a series of essential validation checks across vector layers and standalone attribute tables within a QGIS project. It helps users identify data quality issues by checking shared keys, spatial relationships, and duplicate values, with results available in clear, exportable reports.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-09-28
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Lei Ding
        email                : lleidding@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import (QDialogButtonBox, QHBoxLayout, QPushButton, QWidget, QComboBox, QMessageBox)
from qgis.PyQt.QtCore import Qt
from qgis.gui import QgsMapLayerComboBox, QgsFieldComboBox
from qgis.core import QgsMapLayerProxyModel 
from .core.audit_runner import AuditRunner # 
from qgis.PyQt.QtWidgets import QDialogButtonBox, QFileDialog, QMessageBox # Added QFileDialog, QMessageBox
from qgis.PyQt.QtCore import QDateTime, QDir, QUrl # Added QDateTime, QDir for path handling
from qgis.PyQt.QtGui import QDesktopServices


FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'gis_auditor_report_dialog_base.ui'))



class GISAuditorReportDialog(QtWidgets.QDialog, FORM_CLASS):
    # Set the maximum allowed rows per check section
    MAX_ROWS_PER_CHECK = 10
    
    # Define Spatial Relationship Options for the ComboBox
    SPATIAL_RELATIONSHIPS = [
        ("Select relationship...", ""), # Placeholder
        ("Child must be Within Parent", "within"),
        ("Child must be Contained By Parent", "contains"),
        ("Child must Intersect Parent", "intersects"),
        
    ]
    
    def __init__(self, parent=None):
        """Constructor."""
        super(GISAuditorReportDialog, self).__init__(parent)
        self.setWindowFlags(Qt.WindowStaysOnTopHint)
        self.setupUi(self)
        self.progressBar.setValue(0) # Start from 0 not the default 24, but can use a bigger number when take screenshot for product page
        # Dictionary to store dynamically created row widgets for each check type.
        self.check_rows = {
            'duplicate': [],
            'spatial': [],
            'exclusion': []
        }
        
        # 1. Initialize UI components and connect signals
        self._init_button_box()
        self._init_check_ui()
        self.siteCodeLineEdit = self.findChild(QtWidgets.QLineEdit, "siteCodeLineEdit")
        
        # 2. Add an initial row for each section to start
        self._add_check_row('duplicate')
        self._add_check_row('spatial')
        self._add_check_row('exclusion')


    # ----------------------------------------------------------------------
    # UI Initialization and Signal Connection Methods
    # ----------------------------------------------------------------------

    def _init_button_box(self):
        """Changing 'Ok' to 'Run'."""
        ok_button = self.button_box.button(QDialogButtonBox.Ok)
        if ok_button:
            ok_button.setText("Run")
            ok_button.setObjectName("runButton")
            # Connect 'Run' button to the main execution function
            ok_button.clicked.connect(self.run_audit_checks)
 

    def _init_check_ui(self):
        """Connects the 'Add new row' buttons for all sections."""
        
        # Duplicate Check, use lambda here
        self.addDuplicateLayerButton.clicked.connect(lambda: self._add_check_row('duplicate'))
        
        # Spatial Relationship Check
        self.addSpatialButton.clicked.connect(lambda: self._add_check_row('spatial'))
        
        # Exclusion Zone Check
        self.addExclusionButton.clicked.connect(lambda: self._add_check_row('exclusion'))

    # ----------------------------------------------------------------------
    # Dynamic Row Management (Creation and Removal)
    # ----------------------------------------------------------------------

    def _add_check_row(self, check_type: str):
        """
        Dynamically creates and adds a new check configuration row to the UI.
        Enforces MAX_ROWS_PER_CHECK limit.
        """
        # 1. Row limit check
        if len(self.check_rows[check_type]) >= self.MAX_ROWS_PER_CHECK:
            QMessageBox.warning(self, "Limit Reached", 
                                f"Maximum of {self.MAX_ROWS_PER_CHECK} checks reached for {check_type}.")
            return

        # 2. Create the row container and layout
        row_widget = QWidget()
        row_layout = QHBoxLayout(row_widget)
        row_layout.setContentsMargins(0, 5, 0, 5) # Set spacing
        
        # 3. Add components based on check type
        if check_type == 'duplicate':
            self._setup_duplicate_row(row_layout)
            target_layout = self.duplicateCheckContent
            
        elif check_type == 'spatial':
            self._setup_spatial_row(row_layout)
            target_layout = self.spatialCheckContent
            
        elif check_type == 'exclusion':
            self._setup_exclusion_row(row_layout)
            target_layout = self.exclusionCheckContent
            
        else:
            return

        # 4. Create and connect the Remove Button (common to all)
        remove_btn = QPushButton("Remove")
        remove_btn.setObjectName(f"remove{check_type.capitalize()}RowButton")
        remove_btn.setFixedWidth(80)
        remove_btn.clicked.connect(
            lambda checked, widget=row_widget: self._remove_check_row(widget, check_type)
        )
        row_layout.addWidget(remove_btn, 10) # 10% width share

        # 5. Insert the new row into the target QVBoxLayout, just before the 'Add' button --- IMPORTANT INSERT
        # addWidget is most common, add to end. but this can give me the control of 'add below the last item', which is my 'Add' button!
        target_layout.insertWidget(
            target_layout.count() - 1, 
            row_widget
        )
        
        # 6. Store the row reference
        self.check_rows[check_type].append(row_widget)


    
    def _setup_duplicate_row(self, layout: QHBoxLayout):
        """Sets up QgsMapLayerComboBox and QgsFieldComboBox for Duplicate Check."""
        
        layer_combo = QgsMapLayerComboBox()
        layer_combo.setObjectName("duplicateLayerCombo")
        
        
        
        
        field_combo = QgsFieldComboBox()
        field_combo.setObjectName("duplicateFieldCombo")
        
        # Auto-update field combobox when layer changes
        layer_combo.layerChanged.connect(field_combo.setLayer)

        layout.addWidget(layer_combo, 60)
        layout.addWidget(field_combo, 30)

    def _setup_spatial_row(self, layout: QHBoxLayout):
        """Sets up Parent/Child Layer and Unique Field Comboboxes for Spatial Check."""

        
        # 1. Parent Layer and ID
        parent_layer = QgsMapLayerComboBox()
        parent_layer.setObjectName("spatialParentLayerCombo")
        parent_layer.setFilters(QgsMapLayerProxyModel.PolygonLayer)


        # 2. Child Layer and ID
        child_layer = QgsMapLayerComboBox()
        child_layer.setObjectName("spatialChildLayerCombo")
        child_layer.setFilters(QgsMapLayerProxyModel.VectorLayer)

        child_unique_field_combo = QgsFieldComboBox()
        child_unique_field_combo.setObjectName("spatialChildUniqueFieldCombo")
        child_layer.layerChanged.connect(child_unique_field_combo.setLayer) # Send signal of the new layer, get the list
        
        # 3. Add all the widgets to the layout
   
        layout.addWidget(parent_layer, 25)

        layout.addWidget(child_layer, 25)
  
        layout.addWidget(child_unique_field_combo, 15)




    def _setup_exclusion_row(self, layout: QHBoxLayout):
        """Sets up Target Layer, Exclusion Zone Layer, and Unique Field Comboboxes."""
        
        
        # 1. Target Layer and ID
        target_layer = QgsMapLayerComboBox()
        target_layer.setObjectName("exclusionTargetLayerCombo")
        target_layer.setFilters(QgsMapLayerProxyModel.VectorLayer)

        target_unique_field_combo = QgsFieldComboBox()
        target_unique_field_combo.setObjectName("exclusionTargetUniqueFieldCombo")
        target_layer.layerChanged.connect(target_unique_field_combo.setLayer)

        # 2. Exclusion Zone Layer and ID
        exclusion_layer = QgsMapLayerComboBox()
        exclusion_layer.setObjectName("exclusionZoneLayerCombo")
        exclusion_layer.setFilters(QgsMapLayerProxyModel.VectorLayer)



        # Add all the widgets to the layout
        layout.addWidget(exclusion_layer, 25)
        layout.addWidget(target_layer, 25)
        layout.addWidget(target_unique_field_combo, 15)

        




    def _remove_check_row(self, row_widget: QWidget, check_type: str):
        """Removes a row widget from the UI and the internal list."""
        if row_widget in self.check_rows[check_type]:
            # Remove widget from layout and delete
            row_widget.setParent(None)
            row_widget.deleteLater()
            
            # Remove reference from the list
            self.check_rows[check_type].remove(row_widget)

    # User has to provide a site code
    def _validate_site_code(self) -> bool:
        """Validates that the site code is not empty."""
        site_code = self.siteCodeLineEdit.text().strip()
        if not site_code:
            QMessageBox.warning(
                self, 
                "Title Required", 
                "Please enter a Project Name/Site Code/Report Title before running the audit."
            )
            self.siteCodeLineEdit.setFocus()
            return False
        return True

    # ----------------------------------------------------------------------
    # Configuration Collection Methods
    # ----------------------------------------------------------------------

    def get_duplicate_check_configs(self) -> list:
        """Collects Layer ID and Field Name for all Duplicate Check rows."""
        configs = []
        for row_widget in self.check_rows['duplicate']:
            layer_combo = row_widget.findChild(QgsMapLayerComboBox, "duplicateLayerCombo")
            field_combo = row_widget.findChild(QgsFieldComboBox, "duplicateFieldCombo")
            
            if layer_combo and field_combo and layer_combo.currentLayer() and field_combo.currentField():
                layer = layer_combo.currentLayer()
                configs.append({
                    'check_type': 'duplicate',
                    'layer_id': layer.id(),
                    'layer_name': layer.name(),
                    'field_name': field_combo.currentField()
                })
        return configs
        
    def get_spatial_check_configs(self) -> list:
        """Collects Parent/Child Layers, Relationship, and Unique Field Names."""
        configs = []
        for row_widget in self.check_rows['spatial']:
            parent_combo = row_widget.findChild(QgsMapLayerComboBox, "spatialParentLayerCombo")
            child_combo = row_widget.findChild(QgsMapLayerComboBox, "spatialChildLayerCombo")
            child_id_combo = row_widget.findChild(QgsFieldComboBox, "spatialChildUniqueFieldCombo")

            if parent_combo and child_combo and child_id_combo:
                parent_layer = parent_combo.currentLayer()
                child_layer = child_combo.currentLayer()
                child_unique_field = child_id_combo.currentField()
                
                # Ensure both layers are selected and a unique field is chosen
                if parent_layer and child_layer and child_unique_field:
                    configs.append({
                        'check_type': 'spatial',
                        'parent_id': parent_layer.id(),
                        'parent_name': parent_layer.name(),
                        'child_id': child_layer.id(),
                        'child_name': child_layer.name(),
                        'child_unique_field': child_unique_field
                    })
                    print(f"[DEBUG] Spatial config added: {parent_layer.name()} -> {child_layer.name()}")
                else:
                    print(f"[WARNING] Skipping incomplete spatial check row")
                    
        return configs


    def get_exclusion_check_configs(self) -> list:
        """Collects Target Layer and Exclusion Layer for all Exclusion Zone Check rows."""
        configs = []
        for row_widget in self.check_rows['exclusion']:
            target_combo = row_widget.findChild(QgsMapLayerComboBox, "exclusionTargetLayerCombo")
            exclusion_combo = row_widget.findChild(QgsMapLayerComboBox, "exclusionZoneLayerCombo")
            target_id_combo = row_widget.findChild(QgsFieldComboBox, "exclusionTargetUniqueFieldCombo")

            # Check if the widgets were found and if layers and fields are selected
            if (target_combo and exclusion_combo and target_id_combo):
                target_layer = target_combo.currentLayer()
                exclusion_layer = exclusion_combo.currentLayer()
                target_unique_field = target_id_combo.currentField()
                
                # Ensure all required values are present
                if target_layer and exclusion_layer and target_unique_field:
                    configs.append({
                        'check_type': 'exclusion',
                        'exclusion_id': exclusion_layer.id(),  # ✅ FIXED: Added this!
                        'exclusion_name': exclusion_layer.name(),
                        'target_id': target_layer.id(),
                        'target_name': target_layer.name(),
                        'target_unique_field': target_unique_field
                    })
                    print(f"[DEBUG] Exclusion config added: {target_layer.name()} vs {exclusion_layer.name()}")
                else:
                    print(f"[WARNING] Skipping incomplete exclusion check row")
                    
        return configs

    # Get a select directory
    def _get_report_save_path(self):
        """Opens a file dialog to let the user choose the output HTML report path."""
        
        # Get site code for the default filename
        site_code = self.siteCodeLineEdit.text().strip()
        if site_code:
            default_name = f"{site_code}_GIS_Audit_Report_{QDateTime.currentDateTime().toString('yyyyMMdd_hhmmss')}.html"
        else:
            default_name = f"GIS_Audit_Report_{QDateTime.currentDateTime().toString('yyyyMMdd_hhmmss')}.html"
        
        # Use QFileDialog to get a save file name
        filepath, _ = QFileDialog.getSaveFileName(
            self,
            "Save Audit Report",
            os.path.join(QDir.homePath(), default_name),
            "HTML Report (*.html)"
        )
    
        return filepath

    def _handle_audit_completion(self, report_path: str):
        """
        Handles the actions taken when the AuditRunner finishes.
        This replaces the old self.accept() call.
        """
        
        # 1. Hide the progress bar
        #self.progressBar.hide()
        
        # 2. Show the success message (QMessageBox)
        QMessageBox.information(
            self,
            "Audit Completed",
            f"The GIS Audit has successfully finished.\n\nReport saved to:\n{report_path} and opened in your default website."
        )
        QDesktopServices.openUrl(QUrl.fromLocalFile(report_path))
    # ----------------------------------------------------------------------
    # Main Execution
    # ----------------------------------------------------------------------

    def run_audit_checks(self):
        """Main function to collect all configurations and initiate the audit process."""
        # 1. Validate site code first (NEW)
        print('[DEBUG] Run button clicked')
        if not self._validate_site_code():
            print('[DEBUG] Site code validation failed')
            return
        print('[DEBUG] Passed site code validation')
        # 2. Collect all configurations
        all_configs = []
        all_configs.extend(self.get_duplicate_check_configs())
        all_configs.extend(self.get_spatial_check_configs())
        all_configs.extend(self.get_exclusion_check_configs())

        # 3. Check for tasks
        if not all_configs:
            QMessageBox.warning(
                self, 
                "No Checks Configured", 
                "No check configurations have been selected.\n\n"
                "Please add and configure at least one check in any section."
            )
            return

        # 4 Get the report save path from the user
        report_path = self._get_report_save_path()
        if not report_path:
            # User cancelled the save dialog
            return
            
        # 5. Collect the high-level report configuration
        report_config = {
            'site_code': self.siteCodeLineEdit.text().strip()
        }
                        
        # 6. Start the audit process: Instantiate the Runner
        self.runner = AuditRunner(
            all_configs=all_configs,
            progress_bar=self.progressBar,
            report_path=report_path,
            report_config=report_config, 
            parent=self
        )
        
        # 7. Connect the Runner's signal to the completion handler
        self.runner.report_generated.connect(self._handle_audit_completion)

        # 6. Show the progress bar and Start the audit work
        self.progressBar.setValue(0)
        self.progressBar.show()
        self.runner.run_checks()