# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GifClicker
                                 A QGIS plugin
 Select GIFs to play when you click on the canvas with different map tools
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-08-08
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Dror Bogin
        email                : dror.bogin@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.gui import QgsMapMouseEvent, QgsMapToolPan
from qgis.PyQt.QtGui import QMovie
from qgis.PyQt.QtWidgets import QLabel
from qgis.PyQt.QtCore import pyqtSlot, pyqtSignal, Qt, QSize
from qgis.PyQt import sip
from qgis.core import QgsMessageLog, Qgis

MESSAGE_CATEGORY = 'QGIS GIF Clicker Plugin'

class GifClickerMapToolPan(QgsMapToolPan):

    gifChanged = pyqtSignal(QMovie)
    enabled = True

    def setEnabled(self, enabled: bool):
        self.enabled = enabled
        if not self.enabled:
            self.canvas().unsetMapTool(self)
            if( hasattr(self, 'gifInstance') and self.gifInstance is not None):
                self.gifInstance.jumpToFrame(self.gifInstance.frameCount() - 1)
            self.gifInstance = QMovie()
            QgsMessageLog.logMessage('GifClicker: Map tool disabled', MESSAGE_CATEGORY, level=Qgis.Info)
        else:
            self.canvas().setMapTool(self)
            if self.gif is not None:
                self.gifInstance = QMovie(self.gif.fileName())
            QgsMessageLog.logMessage('GifClicker: Map tool enabled', MESSAGE_CATEGORY, level=Qgis.Info)
        return self.enabled
    
    def isEnabled(self):
        return self.enabled
    
    def getGif(self) -> QMovie:
        if self.gif is None:
            return QMovie()
        return self.gif
        

    def setGif(self, gif: QMovie):
        if not isinstance(gif, QMovie):
            QgsMessageLog.logMessage('GifClicker: Invalid GIF instance', MESSAGE_CATEGORY, level=Qgis.Warning)
            return
        if gif.isValid() is False:
            QgsMessageLog.logMessage('GifClicker: GIF instance is not valid', MESSAGE_CATEGORY, level=Qgis.Warning)
            return
        if self.gif is not None:
            self.gif.stop()
        self.gif = gif
        self.gifChanged.emit(self.gif)
    
    def setGifUrl(self, gif_url: str):
        self.gif = QMovie(gif_url)
        self.gif.setScaledSize(QSize(50, 50))
        self.gifChanged.emit(self.gif)

    @pyqtSlot(QMovie)
    def gifChangedSlot(self, gif: QMovie):
        self.gif = gif

    def setupLabel(self):
        try:
            if not hasattr(self, 'labels'):
                self.labels = {}
            self.labelIdx = self.labelIdx + 1 if self.labelIdx else 0
            self.labels[self.labelIdx] = QLabel()
            self.labels[self.labelIdx].setObjectName('gifLabel')
            self.labels[self.labelIdx].setStyleSheet("QLabel#gifLabel { background-color: rgba(0,0,0,0) }")
            return self.labelIdx
        except Exception as e:
            QgsMessageLog.logMessage(f'{str(e)}', MESSAGE_CATEGORY, level=Qgis.Critical)

    def setGifSize(self, width, height):
        self.gif.setScaledSize(width, height)

    @staticmethod
    def onMovieFinished(scene,itemIdx):
        # For debugging purposes
        #QgsMessageLog.logMessage(f'Gif {itemIdx} played on scene {scene}', MESSAGE_CATEGORY, level=Qgis.Info)
        try:
            if scene is None or itemIdx is None:
                return
            scene.removeItem(itemIdx)

        except Exception as e:
            QgsMessageLog.logMessage(f'Error removing item from scene: {str(e)}', MESSAGE_CATEGORY, level=Qgis.Critical)


    def onFrameChanged(self, frame, scene=None, itemIdx=None):
        """Check frame number and stop the movie if it reaches the last frame and has a higher loop count than allowed.
        This is to prevent the movie from looping indefinitely if the loop count is set to a value greater than 1.
        Args:
            frame (int): The current frame number of the movie.
        """
        if self.gifInstance is None or not self.gifInstance.isValid():
            QgsMessageLog.logMessage('GifClicker: Invalid GIF instance', MESSAGE_CATEGORY, level=Qgis.Warning)
            return
        if(frame == (self.gifInstance.frameCount() - 1) and (self.gifInstance.loopCount() > 0 or self.gifInstance.loopCount() == -1)):
            self.gifInstance.stop()
            self.gifInstance.deleteLater()
            self.onMovieFinished(scene, itemIdx)


    def canvasReleaseEvent(self, event: QgsMapMouseEvent):
        super().canvasReleaseEvent(event)
        if self.scene is None:
            self.scene = self.canvas().scene()
        if self.gif is None:
            return
        try:
            if self.enabled is False:
                return
            labelIdx = self.setupLabel()
            point = event.pixelPoint()
            size = self.gif.scaledSize()
            self.gifInstance = QMovie(self.gif.fileName())
            self.gifInstance.setParent(self)
            self.gifInstance.setScaledSize(size)
            self.gifInstance.setSpeed(self.gif.speed())
            w = size.width()
            h = size.height()
            itemIdx = self.scene.addWidget(self.labels[self.labelIdx], Qt.WindowTransparentForInput)
            self.gifInstance.finished.connect(lambda: self.onMovieFinished(self.scene, itemIdx))
            if self.gifInstance is not None and not sip.isdeleted(self.gifInstance):
                frame_number = self.gifInstance.currentFrameNumber()
                self.gifInstance.frameChanged.connect(lambda: self.onFrameChanged(frame_number, self.scene, itemIdx))
            self.gifInstance.setCacheMode(QMovie.CacheAll)
            self.labels[labelIdx].setGeometry(round(point.x()-(w/2)), round(point.y()-(h/2)), w, h)
            
            self.labels[labelIdx].setMovie(self.gifInstance)
            self.gifInstance.start()
            # For debugging purposes
            #QgsMessageLog.logMessage('Gif added to scene', MESSAGE_CATEGORY, level=Qgis.Info)
            

        except Exception as e:
            QgsMessageLog.logMessage(f'{str(e)}', MESSAGE_CATEGORY, level=Qgis.Critical)


    def __init__(self, canvas):
        super(QgsMapToolPan, self).__init__(canvas)
        self.scene = self.canvas().scene()
        self.labelIdx = 0
        self.labels = {}
        self.gifChanged.connect(self.gifChangedSlot)
        
        
        



