# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GifClicker
                                 A QGIS plugin
 Select GIFs to play when you click on the canvas with different map tools
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-08-08
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Dror Bogin
        email                : dror.bogin@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QCheckBox, QComboBox, QDialog
from qgis.gui import QgsMapToolPan
from qgis.core import QgsSettings, QgsMessageLog, Qgis
import json

# Initialize Qt resources from file resources.py
from .resources import * # noqa: F403
from .map_tools import GifClickerMapToolPan
# Import the code for the dialog
from .gif_clicker_dialog import GifClickerDialog, GifClickerAddGifDialog
import os.path

MESSAGE_CATEGORY = 'QGIS GIF Clicker Plugin'

GIF_DICT = {
    'explosion': {'path':'gifs/explosion.gif',
                  'label': 'Explosion',
                  'type': 'base',
                  'attribution': 'created from assets in the kenny.nl smoke particale pack - https://kenney.nl/assets/smoke-particles'},
    'star': {'path':'gifs/star.gif',
                'label': 'Star',
                'type': 'base',
                'attribution': 'created from assets in the kenny.nl particale pack - https://kenney.nl/assets/particle-pack'},
    'circle': {'path':'gifs/circle.gif',
                'label': 'Circle',
                'type': 'base',
                'attribution': 'created from assets in the kenny.nl particale pack - https://kenney.nl/assets/particle-pack'},
}

class GifClicker:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GifClicker_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Write default GIF configuration if it does not exist
        if not os.path.exists(os.path.join(self.plugin_dir, 'gifs.json')):
            with open(os.path.join(self.plugin_dir, 'gifs.json'), 'w') as f:
                json.dump(GIF_DICT, f, indent=4)

        with open(os.path.join(self.plugin_dir,'gifs.json')) as f:
            self.gifs_config = json.load(f)
            f.close()


        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Gif Clicker')
        self.tools = []
        self.toolbar = None
        self.config = None        
        self.mb = self.iface.messageBar()
        self.enabled = self.get_setting('enabled', True)  # Default value for GIFs enabled
        self.pan_tool = GifClickerMapToolPan(self.iface.mapCanvas())
        self.pan_tool_gif = self.get_setting('panGif', 'star')  # Default GIF for pan tool
        if self.pan_tool_gif not in self.gifs_config:
                QgsMessageLog.logMessage(
                f'Invalid GIF selected: {self.pan_tool_gif}. Using default GIF.',
                MESSAGE_CATEGORY,
                Qgis.Warning)
                self.pan_tool_gif = 'star'  # Fallback to default if not found
        
        # Set the GIF URL for the pan tool
        if not hasattr(self.gifs_config[self.pan_tool_gif],'type') or self.gifs_config[self.pan_tool_gif]['type'] == 'base':
            self.pan_tool.setGifUrl(os.path.join(self.plugin_dir, self.gifs_config[self.pan_tool_gif]['path']))
        else:
            if self.gifs_config[self.pan_tool_gif]['type'] != 'base' and os.path.exists(self.gifs_config[self.pan_tool_gif]['path']):
                self.pan_tool.setGifUrl(self.gifs_config[self.pan_tool_gif]['path'])
            else:
                QgsMessageLog.logMessage(
                    f'GIF path not found: {self.gifs_config[self.pan_tool_gif]["path"]}. Using default GIF.',
                    MESSAGE_CATEGORY,
                    Qgis.Warning)
                self.mb.pushWarning('GIF Clicker Plugin', f'GIF path not found: {self.gifs_config[self.pan_tool_gif]["path"]}. Using default GIF.')
                # Fallback to default GIF if path not found
                self.pan_tool.setGifUrl(os.path.join(self.plugin_dir, self.gifs_config['star']['path']))
        # Initialize the map tool with a default GIF
        


        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GifClicker', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        
        self.toolbar = self.iface.addToolBar('GIF Clicker')
        self.toolbar.setObjectName('GIF Clicker')

        icon_path = os.path.join(self.plugin_dir,'qgs_gifs_icon.png')
        self.add_gif_dlg = GifClickerAddGifDialog()
        
        # Add the toolbar
        self.toolbar = self.iface.addToolBar('GIFs Toolbar')
        self.toolbar.setObjectName('GIFs')

        # Add Icon
        self.icon = QIcon(icon_path)

        # Add the GIFs toggle
        self.gif_toggle = QCheckBox('Enable GIFs')
        self.gif_toggle.setToolTip(self.tr(u'Enable GIFs on map click with the pan tool'))
        self.gif_toggle.setIcon(self.icon) 
        gifs_enabled = self.get_setting('enabled', True)

        if gifs_enabled:
            self.gif_toggle.setChecked(True)
        else:
            self.gif_toggle.setChecked(False)
        self.gif_toggle.toggled.connect(self.toggle_gif_enabled)
        self.toolbar.addWidget(self.gif_toggle)

        # Add a ComboBox to select the GIF
        self.gif_combo = QComboBox()

        # Populate the ComboBox with available GIFs
        self.pan_tool_gif = self.restore_settings()
        self.gif_combo.setToolTip(self.tr(u'Select the GIF to play when using the pan tool'))
        self.gif_combo.currentTextChanged.connect(self.onGifSelected)
        self.toolbar.addWidget(self.gif_combo)

        # Add the button to open the "Add Custom Gif" dialog
        #add_icon = QIcon(":images/themes/default/mActionAdd.svg")
        self.add_gif_button = self.add_action(":images/themes/default/mActionAdd.svg",
            text=self.tr(u'Add Custom GIF'),
            callback=self.display_add_gif_dialog,
            parent=self.iface.mainWindow(),
            add_to_menu=False,
            add_to_toolbar=False,
            status_tip=self.tr(u'Add a custom GIF to the plugin'))
        self.toolbar.addAction(self.add_gif_button)
        

        #TODO: add this as an action that opens the config dialog
        """
        self.add_action(
            icon_path,
            text=self.tr(u'Use GIFs'),
            callback=self.run,
            parent=self.iface.mainWindow())
        """

        self.iface.mapCanvas().mapToolSet.connect(self.onMapToolSet)
        # will be set False in run()
        self.first_start = True

    
    def display_add_gif_dialog(self):
        """Display the dialog to add a custom GIF."""
        try:
            self.add_gif_dlg.show()
            self.add_gif_dlg.gif_path.setFilePath('')  # Clear the input field
            self.add_gif_dlg.gif_label.clear()  # Clear the label field

            if self.add_gif_dlg.exec_() == QDialog.Accepted:
                gif_path = self.add_gif_dlg.gif_path.filePath()
                gif_label = self.add_gif_dlg.gif_label.text()
                if gif_path and gif_label:
                    self.add_custom_gif(gif_path, gif_label)
                    self.mb.pushInfo('GIF Clicker Plugin', f'GIF {gif_label} added successfully.')
                else:
                    self.mb.pushWarning('GIF Clicker Plugin', 'Please provide both GIF path and label.')
            else:
                self.add_gif_dlg.reject()  # Close the dialog without saving changes
        except Exception as e:
            QgsMessageLog.logMessage(f'Error displaying add GIF dialog: {str(e)}', MESSAGE_CATEGORY, Qgis.Critical)


    
    def onMapToolSet(self, newToolSet, oldToolSet):
        """Called when the map tool is set."""
        if isinstance(newToolSet, QgsMapToolPan):
            self.iface.mapCanvas().setMapTool(self.pan_tool)


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        try:
            # disconnect the map tool set signal
            self.iface.mapCanvas().mapToolSet.disconnect(self.onMapToolSet)
            self.pan_tool.gifChanged.disconnect(self.pan_tool.gifChangedSlot)
            self.pan_tool.setEnabled(False)  # Disable the pan tool
            # remove the toolbar
            if self.toolbar is not None:
                self.toolbar = None
        except Exception as e:
            # no need to disconnect if not connected
            QgsMessageLog.logMessage(
                'Error unloading plugin: {}'.format(str(e)),
                MESSAGE_CATEGORY,
                Qgis.Critical)
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Gif Clicker'),
                action)
            self.iface.removeToolBarIcon(action)

    def onGifSelected(self, gif_label):
        """Handle the GIF selection from the combo box."""
        # Find the selected GIF in the self.gifs_config
        selected_gif = next((gif for gif in self.gifs_config.values() if gif['label'] == gif_label), None)
        if selected_gif:
            # Find the GIF name in the configuration by the label
            for gif_name, gif_info in self.gifs_config.items():
                if gif_info['label'] == gif_label:
                    self.pan_tool_gif = gif_name
                    break
            self.pan_tool.setGifUrl(os.path.join(self.plugin_dir, self.gifs_config[self.pan_tool_gif]['path']))
            self.pan_tool.setGif(self.pan_tool.getGif())
            self.set_setting('panGif', self.pan_tool_gif)
            QgsMessageLog.logMessage(
                f'Selected GIF: {gif_label}',
                MESSAGE_CATEGORY,
                Qgis.Info)
            


    def toggle_gif_enabled(self):
        """Toggle the GIFs enabled setting."""
        try:
            self.enabled = not self.enabled
            self.pan_tool.setEnabled(self.enabled)
            self.set_setting('enabled', self.enabled)
            if self.enabled:
                QgsMessageLog.logMessage(
                    'GIFs enabled',
                    MESSAGE_CATEGORY,
                    Qgis.Info)
                self.pan_tool.setGifUrl(os.path.join(self.plugin_dir, self.pan_tool_gif))
                self.iface.mapCanvas().setMapTool(self.pan_tool)
            else:
                QgsMessageLog.logMessage(
                    'GIFs disabled',
                    MESSAGE_CATEGORY,
                    Qgis.Info)
        except Exception as e:
            QgsMessageLog.logMessage('Error toggling GIFs: {}'.format(str(e)),MESSAGE_CATEGORY,Qgis.Critical)


    def configure(self):
        """Configure the plugin"""
        try:
            if self.config is None:
                self.config = self.restore_settings()

        except Exception as e:
            self.mb.pushCritical('Error configuring GIF Clicker plugin', str(e))
            QgsMessageLog.logMessage('Error unloading: {}'.fomrat(str(e)),MESSAGE_CATEGORY,Qgis.Critical)

    def restore_settings(self, new_gif_label=None):
        try:
            self.enabled = self.get_setting('enabled', True)
            default_pan_gif = 'star'
            self.pan_tool_gif = self.get_setting('panGif', default_pan_gif)
            # Check if the GIFs configuration file exists, if not create it
            if not os.path.exists(os.path.join(self.plugin_dir, 'gifs.json')):
                with open(os.path.join(self.plugin_dir, 'gifs.json'), 'w') as f:
                    json.dump(GIF_DICT, f, indent=4)

            # Load the GIFs configuration from the JSON file
            with open(os.path.join(self.plugin_dir, 'gifs.json')) as f:
                self.gifs_config = json.load(f)
                f.close()

            if self.pan_tool_gif not in self.gifs_config:
                self.pan_tool_gif = default_pan_gif

            # Update the combo box with the available GIFs
            self.gif_combo.clear()  # Clear existing items
            self.gif_combo.addItems([gif['label'] for gif in self.gifs_config.values()])
            # If a new GIF label is provided, update the pan tool GIF
            if new_gif_label is not None:
                if new_gif_label in self.gifs_config:
                    self.pan_tool_gif = new_gif_label
                else:
                    QgsMessageLog.logMessage(
                        f'GIF {new_gif_label} not found in configuration. Using default GIF.',
                        MESSAGE_CATEGORY,
                        Qgis.Warning)
                    self.pan_tool_gif = default_pan_gif
            self.gif_combo.setCurrentText(self.gifs_config[self.pan_tool_gif]['label'])

            self.pan_tool.setEnabled(self.enabled)
            return self.pan_tool_gif
        except Exception as e:
            print(f'Error restoring settings: {str(e)}')
            QgsMessageLog.logMessage('Error restoring settings: {}'.format(str(e)), MESSAGE_CATEGORY, Qgis.Critical)
            return default_pan_gif
        

    def save_gif_settings(self,new_gif_label=None):
        """Save the plugin settings to a JSON file.

        :param settings: The settings to save.
        :type settings: dict
        """
        try:
            with open(os.path.join(self.plugin_dir, 'gifs.json'), 'w') as f:
                json.dump(self.gifs_config, f, indent=4)
                f.close()

            if new_gif_label is not None:
                self.restore_settings(new_gif_label)  # Refresh the settings after saving
            else:
                self.restore_settings()
            QgsMessageLog.logMessage('GIF settings saved successfully', MESSAGE_CATEGORY, Qgis.Info)
        except Exception as e:
            QgsMessageLog.logMessage('Error saving GIF settings: {}'.format(str(e)), MESSAGE_CATEGORY, Qgis.Critical)


    def add_custom_gif(self, gif_path: str, gif_label: str):
        """Add a custom GIF to the plugin configuration.

        :param gif_path: The path to the GIF file.
        :type gif_path: str

        :param gif_label: The label for the GIF.
        :type gif_label: str

        :param gif_type: The type of the GIF, can be 'base' or 'custom'.
        :type gif_type: str
        """
        try:
            if not os.path.exists(gif_path):
                QgsMessageLog.logMessage(f'GIF file not found: {gif_path}', MESSAGE_CATEGORY, Qgis.Critical)
                self.mb.pushCritical('GIF Clicker Plugin', f'GIF file not found: {gif_path}')
                return
            
            if gif_label in self.gifs_config:
                if self.gifs_config[gif_label]['type'] == 'base':
                    QgsMessageLog.logMessage(f'GIF name already exists as base: {gif_label}', MESSAGE_CATEGORY, Qgis.Critical)
                    self.mb.pushCritical('GIF Clicker Plugin', f'GIF name already exists as base: {gif_label}')
                    return
                else:
                    QgsMessageLog.logMessage(f'GIF {gif_label} already exists, updating path.', MESSAGE_CATEGORY, Qgis.Info)
                    self.mb.pushInfo('GIF Clicker Plugin', f'GIF {gif_label} already exists, updating path.')
                    self.gifs_config[gif_label]['path'] = gif_path
            else:
                # Add the new GIF to the configuration
                self.gifs_config[gif_label] = {
                    'path': gif_path,
                    'label': gif_label,
                    'type': 'custom'
                }
            # Save the updated configuration
            self.set_setting('panGif', gif_label) 
            self.save_gif_settings(gif_label)
        except Exception as e:
            QgsMessageLog.logMessage(f'Error adding custom GIF: {str(e)}', MESSAGE_CATEGORY, Qgis.Critical)
            self.mb.pushCritical('GIF Clicker Plugin', f'Error adding custom GIF: {str(e)}')

        
        
    def get_setting(self, key: str, default: str = None):
        """Get a value in the QgsSettings.

        :param key: The key to fetch in the QgsSettings
        :type key: basestring

        :param default: The default value if the key is not found.
        :type default: basestring

        :return: The value or the default value.
        :rtype: basestring
        """
        q_setting = QgsSettings()
        prefix = '/GifClicker/'
        value = q_setting.value(prefix + key)

        if value is not None:
            return value

        return default


    def set_setting(self, key: str, value: str):
        """
        Set a value in the QgsSettings
        :param key: key
        :type key: str

        :param value: value
        :type value: str

        :return: result
        :rtype: bool
        """
        q_setting = QgsSettings()
        prefix = '/GifClicker/'
        return q_setting.setValue(prefix + key, value)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start is True:
            self.first_start = False
            self.dlg = GifClickerDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
