#
# This file is part of GHydraulics
#
# ghydraulicsplugin.py - The GHydraulics plugin
#
# Copyright 2007 - 2013 Steffen Macke <sdteffen@sdteffen.de>
#
# GHydraulics is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GHydraulics is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#

from pickle import * 
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *
from ghyeconomicdiameter import *
from EpanetModel import *
from GHydraulicsModelChecker import *
from GHydraulicsModelMaker import *
from GHydraulicsException import *
from GHydraulicsInpWriter import *
from GHydraulicsSettingsDialog import *

# initialize Qt resources from file resouces.py
import resources

class GHydraulicsPlugin:
    # Store settings in QGIS projects under this key
    SETTINGS = "ghydraulics"

    def __init__(self, iface):
        # save reference to the QGIS interface
        self.iface = iface
    
    def initGui(self):
        # create actions
        self.action = QAction(QIcon(":/python/plugins/ghydraulic/icon.xpm"), "Calculate economic diameters", self.iface.mainWindow())
        self.action.setWhatsThis("Calculate economic pipe diameters based on flow data.")
        self.settingsAction = QAction(QIcon(':/python/plugins/ghydraulic/icon.xpm'), 'Settings', self.iface.mainWindow())
        self.checkModelAction = QAction(QIcon(':/python/plugins/ghydraulic/icon.xpm'), 'Check Model', self.iface.mainWindow())
        self.makeModelAction = QAction(QIcon(':/python/plugins/ghydraulic/icon.xpm'), 'Make EPANET Model', self.iface.mainWindow())
        self.writeInpAction = QAction(QIcon(':/python/plugins/ghydraulic/icon.xpm'), 'Write EPANET INP file', self.iface.mainWindow())
        self.aboutAction = QAction(QIcon(":/plugins/ghydraulic/about_icon.png"), QCoreApplication.translate('GHydraulics', "&About"), self.iface.mainWindow())    
        QObject.connect(self.action, SIGNAL("triggered()"), self.run)
        QObject.connect(self.settingsAction, SIGNAL('triggered()'), self.showSettings)
        QObject.connect(self.checkModelAction, SIGNAL('triggered()'), self.checkModel)
        QObject.connect(self.makeModelAction, SIGNAL('triggered()'), self.makeModel)
        QObject.connect(self.writeInpAction, SIGNAL('triggered()'), self.writeInpFile)
        QObject.connect(self.aboutAction, SIGNAL("triggered()"), self.about)
    
        # add toolbar button and menu item
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&GHydraulics", self.action)
        self.iface.addPluginToMenu("&GHydraulics", self.settingsAction)
        self.iface.addPluginToMenu("&GHydraulics", self.checkModelAction)
        self.iface.addPluginToMenu('&GHydraulics', self.makeModelAction)
        self.iface.addPluginToMenu("&GHydraulics", self.writeInpAction)
        self.iface.addPluginToMenu("&GHydraulics", self.aboutAction)
    
    def unload(self):
        # remove the plugin menu item and icon
        self.iface.removePluginMenu("&GHydraulics", self.aboutAction)
        self.iface.removePluginMenu("&GHydraulics", self.writeInpAction)
        self.iface.removePluginMenu('&GHydraulics', self.checkModelAction)
        self.iface.removePluginMenu('&GHydraulics', self.makeModelAction)
        self.iface.removePluginMenu("&GHydraulics", self.settingsAction)
        self.iface.removePluginMenu("&GHydraulics", self.action)
        self.iface.removeToolBarIcon(self.action)
    
    def run(self):
        # Diameter field name
        # TODO: make this configurable
        self.diafieldname = "DIAMETER"
        # TODO: make this configurable
        # Flow result field name
        self.flofieldname = "RESULT_FLO"
        # Check that there is an active vector layer
        activeLayer = self.iface.activeLayer()
        if not type(activeLayer) is QgsVectorLayer:
            QMessageBox.warning(self.iface.mainWindow(), "GHydraulics", "Please activate a vector layer.", QMessageBox.Ok, QMessageBox.Ok)
            return
              
        if not activeLayer.isEditable():
            QMessageBox.warning(self.iface.mainWindow(), "GHydraulics", "Active layer is not editable.", QMessageBox.Ok, QMessageBox.Ok)
            return
          
        # Check that the DIAMETER and RESULT_FLO fields are available
        provider = activeLayer.dataProvider()
    
        # Locate fields
        diafieldidx = -1
        flowfieldidx = -1
        fields = provider.fields()
        for (k, field) in fields.iteritems():
            if self.diafieldname == field.name():
                diafieldidx = k
            if self.flofieldname == field.name():
                flowfieldidx = k
        
        if -1 == diafieldidx:
            QMessageBox.warning(self.iface.mainWindow(), "GHydraulics", "Failed to locate the DIAMETER field.", QMessageBox.Ok, QMessageBox.Ok)
            return
      
        if -1 == flowfieldidx:
            QMessageBox.warning(self.iface.mainWindow(), "GHydraulics", "Failed to locate the RESULT_FLO field.", QMessageBox.Ok, QMessageBox.Ok)
            return
        # Let user agree to the change
        selectedbutton = QMessageBox.question(self.iface.mainWindow(), "GHydraulics", "This will overwrite all DIAMETER field values. Do you want to continue?", QMessageBox.Ok|QMessageBox.Cancel, QMessageBox.Cancel)
        if QMessageBox.Cancel == selectedbutton:
            return
      
        # Execute the action
        ecodia = GhyEconomicDiameter(self.flofieldname, self.diafieldname)
        ecodia.commitEconomicDiametersForLayer(activeLayer)
    
    # Display the About dialog   
    def about(self):
        infoString = QString(QCoreApplication.translate('GHydraulics', "GHydraulics Plugin 2.0.9<br />This plugin integrates EPANET with QGIS.<br />Copyright (c) 2007 - 2013 Steffen Macke<br /><a href=\"http://epanet.de/ghydraulics\">epanet.de/ghydraulics</a>\n"))
        QMessageBox.information(self.iface.mainWindow(), "About GHydraulics", infoString)
    
    # Display the settings dialog
    def showSettings(self):
        dlg = GHydraulicsSettingsDialog()
        layers = QgsMapLayerRegistry.instance().mapLayers()
        proj = QgsProject.instance()
        # Restore selected layers
        for layer_id,layer in layers.iteritems():
            if layer.type() == QgsMapLayer.VectorLayer:
                item = 6
                name = layer.name()
                for i in range(0,6):
                    pickle_list = str(proj.readEntry(GHydraulicsPlugin.SETTINGS, EpanetModel.GIS_SECTIONS[i], "")[0])
                    if '' != pickle_list:
                        # Windows QGIS injects some carriage returns here
                        pickle_list = pickle_list.replace('\r','')
                        try:
                            l = loads(pickle_list)
                            if name in l:
                                item = i
                                break
                        except(KeyError):
                            continue
                num_layers = dlg.ui.treeWidget.topLevelItem(item).childCount()
                QtGui.QTreeWidgetItem(dlg.ui.treeWidget.topLevelItem(item))
                dlg.ui.treeWidget.topLevelItem(item).child(num_layers).setText(0, name)
        # Restore template inp file
        template = dlg.getTemplate()    
        dlg.ui.inpFileLineEdit.setText(template)
    
        dlg.show()
        result = dlg.exec_()
    
    # Display a message once
    def explainOnce(self, key, title, message):
        proj = QgsProject.instance()
        if "True" == proj.readEntry(GHydraulicsPlugin.SETTINGS, key, "False"):
            return True
        reply = QtGui.QMessageBox.question(self.iface.mainWindow(), title,
                                            message, QtGui.QMessageBox.Yes |
                                            QtGui.QMessageBox.No, QtGui.QMessageBox.Yes)
        if QtGui.QMessageBox.Yes == reply:
            proj.writeEntry(GHydraulicsPlugin.SETTINGS, key, "True");
            return True
        return False
    
    # Check if all fields are in place
    def checkModel(self):
        checker = GHydraulicsModelChecker()
        missing = checker.check()
        if 0 == len(missing):
            QtGui.QMessageBox.information(self.iface.mainWindow(), GHydraulicsModelChecker.TITLE,
                                              'All required fields exist.')
            return
        fieldlist = []
        for name in missing.keys():
            fieldlist.append('<br/>Layer "'+name+'": '+ ', '.join(missing[name]))
        message = 'Your model is missing some fields.'+''.join(fieldlist)+'<br/>Would you like to add them?'
        reply = QtGui.QMessageBox.question(self.iface.mainWindow(), GHydraulicsModelChecker.TITLE,
                                                       message, QtGui.QMessageBox.Yes |
                                                       QtGui.QMessageBox.No, QtGui.QMessageBox.Yes)
        if QtGui.QMessageBox.Yes != reply:
            return
        if not checker.addFields(missing):
            QtGui.QMessageBox.critical(self.iface.mainWindow(), GHydraulicsModelChecker.TITLE,
                                      'Not all fields could be added.', QtGui.QMessageBox.Ok)
    
    # Fill the node1, node2 fields    
    def makeModel(self):
        if self.explainOnce('makeModelExplanation', 'Make EPANET Model', "Overwrite the fields NODE1 and NODE2 in all line tables?"):
            maker = GHydraulicsModelMaker()
            maker.make()
    
    # Write out a file in EPANET INP format
    def writeInpFile(self):
        # Modified layers may not be exprted correctly
        checker = GHydraulicsModelChecker()
        modified = checker.getModifiedLayers()
        if 0 < len(modified):
            message = 'Some of your model layers have not been saved (' + ', '.join(modified) + ').<br/>Do you want to save them now?'
            reply = QtGui.QMessageBox.question(self.iface.mainWindow(), GHydraulicsInpWriter.TITLE,
                                      message, QtGui.QMessageBox.Yes|
                                      QtGui.QMessageBox.No, QtGui.QMessageBox.Yes)
            if QtGui.QMessageBox.Yes == reply:
                checker.commitChanges()
        # select a file
        f =  QFileDialog.getSaveFileName(self.iface.mainWindow(), GHydraulicsInpWriter.TITLE,
                                QString(),
                                'EPANET INP file (*.inp)');
    
        if 0 < len(f):
            dlg = GHydraulicsSettingsDialog()
            template = dlg.getTemplate()
            try:
                writer = GHydraulicsInpWriter(template)
                writer.write(f)
            except GHydraulicsException as e:
                QMessageBox.warning(self.iface.mainWindow(), 'Saving an INP file failed', str(e), QMessageBox.Ok, QMessageBox.Ok)

