#
# This file is part of GHydraulics
#
# GHydraulicsInpWriter.py - Write INP files
#
# Copyright 2007 - 2013 Steffen Macke <sdteffen@sdteffen.de>
#
# GHydraulics is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GHydraulics is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
#

import re
import numpy
import numpy.linalg
from pickle import *
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *
from qgis import *
from EpanetModel import *
from GHydraulicsCommon import *
from GHydraulicsException import *
from GHydraulicsModel import *

# Write EPANET INP file
class GHydraulicsInpWriter(GHydraulicsCommon):
    TITLE = 'Save EPANET INP file'

    # Write to the given filename
    def write(self, filename):
        template = open(self.templateFilename)
        self.inpfile = open(filename, 'w')
        section = None
        for line in template.readlines():
            sectionname = re.match(' *\[([A-Z]+)\].*', line)
            if None != sectionname:
                section = self.writeSection(sectionname.group(1))
            if None == section:
                self.inpfile.write(line)
        self.inpfile.close()
        template.close()

    # Write the given section to the INP file
    def writeSection(self, section):
        if self.sections.has_key(section) and 0 < len(self.sections[section]):
            self.writeSectionLabel(section)
            if section in self.sections:
                self.inpfile.write(self.sections[section])
                return section
        elif 'BACKDROP' == section:
            self.writeBackdropSection()
            return section
        return None

    # Extract nodes from the model and write to string
    def getNodes(self, section):
        nodes = ''
        if self.sections.has_key(section):
            nodes = self.sections[section]
        for name in self.layers[section]:
            maplayers = QgsMapLayerRegistry.instance().mapLayers()
            for l,layer in maplayers.iteritems():
                if layer.type() == QgsMapLayer.VectorLayer and layer.name() == name:
                    feature = QgsFeature()
                    provider = layer.dataProvider()
                    allAttrs = provider.attributeIndexes()
                    fieldIndices = []
                    for field in EpanetModel.COLUMNS[section]:
                        fieldidx = provider.fieldNameIndex(field)
                        if -1 == fieldidx:
                            raise GHydraulicsException('ERROR: Failed to locate '+field+' field in layer '+name)
                        fieldIndices.append(fieldidx)
                    provider.select(allAttrs)
                    # Loop over all features
                    while provider.nextFeature(feature):
                        geometry = feature.geometry()
                        if geometry.type() == QGis.Point:
                            attrs = feature.attributeMap()
                            # write node
                            id = ''
                            for fieldidx in fieldIndices:
                                attribute = attrs[fieldidx].toString()
                                nodes = nodes + attribute + ' '
                                if '' == id:
                                    id = attribute
                            nodes = nodes + '\n'
                            # write coordinate
                            point = geometry.asPoint()
                            x = point.x()
                            y = point.y()
                            p = str(x) + ' ' + str(y)
                            self.sections['COORDINATES'] = self.sections['COORDINATES'] + id + ' ' + p + '\n'
                            self.xcoords.append(x)
                            self.ycoords.append(y)
        self.sections[section] = nodes + '\n'

    # Extract pipes from model, write to INP format string
    def getPipes(self):
        pipes = self.sections[EpanetModel.PIPES]
        for name in self.layers[EpanetModel.PIPES]:
            maplayers = QgsMapLayerRegistry.instance().mapLayers()
            for l,layer in maplayers.iteritems():
                if layer.type() == QgsMapLayer.VectorLayer and layer.name() == name:
                    feature = QgsFeature()
                    provider = layer.dataProvider()
                    allAttrs = provider.attributeIndexes()
                    fieldIndices = []
                    node1idx = -1
                    for field in EpanetModel.COLUMNS[EpanetModel.PIPES]:
                        fieldidx = provider.fieldNameIndex(field)
                        if -1 == fieldidx:
                            raise GHydraulicsException('ERROR: Failed to locate '+field+' field in layer '+name)
                        fieldIndices.append(fieldidx)
                        if EpanetModel.NODE1 == field:
                            node1idx = fieldidx
                    provider.select(allAttrs)
                    # Loop over all features
                    while provider.nextFeature(feature):
                        geometry = feature.geometry()
                        if geometry.type() == QGis.Line:
                            attrs = feature.attributeMap()
                            # write node
                            id = ''
                            for fieldidx in fieldIndices:
                                attribute = attrs[fieldidx].toString()
                                # Use dynamic nodes where necessary
                                if fieldidx == node1idx and self.virtualnodes.has_key(attribute):
                                    attribute = self.virtualnodes[attribute]
                                pipes = pipes + attribute + ' '
                                if '' == id:
                                    id = attribute
                            pipes = pipes + '\n'
                            # vertices
                            line = geometry.asPolyline()
                            for p in range(1,len(line)-1):
                                x = str(line[p].x())
                                y = str(line[p].y())
                                self.xcoords.append(x)
                                self.ycoords.append(y)
                                self.sections['VERTICES'] = self.sections['VERTICES'] + id + ' ' + x + ' ' + y + '\n'
        self.sections['PIPES'] = pipes + '\n'

    # Virtual lines are lines in EPANET and nodes in QGIS
    def getVirtualLines(self, section):
        lines = ''
        for name in self.layers[section]:
            maplayers = QgsMapLayerRegistry.instance().mapLayers()
            for l,layer in maplayers.iteritems():
                if layer.type() == QgsMapLayer.VectorLayer and layer.name() == name:
                    feature = QgsFeature()
                    provider = layer.dataProvider()
                    allAttrs = provider.attributeIndexes()
                    fieldIndices = []
                    for field in EpanetModel.COLUMNS[section]:
                        fieldidx = provider.fieldNameIndex(field)
                        if -1 == fieldidx:
                            raise GHydraulicsException('ERROR: Failed to locate '+field+' field in layer '+name)
                        fieldIndices.append(fieldidx)
                    provider.select(allAttrs)
                    # Loop over all features
                    while provider.nextFeature(feature):
                        geometry = feature.geometry()
                        if geometry.type() == QGis.Point:
                            attrs = feature.attributeMap()
                            # write node
                            id = ''
                            virtual_id = ''
                            # Leave out the elevation
                            for i in range(1,len(fieldIndices)):
                                attribute = attrs[fieldIndices[i]].toString()
                                lines = lines + attribute + ' '
                                if '' == id:
                                    id = attribute
                                    virtual_id = id + GHydraulicsModel.VIRTUAL_POSTFIX
                                    self.virtualnodes[id] = virtual_id
                                    lines = lines + id + ' ' + virtual_id + ' '
                            lines = lines + '\n'
                            # write first point
                            point = geometry.asPoint()
                            elevation = attrs[fieldIndices[0]].toString()
                            self.addJunction(id, elevation, '0', '', point.x(), point.y())
                            # write the second point
                            self.getSecondVirtualLineJunction(id, elevation)

        self.sections[section] = lines + '\n'

    # Insert virtual node into line
    def getSecondVirtualLineJunction(self, id, elevation):
        node = ''
        virtual_id = self.virtualnodes[id]
        # Find the referenced pipe
        for name in self.layers[EpanetModel.PIPES]:
            maplayers = QgsMapLayerRegistry.instance().mapLayers()
            for l,layer in maplayers.iteritems():
                if layer.type() == QgsMapLayer.VectorLayer and layer.name() == name:
                    feature = QgsFeature()
                    provider = layer.dataProvider()
                    allAttrs = provider.attributeIndexes()
                    node1idx = provider.fieldNameIndex(EpanetModel.NODE1)
                    if -1 == node1idx:
                        raise GHydraulicsException('ERROR: Failed to locate '+EpanetModel.NODE1+' field in layer '+name)
                    provider.select(allAttrs)
                    # Loop over all features
                    while provider.nextFeature(feature):
                        geometry = feature.geometry()
                        if geometry.type() == QGis.Line:
                            attrs = feature.attributeMap()
                            if attrs[node1idx].toString() == id:
                                line = geometry.asPolyline()
                                s = line[0]
                                e = line[1]
                                if GHydraulicsModel.VIRTUAL_LINE_LENGTH > s.sqrDist(e):
                                    p = [(s.x()+e.x())/2,(s.y()+e.y())/2]
                                else:
                                    sv = numpy.array([s.x(), s.y()])
                                    ev = numpy.array([e.x(), e.y()])
                                    dv = ev - sv
                                    nv = dv/numpy.linalg.norm(dv)
                                    p = sv + nv
                                self.addJunction(virtual_id, elevation, '0', '', p[0], p[1])
                                return
        raise GHydraulicsException('ERROR: Failed to locate Pipe with NODE1 named '+id)

    # Add a junction to the buffer
    def addJunction(self, id, elevation, demand, pattern, x, y):
        self.sections[EpanetModel.JUNCTIONS] = self.sections[EpanetModel.JUNCTIONS] + id + ' ' + elevation+ ' ' + demand + ' ' + pattern + '\n'
        self.addXY(EpanetModel.COORDINATES, id, x, y)

    # section is one of COORDINATES or VERTICES
    def addXY(self, section, id, x, y):
        self.sections[section] = self.sections[section] + id + ' ' + str(x) + ' ' + str(y) + '\n'

    # Write a section label to the INP file
    def writeSectionLabel(self, section):
        self.inpfile.write('['+section+'] ; created by GHydraulics\n')

    # Write out the backdrop section
    def writeBackdropSection(self):
        self.writeSectionLabel('BACKDROP')
        mins = maxs = '0 0 '
        if 0 < len(self.xcoords) and 0 < len(self.ycoords):
            mins = str(min(self.xcoords)) + ' '  + str(min(self.ycoords)) + ' '
            maxs = str(max(self.xcoords)) + ' ' + str(max(self.ycoords))
        self.inpfile.write('DIMENSIONS ' + mins + maxs + '\n')
        self.inpfile.write('UNITS              None\n')
        self.inpfile.write('FILE\n')
        self.inpfile.write('OFFSET             0.00                0.00\n\n')

    def __init__(self, templateFilename):
        self.templateFilename = templateFilename
        # Calculate map extent
        self.xcoords = []
        self.ycoords = []
        self.getLayers()
        self.sections = {EpanetModel.JUNCTIONS: '', EpanetModel.PIPES: ''}
        # Dictionary of those node1 values that change because of virtual lines
        self.virtualnodes = {}
        for section in EpanetModel.COORDINATE_DATA_SECTIONS:
            self.sections[section] = ''
        for section in EpanetModel.VIRTUAL_LINE_SECTIONS:
            self.getVirtualLines(section)
        for section in EpanetModel.COORDINATE_SECTIONS:
            self.getNodes(section)
        self.getPipes()
