#
# This file is part of GHydraulics
#
# GHydraulictModelMaker.py - Create an EPANET model
#
# Copyright 2012 Steffen Macke <sdteffen@sdteffen.de>
#
# GHydraulics is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GHydraulics is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# QGIS 1.0.0 or better required to run this file
#

from EpanetModel import *
from GHydraulicsCommon import *
from GHydraulicsException import *
from GHydraulicsModel import *
from math import *

# Create the node1 node2 topology
class GHydraulicsModelMaker(GHydraulicsCommon):
    # ID field in the spatial index data
    ID = 0

    # Fill the node1, node2
    def make(self):
        self.buildNodeSpatialIndex()
        # loop over all pipe layers
        for name in self.layers[EpanetModel.PIPES]:
            maplayers = QgsMapLayerRegistry.instance().mapLayers()
            for l,layer in maplayers.iteritems():
                if layer.type() == QgsMapLayer.VectorLayer and layer.name() == name:
                    if not layer.isEditable() and not layer.startEditing():
                        raise GHydraulicsException('ERROR: Unable to edit layer '+name)
                    # Map node fields
                    nodes = {}
                    provider = layer.dataProvider()
                    feature = QgsFeature()
                    for fieldname in EpanetModel.NODE_FIELDS:
                        nodeidx = provider.fieldNameIndex(fieldname)
                        if -1 == nodeidx:
                            raise GHydraulicsException('ERROR: Failed to locate '+fieldname+' field in layer '+name)
                        nodes[fieldname] = nodeidx
                    # loop over all features
                    provider.select()
                    while provider.nextFeature(feature):
                        geometry = feature.geometry()
                        if geometry.type() == QGis.Line:
                            attrs = feature.attributeMap()
                            line = geometry.asPolyline()
                            points = {EpanetModel.NODE1: line[0], EpanetModel.NODE2: line.pop()}
                            for fieldname in EpanetModel.NODE_FIELDS:
                                nodeidx = nodes[fieldname]
                                nodeid = self.getNodeId(points[fieldname])
                                layer.changeAttributeValue(feature.id(), nodeidx ,QVariant(nodeid))
        del self.allnodes
        del self.nodemap

    # Get the node id for a certain point
    def getNodeId(self, point):
        id = self.allnodes.nearestNeighbor(point, 1)
        feature = self.nodemap[id.pop()]
        geometry = feature.geometry()
        nearest = geometry.asPoint()
        if sqrt(nearest.sqrDist(point)) > GHydraulicsModel.SNAP_RADIUS:
            raise GHydraulicsException('ERROR: Failed to locate node for point')
        attrs = feature.attributeMap()
        return attrs[self.ID].toString()

    # Get a QgsSpatialIndex object over all node layers and the given field list
    def buildNodeSpatialIndex(self):
        self.allnodes = QgsSpatialIndex()
        self.nodemap = {}
        nodeid = 0
        maplayers = QgsMapLayerRegistry().instance().mapLayers()
        for name in GHydraulicsModel.NODE_SECTIONS:
            for layername in self.layers[name]:
                for l,layer in maplayers.iteritems():
                    if layer.type() == QgsMapLayer.VectorLayer and layer.name() == layername:
                        provider = layer.dataProvider()
                        feature = QgsFeature()
                        ididx = provider.fieldNameIndex(GHydraulicsModel.ID_FIELD)
                        provider.select([ididx])
                        while provider.nextFeature(feature):
                            geometry = feature.geometry()
                            if geometry.type() == QGis.Point:
                                point = QgsFeature(nodeid)
                                nodeid = nodeid+1
                                point.setGeometry(geometry)
                                attrs = feature.attributeMap()
                                point.addAttribute(self.ID, attrs[ididx])
                                self.allnodes.insertFeature(point)
                                self.nodemap[point.id()] = point

    def __init__(self):
        self.getLayers()
