import os
import xlsxwriter
import csv



from qgis.PyQt.QtWidgets import QTableWidgetItem, QMessageBox
from qgis.PyQt.QtCore import QDate
from qgis.utils import iface
from qgis.core import QgsProject

from .utils import tr, get_finances_csv_path


class FinanceManager:
    def __init__(self, dlg):
        self.dlg = dlg
        # On crée le dossier data une seule fois au cas où il n'existe pas
        plugin_dir = os.path.dirname(__file__)
        os.makedirs(os.path.join(plugin_dir, 'data'), exist_ok=True)


    def initialize(self):
        """Charge les données depuis CSV dans la table au démarrage."""
        self.load_finance_from_csv()


    def create_finance_excel(self, path, langue='fr'):
        if os.path.exists(path):
            return  # Ne pas recréer le fichier s’il existe déjà

        # Créer un nouveau classeur Excel
        workbook = xlsxwriter.Workbook(path)
        worksheet = workbook.add_worksheet("Finances")

        # Définir les en-têtes selon la langue
        colonnes_fr = ["ID Parcelle", "Date", "Libellé","Type", "Montant", "Commentaire"]
        colonnes_en = ["Parcel ID", "Date", "Label", "Type", "Amount", "Comment"]

        colonnes = colonnes_fr if langue == 'fr' else colonnes_en

        # Écrire les en-têtes dans la première ligne
        for col_index, header in enumerate(colonnes):
            worksheet.write(0, col_index, header)

        workbook.close()

    def save_finance_to_csv(self):
        csv_path = get_finances_csv_path()
        if not csv_path:
            return

        table_widget = self.dlg.tableWidgetFinance
        with open(csv_path, mode='w', newline='', encoding='utf-8') as f:
            writer = csv.writer(f)
            # Écrire l'en-tête
            writer.writerow(["ID Parcelle", "Date", "Libellé", "Type", "Montant", "Commentaire"])
            # Écrire les données
            for row in range(table_widget.rowCount()):
                row_data = []
                for col in range(table_widget.columnCount()):
                    item = table_widget.item(row, col)
                    row_data.append(item.text() if item else "")
                writer.writerow(row_data)

        # ✅ Message de confirmation avec tr
        QMessageBox.information(
            self.dlg,
            tr("Sauvegarde réussie"),
            tr("Les données ont été enregistrées dans\n{}").format(os.path.basename(csv_path))
        )

    def load_finance_from_csv(self):
        table_widget = self.dlg.tableWidgetFinance
        csv_path = get_finances_csv_path()
        if not csv_path or not os.path.exists(csv_path):
            return

        with open(csv_path, newline='', encoding='utf-8') as f:
            reader = csv.reader(f)
            next(reader, None)  # sauter l'en-tête
            table_widget.setRowCount(0)
            for row_data in reader:
                row = table_widget.rowCount()
                table_widget.insertRow(row)
                for col, val in enumerate(row_data):
                    table_widget.setItem(row, col, QTableWidgetItem(val))

    def add_finance_entry(self):
        print("==> lance entrée finance")
        date_str = self.dlg.dateEditFinance.date().toString("dd-MM-yyyy")

        # Déterminer le type (recette, dépense ou libellé)
        if self.dlg.radioRecette.isChecked():
            type_finance = tr("Recette")
        elif self.dlg.radioDepense.isChecked():
            type_finance = tr("Dépense")
        else:
            type_finance = self.dlg.comboBoxFinance.currentText().strip()

        libelle = self.dlg.comboBoxFinance.currentText().strip()
        commentaire = self.dlg.textEditFinance.toPlainText()
        montant_str = self.dlg.lineEditCout.text().replace(",", ".")

        if not montant_str:
            QMessageBox.warning(self.dlg, tr("Erreur"), tr("Veuillez entrer un montant."))
            return

        try:
            montant = "{:.2f}".format(float(montant_str))
        except ValueError:
            QMessageBox.warning(
                self.dlg,
                tr("Erreur"),
                tr("Veuillez entrer un montant valide au format x,xx.")
            )
            return

        # ID de la parcelle
        id_parcelle = str(getattr(self.dlg, "current_parcelle_id", ""))

        # Ajout dans le tableau
        table = self.dlg.tableWidgetFinance
        row = table.rowCount()
        table.insertRow(row)
        table.setItem(row, 0, QTableWidgetItem(id_parcelle))
        table.setItem(row, 1, QTableWidgetItem(date_str))
        table.setItem(row, 2, QTableWidgetItem(libelle))
        table.setItem(row, 3, QTableWidgetItem(type_finance))
        table.setItem(row, 4, QTableWidgetItem(montant))
        table.setItem(row, 5, QTableWidgetItem(commentaire))

        # Réinitialisation des champs
        self.dlg.textEditFinance.clear()
        self.dlg.lineEditCout.clear()
        self.dlg.dateEditFinance.setDate(QDate.currentDate())
        self.dlg.radioRecette.setChecked(True)

        # Sauvegarde dans le CSV
        self.save_finance_to_csv()