# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeosysPluginDockWidget
                                 A QGIS plugin
 Discover, request and use aggregate imagery products based on landsat-8,
 Sentinel 2 and other sensors from within QGIS, using the GEOSYS API.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2019-03-11
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Kartoza (Pty) Ltd
        email                : andre@kartoza.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
import sys
import json

from PyQt5 import QtGui, QtWidgets
from PyQt5.QtCore import pyqtSignal, QSettings, QMutex, QDate
from PyQt5.QtGui import QCursor
from PyQt5.QtWidgets import QLabel, QListWidgetItem, QMessageBox, QApplication

from qgis.core import (
    QgsProject,
    QgsFeatureRequest,
    QgsVectorLayer,
    QgsRasterLayer,
    QgsCoordinateReferenceSystem
)
from qgis.PyQt.QtCore import Qt

from geosys.bridge_api.default import (
    VECTOR_FORMAT,
    PNG,
    PNG_KMZ,
    ZIPPED_TIFF,
    ZIPPED_SHP,
    KMZ,
    KML,
    VALID_QGIS_FORMAT,
    YIELD_AVERAGE,
    YIELD_MINIMUM,
    YIELD_MAXIMUM,
    ORGANIC_AVERAGE,
    POSITION,
    FILTER,
    SAMZ_ZONE,
    SAMZ_ZONING,
    HOTSPOT,
    ZONING_SEGMENTATION,
    MAX_FEATURE_NUMBERS,
    DEFAULT_ZONE_COUNT,
    GAIN,
    OFFSET,
    DEFAULT_N_PLANNED,
    DEFAULT_AVE_YIELD,
    DEFAULT_MIN_YIELD,
    DEFAULT_MAX_YIELD,
    DEFAULT_ORGANIC_AVE,
    DEFAULT_GAIN,
    DEFAULT_OFFSET,
    DEFAULT_COVERAGE_PERCENT)
from geosys.bridge_api.definitions import (
    ARCHIVE_MAP_PRODUCTS,
    ALL_SENSORS,
    SENSORS,
    NDVI,
    EVI,
    SAMZ,
    SOIL,
    SLOPE,
    ELEVATION,
    REFLECTANCE,
    LANDSAT_8,
    LANDSAT_9,
    SENTINEL_2,
    INSEASONFIELD_AVERAGE_NDVI,
    INSEASONFIELD_AVERAGE_REVERSE_NDVI,
    INSEASONFIELD_AVERAGE_LAI,
    INSEASONFIELD_AVERAGE_REVERSE_LAI,
    COLOR_COMPOSITION,
    SAMPLE_MAP,
    IGNORE_LAYER_FIELDS,
    MASK_PARAMETERS,
    ALLOWED_FIELD_TYPES)
from geosys.bridge_api_wrapper import BridgeAPI
from geosys.bridge_api.utilities import get_definition
from geosys.ui.help.help_dialog import HelpDialog
from geosys.ui.widgets.geosys_coverage_downloader import (
    CoverageSearchThread,
    create_map,
    create_difference_map,
    create_samz_map,
    create_rx_map,
    fetch_map,
    fetch_samz_map,
    credentials_parameters_from_settings)
from geosys.ui.widgets.geosys_itemwidget import CoverageSearchResultItemWidget
from geosys.utilities.gui_utilities import (
    add_ordered_combo_item, layer_icon, is_polygon_layer, layer_from_combo,
    add_layer_to_canvas, reproject, item_data_from_combo,
    wkt_geometries_from_feature_iterator, item_text_from_combo,
    is_point_layer, attribute_from_feature_iterator
)
from geosys.utilities.qgis_settings import QGISSettings
from geosys.utilities.resources import get_ui_class
from geosys.utilities.settings import setting, set_setting
from geosys.utilities.utilities import check_if_file_exists, log, clean_filename
FORM_CLASS = get_ui_class('geosys_dockwidget_base.ui')


class GeosysPluginDockWidget(QtWidgets.QDockWidget, FORM_CLASS):
    closingPlugin = pyqtSignal()

    def __init__(self, iface, parent=None):
        """Constructor."""
        super(GeosysPluginDockWidget, self).__init__(parent)
        # Set up the user interface from Designer.
        # After setupUI you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://doc.qt.io/qt-5/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        # Save reference to the QGIS interface and parent
        self.iface = iface
        self.parent = parent
        self.settings = QSettings()
        self.one_process_work = QMutex()
        self.search_threads = None
        self.max_stacked_widget_index = self.stacked_widget.count() - 1
        self.current_stacked_widget_index = 0

        # Coverage parameters from input values
        self.wkt_geometries = None
        self.wkt_point_geometries = None
        self.attributes = None
        self.map_product = None
        self.sensor_type = None
        self.mask_type = None
        self.start_date = None
        self.end_date = None
        self.coverage_percent = DEFAULT_COVERAGE_PERCENT
        self.coverage_percent_value_spinbox.setValue(self.coverage_percent)

        # Nitrogen map type parameter
        self.n_planned_value = DEFAULT_N_PLANNED
        self.n_planned_value_spinbox.setValue(self.n_planned_value)

        # Sample map parameters
        self.sample_map_point_layer = None
        self.sample_map_field = None

        # Map creation parameters from input values
        self.yield_average = None
        self.yield_minimum = None
        self.yield_maximum = None
        self.organic_average = None
        self.samz_zone = None
        self.samz_zoning = None
        self.hotspot_fetch = None
        self.hotspot_polygon = None
        self.hotspot_polygon_part = None
        self.hotspot_position = None
        self.hot_spot_point_on_surface = True
        self.hot_spot_min = False
        self.hot_spot_ave = False
        self.hot_spot_med = False
        self.hot_spot_max = False
        self.zoning_segmentation = None
        self.output_map_format = None
        self.gain = DEFAULT_GAIN
        self.offset = DEFAULT_OFFSET
        self.map_creation_parameters_settings = {
            YIELD_AVERAGE: self.yield_average_form,
            YIELD_MINIMUM: self.yield_minimum_form,
            YIELD_MAXIMUM: self.yield_maximum_form,
            ORGANIC_AVERAGE: self.organic_average_form,
            SAMZ_ZONE: self.samz_zone_form
        }
        # For rx zone
        self.fetch_rx_map = None
        # TODO: Handle the RX zone creation parameters similarly to the SAMZ
        # zone
        self.rx_zone = None
        self.rx_map_json = None
        self.selected_coverage_results = []

        # Stores the selected layer text for when a coverage search is done
        self.current_selected_layer = None

        # Coverage parameters from settings
        self.crop_type = setting(
            'crop_type', expected_type=str, qsettings=self.settings)
        self.sowing_date = setting(
            'sowing_date', expected_type=str, qsettings=self.settings)
        self.output_directory = setting(
            'output_directory', expected_type=str, qsettings=self.settings)

        # Flag used to prevent recursion and allow bulk loads of layers to
        # trigger a single event only
        self.get_layers_lock = False

        # Set connectors
        self.setup_connectors()
        self.update_button_states()  # Ensure buttons are correctly initialized

        # Populate layer combo box
        self.connect_layer_listener()
        self.connect_point_layer_listener()

        # Set checkbox label based on MAX_FEATURE_NUMBERS constant
        if MAX_FEATURE_NUMBERS:
            label_format = self.tr('Selected features only (max {} features)')
            self.selected_features_checkbox.setText(
                label_format.format(MAX_FEATURE_NUMBERS))

        # Populate map product combo box
        self.populate_map_products()

        # Populate sensor combo box
        self.clear_combo_box(self.sensor_combo_box)
        self.populate_sensors()
        self.populate_mask_types()

        # Set default date value
        self.populate_date()

        # Set default behaviour
        # self.help_push_button.setEnabled(False)
        self.back_push_button.setEnabled(False)
        self.next_push_button.setEnabled(True)
        self.difference_map_push_button.setVisible(False)
        self.samz_zone_form.setValue(DEFAULT_ZONE_COUNT)
        self.stacked_widget.setCurrentIndex(self.current_stacked_widget_index)
        self.set_next_button_text(self.current_stacked_widget_index)

    def populate_sensors(self):
        """Obtain a list of sensors from Bridge API definition."""
        for sensor in [ALL_SENSORS] + SENSORS:
            add_ordered_combo_item(
                self.sensor_combo_box, sensor['name'], sensor['key'])

    def populate_mask_types(self):
        """Populates the the Mask type combobox."""
        for mask_type in MASK_PARAMETERS:
            self.cb_mask.addItem(mask_type)

            # Set 'Auto' as the default selected option
        index = self.cb_mask.findText('Auto')
        if index != -1:
            self.cb_mask.setCurrentIndex(index)

    def populate_map_products(self):
        """Obtain a list of map products from Bridge API definition.
        If the US zone has been selected the soil option will be included, otherwise excluded.
        """
        # Checks if the US zone option is selected/activate
        key = 'geosys_region_na'
        us_option = setting(key, expected_type=bool, qsettings=self.settings)

        self.clear_combo_box(self.map_product_combo_box)

        for map_product in ARCHIVE_MAP_PRODUCTS:
            product_name = map_product['name']
            if us_option:  # If US zone is selected the SOILMAP option will be added
                add_ordered_combo_item(
                    self.map_product_combo_box,
                    map_product['name'],
                    map_product['key'])
            else:  # If EU area is selected the SOILMAP option will not be added
                if product_name != SOIL['name']:
                    add_ordered_combo_item(
                        self.map_product_combo_box,
                        map_product['name'],
                        map_product['key'])

    def populate_date(self):
        """Set default value of start and end date to last week."""
        current_date = QDate.currentDate()
        last_year_date = current_date.addDays(-365)
        self.start_date_edit.setDate(last_year_date)
        self.end_date_edit.setDate(current_date)

    def show_help(self):
        """Open the help dialog."""
        # noinspection PyTypeChecker
        dialog = HelpDialog(self)
        dialog.show()

    def show_previous_page(self):
        """Open previous page of stacked widget."""
        if self.current_stacked_widget_index > 0:
            # If the current map type is elevation or soil map and the
            # widget is on the map creation page, the back button should
            # take the user to the coverage parameters page
            if ((self.map_product == ELEVATION['key']
                 or self.map_product == SOIL['key']
                 or self.map_product == SLOPE['key'])
                    and self.current_stacked_widget_index == 2):
                self.current_stacked_widget_index -= 2
            else:
                self.current_stacked_widget_index -= 1
            self.stacked_widget.setCurrentIndex(
                self.current_stacked_widget_index)
            self.next_push_button.setEnabled(True)

            # Sets the current selection to an empty string so
            # that layer changes will not change back to the original selection
            # as it's before the search has been done
            self.current_selected_layer = None
        if self.current_stacked_widget_index == 0:
            self.back_push_button.setEnabled(False)

        self.handle_difference_map_button()

    def show_next_page(self):
        """Open next page of stacked widget."""
        # If current page is coverage parameters page, run coverage searcher.
        if self.current_stacked_widget_index == 0:
            self.start_coverage_search()
            self.next_push_button.setEnabled(False)

            # Stores the current selected layer name
            # This will be used to keep this layer selected for use in the
            # widget display
            self.current_selected_layer = self.geometry_combo_box.currentText()

        # Default behavior when fetch_rx_group is not checked
        # If current page is coverage results page, prepare map creation
        # parameters.
        if self.current_stacked_widget_index == 1:
            if self.map_product == REFLECTANCE['key']:
                # Reflectance can only make use of the tiff.zip format
                self.tiff_radio_button.setChecked(True)

                # Unchecks and disabled all other options for reflectance maps
                self.png_radio_button.setChecked(False)
                self.png_radio_button.setEnabled(False)

                self.shp_radio_button.setChecked(False)
                self.shp_radio_button.setEnabled(False)

                self.kmz_radio_button.setChecked(False)
                self.kmz_radio_button.setEnabled(False)
            elif self.map_product == COLOR_COMPOSITION['key']:
                # Color composition can only work with png
                self.png_radio_button.setChecked(True)
                self.png_radio_button.setEnabled(True)

                # These formats are not available for color composition
                self.tiff_radio_button.setChecked(False)
                self.tiff_radio_button.setEnabled(False)

                self.shp_radio_button.setChecked(False)
                self.shp_radio_button.setEnabled(False)

                self.kmz_radio_button.setChecked(False)
                self.kmz_radio_button.setEnabled(False)

                # Hide groups that are not needed for color composition
                self.hotspots_group.hide()
                self.fetch_rx_group.hide()
            else:
                # If these radio buttons has been disabled, it is reenabled
                self.png_radio_button.setEnabled(True)
                self.tiff_radio_button.setEnabled(True)
                self.shp_radio_button.setEnabled(True)
                self.kmz_radio_button.setEnabled(True)

            self.set_gain_offset_state()  # Disabled gain and offset for some map product types
            self.set_parameter_values_as_default()
            # self.restore_parameter_values_from_setting()

        # Handle the case when fetch_rx_group is checked
        if self.current_stacked_widget_index == 2 and self.fetch_rx_group.isChecked():
            # If on the first page, go to the next page
            self.set_zone_visibility()
            self.update_zone_areas()
            self.current_stacked_widget_index += 1
            self.stacked_widget.setCurrentIndex(
                self.current_stacked_widget_index
            )
            self.set_next_button_text(self.current_stacked_widget_index)
            return
        elif self.current_stacked_widget_index == self.max_stacked_widget_index:
            # Dynamically show/hide elements based on the RX zone count
            self.set_zone_visibility()
            # If on the last page, perform the map creation task
            self.png_radio_button_2.setEnabled(True)
            self.tiff_radio_button_2.setEnabled(True)
            self.shp_radio_button_2.setEnabled(True)
            self.kmz_radio_button_2.setEnabled(True)
            self.back_push_button.setEnabled(True)
            self.start_map_creation()
            return

        # If current page is map creation parameters page, create map without
        # increasing index.
        if self.current_stacked_widget_index == 2:
            self.start_map_creation()
            return

        if self.current_stacked_widget_index < self.max_stacked_widget_index:
            self.current_stacked_widget_index += 1
            self.stacked_widget.setCurrentIndex(
                self.current_stacked_widget_index)
            self.back_push_button.setEnabled(True)

        self.handle_difference_map_button()

    def set_zone_visibility(self):
        """Shows or hides zone-related UI elements based on the RX zone count."""
        rx_zone_count = self.fetch_rx_zones.value()
        for zone_index in range(1, 21):
            # Dynamically construct object names
            label_name = f"zone_x_{zone_index}"
            line_edit_name = f"zone_{zone_index}_val"
            spinbox_name = f"zone_{zone_index}_sb"

            # Retrieve the UI elements using getattr
            label = getattr(self, label_name, None)
            line_edit = getattr(self, line_edit_name, None)
            spinbox = getattr(self, spinbox_name, None)

            # Show or hide elements based on the RX zone count
            if zone_index >= rx_zone_count + 1:
                # Hide
                label.hide()
                line_edit.hide()
                spinbox.hide()
            else:
                # Show
                label.show()
                line_edit.show()
                spinbox.show()

    def fetch_rx_json(self, map_specifications, map_product):
        bridge_api = BridgeAPI(
            *credentials_parameters_from_settings(),
            proxies=QGISSettings.get_qgis_proxy())

        geometry = self.wkt_geometries[0]

        if map_product == SAMZ['key']:
            image_dates = []
            image_ids = []
            # Check if images are provided; raise an error if none are selected
            if not map_specifications:
                QMessageBox.critical(
                    self,
                    'Image Selection Required',
                    'At least one image must be selected to generate a SAMZ map.')
                return
            # Proceed with custom SAMZ using selected images
            geometry = self.wkt_geometries[0]
            zone_cnt = self.fetch_rx_zones.value()

            if len(map_specifications) == 1:
                # Log and use the single image provided
                single_specification = map_specifications[0]
                image_dates.append(single_specification['image']['date'])
                image_ids.append(single_specification['image']['id'])
            else:
                # Iterate through multiple specifications
                for map_specification in map_specifications:
                    image_dates.append(map_specification['image']['date'])
                    image_ids.append(map_specification['image']['id'])
            try:
                map_response = fetch_samz_map(
                    geometry,
                    image_ids,
                    image_dates,
                    zone_cnt
                )

                if map_response and 'id' in map_response:
                    source_map_id = (map_response['id'])
            except Exception as e:
                QMessageBox.critical(
                    self,
                    'RX Map',
                    f'Error fetching samz map data: {e}'
                )
                return

            rx_map_json = bridge_api.get_rx_map(
                url=bridge_api.bridge_server,
                source_map_id=source_map_id,
                zone_count=zone_cnt
            )

            return rx_map_json

        else:

            # Extract season field and image IDs
            source_map_id = None
            zone_count = self.fetch_rx_zones.value()
            self.gain = self.spinBox_gain.value()
            self.offset = self.spinBox_offset.value()

            data = {
                YIELD_AVERAGE: self.yield_average,
                YIELD_MINIMUM: self.yield_minimum,
                YIELD_MAXIMUM: self.yield_maximum,
                ORGANIC_AVERAGE: self.organic_average,
                SAMZ_ZONE: zone_count,
                GAIN: self.gain,
                OFFSET: self.offset
            }
            for map_specification in map_specifications:
                sample_map_data = None
                if self.map_product == SAMPLE_MAP['key']:
                    sample_data = []
                    i = 0
                    # Create the request data from the points and its
                    # values
                    for geom in self.wkt_point_geometries:
                        val = self.attributes[i]
                        data_item = {
                            "geometry": geom,
                            "value": val
                        }
                        sample_data.append(data_item)

                        i += 1
                    # The final request data
                    sample_map_data = {
                        "seasonField": {
                            "Id": None,
                            "geometry": geometry,
                        },
                        "properties": {
                            "nutrientType": self.sample_map_field
                        },
                        "data": sample_data
                    }

                    data = sample_map_data

                try:
                    map_response = fetch_map(
                        map_specification, map_product, geometry,
                        n_planned_value=self.n_planned_value,
                        yield_val=self.yield_average_form.value(),
                        min_yield_val=self.yield_minimum_form.value(),
                        max_yield_val=self.yield_maximum_form.value(),
                        params=data, data=data,
                        crop_type=self.crop_type,
                        gain=self.gain, offset=self.offset,
                        zone_count=zone_count
                    )

                    if map_response and 'id' in map_response:
                        source_map_id = (map_response['id'])
                except Exception as e:
                    QMessageBox.critical(
                        self,
                        'RX Map',
                        f'Error fetching map data: {e}'
                    )
                    return

                rx_map_json = bridge_api.get_rx_map(
                    url=bridge_api.bridge_server,
                    source_map_id=source_map_id,
                    zone_count=zone_count
                )

                return rx_map_json

    def get_areas_from_rx_map(self, rx_map_json):
        """Collect areas of each zone from the rx_map_json."""
        areas = []

        # Check if 'zones' is in rx_map_json
        if 'zones' in rx_map_json:
            # Iterate through each zone in the 'zones' list
            for zone in rx_map_json['zones']:
                # Check if 'stats' and 'area' exist for the zone
                if 'stats' in zone and 'area' in zone['stats']:
                    areas.append(zone['stats']['area'])
        return areas

    def update_zone_areas(self):
        """Updates the area values in the corresponding line_edit fields based on the RX zone data."""

        # Fetch the rx_map_json and extract the areas
        try:
            QApplication.setOverrideCursor(QCursor(Qt.WaitCursor))
            self.rx_map_json = self.fetch_rx_json(
                self.selected_coverage_results,
                self.map_product
            )
            areas = self.get_areas_from_rx_map(
                self.rx_map_json)  # Call the method to get areas

            for zone_index in range(1, 21):
                # Dynamically construct object names for line edits
                line_edit_name = f"zone_{zone_index}_val"

                # Retrieve the line_edit element using getattr
                line_edit = getattr(self, line_edit_name, None)

                # Only update if the line_edit exists and the area data is
                # available for the zone
                if line_edit and zone_index <= len(areas):
                    # Adjust for 0-based index
                    area_value = areas[zone_index - 1]
                    line_edit.setText(f"{area_value:.3f} Ha")
                    line_edit.setReadOnly(True)
        except Exception as e:
            log(f"Error updating area values: {e}")
        finally:
            QApplication.restoreOverrideCursor()

    def set_gain_offset_state(self):
        """Disables the gain and offset options in the parameters menu for the COLORCOMPOSITION, ELEVATION,
        OM, SLOPE, SOILMAP, SAMZ, YGM, and YPM map product types.
        """
        selected_map_product = self.map_product  # Map product type selected by the user
        list_products_to_exclude = [
            'COLORCOMPOSITION',
            'ELEVATION',
            'SLOPE',
            'OM',
            'SOILMAP',
            'SAMZ',
            'YGM',
            'YPM']

        for map_product_to_exclude in list_products_to_exclude:
            if selected_map_product == map_product_to_exclude:
                # The gain and offset options will be hidden
                self.gain_label.hide()
                self.offset_label.hide()
                self.spinBox_gain.hide()
                self.spinBox_offset.hide()
                return

        # If the gain and offset options should be shown
        self.gain_label.show()
        self.offset_label.show()
        self.spinBox_gain.show()
        self.spinBox_offset.show()

    def set_next_button_text(self, index):
        """Programmatically change next button text based on current page."""
        text_rule = {
            0: 'Search Map',
            1: 'Next',
            2: 'Create Map'
        }
        if self.fetch_rx_group.isChecked():
            # When fetch_rx_group is active, set "Next" or "Create Map"
            if index == self.max_stacked_widget_index:
                self.next_push_button.setText("Create Map")
            else:
                self.next_push_button.setText("Next")
        else:
            self.next_push_button.setText(text_rule[index])

    def handle_difference_map_button(self):
        """Handle difference map button behavior."""
        if self.current_stacked_widget_index == 2:
            # Show difference map button only if 2 items are being selected
            # and has same SeasonField ID.

            # check SeasonField ID
            has_same_id = False
            season_field_id = None
            for coverage_result in self.selected_coverage_results:
                if not season_field_id:
                    season_field_id = coverage_result['seasonField']['id']
                else:
                    has_same_id = season_field_id == (
                        coverage_result['seasonField']['id'])

            if len(self.selected_coverage_results) == 2 and has_same_id and (
                    self.map_product in [NDVI['key'],
                                         EVI['key']]):
                self.difference_map_push_button.setVisible(True)
            else:
                self.difference_map_push_button.setVisible(False)
        else:
            self.difference_map_push_button.setVisible(False)

    def update_selection_data(self):
        """Update current selection data."""
        # update data based on selected coverage results
        self.selected_coverage_results = []
        for item in self.coverage_result_list.selectedItems():
            item_json = item.data(Qt.UserRole)
            if self.map_product == REFLECTANCE['key']:
                # NDVI used for coverage. This is a workaround suggested by
                # GeoSys
                item_json['maps'][0]['type'] = REFLECTANCE['key']
            elif self.map_product == SOIL['key']:
                # Used when soilmap type. This is a workaround suggested by
                # GeoSys
                item_json['maps'][0]['type'] = SOIL['key']
            elif self.map_product == ELEVATION['key']:
                # For elevation map type
                item_json['maps'][0]['type'] = ELEVATION['key']
            elif self.map_product == SLOPE['key']:
                # For slope map type
                item_json['maps'][0]['type'] = SLOPE['key']
            elif self.map_product == INSEASONFIELD_AVERAGE_NDVI['key']:
                item_json['maps'][0]['type'] = INSEASONFIELD_AVERAGE_NDVI['key']
            elif self.map_product == INSEASONFIELD_AVERAGE_LAI['key']:
                item_json['maps'][0]['type'] = INSEASONFIELD_AVERAGE_LAI['key']
            elif self.map_product == INSEASONFIELD_AVERAGE_REVERSE_NDVI['key']:
                item_json['maps'][0]['type'] = INSEASONFIELD_AVERAGE_REVERSE_NDVI['key']
            elif self.map_product == INSEASONFIELD_AVERAGE_REVERSE_LAI['key']:
                item_json['maps'][0]['type'] = INSEASONFIELD_AVERAGE_REVERSE_LAI['key']
            elif self.map_product == SAMPLE_MAP['key']:
                item_json['maps'][0]['type'] = SAMPLE_MAP['key']

            self.selected_coverage_results.append(item_json)

        self.handle_difference_map_button()

    def get_layers(self, *args):
        """Obtain a list of layers currently loaded in QGIS.

        Only **polygon vector** layers will be added to the layers list.

        :param *args: Arguments that may have been passed to this slot.
            Typically a list of layers, but depends on which slot or function
            called this function.
        :type *args: list

        ..note:: \*args is only used for debugging purposes.
        """
        _ = args  # NOQA
        # Prevent recursion
        if self.get_layers_lock:
            return

        # Map registry may be invalid if QGIS is shutting down
        project = QgsProject.instance()
        canvas_layers = self.iface.mapCanvas().layers()
        # MapLayers returns a QMap<QString id, QgsMapLayer layer>
        layers = list(project.mapLayers().values())

        self.get_layers_lock = True

        # Make sure this comes after the checks above to prevent signal
        # disconnection without reconnection.
        self.block_signals()
        self.geometry_combo_box.clear()

        for layer in layers:
            # show only active layers
            if layer not in canvas_layers or not is_polygon_layer(layer):
                continue

            layer_id = layer.id()
            title = layer.title() or layer.name()
            icon = layer_icon(layer)

            add_ordered_combo_item(
                self.geometry_combo_box, title, layer_id, icon=icon)

        current_index = self.geometry_combo_box.findText(
            self.current_selected_layer)
        if current_index >= 0:
            # Current text/previous selection, is in the list of active layers
            # The previous selection will still be selected
            self.geometry_combo_box.setCurrentIndex(current_index)

        self.unblock_signals()
        # Note: Don't change the order of the next two lines otherwise there
        # will be a lot of unneeded looping around as the signal is handled
        self.connect_layer_listener()
        self.get_layers_lock = False

    def get_sample_map_point_layers(self):
        """Gets the list of possible point layers in the QGIS project
        and adds them as options for Sample maps
        """
        # Map registry may be invalid if QGIS is shutting down
        project = QgsProject.instance()
        canvas_layers = self.iface.mapCanvas().layers()
        # MapLayers returns a QMap<QString id, QgsMapLayer layer>
        layers = list(project.mapLayers().values())

        self.cb_point_layer.clear()

        for layer in layers:
            # show only active layers
            if layer not in canvas_layers or not is_point_layer(layer):
                continue

            layer_id = layer.id()
            title = layer.title() or layer.name()
            icon = layer_icon(layer)

            add_ordered_combo_item(
                self.cb_point_layer, title, layer_id, icon=icon)

        active_layer = self.iface.activeLayer()
        if active_layer:
            active_layer_name = self.iface.activeLayer().name()
            current_index = self.cb_point_layer.findText(active_layer_name)

            if current_index >= 0:
                # Current text/previous selection, is in the list of active layers
                # The previous selection will still be selected
                self.cb_point_layer.setCurrentIndex(current_index)

    def get_map_format(self):
        """Get selected map format from the radio button."""

        if self.fetch_rx_group.isChecked():
            # If fetch_rx_group is checked, the radio buttons are different
            widget_data = [
                {
                    'widget': self.png_radio_button_2,
                    'data': PNG
                },
                {
                    'widget': self.tiff_radio_button_2,
                    'data': ZIPPED_TIFF
                },
                {
                    'widget': self.shp_radio_button_2,
                    'data': ZIPPED_SHP
                },
                {
                    'widget': self.kmz_radio_button_2,
                    'data': KML
                },
            ]
        else:
            # Default behavior
            widget_data = [
                {
                    'widget': self.png_radio_button,
                    'data': PNG
                },
                {
                    'widget': self.tiff_radio_button,
                    'data': ZIPPED_TIFF
                },
                {
                    'widget': self.shp_radio_button,
                    'data': ZIPPED_SHP
                },
                {
                    'widget': self.kmz_radio_button,
                    'data': KML
                },
            ]
        for wd in widget_data:
            if wd['widget'].isChecked():
                return wd['data']

    def load_layer(self, base_path):
        """Load layer into QGIS map canvas.

        :param base_path: Base path of the layer.
        :type base_path: str
        """
        if self.output_map_format in VALID_QGIS_FORMAT:
            filename = os.path.basename(base_path)
            layer = base_path + self.output_map_format['extension']

            if self.output_map_format in VECTOR_FORMAT:
                map_layer = QgsVectorLayer(layer, filename)
            else:
                if os.path.exists(layer):
                    map_layer = QgsRasterLayer(layer, filename)
                else:
                    if '.tiff' in layer:
                        layer = layer.replace('.tiff', '.tif')
                        map_layer = QgsRasterLayer(layer, filename)
                    else:
                        raise FileNotFoundError(f"File not found: {layer}")
            add_layer_to_canvas(map_layer, filename)

    def save_parameter_values_as_setting(self):
        """Save parameter values as qsettings."""
        for key, form in self.map_creation_parameters_settings.items():
            set_setting(key, form.value(), self.settings)

    def restore_parameter_values_from_setting(self):
        """Restore parameter values from qsettings."""
        for key, form in self.map_creation_parameters_settings.items():
            value = setting(key, expected_type=int, qsettings=self.settings)
            value and form.setValue(value)

    def set_parameter_values_as_default(self):
        """Set parameter values to default values."""
        for key, form in self.map_creation_parameters_settings.items():
            value = 0  # Default value if none is given
            if key == YIELD_AVERAGE:
                value = DEFAULT_AVE_YIELD
            elif key == YIELD_MINIMUM:
                value = DEFAULT_MIN_YIELD
            elif key == YIELD_MAXIMUM:
                value = DEFAULT_MAX_YIELD
            elif key == ORGANIC_AVERAGE:
                value = DEFAULT_ORGANIC_AVE
            elif key == SAMZ_ZONE:
                value = DEFAULT_ZONE_COUNT

            form.setValue(value)

    def validate_map_creation_parameters(self):
        """Check current state of map creation parameters."""
        self.yield_average = self.yield_average_form.value()
        self.yield_minimum = self.yield_minimum_form.value()
        self.yield_maximum = self.yield_maximum_form.value()
        self.organic_average = self.organic_average_form.value()
        self.samz_zone = self.samz_zone_form.value()
        # self.hotspot_polygon = self.hotspot_polygon_form.isChecked()
        # self.hotspot_polygon_part = self.hotspot_polygon_part_form.isChecked()
        self.output_map_format = self.get_map_format()

        self.hotspot_fetch = self.hotspots_group.isChecked()
        if self.hotspot_fetch:
            self.hotspot_polygon = self.hotspot_polygon_form.isChecked()
            self.hotspot_polygon_part = self.hotspot_polygon_part_form.isChecked()

            self.hot_spot_point_on_surface = self.rb_point_on_surface.isChecked()
            self.hot_spot_min = self.rb_min.isChecked()
            self.hot_spot_ave = self.rb_ave.isChecked()
            self.hot_spot_med = self.rb_med.isChecked()
            self.hot_spot_max = self.rb_max.isChecked()

        # SaMZ map creation accept zero selected results, which means it will
        # trigger automatic SaMZ map creation.
        if len(self.selected_coverage_results) == 0 and (
                self.map_product != SAMZ['key']):
            return False, 'Please select at least one coverage result.'

        return True, ''

    def validate_coverage_parameters(self):
        """Check current state of coverage parameters."""

        # Get map product
        self.map_product = item_data_from_combo(self.map_product_combo_box)
        if not self.map_product:
            # map product is not valid
            return False, 'Map product data is not valid.'

        layer = layer_from_combo(self.geometry_combo_box)
        if not layer:
            # layer is not selected
            return False, 'Layer is not selected.'
        use_selected_features = (
            self.selected_features_checkbox.isChecked() and (
                layer.selectedFeatureCount() > 0))
        use_single_geometry = self.single_geometry_checkbox.isChecked()

        # Reproject layer to EPSG:4326
        if layer.crs().authid() != 'EPSG:4326':
            layer = reproject(
                layer, QgsCoordinateReferenceSystem('EPSG:4326'))

        feature_iterator = layer.getFeatures()
        if use_selected_features:
            request = QgsFeatureRequest()
            request.setFilterFids(layer.selectedFeatureIds())
            feature_iterator = layer.getFeatures(request)

        # Handle multi features
        # Merge features into multi-part polygon
        # TODO use Collect Geometries processing algorithm
        self.wkt_geometries = wkt_geometries_from_feature_iterator(
            feature_iterator, MAX_FEATURE_NUMBERS, use_single_geometry)

        if not self.wkt_geometries:
            # geometry is not valid
            return False, 'Geometry is not valid.'

        # Get the sensor type
        self.sensor_type = item_data_from_combo(self.sensor_combo_box)
        if not self.sensor_type:
            # sensor type is not valid
            return False, 'Sensor data is not valid.'
        if self.sensor_type == ALL_SENSORS['key']:
            self.sensor_type = None

        # Get the mask type
        self.mask_type = item_text_from_combo(self.cb_mask)
        if not self.mask_type:
            # Mask type invalid
            return False, 'Mask type is invalid.'

        if self.map_product == SAMPLE_MAP['name'].lower():
            # These checks are only required for sample maps
            self.sample_map_point_layer = item_data_from_combo(
                self.cb_point_layer)
            if not self.sample_map_point_layer:
                # Point layer
                return False, 'Sample point layer has not been selected.'

            self.sample_map_field = item_text_from_combo(self.cb_column_name)
            if not self.sample_map_field:
                # Field name of point layer attribute table
                return False, 'Sample point layer column has not been selected.'

            layer_points = layer_from_combo(self.cb_point_layer)
            feature_points_iterator = layer_points.getFeatures()
            feature_points_iterator2 = layer_points.getFeatures()
            if use_selected_features:
                request = QgsFeatureRequest()
                request.setFilterFids(layer_points.selectedFeatureIds())
                feature_points_iterator = layer_points.getFeatures(request)
                feature_points_iterator2 = layer_points.getFeatures(request)

            self.wkt_point_geometries = wkt_geometries_from_feature_iterator(
                feature_points_iterator, MAX_FEATURE_NUMBERS, use_single_geometry)

            self.attributes = attribute_from_feature_iterator(
                feature_points_iterator2, self.sample_map_field)

        # Get the start and end date
        self.start_date = self.start_date_edit.date().toString('yyyy-MM-dd')
        self.end_date = self.end_date_edit.date().toString('yyyy-MM-dd')

        # Coverage percent
        self.coverage_percent = self.coverage_percent_value_spinbox.value()

        self.n_planned_value = self.n_planned_value_spinbox.value()

        return True, ''

    def _start_map_creation(self, map_specifications):
        """Actual method to run the map creation task.

        :param map_specifications: List of map specification.
        :type map_specifications: list
        """
        geometry = self.wkt_geometries[0]
        # Checks whether the gain and offset values are allowed

        self.output_directory = setting(
            'output_directory', expected_type=str, qsettings=self.settings)

        # Map product type selected by the user
        selected_map_product = self.map_product
        list_products_to_exclude = [
            'COLORCOMPOSITION',
            'ELEVATION',
            'SLOPE',
            'OM',
            'SOILMAP',
            'SAMZ',
            'YGM',
            'YPM'
        ]
        gain_offset_allowed = True
        for map_product_to_exclude in list_products_to_exclude:
            if selected_map_product == map_product_to_exclude:
                # The gain and offset values will not be included
                gain_offset_allowed = False
                break

        map_product_definition = get_definition(self.map_product)
        if gain_offset_allowed and \
                (self.spinBox_gain.value() > 0
                 or self.spinBox_offset.value() > 0):  # Gain and offset will be added to the data
            self.gain = self.spinBox_gain.value()  # Gain set by user
            self.offset = self.spinBox_offset.value()  # Offset set by user
            data = {
                YIELD_AVERAGE: self.yield_average,
                YIELD_MINIMUM: self.yield_minimum,
                YIELD_MAXIMUM: self.yield_maximum,
                ORGANIC_AVERAGE: self.organic_average,
                SAMZ_ZONE: self.samz_zone,
                GAIN: self.gain,
                OFFSET: self.offset
            }
        else:  # Gain and offset will not be included
            data = {
                YIELD_AVERAGE: self.yield_average,
                YIELD_MINIMUM: self.yield_minimum,
                YIELD_MAXIMUM: self.yield_maximum,
                ORGANIC_AVERAGE: self.organic_average,
                SAMZ_ZONE: self.samz_zone
            }

        if self.samz_zone > 0:
            self.samz_zoning = True
            data.update({
                SAMZ_ZONING: 'true'
            })
            if self.hotspots_group.isChecked():
                if self.hotspot_polygon:
                    data.update({
                        HOTSPOT: 'true'
                    })
                if self.hotspot_polygon_part:
                    data.update({
                        HOTSPOT: self.hotspot_polygon_part,
                        ZONING_SEGMENTATION: 'Zone'
                    })

                position_values = {
                    'PointOnSurface': self.hot_spot_point_on_surface,
                    'Minimum': self.hot_spot_min,
                    'Maximum': self.hot_spot_max,
                    'Average': self.hot_spot_ave,
                    'Median': self.hot_spot_med,
                }
                position = 'Average'
                for key, value in position_values.items():
                    if value:
                        position = f"{key}"
                        break
                data.update({
                    POSITION: position
                })

        zone_cnt = self.samz_zone_form.value()
        if self.fetch_rx_group and self.fetch_rx_group.isChecked():  # RX Map Logic
            rx_zone_count = self.fetch_rx_zones.value()
            rx_json_map = self.rx_map_json

            source_map_id = rx_json_map.get('id')

            filename = f"{self.map_product}_RX_zones_{rx_zone_count}"
            filename = clean_filename(filename)
            filename = check_if_file_exists(
                self.output_directory,
                filename,
                self.output_map_format['extension']
            )

            # Dynamically generate patch_data
            patch_data = []
            value = self.rx_name_line.text() or 'value'
            for i in range(rx_zone_count):
                # Dynamically get the spinbox
                widget = getattr(self, f"zone_{i + 1}_sb", None)
                if widget:
                    patch_data.append({
                        "op": "add",
                        "path": f"/parameters/zones/{i}/attributes/{value}",
                        "value": widget.value()
                    })

            #patch_data = json.dumps(patch_data)

            is_success, message = create_rx_map(
                rx_map_json=rx_json_map,
                source_map_id=source_map_id,
                zone_count=rx_zone_count,
                output_dir=self.output_directory,
                filename=filename,
                output_map_format=self.output_map_format,
                data=data,
                patch_data=patch_data
            )

            if not is_success:
                QMessageBox.critical(
                    self,
                    'RX Map Creation Error',
                    f'Error creating RX Map: {message}'
                )
                return

            # Load the RX map into the QGIS canvas
            self.load_layer(os.path.join(self.output_directory, filename))
            return
        else:
            if map_product_definition == SAMZ:
                image_dates = []
                image_ids = []
                # Check if images are provided; raise an error if none are
                # selected
                if not map_specifications:
                    QMessageBox.critical(
                        self,
                        'Image Selection Required',
                        'At least one image must be selected to generate a SAMZ map.')
                    return
                # Proceed with custom SAMZ using selected images
                season_field_id = map_specifications[0]['seasonField']['id']
                geometry = self.wkt_geometries[0]
                if len(map_specifications) == 1:
                    # Log and use the single image provided
                    single_specification = map_specifications[0]
                    image_dates.append(single_specification['image']['date'])
                    image_ids.append(single_specification['image']['id'])
                else:
                    # Iterate through multiple specifications
                    for map_specification in map_specifications:
                        image_dates.append(map_specification['image']['date'])
                        image_ids.append(map_specification['image']['id'])

                filename = '{}_{}_zones'.format(
                    SAMZ['key'], str(zone_cnt))
                filename = clean_filename(filename)
                filename = check_if_file_exists(
                    self.output_directory,
                    filename,
                    self.output_map_format['extension']
                )

                is_success, message = create_samz_map(
                    geometry, image_ids, image_dates, zone_cnt, self.output_directory, filename,
                    output_map_format=self.output_map_format, params=data)

                if not is_success:
                    QMessageBox.critical(
                        self,
                        'Map Creation Status',
                        'Error creating map. {}'.format(message))
                    return

                # Add map to qgis canvas
                self.load_layer(os.path.join(self.output_directory, filename))
            else:
                for map_specification in map_specifications:
                    filename = '{}_{}_zones_{}_{}'.format(
                        self.map_product,
                        # map_specification['maps'][0]['type'],
                        str(zone_cnt),
                        map_specification['seasonField']['id'] or '',
                        map_specification['image']['date'] or ''
                    )
                    filename = clean_filename(filename)
                    filename = check_if_file_exists(
                        self.output_directory,
                        filename,
                        self.output_map_format['extension']
                    )

                    sample_map_data = None
                    if self.map_product == SAMPLE_MAP['key']:
                        sample_data = []
                        i = 0
                        # Create the request data from the points and its
                        # values
                        for geom in self.wkt_point_geometries:
                            val = self.attributes[i]
                            data_item = {
                                "geometry": geom,
                                "value": val
                            }
                            sample_data.append(data_item)

                            i += 1
                        # The final request data
                        sample_map_data = {
                            "seasonField": {
                                "Id": None,
                                "geometry": geometry,
                            },
                            "properties": {
                                "nutrientType": self.sample_map_field
                            },
                            "data": sample_data
                        }

                        data = sample_map_data

                    is_success, message = create_map(
                        map_specification, self.map_product, geometry, self.output_directory, filename,
                        data=data, output_map_format=self.output_map_format,
                        n_planned_value=self.n_planned_value,
                        yield_val=self.yield_average_form.value(),
                        min_yield_val=self.yield_minimum_form.value(),
                        max_yield_val=self.yield_maximum_form.value(),
                        sample_map_id=None, params=data, crop_type=self.crop_type,
                        gain=self.gain, offset=self.offset, zone_count=self.samz_zone,
                    )

                    if not is_success:
                        QMessageBox.critical(
                            self,
                            'Map Creation Status',
                            'Error creating map. {}'.format(message))
                        return

                    # Add map to qgis canvas
                    self.load_layer(
                        os.path.join(
                            self.output_directory,
                            filename))

    def start_map_creation(self):
        """Map creation starts here."""
        # validate map creation parameters before creating the map
        message_title = 'Map Creation Status'
        try:
            QApplication.setOverrideCursor(QCursor(Qt.WaitCursor))
            is_success, message = self.validate_map_creation_parameters()
            if not is_success:
                QMessageBox.critical(
                    self,
                    message_title,
                    'Error validating map creation parameters. {}'.format(
                        message))
                return
            # store parameters value as qsettings
            self.save_parameter_values_as_setting()

            # start map creation job
            self._start_map_creation(self.selected_coverage_results)
        except Exception as e:
            error_text = "{0}: {1}".format(
                unicode(sys.exc_info()[0].__name__),
                unicode(sys.exc_info()[1]))
            QMessageBox.critical(self, message_title, error_text)
        finally:
            QApplication.restoreOverrideCursor()

    def start_difference_map_creation(self):
        """Difference Map creation starts here."""
        message_title = 'Difference Map Creation Status'
        try:
            QApplication.setOverrideCursor(QCursor(Qt.WaitCursor))

            # Validate map creation parameters
            self.validate_map_creation_parameters()

            # Construct filename
            map_specifications = self.selected_coverage_results
            map_type_definition = get_definition(
                map_specifications[0]['maps'][0]['type'])
            difference_map_definition = map_type_definition['difference_map']
            filename = '{}_{}_{}_{}'.format(
                difference_map_definition['key'],
                map_specifications[0]['seasonField']['id'],
                map_specifications[0]['image']['date'],
                map_specifications[1]['image']['date']
            )

            # Run difference map creation
            is_success, message = create_difference_map(
                map_specifications, self.output_directory,
                filename, output_map_format=self.output_map_format)

            if not is_success:
                QMessageBox.critical(
                    self,
                    'Map Creation Status',
                    'Error creating map. {}'.format(message))
                return

            # Add map to qgis canvas
            self.load_layer(os.path.join(self.output_directory, filename))
        except:
            error_text = "{0}: {1}".format(
                unicode(sys.exc_info()[0].__name__),
                unicode(sys.exc_info()[1]))
            QMessageBox.critical(self, message_title, error_text)
        finally:
            QApplication.restoreOverrideCursor()

    def start_coverage_search(self):
        """Coverage search starts here."""
        # validate coverage parameters before run the coverage searcher
        is_success, message = self.validate_coverage_parameters()
        if not is_success:
            self.show_error(
                'Error validating coverage parameters. {}'.format(message))
            return

        if self.search_threads:
            self.search_threads.data_downloaded.disconnect()
            self.search_threads.search_finished.disconnect()
            self.search_threads.stop()
            self.search_threads.wait()
            self.coverage_result_list.clear()

        # start search thread
        map_product = COLOR_COMPOSITION['key'] if self.map_product == SAMZ['key'] else self.map_product
        searcher = CoverageSearchThread(
            geometries=self.wkt_geometries,
            crop_type=self.crop_type,
            sowing_date=self.sowing_date,
            map_product=map_product,
            sensor_type=self.sensor_type,
            mask_type=self.mask_type,
            end_date=self.end_date,
            start_date=self.start_date,
            geometries_points=self.wkt_point_geometries,
            attributes_points=self.attributes,
            attribute_field=self.sample_map_field,
            mutex=self.one_process_work,
            coverage_percent=self.coverage_percent,
            n_planned_value=self.n_planned_value,
            parent=self.iface.mainWindow())
        searcher.search_started.connect(self.coverage_search_started)
        searcher.search_finished.connect(self.coverage_search_finished)
        searcher.data_downloaded.connect(self.show_coverage_result)
        searcher.error_occurred.connect(self.show_error)
        self.search_threads = searcher
        searcher.start()

    def coverage_search_started(self):
        """Action after search thread started."""
        self.coverage_result_list.clear()
        self.coverage_result_list.insertItem(0, self.tr('Searching...'))

    def coverage_search_finished(self):
        """Action after search thread finished."""
        self.coverage_result_list.takeItem(0)
        coverage_result_empty = self.coverage_result_list.count() == 0
        self.next_push_button.setEnabled(not coverage_result_empty)
        if coverage_result_empty:
            new_widget = QLabel()
            new_widget.setTextFormat(Qt.RichText)
            new_widget.setOpenExternalLinks(True)
            new_widget.setWordWrap(True)
            new_widget.setText(
                u"<div align='center'> <strong>{}</strong> </div>"
                u"<div align='center' style='margin-top: 3px'> {} "
                u"</div>".format(
                    self.tr(u"No results."),
                    self.tr(
                        u"No coverage results available based on given "
                        u"parameters.")))
            new_item = QListWidgetItem(self.coverage_result_list)
            new_item.setSizeHint(new_widget.sizeHint())
            self.coverage_result_list.addItem(new_item)
            self.coverage_result_list.setItemWidget(
                new_item,
                new_widget
            )
        else:
            self.coverage_result_list.setCurrentRow(0)

            # When user selected Elevation or Soil map, we want to skip the coverage
            # results panel and go straight to the map creation panel rather.
            if self.map_product == ELEVATION['key'] or self.map_product == SOIL['key'] or self.map_product == SLOPE['key']:
                self.show_next_page()

    def show_coverage_result(self, coverage_map_json, thumbnail_ba):
        """Translate coverage map result into widget item.

        :param coverage_map_json: Result of single map coverage.
            example: {
                "seasonField": {
                    "id": "zgzmbrm",
                    "customerExternalId": "..."
                },
                "image": {
                    "date": "2018-10-18",
                    "sensor": "SENTINEL_2",
                    "mask": "Auto",
                    "soilMaterial": "BARE"
                },
                "maps": [
                    {
                        "type": "NDVI",
                        "_links": {
                            "self": "the_url",
                            "worldFile": "the_url",
                            "thumbnail": "the_url",
                            "legend": "the_url",
                            "image:image/png": "the_url",
                            "image:image/tiff+zip": "the_url",
                            "image:application/shp+zip": "the_url",
                            "image:application/vnd.google-earth.kmz": "the_url"
                        }
                    }
                ],
                "coverageType": "CLEAR"
            }
        :type coverage_map_json: dict

        :param thumbnail_ba: Thumbnail image data in byte array format.
        :type thumbnail_ba: QByteArray
        """
        if coverage_map_json:
            custom_widget = CoverageSearchResultItemWidget(
                coverage_map_json, thumbnail_ba, self.map_product)
            new_item = QListWidgetItem(self.coverage_result_list)
            new_item.setSizeHint(custom_widget.sizeHint())
            new_item.setData(Qt.UserRole, coverage_map_json)
            self.coverage_result_list.addItem(new_item)
            self.coverage_result_list.setItemWidget(new_item, custom_widget)
        else:
            new_item = QListWidgetItem()
            new_item.setText(self.tr('No results!'))
            new_item.setData(Qt.UserRole, None)
            self.coverage_result_list.addItem(new_item)
        self.coverage_result_list.update()

    def show_error(self, error_message):
        """Show error message as widget item.

        :param error_message: Error message.
        :type error_message: str
        """
        self.coverage_result_list.clear()
        new_widget = QLabel()
        new_widget.setTextFormat(Qt.RichText)
        new_widget.setOpenExternalLinks(True)
        new_widget.setWordWrap(True)
        new_widget.setText(
            u"<div align='center'> <strong>{}</strong> </div>"
            u"<div align='center' style='margin-top: 3px'> {} </div>".format(
                self.tr('Error'), error_message))
        new_item = QListWidgetItem(self.coverage_result_list)
        new_item.setSizeHint(new_widget.sizeHint())
        self.coverage_result_list.addItem(new_item)
        self.coverage_result_list.setItemWidget(new_item, new_widget)

    def connect_layer_listener(self):
        """Establish a signal/slot to listen for layers loaded in QGIS.

        ..seealso:: disconnect_layer_listener
        """
        project = QgsProject.instance()
        project.layersWillBeRemoved.connect(self.get_layers)
        project.layersAdded.connect(self.get_layers)
        project.layersRemoved.connect(self.get_layers)

        self.iface.mapCanvas().layersChanged.connect(self.get_layers) \
            if self.iface is not None else None

    # pylint: disable=W0702
    def disconnect_layer_listener(self):
        """Destroy the signal/slot to listen for layers loaded in QGIS.

        ..seealso:: connect_layer_listener
        """
        project = QgsProject.instance()
        project.layersWillBeRemoved.disconnect(self.get_layers)
        project.layersAdded.disconnect(self.get_layers)
        project.layersRemoved.disconnect(self.get_layers)

        self.iface.mapCanvas().layersChanged.disconnect(self.get_layers)

    def connect_point_layer_listener(self):
        """Establish a signal/slot to listen for point layers loaded in QGIS.

        ..seealso:: disconnect_point_layer_listener
        """
        project = QgsProject.instance()
        project.layersWillBeRemoved.connect(self.get_sample_map_point_layers)
        project.layersAdded.connect(self.get_sample_map_point_layers)
        project.layersRemoved.connect(self.get_sample_map_point_layers)

        self.iface.mapCanvas().layersChanged.connect(
            self.get_sample_map_point_layers) if self.iface is not None else None

    # pylint: disable=W0702
    def disconnect_point_layer_listener(self):
        """Destroy the signal/slot to listen for point layers loaded in QGIS.

        ..seealso:: connect_point_layer_listener
        """
        project = QgsProject.instance()
        project.layersWillBeRemoved.disconnect(
            self.get_sample_map_point_layers)
        project.layersAdded.disconnect(self.get_sample_map_point_layers)
        project.layersRemoved.disconnect(self.get_sample_map_point_layers)

        self.iface.mapCanvas().layersChanged.disconnect(self.get_sample_map_point_layers)

    def populate_sensors_reflectance(self):
        """Obtain a list of sensors from Bridge API definition.
        For reflectance TOC, so only Landsat-8 and Sentinel-8 should be included, otherwise all of the sensors
        """
        for sensor in [ALL_SENSORS] + SENSORS:
            sensor_name = sensor['name']
            if sensor_name == LANDSAT_8['name'] or \
                    sensor_name == LANDSAT_9['name'] or \
                    sensor_name == SENTINEL_2['name']:
                add_ordered_combo_item(
                    self.sensor_combo_box, sensor_name, sensor['key'])

    def clear_combo_box(self, combo_box):
        """Clears/removes all the entries in the provided combo_box
        :param combo_box: Combobox for which all the entries should be removed
        :type combo_box: QComboBox
        """
        cnt = combo_box.count()
        if cnt > 0:  # Skips if there are no items in the combobox
            while cnt >= 0:
                combo_box.removeItem(cnt)

                cnt = cnt - 1

    def product_type_change(self):
        map_product = self.map_product_combo_box.currentText()
        # If TOC reflectance has been chosen, only Sentinel-2 and Landsat-8
        # will be available as an option
        if map_product == REFLECTANCE['name']:
            self.clear_combo_box(self.sensor_combo_box)
            self.populate_sensors_reflectance()
        else:
            self.clear_combo_box(self.sensor_combo_box)
            self.populate_sensors()

        if map_product == SOIL['name'] or map_product == ELEVATION[
                'key'] or map_product == SAMPLE_MAP['name'] or map_product == SLOPE['key']:
            # Mask type not required for soil, elevation, and sample maps
            self.cb_mask.setEnabled(False)
        else:
            # Other maps types makes use of the mask type
            self.cb_mask.setEnabled(True)

        if map_product == SAMZ['name']:
            self.start_date_edit.setEnabled(False)
            self.end_date_edit.setEnabled(False)
        else:
            self.start_date_edit.setEnabled(True)
            self.end_date_edit.setEnabled(True)

        list_nitrogen_maps = [
            INSEASONFIELD_AVERAGE_NDVI['name'],
            INSEASONFIELD_AVERAGE_REVERSE_NDVI['name'],
            INSEASONFIELD_AVERAGE_LAI['name'],
            INSEASONFIELD_AVERAGE_REVERSE_LAI['name']
        ]

        if map_product in list_nitrogen_maps:
            # The n-planned parameter should be enabled for nitrogen map types
            self.n_planned_value_spinbox.setEnabled(True)
        else:
            # The n-planned parameter should be disabled for other map types
            self.n_planned_value_spinbox.setEnabled(False)

        # Sample map parameters
        if map_product == SAMPLE_MAP['name']:
            self.sensor_combo_box.setEnabled(False)
            self.start_date_edit.setEnabled(False)
            self.end_date_edit.setEnabled(False)

            # Enable point layer and field attribute
            self.cb_point_layer.setEnabled(True)
            self.cb_column_name.setEnabled(True)
        else:
            self.sensor_combo_box.setEnabled(True)
            self.start_date_edit.setEnabled(True)
            self.end_date_edit.setEnabled(True)

            # Disable point layer and field attribute
            self.cb_point_layer.setEnabled(False)
            self.cb_column_name.setEnabled(False)

    def point_layer_changed(self):
        """Calls this function when the selected point layer changed in the
        QCombobox. Repopulates the fields Combobox with the newly selected
        point layer.
        """
        self.cb_column_name.clear()

        project = QgsProject.instance()
        canvas_layers = self.iface.mapCanvas().layers()
        layers = list(project.mapLayers().values())

        current_layer = None
        current_index = self.cb_point_layer.currentIndex()
        cb_layer_id = self.cb_point_layer.itemData(current_index)
        for layer in layers:
            # Get the active layer
            if layer not in canvas_layers or not is_point_layer(layer):
                continue
            layer_id = layer.id()
            if layer_id == cb_layer_id:
                current_layer = layer
                break

        if current_layer:
            # Populate the selected layer fields in the QCombobox
            layer_fields = current_layer.fields().toList()
            for field in layer_fields:
                field_name = field.name()
                field_type = field.typeName()
                if field_name not in IGNORE_LAYER_FIELDS \
                        and field_type in ALLOWED_FIELD_TYPES:
                    self.cb_column_name.addItem(field_name)

    def update_button_states(self):
        """Update button states dynamically based on fetch_rx_group state."""
        if self.fetch_rx_group.isChecked():
            # Enable the Next button, disable Create Map button
            self.next_push_button.setEnabled(True)
            self.next_push_button.setVisible(True)
            self.set_next_button_text(1)  # Set "Next" as the button text
        else:
            # Revert to default behavior when fetch_rx_group is not checked
            self.next_push_button.setEnabled(
                self.current_stacked_widget_index < self.max_stacked_widget_index)
            self.set_next_button_text(self.current_stacked_widget_index)

    def setup_connectors(self):
        """Setup signal/slot mechanisms for dock elements."""
        # Button connector
        self.help_push_button.clicked.connect(self.show_help)
        self.back_push_button.clicked.connect(self.show_previous_page)
        self.next_push_button.clicked.connect(self.show_next_page)
        self.difference_map_push_button.clicked.connect(
            self.start_difference_map_creation)

        # Product type has changed
        self.map_product_combo_box.currentIndexChanged.connect(
            self.product_type_change)

        # Fetch RX Group state toggled
        self.fetch_rx_group.toggled.connect(self.update_button_states)

        # Stacked widget connector
        self.stacked_widget.currentChanged.connect(self.set_next_button_text)

        # List widget item connector
        self.coverage_result_list.itemSelectionChanged.connect(
            self.update_selection_data)

        # If the selected point layer for the Sample maps changes
        self.cb_point_layer.currentIndexChanged.connect(
            self.point_layer_changed)

    def unblock_signals(self):
        """Let the combos listen for event changes again."""
        self.geometry_combo_box.blockSignals(False)

    def block_signals(self):
        """Prevent the combos and dock listening for event changes."""
        self.disconnect_layer_listener()
        self.geometry_combo_box.blockSignals(True)

    def closeEvent(self, event):
        self.closingPlugin.emit()
        event.accept()
