# -*- coding: utf-8 -*-

"""
/***************************************************************************
 GeosquareGrid
                                 A QGIS plugin
 Geosquare Grid
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-04-17
        copyright            : (C) 2025 by PT Geo Innovasi Nussantara
        email                : admin@geosquare.ai
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'PT Geo Innovasi Nussantara'
__date__ = '2025-04-17'
__copyright__ = '(C) 2025 by PT Geo Innovasi Nussantara'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

grid_size = {
    '50 m': 50,
    '100 m': 100,
    '500 m': 500,
    '1 km': 1000,
    '5 km': 5000,
    '10 km': 10000,
}

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterField,
                       QgsRasterLayer,
                       QgsProcessingParameterNumber,
                       QgsProcessingUtils)
from .geosquare_grid import GeosquareGrid
from qgis.core import QgsField, QgsFields, QgsCoordinateReferenceSystem, QgsWkbTypes, QgsCoordinateTransform
from PyQt5.QtCore import QVariant
from qgis import processing
from qgis.core import QgsGeometry, QgsFeature, QgsVectorLayer
import os


class FromVectorAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    geosquare_grid = GeosquareGrid()
    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    FIELD = 'FIELD'
    GRIDSIZE = 'GRIDSIZE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        # We add the input vector features source. It can have any kind of
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.FIELD,
                self.tr('Field to use'),
                parentLayerParameterName=self.INPUT,
                allowMultiple=True,
            )
        )

        

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )

        # We add a grid size parameter
        # option select from 50 m, 100 m, 500 m, 1 km, 5 km, 10 km
        self.addParameter(
            QgsProcessingParameterEnum(
                self.GRIDSIZE,
                self.tr('Grid size'),
                options=list(grid_size.keys()),
                defaultValue='50 m',
                allowMultiple=False,
                optional=False
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        fields = QgsFields()
        fields.append(QgsField('gid', QVariant.String))
        
        # Create a CRS using EPSG:4326 (WGS84)
        crs = QgsCoordinateReferenceSystem('EPSG:4326')

        source = self.parameterAsSource(parameters, self.INPUT, context)
        selected_fields = self.parameterAsFields(parameters, self.FIELD, context)
        if selected_fields:
            for field in selected_fields:
                fields.append(source.fields().field(field))
        
        size = grid_size[list(grid_size.keys())[self.parameterAsEnum(parameters, self.GRIDSIZE, context)]]
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
            context, fields, QgsWkbTypes.Polygon, crs)
        
        # Check if the input layer is empty
        if source.featureCount() == 0:
            feedback.pushInfo(self.tr('Input layer is empty.'))
            return {self.OUTPUT: dest_id}
        
        # Check if the input layer has crs
        if source.sourceCrs() is None:
            feedback.pushInfo(self.tr('Input layer has no CRS.'))
            return {self.OUTPUT: dest_id}

        # convert to WGS84 if not already
        if source.sourceCrs() != crs:
            feedback.pushInfo(self.tr('Input layer is not in WGS84. Converting to WGS84.'))
            transform = QgsCoordinateTransform(source.sourceCrs(), crs, context.project())
            for feature in source.getFeatures():
                geom = feature.geometry().simplify(0.0004)
                geom.transform(transform)
                feature.setGeometry(geom)

        count_features = source.featureCount()
        total = 100 / count_features if count_features else 0
        current = 0
        for feature in source.getFeatures():
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # process each feature
            self.processFeature(
                feature,
                fields,
                size,
                sink,
                feedback
            )
            # Update the progress bar
            current += total
            feedback.setProgress(int(current))
        feedback.setProgress(100)
        feedback.pushInfo(self.tr('Processing completed.'))
        return {self.OUTPUT: dest_id}


    def processFeature(self, feature, fields, size, sink, feedback):
        """
        Process each feature and add it to the sink.
        """
        try:
            gid_features = self.geosquare_grid.polyfill(
                feature.geometry(),
                size,
                feedback=feedback,
                fullcover=False,
                as_feature=True,
            )
            # Add each generated feature to the sink
            for gid_feature in gid_features:
                # Get the GID value
                gid_value = gid_feature.attributes()[0]  # First attribute is gid
                
                # Create a new feature with the defined fields
                new_feature = QgsFeature(fields)
                
                # Set the GID field
                new_feature.setAttribute('gid', gid_value)
                
                # Copy selected field values from the original feature
                for field in fields:
                    field_name = field.name()
                    if field_name != 'gid' and feature.fields().indexFromName(field_name) >= 0:
                        new_feature.setAttribute(field_name, feature[field_name])
                
                # Set geometry
                new_feature.setGeometry(gid_feature.geometry())
                
                # Add to sink
                sink.addFeature(new_feature, QgsFeatureSink.FastInsert)
                sink.flushBuffer()
            
            
        except Exception as e:
            feedback.reportError(f"Error during processing: {str(e)}")


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Geosquare grid - from vector'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'vector'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def shortHelpString(self):
        return self.tr("""
This algorithm converts vector polygons to geosquare grid cells.

Input:
- A polygon vector layer
- Field(s) to copy from input polygons to output grid cells
- Grid size (from 50m to 10km)

Output:
- A grid layer where:
  - Each cell has a unique geosquare ID (gid)
  - Selected attribute values from the input polygon are copied to all grid cells that fall within it
  - Only cells that intersect with input polygons are created

This is useful for converting irregular polygons to a regular grid format while preserving attribute data.
        """)

    def createInstance(self):
        return FromVectorAlgorithm()
