# -*- coding: utf-8 -*-

"""
/***************************************************************************
 GeosquareGrid
                                 A QGIS plugin
 Geosquare Grid
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-04-17
        copyright            : (C) 2025 by PT Geo Innovasi Nussantara
        email                : admin@geosquare.ai
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'PT Geo Innovasi Nussantara'
__date__ = '2025-04-17'
__copyright__ = '(C) 2025 by PT Geo Innovasi Nussantara'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

grid_size = {
    '50 m': 50,
    '100 m': 100,
    '500 m': 500,
    '1 km': 1000,
    '5 km': 5000,
    '10 km': 10000,
}

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterRasterLayer,
                       QgsRasterLayer,
                       QgsProcessingParameterNumber,
                       QgsProcessingUtils)
from .geosquare_grid import GeosquareGrid
from qgis.core import QgsField, QgsFields, QgsCoordinateReferenceSystem, QgsWkbTypes, QgsCoordinateTransform
from PyQt5.QtCore import QVariant
from qgis import processing
from qgis.core import QgsGeometry, QgsFeature, QgsVectorLayer
import os


class FromRasterAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    geosquare_grid = GeosquareGrid()
    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    BOUNDARY = 'BOUNDARY'
    CALCULATETYPE = 'CALCULATETYPE'
    GRIDSIZE = 'GRIDSIZE'
    BAND = 'BAND'

    def prepareAlgorithm(self, parameters, context, feedback):
        """
        This method is called before the algorithm is executed.
        It allows you to dynamically adjust parameters based on user input.
        """
        # Get the selected raster layer
        raster_layer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        if raster_layer is not None:
            # Get the number of bands in the raster
            band_count = raster_layer.bandCount()
            if band_count > 0:
                # Update the BAND parameter to reflect the available bands
                self.parameterDefinition(self.BAND).setMetadata({
                    'widget_wrapper': {
                        'min': 1,
                        'max': band_count,
                        'step': 1
                    }
                })
                feedback.pushInfo(self.tr(f"Raster has {band_count} bands. Band selector updated."))
            else:
                feedback.reportError(self.tr("The selected raster has no bands."))
        else:
            feedback.reportError(self.tr("No raster layer selected."))
        return super().prepareAlgorithm(parameters, context, feedback)


    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                self.tr('Input raster layer'),
                [QgsProcessing.TypeRaster]
            )
        )

        # band selector from input raster layer
        self.addParameter(
            QgsProcessingParameterNumber(
                self.BAND,
                self.tr('Raster band'),
                QgsProcessingParameterNumber.Integer,
                defaultValue=1,
                minValue=1
            )
        )

        # We add the input vector features source. It can have any kind of
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.BOUNDARY,
                self.tr('Boundary layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )

        # We add a boolean parameter to determine if we want to only
        # include features that are inside the polygon
        self.addParameter(
            QgsProcessingParameterEnum(
                self.CALCULATETYPE,
                self.tr('Calculate type'),
                options=['Sum', 'Mean', 'Median', 'St Dev', 'Min', 'Max'],
                defaultValue='Median',
                allowMultiple=False,
                optional=False
            )
        )

        # We add a grid size parameter
        # option select from 50 m, 100 m, 500 m, 1 km, 5 km, 10 km
        self.addParameter(
            QgsProcessingParameterEnum(
                self.GRIDSIZE,
                self.tr('Grid size'),
                options=list(grid_size.keys()),
                defaultValue='50 m',
                allowMultiple=False,
                optional=False
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        fields = QgsFields()
        fields.append(QgsField('gid', QVariant.String))
        fields.append(QgsField('value', QVariant.Double))
        
        # Create a CRS using EPSG:4326 (WGS84)
        crs = QgsCoordinateReferenceSystem('EPSG:4326')
        band = self.parameterAsInt(parameters, self.BAND, context)
        feedback.pushInfo(self.tr(f'Using raster band: {band}'))
        
        source = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        boundary = self.parameterAsSource(parameters, self.BOUNDARY, context)
        calculatetype = self.parameterAsEnum(parameters, self.CALCULATETYPE, context)
        size = grid_size[list(grid_size.keys())[self.parameterAsEnum(parameters, self.GRIDSIZE, context)]]
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
            context, fields, QgsWkbTypes.Polygon, crs)
        
        # Check if the input layer has crs
        if source.crs() is None:
            feedback.pushInfo(self.tr('Input layer has no CRS.'))
            return {self.OUTPUT: dest_id}

        boundarygeometry = QgsGeometry.unaryUnion([feature.geometry() for feature in boundary.getFeatures()]).simplify(0.0004)

        # convert to WGS84 if not already
        if boundary.sourceCrs() != crs:
            feedback.pushInfo(self.tr('Input layer is not in WGS84. Converting to WGS84.'))
            transform = QgsCoordinateTransform(boundary.sourceCrs(), crs, context.project())
            boundarygeometry.transform(transform)

        #  convert raster source to WGS84 if not already
        if source.crs() != crs:
            feedback.pushInfo(self.tr('Input layer is not in WGS84. Converting to WGS84.'))
            reproject = processing.run(
                'gdal:warpreproject',
                {
                    'INPUT': source.source(),
                    'TARGET_CRS': crs,
                    'OUTPUT': 'TEMPORARY_OUTPUT'
                },
                context=context,
                feedback=feedback,
            )
            source = QgsRasterLayer(reproject['OUTPUT'], 'reprojected')


        try:
            parrentGID = self.geosquare_grid.polyfill(
                boundarygeometry,
                10000,
                feedback=feedback,
            )
            
            count_10km = len(parrentGID)
            total = 100 / count_10km if count_10km else 0
            current = 0

            # clip big raster to small raster by using the boundary from parentGID and save it to temporary file with filename gid10km
            for g10km in parrentGID:
                # Stop the algorithm if cancel button has been clicked
                if feedback.isCanceled():
                    break
                # process each part
                self.processPart(
                    boundarygeometry,
                    g10km,
                    band,
                    source,
                    calculatetype,
                    size,
                    context,
                    feedback,
                    sink
                )
                # Update the progress bar
                current += total
                sink.flushBuffer()
                feedback.setProgress(int(current))
                
            feedback.setProgress(100)
            feedback.pushInfo(self.tr('Processing completed.'))
            feedback.pushInfo(self.tr('Output layer created.'))
        except Exception as e:
            feedback.reportError(f"Error during processing: {str(e)}")
            
        return {self.OUTPUT: dest_id}
    
    def processPart(self, boundarygeometry, g10km, band, source, calculatetype, size, context, feedback, sink):
        geometry = self.geosquare_grid.gid_to_geometry(g10km)
        vl = None
        temp_output = os.path.join(QgsProcessingUtils.tempFolder(), f'{g10km}.tif')
        
        try:
            vl = QgsVectorLayer("Polygon?crs=EPSG:4326", f"temp_{g10km}", "memory")
            pr = vl.dataProvider()
            f = QgsFeature()
            f.setGeometry(geometry)
            pr.addFeature(f)
            vl.updateExtents()

            clipped_result = processing.run(
                'gdal:cliprasterbymasklayer',
                {
                    'INPUT': source.source(),
                    'MASK': vl,
                    'SOURCE_CRS': source.crs().authid(),
                    'TARGET_CRS': 'EPSG:4326',
                    'NODATA': None,
                    'ALPHA_BAND': False,
                    'CROP_TO_CUTLINE': True,
                    'KEEP_RESOLUTION': True,
                    'SET_RESOLUTION': False,
                    'OUTPUT': temp_output
                },
                context=context,
                is_child_algorithm=True
            )
            
            # process per part
            self.process_zonal_statistics(
                boundarygeometry,
                g10km,
                band, 
                clipped_result['OUTPUT'],
                calculatetype,
                size,
                context,
                feedback,
                sink
            )
        except Exception as e:
            feedback.reportError(f"Error processing part {g10km}: {str(e)}")
        finally:
            # Clean up resources
            if vl is not None and vl.isValid():
                del vl
            
            # Force garbage collection to release memory
            import gc
            gc.collect()
            

    def process_zonal_statistics(self, boundarygeometry, g10km, band, clipped_output, calculatetype, size, context, feedback, sink):
        vl = None
        output_layer = None
        
        try:
            child_grids = self.geosquare_grid.parrent_to_allchildren(
                    g10km,
                    size,
                    as_feature=True,
                )
            # filter out the child grids that are only intersecting with the boundary
            child_grids = [child for child in child_grids if child.geometry().intersects(boundarygeometry)]
            vl = QgsVectorLayer("Polygon?crs=EPSG:4326&field=gid:string(0,0)", f"temp_{g10km}_part", "memory")
            pr = vl.dataProvider()
            for child in child_grids:
                pr.addFeature(child)
            # Map calculation type to statistics number
            stat_value = calculatetype
            output = processing.run(
                "native:zonalstatisticsfb",
                {
                    'INPUT': vl,
                    'INPUT_RASTER': clipped_output,
                    'RASTER_BAND': band,
                    'COLUMN_PREFIX': '_',
                    'STATISTICS': [stat_value + 1],
                    'OUTPUT': 'TEMPORARY_OUTPUT'
                },
                context=context,
                is_child_algorithm=True
            )
            # Load the output using QgsProcessingUtils to handle temp files properly
            output_layer = QgsProcessingUtils.mapLayerFromString(output['OUTPUT'], context)
            for feature in output_layer.getFeatures():
                # Stop the algorithm if cancel button has been clicked
                if feedback.isCanceled():
                    break
                # Add the new feature to the sink
                sink.addFeature(feature, QgsFeatureSink.FastInsert)
        except Exception as e:
            feedback.reportError(f"Error in zonal statistics for {g10km}: {str(e)}")
        finally:
            # Clean up the temporary layers
            if vl is not None and vl.isValid():
                del vl
            if output_layer is not None:
                del output_layer
            
            # Force garbage collection to release memory
            import gc
            gc.collect()


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'geosquare grid - from raster'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'vector'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def shortHelpString(self):
        return self.tr("""This algorithm converts raster data into a geosquare vector grid.
        
The algorithm:
- Takes a raster input and converts it to vector geosquare grids
- Calculates statistics (Sum, Mean, Median, St Dev, Min, Max) for each grid cell
- Supports multiple grid sizes (50m, 100m, 500m, 1km, 5km, 10km)
- Allows selection of specific raster bands for analysis
- Uses a boundary layer to define the area of interest

The output is a vector layer where each grid cell contains the calculated statistic value from the underlying raster pixels.

This is useful for standardizing raster data at different resolutions or for comparative analysis across datasets.
        """)

    def createInstance(self):
        return FromRasterAlgorithm()
