from qgis.PyQt.QtCore import QSettings, QCoreApplication, QTranslator 
from qgis.PyQt.QtWidgets import QMessageBox, QAction, QPushButton, QInputDialog
from qgis.PyQt.QtGui import QColor, QIcon
from qgis.core       import Qgis, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject
from qgis.gui        import QgsMessageBar, QgsVertexMarker
from .geopunt4QgisAdresdialog import geopunt4QgisAdresDialog
from .geopunt4QgisPoidialog import geopunt4QgisPoidialog
from .geopunt4QgisSettingsdialog import geopunt4QgisSettingsDialog
from .geopunt4QgisBatchGeoCode import geopunt4QgisBatcGeoCodeDialog
# from .geopunt4QgisGipod import geopunt4QgisGipodDialog
from .geopunt4QgisElevation import mathplotlibWorks, geopunt4QgisElevationDialog
from .geopunt4QgisDataCatalog import geopunt4QgisDataCatalog
from .geopunt4QgisParcel import geopunt4QgisParcelDlg
from .geopunt import Adres
from .mapTools.reverseAdres import reverseAdresMapTool
from .tools.geometry import geometryHelper
from .tools.settings import settings
import os.path, webbrowser
from threading import Timer

class geopunt4Qgis(object):
    def __init__(self, iface):
        'initialize'
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        
        # initialize locale
        locale = QSettings().value("locale/userLocale", "nl")
        if not locale: locale == 'nl' 
        else: locale = locale[0:2]
        localePath = os.path.join(self.plugin_dir, 'i18n', 'geopunt4qgis_{}.qm'.format(locale))
        if os.path.exists(localePath):
            self.translator = QTranslator()
            self.translator.load(localePath)
            QCoreApplication.installTranslator(self.translator)

        # Create the dialogs (after translation) and keep reference
        self.adresdlg = geopunt4QgisAdresDialog(self.iface)
        self.batchgeoDlg = geopunt4QgisBatcGeoCodeDialog(self.iface) 
        self.poiDlg = geopunt4QgisPoidialog(self.iface)        
        # self.gipodDlg = geopunt4QgisGipodDialog(self.iface)
        self.settingsDlg = geopunt4QgisSettingsDialog()
        if mathplotlibWorks : self.elevationDlg = geopunt4QgisElevationDialog(self.iface)
        self.datacatalogusDlg = geopunt4QgisDataCatalog(self.iface)
        self.parcelDlg = geopunt4QgisParcelDlg(self.iface)
        
    def initGui(self):
        'intialize UI'
        #get settings
        self.s = QSettings()
        self.loadSettings()
        
        self.gh = geometryHelper(self.iface)
        self.gp = Adres()
        self.graphicsLayer = []

        # Create actions that will start plugin configuration
        self.adresAction = QAction(QIcon(":/svg/images/Adres.svg"),
            QCoreApplication.translate("geopunt4Qgis" , u"Zoek een adres"), self.iface.mainWindow())
        self.reverseAction = QAction(QIcon(":/svg/images/Place.svg"),
                QCoreApplication.translate("geopunt4Qgis", u"Prik een adres op de kaart"), 
                self.iface.mainWindow())
        self.batchAction = QAction(QIcon(":/svg/images/Bestand.svg"),
	        QCoreApplication.translate("geopunt4Qgis", u"CSV-adresbestanden geocoderen"),
	        self.iface.mainWindow())
        self.poiAction = QAction(QIcon(":/svg/images/POI.svg"),
                QCoreApplication.translate("geopunt4Qgis" , u"Zoek een interessante plaats"), 
	        self.iface.mainWindow())	
        # self.gipodAction = QAction(QIcon(":/svg/images/Gipod.svg"),
        #         QCoreApplication.translate("geopunt4Qgis" , u"Bevraag GIPOD"), self.iface.mainWindow())

        self.elevationAction =  QAction(QIcon(":/svg/images/Hoogte.svg"),
                QCoreApplication.translate("geopunt4Qgis" , u"Hoogteprofiel"), self.iface.mainWindow())
        self.datacatalogusAction =  QAction(QIcon(":/svg/images/Catalogus.svg"),
                QCoreApplication.translate("geopunt4Qgis" , u"Datavindplaats"), self.iface.mainWindow())
        self.parcelAction =  QAction(QIcon(":/svg/images/Perceel.svg"),
                QCoreApplication.translate("geopunt4Qgis" , u"Zoeken naar perceel"), self.iface.mainWindow())
        
        self.settingsAction = QAction(QIcon(":/svg/images/Settings.svg"),
                QCoreApplication.translate("geopunt4Qgis" , u"Instellingen"), self.iface.mainWindow())  
        self.aboutAction = QAction(QIcon(":/svg/images/Info.svg"),
                QCoreApplication.translate("geopunt4Qgis" , u"Over geopunt4Qgis"), self.iface.mainWindow())
 
        # connect the action to the run method
        self.adresAction.triggered.connect(self.runAdresDlg)
        self.reverseAction.triggered.connect(self.reverse)
        self.batchAction.triggered.connect(self.runBatch)
        self.poiAction.triggered.connect(self.runPoiDlg)
        # self.gipodAction.triggered.connect(self.runGipod)
        self.elevationAction.triggered.connect(self.runElevation)
        self.datacatalogusAction.triggered.connect(self.rundatacatalog)
        self.parcelAction.triggered.connect(self.runParcel)
        self.settingsAction.triggered.connect(self.runSettingsDlg)
        self.aboutAction.triggered.connect(lambda: webbrowser.open_new_tab("https://www.vlaanderen.be/geopunt/plug-ins/qgis-plug-in") )
        
        #Create toolbar
        self.toolbar = self.iface.addToolBar("Geopunt toolbar")
        self.toolbar.setObjectName("Geopunt toolbar")
        # Add to toolbar button
        self.toolbar.addAction(self.adresAction)
        self.toolbar.addAction(self.reverseAction)
        self.toolbar.addAction(self.batchAction)
        self.toolbar.addAction(self.poiAction)        
        # self.toolbar.addAction(self.gipodAction)
        self.toolbar.addAction(self.elevationAction)
        self.toolbar.addAction(self.parcelAction)
        self.toolbar.addAction(self.datacatalogusAction)
        
        # Add to Menu
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.adresAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.reverseAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.batchAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.poiAction)        
        # self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.gipodAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.elevationAction)
        self.iface.addPluginToWebMenu(u'&geopunt4Qgis' ,self.parcelAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.datacatalogusAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.settingsAction)
        self.iface.addPluginToWebMenu(u"&geopunt4Qgis", self.aboutAction)
        
    def unload(self):
        ' Remove the plugin menu items and icons'
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.adresAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.poiAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.reverseAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.batchAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.aboutAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.settingsAction)
        # self.iface.removePluginMenu(u"&geopunt4Qgis", self.gipodAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.elevationAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.datacatalogusAction)
        self.iface.removePluginMenu(u"&geopunt4Qgis", self.parcelAction)
        
        self.iface.removeToolBarIcon( self.adresAction)
        self.iface.removeToolBarIcon( self.poiAction)
        self.iface.removeToolBarIcon( self.reverseAction)
        self.iface.removeToolBarIcon( self.batchAction)
        self.iface.removeToolBarIcon( self.aboutAction)
        # self.iface.removeToolBarIcon( self.gipodAction)
        self.iface.removeToolBarIcon( self.elevationAction)
        self.iface.removeToolBarIcon( self.datacatalogusAction)
        self.iface.removeToolBarIcon( self.parcelAction)
        
        del self.toolbar 

    def loadSettings(self):
        self.saveToFile_reverse = int(self.s.value("geopunt4qgis/reverseSavetoFile", 0))
        layerName_reverse = self.s.value("geopunt4qgis/reverseLayerText", "")
        if layerName_reverse:
           self.layerName_reverse = layerName_reverse
        self.timeout =  int(  self.s.value("geopunt4qgis/timeout" ,15))
        self.proxy = settings().proxy
        self.startDir = self.s.value("geopunt4qgis/startDir", os.path.expanduser("~"))
        
    def runSettingsDlg(self):
        ' show the dialog'
        if self.settingsDlg.isVisible():
           self.settingsDlg.showNormal()
           self.settingsDlg.activateWindow()
           return
        
        self.settingsDlg.show()
        # Run the dialog event loop
        result = self.settingsDlg.exec_()
        if result:
            self.loadSettings()
            
    def runAdresDlg(self):
        ' show the dialog'
        if self.adresdlg.isVisible():
           self.adresdlg.showNormal()
           self.adresdlg.activateWindow()
           return
          
        self.adresdlg.loadSettings()
        self.adresdlg.show()
        # Run the dialog event loop
        self.adresdlg.exec_()
        
    def runPoiDlg(self):
        'show the dialog'
        if self.poiDlg.isVisible():
           self.poiDlg.showNormal()
           self.poiDlg.activateWindow()
           return 
         
        self.poiDlg.loadSettings()
        self.poiDlg.show()
        # Run the dialog event loop
        self.poiDlg.exec_()
  
    # def runGipod(self):
    #     'show the dialog'
    #     if self.gipodDlg.isVisible():
    #        self.gipodDlg.showNormal()
    #        self.gipodDlg.activateWindow()
    #        return 
        
    #     self.gipodDlg.loadSettings()
    #     self.gipodDlg.show()
    #     # Run the dialog event loop
    #     self.gipodDlg.exec_()
  
    def runBatch(self):
        'show the dialog'
        if self.batchgeoDlg.isVisible():
           self.batchgeoDlg.showNormal()
           self.batchgeoDlg.activateWindow()
           return 

        self.batchgeoDlg.loadSettings()
        self.batchgeoDlg.show()
        # Run the dialog event loop
        self.batchgeoDlg.exec_()

    def runElevation(self):
        if mathplotlibWorks == False: 
          QMessageBox.critical(None, "Error",
             QCoreApplication.translate("geopunt4Qgis" ,
            "Deze functie kan niet geladen worden door het ontbreken van of een fout in matplotlib") )
          return
        'show the dialog'
        if self.elevationDlg.isVisible():
           self.elevationDlg.showNormal()
           self.elevationDlg.activateWindow()
           return 
        
        self.elevationDlg.loadSettings()
        self.elevationDlg.show()
        # Run the dialog event loop
        self.elevationDlg.exec_()

    def rundatacatalog(self):
        'show the dialog'
        if self.datacatalogusDlg.isVisible():
           self.datacatalogusDlg.showNormal()
           self.datacatalogusDlg.activateWindow()
           return 

        self.datacatalogusDlg.show()
        # Run the dialog event loop
        self.datacatalogusDlg.exec_()

    def runParcel(self):
        'show the dialog'  
        if self.parcelDlg.isVisible():
           self.parcelDlg.showNormal()
           self.parcelDlg.activateWindow()
           return 
        self.parcelDlg.loadSettings()
        self.parcelDlg.show()
        # Run the dialog event loop
        self.parcelDlg.exec_()

    def reverse(self):
        widget = self.iface.messageBar().createMessage(
                QCoreApplication.translate("geopunt4Qgis" ,"Zoek een Adres: "), 
                QCoreApplication.translate("geopunt4Qgis" ,'Klik op de kaart om het adres op te vragen'))
                    
        helpBtn = QPushButton("Help", widget)
        helpBtn.clicked.connect(self.openReverseHelp)
        widget.layout().addWidget(helpBtn)
        self.iface.messageBar().clearWidgets()
        self.iface.messageBar().pushWidget(widget, level=Qgis.Info)

        reverseAdresTool = reverseAdresMapTool(self.iface, self._reverseAdresCallback) 
        self.iface.mapCanvas().setMapTool(reverseAdresTool)
        
    def _reverseAdresCallback(self, point):
        self._addMarker( point )
        lam72 = QgsCoordinateReferenceSystem("EPSG:31370")
        mapCrs = self.gh.getGetMapCrs(self.iface)
        xform = QgsCoordinateTransform(mapCrs, lam72, QgsProject.instance())
        lam72clickt = xform.transform(point)
        
        #to clear or not clear that is the question
        self.iface.messageBar().clearWidgets()
        
        #fetch Location from geopunt
        adres = self.gp.fetchLocation( str( lam72clickt.x() ) + "," + str( lam72clickt.y() ), 1)
        Timer( 3, self._clearGraphicLayer, ()).start()
    
        if len(adres) and type( adres ) is list:
            #only one result in list, was set in request
            FormattedAddress = adres[0]["FormattedAddress"]
      
            #add a button to the messageBar widget
            xlam72, ylam72 = adres[0]["Location"]["X_Lambert72"], adres[0]["Location"]["Y_Lambert72"]    
            
            diff = int(((xlam72 - lam72clickt.x())**2 +(ylam72 - lam72clickt.y())**2 )**(0.5))
            
            widget = self.iface.messageBar().createMessage(QCoreApplication.translate(
              "geopunt4Qgis", "Resultaat: "), "{0} (verschil: {1}m)".format(FormattedAddress, diff))
            
            xy = self.gh.prjPtToMapCrs([xlam72, ylam72], 31370)            
            self._addMarker( xy, QColor(0,255,200))
            
            button = QPushButton(widget)
            button.clicked.connect(lambda: self._addReverse(adres[0]))
            button.setText(QCoreApplication.translate("geopunt4Qgis" ,"Voeg toe"))
            
            widget.layout().addWidget(button)
            
            self.iface.messageBar().clearWidgets()
            self.iface.messageBar().pushWidget(widget, level=Qgis.Info)
    
        elif len(adres) == 0:
            self.iface.messageBar().pushMessage(QCoreApplication.translate("geopunt4Qgis","Waarschuwing"),
            QCoreApplication.translate("geopunt4Qgis", "Geen resultaten gevonden"), 
                    level=QgsMessageBar.INFO, duration=3)
      
        elif type( adres ) is str:
            self.iface.messageBar().pushMessage(QCoreApplication.translate("geopunt4Qgis", "Waarschuwing"),
                adres, level=QgsMessageBar.WARNING)
        else:
            self.iface.messageBar().pushMessage("Error", 
            QCoreApplication.translate("geopunt4Qgis","onbekende fout"), level=Qgis.Critical)
      
    def _addReverse(self, adres):
        formattedAddress, locationType = adres["FormattedAddress"] , adres["LocationType"]
        xlam72, ylam72 = adres["Location"]["X_Lambert72"] , adres["Location"]["Y_Lambert72"]
    
        if not hasattr(self, 'layerName_reverse'):
           layerName, accept = QInputDialog.getText(None,
             QCoreApplication.translate("geopunt4Qgis", 'Laag toevoegen'),
             QCoreApplication.translate("geopunt4Qgis", 'Geef een naam voor de laag op:'))
           if accept == False: return
           else:  self.layerName_reverse = layerName
           
        xy = self.gh.prjPtToMapCrs([xlam72, ylam72], 31370)
        self.gh.save_adres_point(xy, formattedAddress, locationType, layername=self.layerName_reverse, 
          startFolder=os.path.join( self.startDir, self.layerName_reverse), saveToFile=self.saveToFile_reverse ,
          sender=self.iface.mainWindow())
        self.iface.messageBar().popWidget()	
        self._clearGraphicLayer()
        
    def openReverseHelp(self):
        webbrowser.open_new_tab("https://www.vlaanderen.be/geopunt/plug-ins/qgis-plug-in/functionaliteiten-qgis-plug-in/prik-een-adres-op-kaart-in-qgis")
                
    def _addMarker(self, pnt, clr=QColor(255,255,0)):
        m = QgsVertexMarker(self.iface.mapCanvas())
        m.setCenter( pnt )
        m.setColor(QColor(0,0,0))
        m.setFillColor(clr)
        m.setIconSize(12)
        m.setIconType(QgsVertexMarker.ICON_BOX) 
        m.setPenWidth(1)
        self.graphicsLayer.append(m)
        return m
      
    def _clearGraphicLayer(self):
      for graphic in  self.graphicsLayer: 
        self.iface.mapCanvas().scene().removeItem(graphic)
      self.graphicsLayer = []
