
from qgis.PyQt.QtCore import Qt, QVariant
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import QWidget, QVBoxLayout, QMessageBox
import random

from .core import create_topographic_profile, create_geological_profile, get_figure_size, save_geological_profile
from .gui import GeoProfileGui, ClassificationColorsWidget, ProfileWidget



class GeoProfileDialog(QWidget):

    def __init__(self, iface, tr, parent=None):
        super().__init__(parent)
        # Settings
        self.iface = iface
        self.tr = tr
        self.setWindowTitle('GeoProfile')
        self.setWindowFlags(Qt.Window)
        layout = QVBoxLayout()
        self.ui = GeoProfileGui(tr=tr, parent=self)
        layout.addWidget(self.ui)
        self.setLayout(layout)

        # Variables
        self.lithologyColors = None

        # loads
        self.loadLithologyColors()

        #Signals
        self.ui.topographyButton.clicked.connect(self.showTopographicProfile)
        self.ui.lithologyLayerChanged.connect(self.loadLithologyColors)
        self.ui.lithologyFieldChanged.connect(self.loadLithologyColors)
        self.ui.lithologyColorsButton.clicked.connect(self.showLithologyColors)
        self.ui.profileButton.clicked.connect(self.showGeologicalProfile)
        self.ui.saveButton.clicked.connect(self.saveGeologicalProfile)

    def getRequiredLayer(self, layer, name):

        if not layer:
            raise ValueError(self.tr('Please, select a layer that contains the') + f' {name}.')

        return layer

    def getLineFeature(self, layer, field_name, value):

        layer = self.getRequiredLayer(layer=layer, name=self.tr('section line'))

        if field_name == 'Feature ID':
            feature = layer.getFeature(int(value))
        else:
            if value == 'NULL':
                expr = f'"{field_name}" IS NULL'
            else:
                field = layer.fields()[field_name]
                if field.type() in (QVariant.String, QVariant.Date, QVariant.DateTime):
                    expr = f'"{field_name}" = \'{value}\''
                else:
                    expr = f'"{field_name}" = {value}'

            feature = next(layer.getFeatures(expr), None)

        if not feature:
            raise ValueError(self.tr('Feature not found.'))

        return feature

    def validateNumericField(self, layer, field_name, label):

        field = layer.fields().field(field_name)
        if field.type() not in (QVariant.Int, QVariant.Double):
            raise ValueError(f'{label}: ' + self.tr('field must be numeric (int or double).'))

    def createTopographicProfile(self):

        dem = self.getRequiredLayer(layer=self.ui.demLayer(), name='DEM')
        line_feature = self.getLineFeature(layer=self.ui.sectionLayer(),
                                           field_name=self.ui.sectionField(),
                                           value=self.ui.sectionFeature())
        invert_line = self.ui.invertSection()

        return create_topographic_profile(line=line_feature, dem=dem, invert_line=invert_line)

    def showTopographicProfile(self):

        try:
            topographic_profile = self.createTopographicProfile()

        except Exception as e:
            QMessageBox.critical(self, 'Error', str(e))
            return

        profile_depth = self.ui.profileDepth()

        window = ProfileWidget(tr=self.tr, parent=self)
        window.setWindowTitle(self.tr('Topographic Profile'))
        window.setTopographicProfile(distances=topographic_profile.distances,
                                     elevations=topographic_profile.elevations,
                                     depth=profile_depth)
        window.show()

    def loadLithologyColors(self):

        lithology_layer = self.ui.lithologyLayer()

        if not lithology_layer:
            self.lithologyColors = None
            return

        lithology_name_field = self.ui.lithologyField()

        names = set()
        for feature in lithology_layer.getFeatures():
            names.add(feature.attribute(lithology_name_field))

        lithology_colors = {}
        for name in names:
            r = random.randint(0, 255)
            g = random.randint(0, 255)
            b = random.randint(0, 255)
            lithology_colors[name] = QColor(r, g , b)

        self.lithologyColors = lithology_colors

    def showLithologyColors(self):

        lithology_layer = self.ui.lithologyLayer()

        if not lithology_layer:
            return

        lithology_colors_widget = ClassificationColorsWidget(tr=self.tr, parent=self)
        lithology_colors_widget.setTitle(self.tr('Lithology Colors'))

        for name, color in self.lithologyColors.items():
            lithology_colors_widget.insertItem(item=name, color=color)

        result = lithology_colors_widget.exec_()

        if not result:
            return

        self.lithologyColors = lithology_colors_widget.getColors()

    def createGeologicalProfile(self):

        dem = self.getRequiredLayer(layer=self.ui.demLayer(), name='DEM')
        line_feature = self.getLineFeature(layer=self.ui.sectionLayer(),
                                           field_name=self.ui.sectionField(),
                                           value=self.ui.sectionFeature())
        invert_line = self.ui.invertSection()

        # Lithology
        lithology_layer = self.getRequiredLayer(layer=self.ui.lithologyLayer(), name=self.tr('lithology'))
        lithology_field = self.ui.lithologyField()

        # Structural
        structural_layer = self.ui.structuralLayer()
        structural_field = self.ui.structuralField()
        structural_color = self.ui.structuralColor()

        # Bedding
        bedding_layer = self.ui.beddingLayer()
        bedding_azimuth = self.ui.beddingAzimuthField()
        bedding_dip = self.ui.beddingDipField()
        bedding_buffer = self.ui.beddingBuffer()

        if bedding_layer:
            self.validateNumericField(layer=bedding_layer, field_name=bedding_azimuth, label=self.tr('azimuth'))
            self.validateNumericField(layer=bedding_layer, field_name=bedding_dip, label=self.tr('dip'))

        return create_geological_profile(line=line_feature,
                                         dem=dem,
                                         invert_line=invert_line,
                                         lithology=lithology_layer,
                                         lithology_names=lithology_field,
                                         lithology_colors=self.lithologyColors,
                                         structural=structural_layer,
                                         structural_names=structural_field,
                                         structural_color=structural_color,
                                         bedding=bedding_layer,
                                         bedding_azimuth=bedding_azimuth,
                                         bedding_dip=bedding_dip,
                                         bedding_buffer=bedding_buffer)

    def showGeologicalProfile(self):

        try:
            geological_profile = self.createGeologicalProfile()

        except Exception as e:
            QMessageBox.critical(self, 'Error', str(e))
            return

        profile_depth = self.ui.profileDepth()
        bedding_length = self.ui.beddingLength()

        window = ProfileWidget(tr=self.tr, parent=self)
        window.setWindowTitle(self.tr('Geological Profile'))
        window.setGeologicalProfile(lithology=geological_profile.lithologies,
                                    depth=profile_depth,
                                    x_limits=geological_profile.getXLimits(),
                                    y_limits=geological_profile.getElevationLimits(),
                                    structures=geological_profile.structures,
                                    bedding=geological_profile.bedding,
                                    bedding_length=bedding_length)
        window.show()

    def saveGeologicalProfile(self):

        try:
            geological_profile = self.createGeologicalProfile()

        except Exception as e:
            QMessageBox.critical(self, 'Error', str(e))
            return

        profile_depth = int(self.ui.profileDepth())
        if not profile_depth:
            profile_depth = 0

        bedding_length = float(self.ui.beddingLength())

        path = self.ui.outputPath()
        if not path:
            QMessageBox.critical(self, 'Error', self.tr('Please, select a output path.'))
            return

        title = self.ui.sectionTitle()
        if not title:
            title = self.tr('Geological Profile')

        start_label = self.ui.sectionStartLabel()
        end_label = self.ui.sectionEndLabel()

        size = get_figure_size(size=self.ui.figureSize(), orientation=self.ui.figureOrientation())

        try:
            save_geological_profile(path=path,
                                    lithology=geological_profile.lithologies,
                                    depth=profile_depth,
                                    x_limits=geological_profile.getXLimits(),
                                    y_limits=geological_profile.getElevationLimits(),
                                    structures=geological_profile.structures,
                                    bedding=geological_profile.bedding,
                                    bedding_length=bedding_length,
                                    title=title,
                                    y_label=self.tr('Elevation'),
                                    start_label=start_label,
                                    end_label=end_label,
                                    size=size)

            QMessageBox.information(self, self.tr('Information'), self.tr('the profile was saved correctly.'))

        except Exception as e:
            QMessageBox.critical(self, 'Error', str(e))
            return

