import json

from qgis.core import (
    Qgis,
    QgsAbstractProfileGenerator,
    QgsAbstractProfileSource,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsCsException,
    QgsFeedback,
    QgsGeometry,
    QgsGeometryUtils,
    QgsNetworkAccessManager,
    QgsPoint,
    QgsProfileGenerationContext,
    QgsProfileRequest,
)
from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtNetwork import QNetworkRequest

from geopf_altimetrie.core.profile_results import GeoPFAltimetrieProfileResults
from geopf_altimetrie.core.profile_url import profile_url
from geopf_altimetrie.toolbelt import PlgLogger, PlgOptionsManager
from geopf_altimetrie.toolbelt.utils import url_with_param


class GeoPFAltimetrieProfileGenerator(QgsAbstractProfileGenerator):
    X = "lon"
    Y = "lat"
    Z = "z"

    def __init__(self, request: QgsProfileRequest):
        QgsAbstractProfileGenerator.__init__(self)
        self.__request = request
        self.__profile_curve = request.profileCurve().clone() if request.profileCurve() else None
        self.__transformed_curve = None
        self.__transformation = QgsCoordinateTransform(
            request.crs(),
            QgsCoordinateReferenceSystem("EPSG:4326"),
            request.transformContext(),
        )
        self.__results: GeoPFAltimetrieProfileResults
        self.__feedback = QgsFeedback()
        self.log = PlgLogger().log
        self.plg_settings = PlgOptionsManager()

    def sourceId(self):
        return "geoplateforme-altimetrie-profile"

    def __get_profile_from_rest_api(self, resource_id, profile_mode):
        result = {}
        geojson = self.__transformed_curve.asJson(6)
        sampling = int(min(self.__profile_curve.length() * 2, 5000))
        base_url, base_params = profile_url(geojson, resource_id, profile_mode, sampling)
        url = url_with_param(base_url, base_params)

        network_access_manager = QgsNetworkAccessManager.instance()

        req = QNetworkRequest(QUrl(url))
        reply = network_access_manager.blockingGet(req, feedback=self.__feedback)

        if reply.error():
            result = {"error": reply.errorString()}
        else:
            content = reply.content()
            try:
                result = json.loads(str(content, "utf-8"))["elevations"]
            except json.decoder.JSONDecodeError as e:
                self.log(
                    message=f"Unable to parse results from Profile service. Details: {e.msg}",
                    log_level=2,
                    push=True,
                )

        return result

    def __parse_response_point(self, point):
        return point[self.X], point[self.Y], point[self.Z]

    def feedback(self):
        return self.__feedback

    def generateProfile(self, context: QgsProfileGenerationContext):
        if self.__profile_curve is None:
            return False

        settings = self.plg_settings.get_plg_settings()
        if not settings.display:
            return False

        self.__transformed_curve = self.__profile_curve.clone()
        try:
            self.__transformed_curve.transform(self.__transformation)
        except QgsCsException:
            self.log(
                message="Error transforming profile line to EPSG:4326.",
                log_level=2,
                push=True,
            )
            return False

        self.__results = GeoPFAltimetrieProfileResults()
        self.__results.copyPropertiesFromGenerator(self)

        result = self.__get_profile_from_rest_api(settings.resource_id, settings.profile_mode)

        if "error" in result:
            self.log(message=result["error"], log_level=2, push=True)
            return False

        cartesian_d = 0
        for point in result:
            if self.__feedback.isCanceled():
                return False

            # Note: d is ellipsoidal from the API
            x, y, z = self.__parse_response_point(point)
            z = max(z, -2)  # - 99999 if no data
            point_z = QgsPoint(x, y, z)
            point_z.transform(self.__transformation, Qgis.TransformDirection.Reverse)

            if self.__results.raw_points:
                # QGIS elevation profile won't calculate distances
                # using 3d, so let's stick to 2d to avoid getting
                # displaced markers or lines in the profile canvas
                cartesian_d += QgsGeometryUtils.distance2D(point_z, self.__results.raw_points[-1])

            self.__results.raw_points.append(point_z)
            self.__results.cartesian_distance_to_height[cartesian_d] = z
            self.__results.cartesian_cross_section_geometries.append(
                QgsGeometry(QgsPoint(cartesian_d, z))
            )

            if z < self.__results.min_z:
                self.__results.min_z = z

            if z > self.__results.max_z:
                self.__results.max_z = z

            self.__results.geometries.append(QgsGeometry(point_z))

        return not self.__feedback.isCanceled()

    def takeResults(self):
        return self.__results


class GeoPFAltimetrieProfileSource(QgsAbstractProfileSource):
    def __init__(self):
        QgsAbstractProfileSource.__init__(self)

    def createProfileGenerator(self, request):
        return GeoPFAltimetrieProfileGenerator(request)
