# -*- coding: utf-8 -*-

"""
/***************************************************************************
 GeoMoveTools
                                 A QGIS plugin
 This plugin installs GeoMove Tools
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-10-02
        copyright            : (C) 2018 by Cartolab
        email                : luipir@gmail.com
        email                : davidfernandezarango@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Cartolab'
__date__ = '2018-10-02'
__copyright__ = '(C) 2018 by Cartolab'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import glob
import shutil
from qgis.core import (
    Qgis,
    QgsProcessingProvider,
    QgsApplication,
    QgsMessageLog,
    QgsProcessingModelAlgorithm
)
from qgis.utils import (
    iface,
    unloadPlugin
)
from PyQt5.QtGui import (
    QIcon
)
from PyQt5.QtWidgets import (
    QMessageBox
)
from processing.modeler.ModelerUtils import ModelerUtils
from processing.core.ProcessingConfig import (
    ProcessingConfig,
    Setting
)
from processing.tools.system import isWindows

# list here all algorithms to import
# no dynamic load is done to avoid malicious import
# one more advantage is that any alg syntax error is discovered as soon as
# it is imported
from .algorithms.crosswalks_from_pointcloud import CrosswalkFromPointclouds
from .algorithms.reclassify_as_sidewalk import ReclassifyAsSidewalk


class GeoMoveToolsProvider(QgsProcessingProvider):

    def __init__(self):
        QgsProcessingProvider.__init__(self)

        self.modelsPath = os.path.join(os.path.dirname(__file__), 'processing_models')
        self.algorithmsPath = os.path.join(os.path.dirname(__file__), 'algorithms')
        self.pipelinesPath = os.path.join(os.path.dirname(__file__), 'pdal_pipelines')
        self.messageTag = type(self).__name__ # e.g. string GeoMoveToolsProvider

        self.alglist = [
            CrosswalkFromPointclouds(), 
            ReclassifyAsSidewalk()
        ]

    def load(self):
        # check dependencies in lazy way because need that all
        # plugin dependency have to be loaded before to check
        iface.initializationCompleted.connect(self.checkDependecies)

        ProcessingConfig.settingIcons[self.name()] = self.icon()
        ProcessingConfig.addSetting(Setting(self.name(), 'ACTIVATE_GeoMoveTOOLS',
                                            self.tr('Activate'), True))
        ProcessingConfig.readSettings()
        self.refreshAlgorithms()

        if QgsApplication.processingRegistry().providerById('model'):
            self.loadModels()
        else:
            # lazy load of models waiting QGIS initialization. This would avoid
            # to load modelas when model provider is still not available in processing
            iface.initializationCompleted.connect(self.loadModels)

        return True

    def checkDependecies(self):
        '''Verfy if all dependencies are availabel. This is the list
        of verification:
        1) existance of PDALtools:pdalpipelineexecutor
        '''
        try:
            iface.initializationCompleted.disconnect(self.checkDependecies)
        except:
            pass

        # 1) check if processing pdaltools:pdalpipelineexecutor is available
        registry = QgsApplication.instance().processingRegistry()
        if not registry.algorithmById('PDALtools:pdalpipelineexecutor'):
            message = self.tr('GeoMove algorithms depends on PDALtools processing provider. Install it using plugin manager')
            if iface.mainWindow().isVisible():
                QMessageBox.information(
                    iface.mainWindow(),
                    self.tr('Dependency error'),
                    message
                )
            else:
                QgsMessageLog.logMessage(message, self.messageTag, Qgis.Critical)
            
            unloadPlugin(self.id())

    def unload(self):
        # remove all models
        modelsFiles = glob.glob(os.path.join(self.modelsPath, '*.model3'))
        for modelFileName in modelsFiles:
            installedModel = os.path.join(ModelerUtils.modelsFolders()[0], os.path.basename(modelFileName))
            try:
                if os.path.exists(installedModel):
                    os.remove(installedModel)
            except Exception as ex:
                QgsMessageLog.logMessage(self.tr('Failed to uninstall model: {} - {}'.format(installedModel, str(ex))), self.messageTag, Qgis.Warning)
                continue
        QgsApplication.processingRegistry().providerById('model').refreshAlgorithms()

        try:
            ProcessingConfig.removeSetting('ACTIVATE_GeoMoveTOOLS')
        except Exception as ex:
            pass

    def loadAlgorithms(self):
        """
        Loads all algorithms belonging to this provider.
        """
        for alg in self.alglist:
            self.addAlgorithm( alg )

    def loadModels(self):
        '''Register models present in models folder of the plugin.'''
        try:
            iface.initializationCompleted.disconnect(self.loadModels)
        except:
            pass

        modelsFiles = glob.glob(os.path.join(self.modelsPath, '*.model3'))

        for modelFileName in modelsFiles:
            alg = QgsProcessingModelAlgorithm()
            if not alg.fromFile(modelFileName):
                QgsMessageLog.logMessage(self.tr('Not well formed model: {}'.format(modelFileName)), self.messageTag, Qgis.Warning)
                continue

            destFilename = os.path.join(ModelerUtils.modelsFolders()[0], os.path.basename(modelFileName))
            try:
                if os.path.exists(destFilename):
                    os.remove(destFilename)

                if isWindows():
                    shutil.copyfile(modelFileName, destFilename)
                else:
                    os.symlink(modelFileName, destFilename)
            except Exception as ex:
                QgsMessageLog.logMessage(self.tr('Failed to install model: {} - {}'.format(modelFileName, str(ex))), self.messageTag, Qgis.Warning)
                continue

        QgsApplication.processingRegistry().providerById('model').refreshAlgorithms()

    def id(self):
        """
        Returns the unique provider id, used for identifying the provider. This
        string should be a unique, short, character only string, eg "qgis" or
        "gdal". This string should not be localised.
        """
        return 'geomovetools'

    def name(self):
        """
        Returns the provider name, which is used to describe the provider
        within the GUI.

        This string should be short (e.g. "Lastools") and localised.
        """
        return self.tr('GeoMoveTools')

    def longName(self):
        """
        Returns the a longer version of the provider name, which can include
        extra details such as version numbers. E.g. "Lastools LIDAR tools
        (version 2.2.1)". This string should be localised. The default
        implementation returns the same string as name().
        """
        return self.name()

    def icon(self):
        iconPath = os.path.join(os.path.dirname(__file__), 'geomovelogo_simple.png')
        return QIcon(iconPath)

    def svgIconPath(self):
        iconPath = os.path.join(os.path.dirname(__file__), 'geomovelogo.svg')
        return iconPath

    def defaultRasterFileExtension(self):
        return 'tif'

    def defaultVectorFileExtension(self):
        return 'las'

    def supportedOutputVectorLayerExtensions(self):
        formats = super().supportedOutputVectorLayerExtensions()
        formats.append(self.defaultVectorFileExtension())
        return formats

    def tr(self, string, context=''):
        if context == '':
            context = 'GeoMoveToolsAlgorithmProvider'
        return QgsApplication.translate(context, string)

