# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeoEASINDockWidget
                                 A QGIS plugin
 Data downloader from EASIN Geospatial Web Service
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2021-12-26
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Patricio Soriano. Geoinnova
        email                : patricio.soriano@geoinnova.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import json
import os
import re
from datetime import datetime
from functools import partial
from urllib import request

from PyQt5.QtCore import Qt
from PyQt5.QtGui import QColor, QFont
from PyQt5.QtWidgets import QTreeWidgetItem, QMessageBox
from qgis.PyQt import QtWidgets, uic
from qgis.PyQt.QtCore import pyqtSignal
from qgis.core import Qgis, QgsMessageLog, QgsVectorLayer, QgsGeometry, QgsFeature, QgsProject

from ..tools.tools import replaceSpaces

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'species_search_dockwidget_base.ui'))


def search_for(d, lst):
    if lst:
        return [i for i in lst if all(i[target_key] == target_value for target_key, target_value in d.items())]


class GeoEASINDockWidget(QtWidgets.QDockWidget, FORM_CLASS):
    closingPlugin = pyqtSignal()

    def __init__(self, iface, parent=None):
        """Constructor."""
        super(GeoEASINDockWidget, self).__init__(parent)
        self.setupUi(self)

        self.data = None
        self.data_filter = None
        self.dict_to_search_for = {}
        self.total = 0
        self.sub_total = 0
        self.url = None

        self.iface = iface

        #  search DOCK WIDGETS

        # Search Button
        self.btnSearch.setEnabled(False)
        self.btnSearch.clicked.connect(self.fetch_specie)

        # Clean Button
        self.btnCleanResults.clicked.connect(self.clean_results)

        # Search text
        self.lineSpecieText.textChanged.connect(self.enable_button)

        # TreeData
        self.treeWidgetData.setColumnCount(2)
        self.treeWidgetData.setHeaderItem(QTreeWidgetItem(["Specie", ""]))
        self.treeWidgetData.setColumnWidth(0, 300)
        self.treeWidgetData.itemDoubleClicked.connect(partial(self.add_grid_layer))
        self.treeWidgetData.sortByColumn(0, Qt.AscendingOrder)

        # Tab
        self.tabWidget.setCurrentIndex(0)

        # CheckBox filters
        self.cb_IsEUConcern.setCheckState(1)
        self.cb_IsDeleted.setCheckState(1)
        self.cb_IsParasite.setCheckState(1)

        self.cb_IsEUConcern.setEnabled(False)
        self.cb_IsDeleted.setEnabled(False)
        self.cb_IsParasite.setEnabled(False)

        self.cb_IsEUConcern.stateChanged.connect(lambda: self.apply_filters(self.cb_IsEUConcern))
        self.cb_IsDeleted.stateChanged.connect(lambda: self.apply_filters(self.cb_IsDeleted))
        self.cb_IsParasite.stateChanged.connect(lambda: self.apply_filters(self.cb_IsParasite))

        # Radiobutton filters
        self.rb_Impact_All.setChecked(True)
        self.rb_Impact_High.setChecked(False)
        self.rb_Impact_Low.setChecked(False)

        self.rb_Impact_All.setEnabled(False)
        self.rb_Impact_High.setEnabled(False)  # Hi
        self.rb_Impact_Low.setEnabled(False)  # Lo

        self.rb_Impact_All.toggled.connect(self.onClickedImpact)
        self.rb_Impact_High.toggled.connect(self.onClickedImpact)
        self.rb_Impact_Low.toggled.connect(self.onClickedImpact)

        self.rb_Status_All.setChecked(True)
        self.rb_Status_Alien.setChecked(False)  # A
        self.rb_Status_Cryptogenic.setChecked(False)  # C
        self.rb_Status_Unkhow.setChecked(False)  # N
        self.rb_Status_Questionable.setChecked(False)  # Q

        self.rb_Status_All.setEnabled(False)
        self.rb_Status_Alien.setEnabled(False)  # A
        self.rb_Status_Cryptogenic.setEnabled(False)  # C
        self.rb_Status_Unkhow.setEnabled(False)  # N
        self.rb_Status_Questionable.setEnabled(False)  # Q

        self.rb_Status_All.toggled.connect(self.onClickedStatus)
        self.rb_Status_Alien.toggled.connect(self.onClickedStatus)
        self.rb_Status_Cryptogenic.toggled.connect(self.onClickedStatus)
        self.rb_Status_Unkhow.toggled.connect(self.onClickedStatus)
        self.rb_Status_Questionable.toggled.connect(self.onClickedStatus)

    def closeEvent(self, event):
        self.closingPlugin.emit()
        event.accept()

    def enable_button(self):
        text_length = len(self.lineSpecieText.text())
        if text_length > 3:
            self.btnSearch.setEnabled(True)
        else:
            self.btnSearch.setEnabled(False)

    def enable_filter(self, active: True):
        """
        Manage the checkboxes and the radio buttons state
        @param active:
        @return:
        """
        self.cb_IsEUConcern.setEnabled(active)
        self.cb_IsDeleted.setEnabled(active)
        self.cb_IsParasite.setEnabled(active)
        self.rb_Impact_All.setEnabled(active)
        self.rb_Impact_High.setEnabled(active)
        self.rb_Impact_Low.setEnabled(active)
        self.rb_Status_All.setEnabled(active)
        self.rb_Status_Alien.setEnabled(active)  # A
        self.rb_Status_Cryptogenic.setEnabled(active)  # C
        self.rb_Status_Unkhow.setEnabled(active)  # N
        self.rb_Status_Questionable.setEnabled(active)  # N

    def clean_results(self):
        """
        Clean the research results, reset the the default widgets values and initialize the main variables
        @return:
        """

        self.data = None
        self.data_filter = None
        self.total = 0
        self.sub_total = 0

        # self.lineSpecieText.clear()
        self.treeWidgetData.clear()

        self.requestInfo.setText(f'Info')

        self.enable_filter(False)

        self.cb_IsEUConcern.setCheckState(1)
        self.cb_IsDeleted.setCheckState(1)
        self.cb_IsParasite.setCheckState(1)

        self.rb_Impact_All.setChecked(True)
        self.rb_Impact_High.setChecked(False)
        self.rb_Impact_Low.setChecked(False)

        self.rb_Status_All.setChecked(True)
        self.rb_Status_Alien.setChecked(False)  # A
        self.rb_Status_Cryptogenic.setChecked(False)  # C
        self.rb_Status_Unkhow.setChecked(False)  # N
        self.rb_Status_Questionable.setChecked(False)  # Q

    def fetch_specie(self):
        """

        @return:
        """
        self.clean_results()

        term = self.lineSpecieText.text()
        term2 = re.sub('\\s+', ' ', term)
        term3 = replaceSpaces(term2)

        self.logText.appendPlainText(datetime.now().strftime("%Y/%m/%d, %H:%M:%S"))

        try:
            self.url = f'https://easin.jrc.ec.europa.eu/api/cat/term/{term3}'
            req = request.Request(self.url)

            with request.urlopen(req) as f:
                data_json = json.loads(f.read().decode('utf-8'))
                self.data = data_json['results']
                self.searchAPI(self.data, True)

        except Exception as error:
            print(f'Error: {error}')
            self.requestInfo.setText("Connection error")
            self.logText.appendPlainText("Connection error")
            QgsMessageLog.logMessage(
                f'Error: {error}', level=Qgis.Critical)

    def searchAPI(self, data, add_log=False):
        """

        @param data:
        @param add_log:
        @return:
        """
        msg = 'There are no species corresponding to your search criteria'

        # self.sub_total = len(data) if data and data != msg else 0

        if data == msg:
            self.requestInfo.setText(f'{msg}')
            self.logText.appendPlainText(f'- Term of search: {self.lineSpecieText.text()} ')
            self.logText.appendPlainText(f'- API URL: {self.url} ')
            self.logText.appendPlainText(f'{msg}')
            return None

        else:
            if add_log:
                self.total = len(data) if data and data != msg else 0
                self.logText.appendPlainText(f'- Term of search: {self.lineSpecieText.text()} ')
                self.logText.appendPlainText(f'- API URL: {self.url} ')
                self.logText.appendPlainText(f'- Total results: {self.total}')
                self.requestInfo.setText(f'Total results: {self.total}')
            else:
                self.requestInfo.setText(f'Filter results: {self.sub_total} of {self.total} ')

        try:
            self.enable_filter(True)
            for dataLevel0 in data:
                speciesName = dataLevel0['SpeciesName']
                speciesCatalogueId = dataLevel0['SpeciesCatalogueId']
                item_level0 = QTreeWidgetItem(self.treeWidgetData,
                                              [speciesName, 'Add grid to map: ' + speciesCatalogueId, ''])
                font = QFont()
                font.setBold(True)
                item_level0.setFont(0, font)
                item_level0.setForeground(1, QColor("blue"))
                item_level0.setToolTip(1, 'Double click to add')

                print(item_level0)
                self.treeWidgetData.addTopLevelItem(item_level0)

                for dataLevel1 in dataLevel0.items():
                    key_level_1 = dataLevel1[0]
                    value_level_1 = dataLevel1[1]
                    if type(value_level_1) is list:
                        item_level_1 = QTreeWidgetItem(item_level0, [key_level_1])
                        self.treeWidgetData.addTopLevelItem(item_level_1)
                        for dataLevel2 in value_level_1:
                            for datalevel3 in dataLevel2.items():
                                key_level_3 = datalevel3[0]
                                value_level_3 = str(datalevel3[1])
                                item_level2 = QTreeWidgetItem(item_level_1, [key_level_3, value_level_3])
                                self.treeWidgetData.addTopLevelItem(item_level2)

                    elif type(value_level_1) is dict:
                        item_level_1 = QTreeWidgetItem(item_level0, [key_level_1])
                        self.treeWidgetData.addTopLevelItem(item_level_1)
                        for datalevel3 in value_level_1.items():
                            key_level_3 = datalevel3[0]
                            value_level_3 = str(datalevel3[1])
                            item_level2 = QTreeWidgetItem(item_level_1, [key_level_3, value_level_3])
                            self.treeWidgetData.addTopLevelItem(item_level2)
                    else:
                        item_level_1 = QTreeWidgetItem(item_level0, [key_level_1, str(value_level_1)])
                        self.treeWidgetData.addTopLevelItem(item_level_1)
        except:
            self.requestInfo.setText(f'Info')
            # print('except')

    def add_grid_layer(self, treeitem, item):
        """

        @param treeitem:
        @param item:
        @return:
        """

        get_selected = self.treeWidgetData.selectedItems()

        if get_selected:
            base_node = get_selected[0]
            getChildNode = base_node.text(item)
            if "Add grid to map" in getChildNode:
                name_layer = base_node.text(0)
                idgrid = base_node.text(1).split(":")[1].strip()
                advice_message = 'Depending on the number of grids to download, the process may take some time. \n\n' \
                                 'Do you wish to continue? '
                reply = QMessageBox.question(self.iface.mainWindow(), 'Add grid to map', advice_message,
                                             QMessageBox.Yes,
                                             QMessageBox.Cancel)
                if reply == QMessageBox.Yes:
                    self.create_layer(idgrid, name_layer)
                else:
                    print('Cancel')

    def apply_filters(self, control_name):
        """

        @param control_name:
        @return:
        """

        self.treeWidgetData.clear()

        if control_name.text() == 'Is EU concern':
            data_filter = []

            if control_name.checkState() == 2:
                item = {"IsEuConcern": True, }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)

            elif control_name.checkState() == 0:
                item = {"IsEuConcern": False, }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)
            else:
                del self.dict_to_search_for["IsEuConcern"]
                data_filter = search_for(self.dict_to_search_for, self.data)

            self.treeWidgetData.clear()
            self.sub_total = len(data_filter) if data_filter else 0
            self.searchAPI(data_filter)

        if control_name.text() == 'Is Deleted':
            data_filter = []
            if control_name.checkState() == 2:
                item = {"IsDeleted": True, }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)
            elif control_name.checkState() == 0:
                item = {"IsDeleted": False, }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)
            else:
                del self.dict_to_search_for["IsDeleted"]
                data_filter = search_for(self.dict_to_search_for, self.data)

            self.treeWidgetData.clear()
            self.sub_total = len(data_filter) if data_filter else 0
            self.searchAPI(data_filter)

        if control_name.text() == 'Is Parasite':
            data_filter = []
            if control_name.checkState() == 2:
                item = {"IsParasite": True, }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)
            elif control_name.checkState() == 0:
                item = {"IsParasite": False, }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)
            else:
                del self.dict_to_search_for["IsParasite"]
                data_filter = search_for(self.dict_to_search_for, self.data)

            self.sub_total = len(data_filter) if data_filter else 0
            self.searchAPI(data_filter)

    def onClickedImpact(self):
        """

        @return:
        """

        radio_btn = self.sender()
        data_filter = []

        if radio_btn.isChecked():
            if radio_btn.text() == 'High':
                item = {"ImpactId": 'Hi', }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)


            elif radio_btn.text() == 'Low/Unknown':
                item = {"ImpactId": 'Lo', }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)

            else:
                del self.dict_to_search_for["ImpactId"]
                data_filter = search_for(self.dict_to_search_for, self.data)

        self.treeWidgetData.clear()
        self.sub_total = len(data_filter) if data_filter else 0
        self.searchAPI(data_filter)

    def onClickedStatus(self):
        """

        @return:
        """

        radio_btn = self.sender()
        data_filter = []

        if radio_btn.isChecked():
            if radio_btn.text() == 'Alien':
                item = {"StatusId": "A", }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)

            elif radio_btn.text() == 'Cryptogenic':
                item = {"StatusId": "C", }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)

            elif radio_btn.text() == 'Questionable':
                item = {"StatusId": "Q", }
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)

            elif radio_btn.text() == 'Unkhow':
                item = {"StatusId": "N"}
                self.dict_to_search_for.update(item)
                data_filter = search_for(self.dict_to_search_for, self.data)

            else:
                del self.dict_to_search_for["StatusId"]
                data_filter = search_for(self.dict_to_search_for, self.data)

        self.treeWidgetData.clear()
        # self.total = len(data) if data and data != msg else 0
        self.sub_total = len(data_filter) if data_filter else 0
        self.searchAPI(data_filter)

    def create_layer(self, species_catalogue_id, species_name=''):
        """

        @param species_catalogue_id:
        @param species_name:
        @return:
        """
        speciesid = species_catalogue_id
        speciesname = species_name
        skip = 0
        len_results = 1

        layer_name = f"{speciesid}_{speciesname}"

        def fetch_data(speciesid, skip):

            url_grid = f'https://easin.jrc.ec.europa.eu/api/geo/speciesid/' \
                       f'{speciesid}/layertype/grid/take/50/skip/{skip}'
            req = request.Request(url_grid)

            self.requestInfo.setText(f'...requesting grid info')

            with request.urlopen(req) as f:
                return json.loads(f.read().decode(
                    'utf-8'))

        temp = QgsVectorLayer(
            "Polygon?crs=epsg:3035"
            "&field=LayerRecordId:string&index=yes"
            "&field=SpeciesId:string"
            "&field=SpeciesName:string"
            "&field=YearMin:int"
            "&field=YearMax:int"
            "&field=Reference:string(400)"
            "&field=Native:boolean"
            "&field=DataPartner:string",
            layer_name, "memory")

        def addGrid(temp, apidata):
            temp.startEditing()

            for feature in apidata:
                geom = QgsGeometry.fromWkt(feature['Wkt'])
                feat = QgsFeature()
                feat.setGeometry(geom)

                year_min = feature['YearMin'].replace("    ", "0")
                year_max = feature['YearMax'].replace("    ", "0")

                feat.setAttributes([
                    feature['LayerRecordId'],
                    feature['SpeciesId'],
                    feature['SpeciesName'],
                    int(year_min),
                    int(year_max),
                    feature['Reference'],
                    feature['Native'],
                    feature['DataPartner'],
                ])
                temp.dataProvider().addFeatures([feat])

            temp.commitChanges()

        while len_results > 0:
            new_data = fetch_data(speciesid, skip)
            skip += 50
            results = new_data['results']
            len_results = len(results)

            addGrid(temp, results)

        if temp.featureCount() > 0:
            # temp.renderer().symbol().setColor(QColor("red"))
            # temp.triggerRepaint()
            QgsProject.instance().addMapLayer(temp)

            self.requestInfo.setText(f'{layer_name} layer added.')
            self.iface.messageBar().pushMessage("INFO", f'{layer_name} layer added.', Qgis.Info, 5)

            self.logText.appendPlainText('')
            self.logText.appendPlainText(datetime.now().strftime("%Y/%m/%d, %H:%M:%S"))
            self.logText.appendPlainText(f'- Grid layer added: {layer_name}')
            self.logText.appendPlainText(
                f'- API URL: https://easin.jrc.ec.europa.eu/api/geo/speciesid/{speciesid}/layertype/grid/take/50/skip/0')
            self.logText.appendPlainText(f'- Total grids: {temp.featureCount()}')

            advice_message = 'Do you want to zoom in the new layer?'
            reply = QMessageBox.question(self.iface.mainWindow(), 'Add grid to map', advice_message,
                                         QMessageBox.Yes,
                                         QMessageBox.Cancel)
            if reply == QMessageBox.Yes:
                self.iface.setActiveLayer(temp)
                self.iface.zoomToActiveLayer()
            else:
                print('Cancel')

        else:
            print('No results')
