# -*- coding: utf-8 -*-
"""
/***************************************************************************
GeodiversityCalculator v1.00
                                 A QGIS plugin
 This plugin calculates geodiversity over a custom polygon. These are the thematic subindices: geology, soil, relief, hydrography (line and polygon features), mineralogy and palaeontology.
 Mostly QGIS proccessing tools and Saga tools are implemented in the code.
 Important remarks:
    1. You can only use SHP and GPKG files.
    2. The files have to be in the same CRS, otherwise the plugin will run into an error.
    3. You can omit thematic indices - this way you can calculate each subindex or a combination of subindices separately.
    4. The methodology is based on: https://www.researchgate.net/publication/239942809_Geodiversity_Assessment_of_Parana_State_Brazil_An_Innovative_Approach
    5. The methodology was refined for QGIS by: https://www.researchgate.net/publication/354059504_Refinement_Proposals_for_Geodiversity_Assessment_-_A_Case_Study_in_the_Bakony-Balaton_UNESCO_Global_Geopark_Hungary
 In case of any questions, do not hesitate to contact me!
 
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-11-05
        git sha              : $Format:%H$
        copyright            : (C) 2021-22 by Marton Pal
        email                : pal.marton@inf.elte.hu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QApplication, QWidget, QProgressBar, QPushButton, QVBoxLayout, QDialogButtonBox
from qgis.analysis import QgsNativeAlgorithms
from qgis.core import *
from osgeo import *
from PyQt5.QtCore import *
import processing
import sys
import math
QgsApplication.processingRegistry().addProvider(QgsNativeAlgorithms())

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .geodiversity_calculator_dialog import GeodiversityCalculatorDialog
import os.path


class GeodiversityCalculator:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GeodiversityCalculator_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Geodiversity Calculator')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GeodiversityPlugin', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = QIcon(os.path.join(os.path.dirname(__file__), "geodiversity_calculator.png"))
        self.add_action(
            icon_path,
            text=self.tr(u'Geodiversity Calculator'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Geodiversity Calculator'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def empty(self):
        self.dlg.lineEdit_16.clear(),
        self.dlg.lineEdit.clear(),
        self.dlg.lineEdit_3.clear(),
        self.dlg.lineEdit_4.clear(),
        self.dlg.lineEdit_5.clear(),
        self.dlg.lineEdit_6.clear(),
        self.dlg.lineEdit_7.clear(),
        self.dlg.lineEdit_8.clear(),
        self.dlg.lineEdit_9.clear(),
        self.dlg.lineEdit_17.clear(),
        self.dlg.lineEdit_12.clear(),
        self.dlg.lineEdit_10.clear(),
        self.dlg.lineEdit_13.clear(),
        self.dlg.lineEdit_11.clear(),
        self.dlg.lineEdit_14.clear()

    def select_input_file(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input boundary polygon file: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit.setText(filename)

    """def select_output_file(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output file: ", "", '*.gpkg')
        self.dlg.lineEdit_2.setText(filename)"""
        
    def select_input_geology_vector(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input geological polygons: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_5.setText(filename)
        
    """def select_output_geology_raster(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output geological raster: ", "", '*.tif')
        self.dlg.lineEdit_6.setText(filename)"""
        
    def select_input_pedology_vector(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input soil type polygons: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_7.setText(filename)
        
    """def select_output_pedology_raster(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output raster soil map: ", "", '*.tif')
        self.dlg.lineEdit_8.setText(filename)"""
        
    def select_input_geomorphon_raster(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input DEM: ", "", '*.tif')
        self.dlg.lineEdit_9.setText(filename)
    
    """def select_output_geomorphon_raster(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output geomorphon map: ", "", '*.tif')
        self.dlg.lineEdit_10.setText(filename)"""
    
    """def select_output_cutdem_raster(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output cut DEM: ", "", '*.tif')
        self.dlg.lineEdit_15.setText(filename)"""
        
    def select_working_folder(self):
        folder = QFileDialog.getExistingDirectory(
            self.dlg, caption="Select working folder: ", directory=os.getcwd())
        self.dlg.lineEdit_16.setText(folder)
        
    """def select_input_watercourses_vector(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input watercourse line features: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_11.setText(filename)"""
        
    def select_input_lakes_vector(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input lakes/seas polygon features: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_17.setText(filename)        
        
    def select_input_mineralogy_vector(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input mineralogy point features: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_12.setText(filename)       
        
    def select_input_palaeontology_vector(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input palaeontology point features: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_13.setText(filename)
    
    def select_output_geodiv_grid(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output geodiversity grid: ", "", '*.gpkg; *.shp')
        self.dlg.lineEdit_14.setText(filename)
                
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = GeodiversityCalculatorDialog()
            self.dlg.pushButton.clicked.connect(self.select_input_file)
            self.dlg.pushButton_3.clicked.connect(self.select_input_geology_vector)
            self.dlg.pushButton_5.clicked.connect(self.select_input_pedology_vector)
            self.dlg.pushButton_7.clicked.connect(self.select_input_geomorphon_raster)
            self.dlg.pushButton_14.clicked.connect(self.select_working_folder)
            self.dlg.pushButton_10.clicked.connect(self.select_input_mineralogy_vector)
            self.dlg.pushButton_11.clicked.connect(self.select_input_palaeontology_vector)
            self.dlg.pushButton_13.clicked.connect(self.select_input_lakes_vector)
            self.dlg.button_box.button(QDialogButtonBox.Reset).clicked.connect(self.empty)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec()
        

        # See if OK was pressed
        if result:
            
            #Push message in QGIS when
            self.iface.messageBar().clearWidgets()
            progressMessageBar = self.iface.messageBar()
            progressbar = QProgressBar()
            progressMessageBar.pushWidget(progressbar)
            # Processing feedback
            
            working_dir = self.dlg.lineEdit_16.text()
            #################
            #GridMaker data:#
            #################
            path0 = self.dlg.lineEdit.text()
            name0 = "boundary"
            grid0name = self.dlg.lineEdit_14.text()         
            grid0 = working_dir + "/"+self.dlg.lineEdit_14.text()+".gpkg"
            hatar0 = QgsVectorLayer(path0, name0, "ogr")
            QgsProject.instance().addMapLayer(hatar0)
            crs0 = hatar0.crs().authid()
            
            extent0 = hatar0.extent()
            xmin0 = extent0.xMinimum()
            xmax0 = extent0.xMaximum()
            ymin0 = extent0.yMinimum()
            ymax0 = extent0.yMaximum()
            hspacing0 = self.dlg.lineEdit_3.text()
            vspacing0 = self.dlg.lineEdit_4.text()
            
            processing.run("native:creategrid", {
                'TYPE': 2,
                'EXTENT': "%f,%f,%f,%f" % (xmin0, xmax0, ymin0, ymax0),
                'HSPACING': hspacing0,
                'VSPACING': vspacing0,
                'HOVERLAY': 0,
                'VOVERLAY': 0,
                'CRS': crs0,
                'OUTPUT': grid0})
            addGrid0 = QgsVectorLayer(grid0, grid0name, "ogr")

            ########################################################
            #Setting r_values and rasterizing vector thematic data:#
            ########################################################
            #GEOLOGY#
            #########
            if self.dlg.lineEdit_5.text():
                self.dlg.lineEdit_6.setEnabled(True)
                path11 = self.dlg.lineEdit_5.text()
                name11 = "geology"
                layer11 = QgsVectorLayer(path11, name11, "ogr")
                geol_index = self.dlg.lineEdit_6.text()
                QgsProject.instance().addMapLayer(layer11)
                # adding the new index field
                newField11 = QgsField('r_value', QVariant.Int)
                layer11.dataProvider().addAttributes([newField11])
                layer11.updateFields()
                idx11 = layer11.fields().indexOf('r_value')
                # filling up the new field
                unique11 = []
                with edit(layer11):
                    for feature in layer11.getFeatures():
                        if feature[geol_index] not in unique11:
                            unique11.append(feature[geol_index])
                        new_value = unique11.index(feature[geol_index])+1 #+1 in order to avoid 0 raster value
                        feature.setAttribute(feature.fieldNameIndex('r_value'), new_value)
                        layer11.updateFeature(feature)
                # rasterize
                raster11 = working_dir + "/geology_raster.tif"
                crs11 = layer11.crs().authid()
                extent11 = layer11.extent()
                xmin11 = extent11.xMinimum()
                xmax11 = extent11.xMaximum()
                ymin11 = extent11.yMinimum()
                ymax11 = extent11.yMaximum()
                processing.run("gdal:rasterize", {
                    'INPUT':layer11,
                    'FIELD':'r_value',
                    'BURN':0,
                    'USE_Z':False,
                    'UNITS':1,
                    'WIDTH':5,
                    'HEIGHT':5,
                    'EXTENT': "%f,%f,%f,%f" % (xmin11, xmax11, ymin11, ymax11),
                    'NODATA':0,
                    'OPTIONS':'',
                    'DATA_TYPE':5,
                    'INIT':None,
                    'INVERT':False,
                    'EXTRA':'',
                    'OUTPUT':raster11})
                rastername11 = "geology_raster"
                addRaster11 = QgsRasterLayer(raster11, rastername11)
                QgsProject.instance().addMapLayer(addRaster11)
            else:
                pass

            ########################################################
            #Setting r_values and rasterizing vector thematic data:#
            ########################################################
            #PEDOLOGY#
            ##########
            if self.dlg.lineEdit_7.text():
                path12 = self.dlg.lineEdit_7.text()
                name12 = "pedology"
                layer12 = QgsVectorLayer(path12, name12, "ogr")
                pedo_index = self.dlg.lineEdit_8.text()
                QgsProject.instance().addMapLayer(layer12)
                # adding the new index field
                newField12 = QgsField('r_value', QVariant.Int)
                layer12.dataProvider().addAttributes([newField12])
                layer12.updateFields()
                idx12 = layer12.fields().indexOf('r_value')
                # filling up the new field
                unique12 = []
                with edit(layer12):
                    for feature in layer12.getFeatures():
                        if feature[pedo_index] not in unique12:
                            unique12.append(feature[pedo_index])
                        new_value = unique12.index(feature[pedo_index])+1 #+1 in order to avoid 0 raster value
                        feature.setAttribute(feature.fieldNameIndex('r_value'), new_value)
                        layer12.updateFeature(feature)
                # rasterize
                raster12 = working_dir + "/pedology_raster.tif"  
                crs12 = layer12.crs().authid()
                extent12 = layer12.extent()
                xmin12 = extent12.xMinimum()
                xmax12 = extent12.xMaximum()
                ymin12 = extent12.yMinimum()
                ymax12 = extent12.yMaximum()
                processing.run("gdal:rasterize", {
                    'INPUT':layer12,
                    'FIELD':'r_value',
                    'BURN':0,
                    'USE_Z':False,
                    'UNITS':1,
                    'WIDTH':5,
                    'HEIGHT':5,
                    'EXTENT': "%f,%f,%f,%f" % (xmin12, xmax12, ymin12, ymax12),
                    'NODATA':0,
                    'OPTIONS':'',
                    'DATA_TYPE':5,
                    'INIT':None,
                    'INVERT':False,
                    'EXTRA':'',
                    'OUTPUT':raster12})
                rastername12 = "pedology_raster"
                addRaster12 = QgsRasterLayer(raster12, rastername12)
                QgsProject.instance().addMapLayer(addRaster12)
            else:
                pass
            
            ###########################################
            #Cutting DEM and producing geomorphon map:#
            ###########################################
            #GEOMORPHOLOGY#
            ###############
            if self.dlg.lineEdit_9.text():
                #DEM cut input
                full_dem2 = self.dlg.lineEdit_9.text()
                name2 = "full_dem"
                mask2 = self.dlg.lineEdit.text()
                cut_dem2 = working_dir + "/cut_dem.tif"
                dem2 = QgsVectorLayer(full_dem2, name2, "ogr")
                crs2 = dem2.crs().authid()
                #DEM cut
                processing.run("gdal:cliprasterbymasklayer",
                    {'INPUT':full_dem2,
                    'MASK':mask2,
                    'SOURCE_CRS':crs2,
                    'TARGET_CRS':crs2,
                    'NODATA':None,
                    'ALPHA_BAND':False,
                    'CROP_TO_CUTLINE':True,
                    'KEEP_RESOLUTION':False,
                    'SET_RESOLUTION':False,
                    'X_RESOLUTION':None,
                    'Y_RESOLUTION':None,
                    'MULTITHREADING':False,
                    'OPTIONS':'',
                    'DATA_TYPE':0,
                    'EXTRA':'',
                    'OUTPUT':cut_dem2})
                #geomorphon output
                geom2 = working_dir + "/geomorphon.tif"
                geomname2 = "geomorphon_raster"            
                #geomorphon
                processing.run("grass7:r.geomorphon",
                    {'elevation':cut_dem2,
                    'search':3,
                    'skip':0,
                    'flat':3,
                    'dist':0,
                    'forms':geom2,
                    '-m':False,
                    '-e':False,
                    'GRASS_REGION_PARAMETER':None,
                    'GRASS_REGION_CELLSIZE_PARAMETER':0,
                    'GRASS_RASTER_FORMAT_OPT':'',
                    'GRASS_RASTER_FORMAT_META':''})
                addRaster2 = QgsRasterLayer(geom2, geomname2)
                QgsProject.instance().addMapLayer(addRaster2)
            else:
                pass
            
            ###################
            #Raster statistics#
            ###################
            #GEOLOGY#
            #########
            if self.dlg.lineEdit_5.text():
                name31 = "geology_grid"
                input_raster31 = raster11
                output_grid31 = working_dir + "/geology_grid.gpkg"
                processing.run("qgis:zonalstatisticsfb",
                    {'INPUT':grid0,
                    'INPUT_RASTER':input_raster31,
                    'RASTER_BAND':1,
                    'COLUMN_PREFIX':'_geol_',
                    'STATISTICS':[1,2,3,4,5,6,7,8,9,10],
                    'OUTPUT':output_grid31})
                output31 = QgsVectorLayer(output_grid31, name31, "ogr")
                output31.dataProvider().deleteAttributes([6,7,8,9,10,11,12,13,14])
                output31.updateFields()
                QgsProject.instance().addMapLayer(output31)
            else:
                pass
            
            ###################
            #Raster statistics#
            ###################
            #PEDOLOGY#
            ##########
            if self.dlg.lineEdit_7.text():
                name32 = "pedology_grid"
                input_raster32 = raster12
                output_grid32 = working_dir + "/" + "pedology_grid.gpkg"
                processing.run("qgis:zonalstatisticsfb",
                    {'INPUT':grid0,
                    'INPUT_RASTER':input_raster32,
                    'RASTER_BAND':1,
                    'COLUMN_PREFIX':'_pedo_',
                    'STATISTICS':[1,2,3,4,5,6,7,8,9,10],
                    'OUTPUT':output_grid32})
                output32 = QgsVectorLayer(output_grid32, name32, "ogr")
                output32.dataProvider().deleteAttributes([6,7,8,9,10,11,12,13,14])
                output32.updateFields()
                QgsProject.instance().addMapLayer(output32)
            else:
                pass
            
            ###################
            #Raster statistics#
            ###################
            #GEOMORPHON#
            ############
            if self.dlg.lineEdit_9.text():
                name33 = "geomorphon_grid"
                input_raster33 = geom2
                output_grid33 = working_dir + "/" + "geomorphon_grid.gpkg"
                processing.run("qgis:zonalstatisticsfb",
                    {'INPUT':grid0,
                    'INPUT_RASTER':input_raster33,
                    'RASTER_BAND':1,
                    'COLUMN_PREFIX':'_geom_',
                    'STATISTICS':[1,2,3,4,5,6,7,8,9,10],
                    'OUTPUT':output_grid33})
                output33 = QgsVectorLayer(output_grid33, name33, "ogr")
                output33.dataProvider().deleteAttributes([6,7,8,9,10,11,12,13,14])
                output33.updateFields()
                QgsProject.instance().addMapLayer(output33)
            else:
                pass

            #######################################################
            #Processing watercourses and evaluate Strahler's order#
            #######################################################            
            if self.dlg.lineEdit_9.text():
                #fill DEM
                filled_output5 = working_dir + "/filled_dem.sdat"
                processing.run("saga:fillsinkswangliu",
                    {'ELEV':full_dem2,
                    'FILLED':filled_output5,
                    'FDIR':'TEMPORARY_OUTPUT',
                    'WSHED':'TEMPORARY_OUTPUT',
                    'MINSLOPE':0.1})
                #Strahler order
                strahler_output5 = working_dir + "/strahler.sdat"
                processing.run("saga:strahlerorder",
                    {'DEM':filled_output5,
                    'STRAHLER':strahler_output5})
                #Zonal stats - Strahler
                name5 = "strahler_grid"
                input_raster5 = strahler_output5
                output_grid5 = working_dir + "/strahler_grid.gpkg"
                processing.run("qgis:zonalstatisticsfb",
                    {'INPUT':grid0,
                    'INPUT_RASTER':input_raster5,
                    'RASTER_BAND':1,
                    'COLUMN_PREFIX':'_stra_',
                    'STATISTICS':[6],
                    'OUTPUT':output_grid5})
                output5 = QgsVectorLayer(output_grid5, name5, "ogr")
                output5.updateFields()
                QgsProject.instance().addMapLayer(output5)
                #Strahler index divide by 2
                with edit(output5):
                    for feature in output5.getFeatures():
                        new_value = math.ceil(feature['_stra_max']/2)
                        feature.setAttribute(feature.fieldNameIndex('_stra_max'), new_value)
                        output5.updateFeature(feature)
            else:
                pass
            
            ##################################################
            #Adding values to grid cells with unique MINERALS#
            ##################################################
            if self.dlg.lineEdit_12.text():
                path61 = self.dlg.lineEdit_12.text()
                name61 = "mineral_occurences"
                layer61 = QgsVectorLayer(path61, name61, "ogr")
                mineral_index = self.dlg.lineEdit_10.text()
                QgsProject.instance().addMapLayer(layer61)
                # adding the new index field
                newField61 = QgsField('r_value', QVariant.Int)
                layer61.dataProvider().addAttributes([newField61])
                layer61.updateFields()
                idx61 = layer61.fields().indexOf('r_value')
                # filling up the new field
                unique61 = []
                with edit(layer61):
                    for feature in layer61.getFeatures():
                        if feature[mineral_index] not in unique61:
                            unique61.append(feature[mineral_index])
                        new_value = unique61.index(feature[mineral_index])+1 #+1 in order to avoid 0 value
                        feature.setAttribute(feature.fieldNameIndex('r_value'), new_value)
                        layer61.updateFeature(feature)
                # count unique mineral features in each grid cell
                mineral_grid = working_dir + '/mineral_grid.gpkg'
                processing.run("native:countpointsinpolygon",
                    {'POLYGONS':grid0,
                    'POINTS':layer61,
                    'WEIGHT':'',
                    'CLASSFIELD':'r_value',
                    'FIELD':'_mineral_idx',
                    'OUTPUT':mineral_grid})
                name_mg = 'mineral_grid'
                mineralgrid = QgsVectorLayer(mineral_grid, name_mg, "ogr")
                QgsProject.instance().addMapLayer(mineralgrid)
            else:
                pass
                    
            #################################################
            #Adding values to grid cells with unique FOSSILS#
            #################################################          
            if self.dlg.lineEdit_13.text():
                path62 = self.dlg.lineEdit_13.text()
                name62 = "fossil_occurences"
                layer62 = QgsVectorLayer(path62, name62, "ogr")
                fossil_index = self.dlg.lineEdit_11.text()
                QgsProject.instance().addMapLayer(layer62)
                # adding the new index field
                newField62 = QgsField('r_value', QVariant.Int)
                layer62.dataProvider().addAttributes([newField62])
                layer62.updateFields()
                idx62 = layer62.fields().indexOf('r_value')
                # filling up the new field
                unique62 = []
                with edit(layer62):
                    for feature in layer62.getFeatures():
                        if feature[fossil_index] not in unique62:
                            unique62.append(feature[fossil_index])
                        new_value = unique62.index(feature[fossil_index])+1 #+1 in order to avoid 0 value
                        feature.setAttribute(feature.fieldNameIndex('r_value'), new_value)
                        layer62.updateFeature(feature)
                # count unique fossil features in each grid cell
                fossil_grid = working_dir + '/fossil_grid.gpkg'
                processing.run("native:countpointsinpolygon",
                    {'POLYGONS':grid0,
                    'POINTS':layer62,
                    'WEIGHT':'',
                    'CLASSFIELD':'r_value',
                    'FIELD':'_fossil_idx',
                    'OUTPUT':fossil_grid})
                name_fg = 'fossil_grid'
                fossilgrid = QgsVectorLayer(fossil_grid, name_fg, "ogr")
                QgsProject.instance().addMapLayer(fossilgrid)
            else:
                pass

            ################################################
            #Joining thematic fields to the base grid layer#
            ################################################

            if self.dlg.lineEdit_5.text():
                geol = output31
            else:  
                pass
            if self.dlg.lineEdit_7.text():
                pedo = output32
            else:
                pass
            if self.dlg.lineEdit_9.text():
                geom = output33
                stra = output5
            else:
                pass            
            if self.dlg.lineEdit_12.text():
                mine = mineralgrid
            else:
                pass
            if self.dlg.lineEdit_13.text():
                foss = fossilgrid
            else:
                pass
            grid = addGrid0

            if self.dlg.lineEdit_5.text():
                field_0 = 'id'
                field_1 = 'id'
                joinObject1 = QgsVectorLayerJoinInfo()
                joinObject1.setJoinFieldName(field_0)
                joinObject1.setTargetFieldName(field_1)
                joinObject1.setJoinLayerId(geol.id())
                joinObject1.setUsingMemoryCache(True)
                joinObject1.setJoinLayer(geol)
                joinObject1.setPrefix('J')
                joinObject1.setJoinFieldNamesSubset(['_geol_variety'])
                grid.addJoin(joinObject1)
            else:   
                pass

            if self.dlg.lineEdit_7.text():
                field_0 = 'id'
                field_1 = 'id'
                joinObject2 = QgsVectorLayerJoinInfo()
                joinObject2.setJoinFieldName(field_0)
                joinObject2.setTargetFieldName(field_1)
                joinObject2.setJoinLayerId(pedo.id())
                joinObject2.setUsingMemoryCache(True)
                joinObject2.setJoinLayer(pedo)
                joinObject2.setPrefix('J')
                joinObject2.setJoinFieldNamesSubset(['_pedo_variety'])
                grid.addJoin(joinObject2)
            else:
                pass

            if self.dlg.lineEdit_9.text():
                field_0 = 'id'
                field_1 = 'id'
                joinObject3 = QgsVectorLayerJoinInfo()
                joinObject3.setJoinFieldName(field_0)
                joinObject3.setTargetFieldName(field_1)
                joinObject3.setJoinLayerId(geom.id())
                joinObject3.setUsingMemoryCache(True)
                joinObject3.setJoinLayer(geom)
                joinObject3.setPrefix('J')
                joinObject3.setJoinFieldNamesSubset(['_geom_variety'])
                grid.addJoin(joinObject3)
                joinObject4 = QgsVectorLayerJoinInfo()
                joinObject4.setJoinFieldName(field_0)
                joinObject4.setTargetFieldName(field_1)
                joinObject4.setJoinLayerId(stra.id())
                joinObject4.setUsingMemoryCache(True)
                joinObject4.setJoinLayer(stra)
                joinObject4.setPrefix('J')
                joinObject4.setJoinFieldNamesSubset(['_stra_max'])
                grid.addJoin(joinObject4)
            else:
                pass

            if self.dlg.lineEdit_12.text():
                field_0 = 'id'
                field_1 = 'id'
                joinObject5 = QgsVectorLayerJoinInfo()
                joinObject5.setJoinFieldName(field_0)
                joinObject5.setTargetFieldName(field_1)
                joinObject5.setJoinLayerId(mine.id())
                joinObject5.setUsingMemoryCache(True)
                joinObject5.setJoinLayer(mine)
                joinObject5.setPrefix('J')
                joinObject5.setJoinFieldNamesSubset(['_mineral_idx'])
                grid.addJoin(joinObject5)
            else:
                pass

            if self.dlg.lineEdit_13.text():
                field_0 = 'id'
                field_1 = 'id'
                joinObject6 = QgsVectorLayerJoinInfo()
                joinObject6.setJoinFieldName(field_0)
                joinObject6.setTargetFieldName(field_1)
                joinObject6.setJoinLayerId(foss.id())
                joinObject6.setUsingMemoryCache(True)
                joinObject6.setJoinLayer(foss)
                joinObject6.setPrefix('J')
                joinObject6.setJoinFieldNamesSubset(['_fossil_idx'])
                grid.addJoin(joinObject6)
            else:
                pass

            #########################################
            #Adding values to grid cells with  lakes#
            #########################################
            if self.dlg.lineEdit_17.text():
                lakes7 = self.dlg.lineEdit_17.text()
                lakes7_name = "lakes"
                lakes7_input = QgsVectorLayer(lakes7, lakes7_name, "ogr")
                newField7 = QgsField('_lakes', QVariant.Int)
                grid.dataProvider().addAttributes([newField7])
                grid.updateFields()      
                selection = processing.run("native:selectbylocation",
                    {'INPUT':grid,
                    'INTERSECT':lakes7_input,
                    'METHOD':0,
                    'PREDICATE':[0]})
                with edit(grid):
                    for id in selection['OUTPUT'].selectedFeatureIds():
                        feature = grid.getFeature(id)
                        feature['_lakes'] = 3
                        grid.updateFeature(feature)
                grid.removeSelection()
            else:
                pass

            #########################################
            #Deleting features where geology is NULL#
            #########################################
            if self.dlg.lineEdit_5.text():
                selection = processing.run("qgis:selectbyattribute", 
                    {'INPUT':grid,
                    'FIELD':'J_geol_variety',
                    'OPERATOR':8,
                    'VALUE':'',
                    'METHOD':0})    
                with edit(grid):
                    for id in selection['OUTPUT'].selectedFeatureIds():
                        feature = grid.getFeature(id)
                        if feature['J_geol_variety'] == NULL:
                            grid.deleteFeature(feature.id())
                        grid.updateFeature(feature)
            else:
                pass
            
            ######################################
            #Determining final geodiversity index#
            ######################################
            prov = grid.dataProvider()
            newField8 = QgsField('_GEODIV', QVariant.Int)
            prov.addAttributes([newField8])
            grid.updateFields()
            idx = grid.fields().lookupField('_GEODIV')
            context = QgsExpressionContext()
            expression = QgsExpression('(CASE WHEN "_lakes" IS NOT NULL THEN "_lakes" ELSE 0 END) + (CASE WHEN "J_geol_variety" IS NOT NULL THEN "J_geol_variety" ELSE 0 END) + (CASE WHEN "J_pedo_variety" IS NOT NULL THEN "J_pedo_variety" ELSE 0 END) + (CASE WHEN "J_geom_variety" IS NOT NULL THEN "J_geom_variety" ELSE 0 END) + (CASE WHEN "J_stra_max" IS NOT NULL THEN "J_stra_max" ELSE 0 END) + (CASE WHEN "J_mineral_idx" IS NOT NULL THEN "J_mineral_idx" ELSE 0 END) + (CASE WHEN "J_fossil_idx" IS NOT NULL THEN "J_fossil_idx" ELSE 0 END)')
            scope = QgsExpressionContextScope()
            scope.setFields(grid.fields())
            context.appendScope(scope)
            expression.prepare(context)
            with edit(grid):
                for feature in grid.getFeatures():
                    context.setFeature(feature)
                    geodiv = expression.evaluate(context)
                    atts = {idx: geodiv}
                    grid.dataProvider().changeAttributeValues({feature.id():atts})
            

            #Adding the final grid into the open QGIS project
            QgsProject.instance().addMapLayer(addGrid0)
            
            #Push message in QGIS when done
            self.iface.messageBar().pushMessage(
                "Success", "Output files are written!",
                level=Qgis.Success, duration=10)
                
                
            self.iface.messageBar().clearWidgets()
            
            
            pass
