# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeodbIO
                                 A QGIS plugin
 Connects to spatial data from geodb.io via api
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-02-05
        git sha              : $Format:%H$
        copyright            : (C) 2025 by geodb.io
        email                : admin@geodb.io
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.core import QgsApplication

# Initialize Qt resources from file resources.py
from .resources import *
from .ui.geodb_modern_dialog import GeodbModernDialog
from .managers.storage_manager import StorageManager


class GeodbIO:
    """QGIS Plugin Implementation - Modern Architecture (v2.0+)"""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GeodbIO_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Geodb.io Connection')

        # Modern dialog instance (created on first run)
        self.modern_dialog = None

        # Storage manager for checking unsaved data
        self.storage_manager = StorageManager()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GeodbIO', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToDatabaseMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/geodb/icons/database-fill-check.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'Geodb.io Sync'),
            callback=self.run_modern,
            parent=self.iface.mainWindow())

        # Connect to QGIS aboutToQuit signal to warn about unsaved data
        QgsApplication.instance().aboutToQuit.connect(self._on_qgis_closing)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginDatabaseMenu(
                self.tr(u'&Geodb.io Connection'),
                action)
            self.iface.removeToolBarIcon(action)

        # Disconnect QGIS closing signal
        try:
            QgsApplication.instance().aboutToQuit.disconnect(self._on_qgis_closing)
        except (TypeError, RuntimeError):
            pass  # Already disconnected or never connected

        # Clean up the dialog to prevent crashes on plugin reload
        if self.modern_dialog is not None:
            try:
                self.modern_dialog.cleanup()
            except Exception:
                pass
            try:
                self.modern_dialog.close()
                self.modern_dialog.deleteLater()
            except Exception:
                pass
            self.modern_dialog = None

    def run_modern(self):
        """Run the modern v2.0+ dialog with new architecture."""
        if self.modern_dialog is None:
            self.modern_dialog = GeodbModernDialog(parent=self.iface.mainWindow())

        self.modern_dialog.show()
        self.modern_dialog.exec_()

    def _on_qgis_closing(self):
        """
        Handle QGIS application closing.

        Warns user if there are unsaved memory layers with Geodb data.
        """
        if self.storage_manager.has_unsaved_memory_layers():
            layer_names = self.storage_manager.get_memory_layer_names()
            layers_text = "\n• ".join(layer_names)

            QMessageBox.warning(
                self.iface.mainWindow(),
                "Geodb.io - Unsaved Data Warning",
                f"The following Geodb layers are stored in memory and will be lost:\n\n"
                f"• {layers_text}\n\n"
                f"To preserve your data, open the Geodb.io plugin and configure "
                f"GeoPackage storage before closing QGIS.\n\n"
                f"Alternatively, you can save your QGIS project to preserve "
                f"memory layers temporarily.",
                QMessageBox.StandardButton.Ok
            )
