# -*- coding: utf-8 -*-
"""
Layer Manager Utility
Handles creation, styling, and management of QGIS layers from GeoConfirmed data.

Alpha version with faction extraction and statistics support.
"""

import os
import re
from datetime import datetime
from typing import List, Dict, Optional, Any, Tuple, Callable

from qgis.PyQt.QtCore import QVariant
from qgis.PyQt.QtGui import QColor
from qgis.core import (
    QgsVectorLayer,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsPointXY,
    QgsProject,
    QgsVectorFileWriter,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransformContext,
    QgsCoordinateTransform,
    QgsSymbol,
    QgsCategorizedSymbolRenderer,
    QgsRendererCategory,
    QgsMarkerSymbol,
    QgsSingleSymbolRenderer,
    QgsVectorLayerTemporalProperties,
    QgsAction,
    QgsMessageLog,
    QgsRectangle,
    QgsDistanceArea,
    Qgis
)


class LayerManager:
    """Manages QGIS layers for GeoConfirmed data."""

    # Field definitions for placemarks (v2 API returns full details)
    # Format: (name, type, length)
    # Note: GeoPackage enforces field length constraints. Use large values for text
    # fields that may contain long content (URLs can be 200+ chars each).
    # SQLite/GeoPackage TEXT has no practical limit (up to 1 billion bytes).
    PLACEMARK_FIELDS = [
        ('id', QVariant.String, 100),
        ('date', QVariant.DateTime, 0),
        ('date_str', QVariant.String, 50),
        ('date_created', QVariant.String, 50),
        ('name', QVariant.String, 500),
        ('description', QVariant.String, 10000),  # Descriptions can be very long
        ('plus_code', QVariant.String, 100),
        ('icon', QVariant.String, 500),
        ('latitude', QVariant.Double, 0),
        ('longitude', QVariant.Double, 0),
        # Source fields - can contain multiple URLs (each URL can be 200+ chars)
        ('original_source', QVariant.String, 10000),  # Multiple URLs, newline-separated
        ('geolocation', QVariant.String, 10000),  # Multiple URLs, newline-separated
        # Equipment and origin
        ('origin', QVariant.String, 200),
        ('gear', QVariant.String, 2000),  # Equipment descriptions can be long
        # Unit info
        ('units', QVariant.String, 2000),  # Unit names (resolved from ORBAT IDs)
        # Derived fields from icon path
        ('faction', QVariant.String, 200),
        ('faction_color', QVariant.String, 10),
        ('is_destroyed', QVariant.Bool, 0),
        ('equipment_type', QVariant.String, 100),
    ]

    # Equipment type mapping based on icon filename patterns
    EQUIPMENT_TYPES = {
        # Numeric icons from transparent folder
        '10': 'Tank', '11': 'Tank', '12': 'Tank', '13': 'Tank',
        '20': 'APC/IFV', '21': 'APC/IFV', '22': 'APC/IFV',
        '30': 'Artillery', '31': 'Artillery', '32': 'Artillery',
        '40': 'MLRS', '41': 'MLRS', '42': 'MLRS',
        '50': 'Vehicle', '51': 'Vehicle', '52': 'Vehicle',
        '60': 'Aircraft', '61': 'Aircraft', '62': 'Aircraft',
        '70': 'Helicopter', '71': 'Helicopter', '72': 'Helicopter',
        '80': 'Air Defense', '81': 'Air Defense',
        '90': 'Drone/UAV', '91': 'Drone/UAV', '92': 'Drone/UAV', '93': 'Drone/UAV',
        '100': 'Infantry/Position', '101': 'Infantry/Position', '102': 'Infantry/Position', '103': 'Infantry/Position',
        '110': 'Truck', '111': 'Truck', '112': 'Truck',
        '120': 'Engineering', '121': 'Engineering',
        '170': 'Naval', '171': 'Naval', '172': 'Naval', '173': 'Naval',
        '180': 'Infrastructure', '181': 'Infrastructure',
        '190': 'Explosion/Strike', '191': 'Explosion/Strike', '192': 'Explosion/Strike', '193': 'Explosion/Strike',
        '200': 'Building', '201': 'Building', '202': 'Building',
        # Special icons
        'ugv': 'UGV (Ground Robot)',
        'quad': 'Quadcopter',
        'droneintercept': 'Drone Intercept',
        'radar': 'Radar/EW',
        'harbor': 'Naval/Port',
        'moto': 'Motorcycle',
    }

    # Field definitions for mentions
    # Format: (name, type, length)
    # Note: GeoPackage enforces field length constraints.
    MENTION_FIELDS = [
        ('id', QVariant.String, 100),
        ('username', QVariant.String, 200),
        ('name', QVariant.String, 500),
        ('text', QVariant.String, 10000),  # Social media posts can be long
        ('date', QVariant.DateTime, 0),
        ('date_str', QVariant.String, 50),
        ('link', QVariant.String, 2000),  # URLs can be very long
        ('remark', QVariant.String, 5000),
        ('processed', QVariant.Bool, 0),
        ('in_slack', QVariant.Bool, 0),
        ('is_deleted', QVariant.Bool, 0),
        ('doubt', QVariant.Bool, 0),
        ('is_wrong', QVariant.Bool, 0),
        ('conflict', QVariant.String, 100),
    ]

    # Icon categories for styling
    ICON_CATEGORIES = {
        # Military equipment
        1: {'name': 'Tank', 'color': '#e74c3c'},
        2: {'name': 'APC/IFV', 'color': '#c0392b'},
        3: {'name': 'Artillery', 'color': '#9b59b6'},
        4: {'name': 'MLRS', 'color': '#8e44ad'},
        5: {'name': 'Air Defense', 'color': '#3498db'},
        6: {'name': 'Aircraft', 'color': '#2980b9'},
        7: {'name': 'Helicopter', 'color': '#1abc9c'},
        8: {'name': 'Drone', 'color': '#16a085'},
        9: {'name': 'Naval', 'color': '#2c3e50'},
        10: {'name': 'Vehicle', 'color': '#f39c12'},
        11: {'name': 'Truck', 'color': '#d35400'},
        12: {'name': 'Infantry', 'color': '#27ae60'},
        13: {'name': 'Position', 'color': '#2ecc71'},
        14: {'name': 'Explosion', 'color': '#e67e22'},
        15: {'name': 'Fire', 'color': '#e74c3c'},
        16: {'name': 'Building', 'color': '#95a5a6'},
        17: {'name': 'Infrastructure', 'color': '#7f8c8d'},
        # Default
        0: {'name': 'Unknown', 'color': '#bdc3c7'},
    }

    def __init__(self, iface):
        """Initialize the layer manager.

        Args:
            iface: QGIS interface instance
        """
        self.iface = iface
        # Faction color -> name lookup (populated from conflict details)
        self._faction_lookup: Dict[str, str] = {}
        # Faction color -> display color lookup
        self._faction_colors: Dict[str, str] = {}

    def set_faction_lookup(self, factions: List[Dict]):
        """Set the faction lookup table from conflict details.

        Args:
            factions: List of faction dicts from /Conflict/{name} API
        """
        self._faction_lookup = {}
        self._faction_colors = {}
        for faction in factions:
            # API uses 'backgroundColor' not 'color'
            color = faction.get('backgroundColor', faction.get('color', '')).lstrip('#').upper()
            name = faction.get('name', 'Unknown')
            if color:
                self._faction_lookup[color] = name
                self._faction_colors[color] = f"#{color}"
        self._log(f"Loaded {len(self._faction_lookup)} factions into lookup table: {self._faction_lookup}")

    def get_faction_names(self) -> List[str]:
        """Get list of all faction names.

        Returns:
            Sorted list of faction names
        """
        return sorted(set(self._faction_lookup.values()))

    def get_faction_colors(self) -> Dict[str, str]:
        """Get faction name to color mapping.

        Returns:
            Dict mapping faction name to hex color
        """
        # Invert the lookup to get name -> color
        name_to_color = {}
        for color_code, name in self._faction_lookup.items():
            if name not in name_to_color:
                name_to_color[name] = f"#{color_code}"
        return name_to_color

    def _parse_icon_path(self, icon_path: str) -> Dict[str, Any]:
        """Parse an icon path to extract faction and equipment info.

        Icon path format: /icons/{COLOR}/{DESTROYED}/icons/{FOLDER}/{FILENAME}.png

        Args:
            icon_path: The icon path from API

        Returns:
            Dict with faction, faction_color, is_destroyed, equipment_type
        """
        result = {
            'faction': 'Unknown',
            'faction_color': '#666666',
            'is_destroyed': False,
            'equipment_type': 'Unknown',
        }

        if not icon_path:
            return result

        # Parse path components
        # Example: /icons/E00000/False/icons/Ukraine/ugv_1.png
        parts = icon_path.split('/')
        if len(parts) >= 3:
            color_code = parts[2].upper()
            result['faction'] = self._faction_lookup.get(color_code, 'Unknown')
            result['faction_color'] = f"#{color_code}" if len(color_code) == 6 else '#666666'

        if len(parts) >= 4:
            result['is_destroyed'] = parts[3].lower() == 'true'

        # Extract equipment type from filename
        if len(parts) >= 7:
            filename = parts[-1].replace('.png', '').lower()

            # Check for special icon names first
            for key, eq_type in self.EQUIPMENT_TYPES.items():
                if key in filename:
                    result['equipment_type'] = eq_type
                    break
            else:
                # Try numeric matching
                match = re.search(r'(\d+)', filename)
                if match:
                    num = match.group(1)
                    result['equipment_type'] = self.EQUIPMENT_TYPES.get(num, 'Other')

        return result

    def compute_statistics(self, items: List[Dict]) -> Dict[str, Any]:
        """Compute statistics from a list of placemark items.

        Args:
            items: List of placemark dicts from API

        Returns:
            Dict with statistics by faction, equipment type, and status
        """
        stats = {
            'total': len(items),
            'by_faction': {},
            'by_equipment': {},
            'by_status': {'active': 0, 'destroyed': 0},
            'faction_equipment': {},  # Nested: faction -> equipment -> count
        }

        for item in items:
            icon_path = item.get('icon', '')
            parsed = self._parse_icon_path(icon_path)

            faction = parsed['faction']
            equipment = parsed['equipment_type']
            is_destroyed = parsed['is_destroyed']

            # Count by faction
            stats['by_faction'][faction] = stats['by_faction'].get(faction, 0) + 1

            # Count by equipment
            stats['by_equipment'][equipment] = stats['by_equipment'].get(equipment, 0) + 1

            # Count by status
            if is_destroyed:
                stats['by_status']['destroyed'] += 1
            else:
                stats['by_status']['active'] += 1

            # Nested faction -> equipment
            if faction not in stats['faction_equipment']:
                stats['faction_equipment'][faction] = {}
            stats['faction_equipment'][faction][equipment] = \
                stats['faction_equipment'][faction].get(equipment, 0) + 1

        return stats

    def filter_items_by_factions(
        self,
        items: List[Dict],
        allowed_factions: List[str]
    ) -> List[Dict]:
        """Filter items to only include specified factions.

        Args:
            items: List of placemark dicts
            allowed_factions: List of faction names to include

        Returns:
            Filtered list of items
        """
        if not allowed_factions:
            return items

        allowed_set = set(allowed_factions)
        filtered = []
        unknown_count = 0
        faction_counts = {}

        for item in items:
            icon_path = item.get('icon', '')
            parsed = self._parse_icon_path(icon_path)
            faction = parsed['faction']

            # Track what factions we're seeing for debugging
            faction_counts[faction] = faction_counts.get(faction, 0) + 1

            if faction in allowed_set:
                filtered.append(item)
            elif faction == 'Unknown':
                unknown_count += 1

        # Log debug info about faction distribution
        self._log(f"Faction filter: allowed={allowed_factions}, found factions={faction_counts}")
        if unknown_count > 0:
            self._log(f"Warning: {unknown_count} items have 'Unknown' faction (color not in lookup table)", Qgis.Warning)

        return filtered

    def filter_items_by_keywords(
        self,
        items: List[Dict],
        search_query: str
    ) -> List[Dict]:
        """Filter items by boolean keyword search.

        Supports: & (AND), | (OR), ! (NOT), () grouping
        Searches in: icon path (which contains equipment/category info)

        Args:
            items: List of placemark dicts
            search_query: Boolean search query string

        Returns:
            Filtered list of items
        """
        if not search_query or not search_query.strip():
            return items

        search_query = search_query.strip()
        self._log(f"Applying keyword filter: '{search_query}' to {len(items)} items")

        # Parse and compile the search expression
        try:
            matcher = self._compile_search_expression(search_query)
        except ValueError as e:
            self._log(f"Invalid search expression: {e}", Qgis.Warning)
            return items

        filtered = []
        for item in items:
            # Build searchable text from available fields
            searchable_text = self._get_searchable_text(item)

            if matcher(searchable_text):
                filtered.append(item)

        self._log(f"Keyword filter: {len(filtered)} of {len(items)} items matched")
        return filtered

    def filter_items_by_spatial(
        self,
        items: List[Dict],
        filter_type: str,
        reference_layer: Optional['QgsVectorLayer'] = None,
        radius_meters: float = 0,
        canvas_extent: Optional[QgsRectangle] = None,
        canvas_crs: Optional[QgsCoordinateReferenceSystem] = None
    ) -> List[Dict]:
        """Filter items by spatial criteria.

        :param items: List of placemark items from API
        :param filter_type: One of "none", "around", "canvas", "within"
        :param reference_layer: Vector layer for "around" or "within" filter types
        :param radius_meters: Radius in meters for "around" filter
        :param canvas_extent: Map canvas extent for "canvas" filter
        :param canvas_crs: CRS of the canvas extent
        :returns: Filtered list of items
        """
        if filter_type == "none" or not items:
            return items

        before_count = len(items)

        if filter_type == "around":
            if reference_layer is None:
                self._log("Spatial filter 'around' requires a reference layer", Qgis.Warning)
                return items
            filtered = self._filter_by_radius(items, reference_layer, radius_meters)

        elif filter_type == "canvas":
            if canvas_extent is None or canvas_crs is None:
                self._log("Spatial filter 'canvas' requires canvas_extent and canvas_crs", Qgis.Warning)
                return items
            filtered = self._filter_by_extent(items, canvas_extent, canvas_crs)

        elif filter_type == "within":
            if reference_layer is None:
                self._log("Spatial filter 'within' requires a reference layer", Qgis.Warning)
                return items
            if not reference_layer.isValid() or reference_layer.featureCount() == 0:
                self._log("Reference layer is invalid or empty", Qgis.Warning)
                return items
            filtered = self._filter_within_layer(items, reference_layer)

        else:
            self._log(f"Unknown spatial filter type: {filter_type}", Qgis.Warning)
            return items

        after_count = len(filtered)
        self._log(f"Spatial filter ({filter_type}): {before_count} -> {after_count} items")
        return filtered

    def _filter_by_radius(
        self,
        items: List[Dict],
        reference_layer: 'QgsVectorLayer',
        radius_meters: float
    ) -> List[Dict]:
        """Filter items to those within radius of reference layer features.

        For point layers: radius is measured from the point
        For polygon/line layers: radius is measured from the boundary (edge)

        :param items: List of placemark items from API
        :param reference_layer: Vector layer containing reference features
        :param radius_meters: Radius in meters to filter by
        :returns: Filtered list of items within radius of any reference feature
        """
        if not reference_layer.isValid() or reference_layer.featureCount() == 0:
            self._log("Reference layer is invalid or empty for radius filter", Qgis.Warning)
            return items

        # Set up distance calculator
        distance_area = QgsDistanceArea()
        distance_area.setSourceCrs(
            QgsCoordinateReferenceSystem('EPSG:4326'),
            QgsProject.instance().transformContext()
        )
        distance_area.setEllipsoid('WGS84')

        # Prepare coordinate transform if reference layer is not EPSG:4326
        ref_crs = reference_layer.crs()
        need_transform = ref_crs.authid() != 'EPSG:4326'
        transform_to_4326 = None
        transform_from_4326 = None
        if need_transform:
            transform_to_4326 = QgsCoordinateTransform(
                ref_crs,
                QgsCoordinateReferenceSystem('EPSG:4326'),
                QgsProject.instance()
            )
            transform_from_4326 = QgsCoordinateTransform(
                QgsCoordinateReferenceSystem('EPSG:4326'),
                ref_crs,
                QgsProject.instance()
            )

        # Determine layer geometry type
        layer_geom_type = reference_layer.geometryType()  # 0=Point, 1=Line, 2=Polygon
        is_point_layer = (layer_geom_type == 0)

        # Collect reference features
        reference_points = []  # For point layers
        reference_geometries = []  # For polygon/line layers (in original CRS for distance calc)

        for feature in reference_layer.getFeatures():
            geom = feature.geometry()
            if geom.isEmpty():
                continue

            if is_point_layer:
                # For points, extract the point and transform to EPSG:4326
                point = geom.asPoint()
                if need_transform and transform_to_4326 is not None:
                    point = transform_to_4326.transform(point)
                reference_points.append(point)
            else:
                # For polygons/lines, store the geometry in its original CRS
                # We'll transform data points TO layer CRS for distance calculation
                reference_geometries.append(geom)

        if is_point_layer:
            if not reference_points:
                self._log("No valid reference points found in layer", Qgis.Warning)
                return items
            self._log(f"Filtering by radius {radius_meters}m from {len(reference_points)} reference points")
        else:
            if not reference_geometries:
                self._log("No valid reference geometries found in layer", Qgis.Warning)
                return items
            self._log(f"Filtering by radius {radius_meters}m from boundary of {len(reference_geometries)} polygon(s)/line(s)")

        filtered = []
        for item in items:
            point = self._get_coordinates(item, is_placemarks=True)
            if point is None:
                continue

            if is_point_layer:
                # Point layer: measure distance from point to reference points
                for ref_point in reference_points:
                    distance = distance_area.measureLine(point, ref_point)
                    if distance <= radius_meters:
                        filtered.append(item)
                        break
            else:
                # Polygon/line layer: measure distance from point to geometry boundary
                # Transform data point to layer CRS for accurate distance calculation
                point_in_layer_crs = point
                if need_transform and transform_from_4326 is not None:
                    point_in_layer_crs = transform_from_4326.transform(point)

                point_geom = QgsGeometry.fromPointXY(point_in_layer_crs)

                for ref_geom in reference_geometries:
                    # distance() returns 0 if point is inside polygon, otherwise distance to nearest edge
                    dist_in_crs_units = ref_geom.distance(point_geom)

                    # Convert CRS units to meters (approximate for projected CRS)
                    # For accurate results, use distance_area to measure
                    if dist_in_crs_units == 0:
                        # Point is inside polygon - include it
                        filtered.append(item)
                        break
                    else:
                        # Find nearest point on boundary and measure geodetic distance
                        nearest_point = ref_geom.nearestPoint(point_geom)
                        if not nearest_point.isEmpty():
                            nearest_pt = nearest_point.asPoint()
                            # Transform back to EPSG:4326 for geodetic distance
                            if need_transform and transform_to_4326 is not None:
                                nearest_pt = transform_to_4326.transform(nearest_pt)
                            distance = distance_area.measureLine(point, nearest_pt)
                            if distance <= radius_meters:
                                filtered.append(item)
                                break

        return filtered

    def _filter_by_extent(
        self,
        items: List[Dict],
        extent: QgsRectangle,
        extent_crs: QgsCoordinateReferenceSystem
    ) -> List[Dict]:
        """Filter items to those within the given extent.

        :param items: List of placemark items from API
        :param extent: Bounding box to filter by
        :param extent_crs: CRS of the extent
        :returns: Filtered list of items within the extent
        """
        # Transform extent to EPSG:4326 if needed
        if extent_crs.authid() != 'EPSG:4326':
            transform = QgsCoordinateTransform(
                extent_crs,
                QgsCoordinateReferenceSystem('EPSG:4326'),
                QgsProject.instance()
            )
            extent = transform.transformBoundingBox(extent)

        filtered = []
        for item in items:
            point = self._get_coordinates(item, is_placemarks=True)
            if point is None:
                continue

            if extent.contains(point):
                filtered.append(item)

        return filtered

    def _filter_within_layer(
        self,
        items: List[Dict],
        reference_layer: 'QgsVectorLayer'
    ) -> List[Dict]:
        """Filter items to those within the polygon geometries of the reference layer.

        :param items: List of placemark items from API
        :param reference_layer: Vector layer containing polygon geometries
        :returns: Filtered list of items that fall within any polygon
        """
        # Prepare coordinate transform if reference layer is not EPSG:4326
        ref_crs = reference_layer.crs()
        need_transform = ref_crs.authid() != 'EPSG:4326'
        transform = None
        if need_transform:
            # Transform FROM EPSG:4326 TO layer CRS (so we can use layer's geometry)
            transform = QgsCoordinateTransform(
                QgsCoordinateReferenceSystem('EPSG:4326'),
                ref_crs,
                QgsProject.instance()
            )

        # Collect all geometries from the reference layer
        geometries = []
        for feature in reference_layer.getFeatures():
            geom = feature.geometry()
            if not geom.isEmpty():
                geometries.append(geom)

        if not geometries:
            self._log("No valid geometries found in reference layer", Qgis.Warning)
            return items

        self._log(f"Filtering within {len(geometries)} polygon(s) from reference layer")

        filtered = []
        for item in items:
            point = self._get_coordinates(item, is_placemarks=True)
            if point is None:
                continue

            # Transform point to layer CRS if needed
            if need_transform and transform is not None:
                point = transform.transform(point)

            # Create point geometry for intersection test
            point_geom = QgsGeometry.fromPointXY(point)

            # Check if point is within any of the reference geometries
            for geom in geometries:
                if geom.contains(point_geom):
                    filtered.append(item)
                    break  # Found one containing polygon, no need to check others

        return filtered

    def _get_searchable_text(self, item: Dict) -> str:
        """Extract searchable text from an item.

        Args:
            item: Placemark dict from API (basic or enriched)

        Returns:
            Lowercase string containing all searchable content
        """
        parts = []

        # Icon path contains equipment type info
        icon = item.get('icon', '')
        if icon:
            parts.append(icon)
            # Also parse and add the equipment type name
            parsed = self._parse_icon_path(icon)
            parts.append(parsed.get('equipment_type', ''))
            parts.append(parsed.get('faction', ''))

        # Enriched fields (from detailed API call)
        # These are the most useful for keyword search
        for field in ['description', 'name', 'plusCode', 'origin', 'gear']:
            value = item.get(field)
            if value:
                parts.append(str(value))

        # Source links (originalSource and geolocation can be strings or lists)
        for field in ['originalSource', 'geolocation']:
            value = item.get(field)
            if value:
                if isinstance(value, list):
                    parts.extend([str(v) for v in value])
                else:
                    parts.append(str(value))

        # Legacy/fallback fields
        for field in ['title', 'text', 'remark']:
            value = item.get(field)
            if value:
                parts.append(str(value))

        return ' '.join(parts).lower()

    def _compile_search_expression(self, query: str):
        """Compile a boolean search expression into a matcher function.

        Supports:
        - & for AND
        - | for OR
        - ! for NOT
        - () for grouping
        - Implicit AND between terms

        Args:
            query: Search query string

        Returns:
            Function that takes text and returns bool

        Raises:
            ValueError: If the expression is invalid
        """
        # Tokenize the query
        tokens = self._tokenize_search(query)
        if not tokens:
            return lambda text: True

        # Parse into an AST and return a matcher
        ast, remaining = self._parse_or_expression(tokens)
        if remaining:
            raise ValueError(f"Unexpected tokens: {remaining}")

        return lambda text: self._evaluate_ast(ast, text)

    def _tokenize_search(self, query: str) -> List[str]:
        """Tokenize a search query into operators and terms.

        Args:
            query: Search query string

        Returns:
            List of tokens
        """
        tokens = []
        i = 0
        query = query.strip()

        while i < len(query):
            char = query[i]

            # Skip whitespace
            if char.isspace():
                i += 1
                continue

            # Single-character operators
            if char in '&|!()':
                tokens.append(char)
                i += 1
                continue

            # Quoted string
            if char in '"\'':
                quote_char = char
                i += 1
                term = ''
                while i < len(query) and query[i] != quote_char:
                    term += query[i]
                    i += 1
                i += 1  # Skip closing quote
                if term:
                    tokens.append(('TERM', term.lower()))
                continue

            # Regular term (word)
            term = ''
            while i < len(query) and not query[i].isspace() and query[i] not in '&|!()':
                term += query[i]
                i += 1
            if term:
                tokens.append(('TERM', term.lower()))

        return tokens

    def _parse_or_expression(self, tokens: List) -> Tuple[Any, List]:
        """Parse an OR expression (lowest precedence).

        Args:
            tokens: List of tokens

        Returns:
            Tuple of (AST node, remaining tokens)
        """
        left, tokens = self._parse_and_expression(tokens)

        while tokens and tokens[0] == '|':
            tokens = tokens[1:]  # consume '|'
            right, tokens = self._parse_and_expression(tokens)
            left = ('OR', left, right)

        return left, tokens

    def _parse_and_expression(self, tokens: List) -> Tuple[Any, List]:
        """Parse an AND expression.

        Args:
            tokens: List of tokens

        Returns:
            Tuple of (AST node, remaining tokens)
        """
        left, tokens = self._parse_not_expression(tokens)

        while tokens:
            # Explicit AND
            if tokens[0] == '&':
                tokens = tokens[1:]
                right, tokens = self._parse_not_expression(tokens)
                left = ('AND', left, right)
            # Implicit AND (adjacent terms without operator)
            elif tokens[0] not in ('|', ')'):
                right, tokens = self._parse_not_expression(tokens)
                left = ('AND', left, right)
            else:
                break

        return left, tokens

    def _parse_not_expression(self, tokens: List) -> Tuple[Any, List]:
        """Parse a NOT expression.

        Args:
            tokens: List of tokens

        Returns:
            Tuple of (AST node, remaining tokens)
        """
        if tokens and tokens[0] == '!':
            tokens = tokens[1:]
            expr, tokens = self._parse_not_expression(tokens)
            return ('NOT', expr), tokens

        return self._parse_primary(tokens)

    def _parse_primary(self, tokens: List) -> Tuple[Any, List]:
        """Parse a primary expression (term or parenthesized expression).

        Args:
            tokens: List of tokens

        Returns:
            Tuple of (AST node, remaining tokens)
        """
        if not tokens:
            return ('TERM', ''), []

        token = tokens[0]

        # Parenthesized expression
        if token == '(':
            tokens = tokens[1:]
            expr, tokens = self._parse_or_expression(tokens)
            if tokens and tokens[0] == ')':
                tokens = tokens[1:]
            return expr, tokens

        # Term
        if isinstance(token, tuple) and token[0] == 'TERM':
            return token, tokens[1:]

        # Unexpected token, treat as empty
        return ('TERM', ''), tokens

    def _evaluate_ast(self, ast: Any, text: str) -> bool:
        """Evaluate an AST node against text.

        Args:
            ast: AST node
            text: Text to search in (already lowercase)

        Returns:
            True if the expression matches
        """
        if not ast:
            return True

        if isinstance(ast, tuple):
            op = ast[0]

            if op == 'TERM':
                term = ast[1]
                return term in text if term else True

            if op == 'AND':
                return self._evaluate_ast(ast[1], text) and self._evaluate_ast(ast[2], text)

            if op == 'OR':
                return self._evaluate_ast(ast[1], text) or self._evaluate_ast(ast[2], text)

            if op == 'NOT':
                return not self._evaluate_ast(ast[1], text)

        return True

    def _log(self, message: str, level: Qgis.MessageLevel = Qgis.Info):
        """Log a message.

        Args:
            message: Message to log
            level: Log level
        """
        QgsMessageLog.logMessage(message, "GeoConfirmed", level)

    def _parse_datetime(self, date_str: Optional[str]):
        """Parse a datetime string from API.

        Args:
            date_str: ISO format date string

        Returns:
            QDateTime object or NULL QVariant if parsing fails
        """
        from qgis.PyQt.QtCore import QDateTime, QDate, QTime

        if not date_str:
            return QVariant()  # Return NULL QVariant

        try:
            # Handle ISO format with timezone
            if 'T' in date_str:
                # Remove timezone suffix for parsing
                clean_str = date_str.replace('Z', '').split('+')[0]
                dt = datetime.fromisoformat(clean_str)
                return QDateTime(
                    QDate(dt.year, dt.month, dt.day),
                    QTime(dt.hour, dt.minute, dt.second)
                )
            else:
                # Just a date
                dt = datetime.strptime(date_str[:10], '%Y-%m-%d')
                return QDateTime(
                    QDate(dt.year, dt.month, dt.day),
                    QTime(0, 0, 0)
                )
        except (ValueError, TypeError):
            return QVariant()  # Return NULL QVariant

    def _get_coordinates(self, item: Dict, is_placemarks: bool) -> Optional[QgsPointXY]:
        """Extract coordinates from an item.

        Args:
            item: Data item from API
            is_placemarks: Whether this is a placemark (vs mention)

        Returns:
            Point geometry or None
        """
        try:
            if is_placemarks:
                # v2 API uses 'coordinates' array [lat, lon]
                coords = item.get('coordinates')
                if coords and isinstance(coords, list) and len(coords) >= 2:
                    lat = coords[0]
                    lng = coords[1]
                else:
                    # Fallback to v1 format (la/lo)
                    lat = item.get('la')
                    lng = item.get('lo')
            else:
                # Mentions have separate lat/lng fields
                lat = item.get('latitude') or item.get('la')
                lng = item.get('longitude') or item.get('lo')

            if lat is not None and lng is not None:
                return QgsPointXY(float(lng), float(lat))
            else:
                self._log(f"Missing coordinates in item: keys={list(item.keys())}", Qgis.Warning)
        except (ValueError, TypeError) as e:
            self._log(f"Failed to parse coordinates: {e}", Qgis.Warning)

        return None

    def _create_fields(self, is_placemarks: bool) -> List[QgsField]:
        """Create field definitions for the layer.

        Args:
            is_placemarks: Whether this is for placemarks

        Returns:
            List of QgsField objects
        """
        field_defs = self.PLACEMARK_FIELDS if is_placemarks else self.MENTION_FIELDS
        fields = []
        for name, var_type, length in field_defs:
            field = QgsField(name, var_type)
            if length > 0:
                field.setLength(length)
            fields.append(field)
        return fields

    def _truncate_to_field_length(self, value: Any, field: QgsField, feature_id: str = None) -> Any:
        """Truncate a string value to fit the field's maximum length.

        Args:
            value: The value to potentially truncate
            field: The QgsField with length constraints
            feature_id: Optional feature ID for logging context

        Returns:
            The value, truncated if necessary
        """
        if value is None:
            return value

        # Only truncate strings
        if field.type() != QVariant.String:
            return value

        # Get field length (0 means unlimited)
        max_length = field.length()
        if max_length <= 0:
            return value

        # Convert to string and truncate if needed
        str_value = str(value) if value is not None else ''
        if len(str_value) > max_length:
            # Log a warning about data loss
            id_context = f" (feature ID: {feature_id})" if feature_id else ""
            self._log(
                f"WARNING: Truncating field '{field.name()}' from {len(str_value)} to {max_length} chars{id_context}. "
                f"Data loss: '{str_value[max_length:max_length+50]}...'",
                Qgis.Warning
            )
            return str_value[:max_length]

        return str_value

    def _create_feature(
        self,
        item: Dict,
        fields: 'QgsFields',
        is_placemarks: bool,
        orbat_lookup: Optional[Dict[int, str]] = None
    ) -> Optional[QgsFeature]:
        """Create a feature from an API item.

        Args:
            item: Data item from API
            fields: QgsFields object from the layer
            is_placemarks: Whether this is a placemark
            orbat_lookup: Optional dict mapping ORBAT node IDs to unit names

        Returns:
            QgsFeature or None if no valid geometry
        """
        point = self._get_coordinates(item, is_placemarks)
        if point is None:
            return None

        feature = QgsFeature(fields)
        feature.setGeometry(QgsGeometry.fromPointXY(point))

        # Set attributes
        if is_placemarks:
            date_val = self._parse_datetime(item.get('date'))
            icon_path = item.get('icon', '')

            # Parse derived fields from icon path
            parsed = self._parse_icon_path(icon_path)

            # Get coordinates (v2 API uses 'coordinates' array [lat, lon])
            coords = item.get('coordinates')
            if coords and isinstance(coords, list) and len(coords) >= 2:
                lat = coords[0]
                lng = coords[1]
            else:
                # Fallback to v1 format
                lat = item.get('la')
                lng = item.get('lo')

            # Handle source fields (can be string or newline-separated)
            original_source = item.get('originalSource', '')
            if isinstance(original_source, list):
                original_source = '\n'.join(original_source)

            geolocation_src = item.get('geolocation', '')
            if isinstance(geolocation_src, list):
                geolocation_src = '\n'.join(geolocation_src)

            # Handle units - resolve IDs to names if lookup provided
            units_data = item.get('units', {})
            unit_names = ''
            if units_data and isinstance(units_data, dict):
                orbat_ids = units_data.get('orbatNodeIds', [])
                if orbat_ids:
                    if orbat_lookup:
                        # Resolve IDs to names
                        names = [orbat_lookup.get(uid, str(uid)) for uid in orbat_ids]
                        unit_names = ', '.join(names)
                    else:
                        # Fallback to IDs if no lookup
                        unit_names = ','.join(str(uid) for uid in orbat_ids)

            attributes = [
                str(item.get('id', '')),
                date_val,
                item.get('date', ''),
                item.get('dateCreated', ''),
                item.get('name', ''),
                item.get('description', ''),
                item.get('plusCode', ''),
                icon_path,
                lat,
                lng,
                # Source fields
                original_source,
                geolocation_src,
                # Equipment and origin
                item.get('origin', ''),
                item.get('gear', ''),
                # Unit info
                unit_names,
                # Derived fields
                parsed['faction'],
                parsed['faction_color'],
                parsed['is_destroyed'],
                parsed['equipment_type'],
            ]
        else:
            date_val = self._parse_datetime(item.get('date'))
            attributes = [
                str(item.get('id', '')),
                item.get('username', ''),
                item.get('name', ''),
                item.get('text', ''),
                date_val,
                item.get('date', ''),
                item.get('link', ''),
                item.get('remark', ''),
                item.get('processed', False),
                item.get('inSlack', False),
                item.get('isDeleted', False),
                item.get('doubt', False),
                item.get('isWrong', False),
                item.get('conflict', ''),
            ]

        # Truncate string attributes to fit field length constraints
        # This is important for GeoPackage layers which enforce field lengths
        feature_id = str(item.get('id', 'unknown'))
        truncated_attributes = []
        for i, attr in enumerate(attributes):
            if i < fields.count():
                field = fields.field(i)
                truncated_attributes.append(self._truncate_to_field_length(attr, field, feature_id))
            else:
                truncated_attributes.append(attr)

        feature.setAttributes(truncated_attributes)
        return feature

    def create_memory_layer(
        self,
        items: List[Dict],
        layer_name: str,
        is_placemarks: bool = True,
        orbat_lookup: Optional[Dict[int, str]] = None
    ) -> Optional[QgsVectorLayer]:
        """Create a memory layer from API items.

        Args:
            items: List of data items from API
            layer_name: Name for the layer
            is_placemarks: Whether these are placemarks (vs mentions)
            orbat_lookup: Optional dict mapping ORBAT node IDs to unit names

        Returns:
            QgsVectorLayer or None on failure
        """
        # Create layer with fields
        fields = self._create_fields(is_placemarks)
        field_str = '&'.join([
            f"field={f.name()}:{self._qvariant_to_type(f.type())}"
            for f in fields
        ])

        layer = QgsVectorLayer(
            f"Point?crs=EPSG:4326&{field_str}",
            layer_name,
            "memory"
        )

        if not layer.isValid():
            self._log("Failed to create memory layer", Qgis.Critical)
            return None

        # Add features using edit session for proper commit
        layer_fields = layer.fields()
        features = []

        for item in items:
            feature = self._create_feature(item, layer_fields, is_placemarks, orbat_lookup)
            if feature:
                features.append(feature)

        self._log(f"Created {len(features)} features from {len(items)} items")

        if features:
            # Start editing session
            layer.startEditing()

            # Add features to the layer directly
            for feature in features:
                layer.addFeature(feature)

            # Commit changes
            if not layer.commitChanges():
                self._log(f"Failed to commit features: {layer.commitErrors()}", Qgis.Warning)
            else:
                self._log(f"Successfully committed {len(features)} features", Qgis.Info)
        else:
            self._log(f"No valid features created from {len(items)} items", Qgis.Warning)

        layer.updateExtents()
        self._log(f"Final layer '{layer_name}' has {layer.featureCount()} features")
        return layer

    def _qvariant_to_type(self, qvariant_type: int) -> str:
        """Convert QVariant type to QGIS field type string.

        Args:
            qvariant_type: QVariant type constant

        Returns:
            Field type string
        """
        type_map = {
            QVariant.String: 'string',
            QVariant.Int: 'integer',
            QVariant.Double: 'double',
            QVariant.Bool: 'integer',  # Booleans stored as int
            QVariant.DateTime: 'datetime',
        }
        return type_map.get(qvariant_type, 'string')

    def create_geopackage_layer(
        self,
        items: List[Dict],
        output_path: str,
        layer_name: str,
        is_placemarks: bool = True,
        orbat_lookup: Optional[Dict[int, str]] = None
    ) -> Optional[QgsVectorLayer]:
        """Create a GeoPackage file from API items.

        Args:
            items: List of data items from API
            output_path: Path for the output GeoPackage
            layer_name: Name for the layer
            is_placemarks: Whether these are placemarks
            orbat_lookup: Optional dict mapping ORBAT node IDs to unit names

        Returns:
            QgsVectorLayer or None on failure
        """
        self._log(f"Creating GeoPackage at {output_path} with {len(items)} items")

        # First create a memory layer
        mem_layer = self.create_memory_layer(items, layer_name, is_placemarks, orbat_lookup)
        if not mem_layer:
            self._log("Failed to create memory layer for GeoPackage", Qgis.Critical)
            return None

        feature_count = mem_layer.featureCount()
        self._log(f"Memory layer created with {feature_count} features, writing to GeoPackage...")

        if feature_count == 0:
            self._log("Memory layer has 0 features, cannot create GeoPackage", Qgis.Warning)
            return None

        # Write to GeoPackage
        transform_context = QgsCoordinateTransformContext()

        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = "GPKG"
        options.layerName = layer_name.replace(' ', '_').replace('-', '_')

        error = QgsVectorFileWriter.writeAsVectorFormatV3(
            mem_layer,
            output_path,
            transform_context,
            options
        )

        if error[0] != QgsVectorFileWriter.NoError:
            self._log(f"Failed to write GeoPackage: {error[1]}", Qgis.Critical)
            return None

        # Load the written layer
        gpkg_layer = QgsVectorLayer(
            f"{output_path}|layername={options.layerName}",
            layer_name,
            "ogr"
        )

        if not gpkg_layer.isValid():
            self._log("Failed to load GeoPackage layer", Qgis.Critical)
            return None

        self._log(f"Created GeoPackage at {output_path}")
        return gpkg_layer

    def apply_categorized_style(
        self,
        layer: QgsVectorLayer,
        is_placemarks: bool = True
    ):
        """Apply categorized styling to a layer.

        Args:
            layer: Layer to style
            is_placemarks: Whether this is a placemark layer
        """
        if is_placemarks:
            self._apply_icon_categorized_style(layer)
        else:
            self._apply_simple_style(layer)

        # Add action for opening source links
        self._add_open_link_action(layer, is_placemarks)

    def _get_icon_cache_dir(self) -> str:
        """Get or create the icon cache directory.

        Returns:
            Path to the icon cache directory
        """
        cache_dir = os.path.join(os.path.dirname(__file__), '..', 'icons_cache')
        os.makedirs(cache_dir, exist_ok=True)
        return cache_dir

    def _download_icon(self, icon_path: str) -> Optional[str]:
        """Download an icon from GeoConfirmed and cache it locally.

        Args:
            icon_path: The icon path from the API (e.g., /icons/E00000/False/icons/transparent/10.png)

        Returns:
            Local file path to the cached icon, or None if download failed
        """
        if not icon_path:
            return None

        # Create a safe filename from the icon path
        safe_name = icon_path.replace('/', '_').replace('\\', '_').lstrip('_')
        cache_dir = self._get_icon_cache_dir()
        local_path = os.path.join(cache_dir, safe_name)

        # Return cached version if it exists
        if os.path.exists(local_path):
            return local_path

        # Download the icon
        try:
            from qgis.core import QgsBlockingNetworkRequest
            from qgis.PyQt.QtNetwork import QNetworkRequest
            from qgis.PyQt.QtCore import QUrl

            url = f"https://geoconfirmed.org{icon_path}"
            request = QNetworkRequest(QUrl(url))
            blocking_request = QgsBlockingNetworkRequest()

            err = blocking_request.get(request)
            if err == QgsBlockingNetworkRequest.NoError:
                reply = blocking_request.reply()
                content = bytes(reply.content())
                if content and len(content) > 0:
                    with open(local_path, 'wb') as f:
                        f.write(content)
                    return local_path
        except Exception as e:
            self._log(f"Failed to download icon {icon_path}: {e}", Qgis.Warning)

        return None

    def _download_emblem(self, emblem_url: str) -> Optional[str]:
        """Download a unit emblem/patch and cache it locally.

        Args:
            emblem_url: Full URL to the emblem (from ORBAT patches/emblems field)

        Returns:
            Local file path to the cached emblem, or None if download failed
        """
        if not emblem_url:
            return None

        # Create a safe filename from the URL
        from urllib.parse import urlparse, quote
        parsed = urlparse(emblem_url)
        # Use the path portion for naming
        safe_name = parsed.path.replace('/', '_').replace('\\', '_').lstrip('_')
        if not safe_name:
            safe_name = 'emblem_' + str(hash(emblem_url)) + '.png'

        cache_dir = self._get_icon_cache_dir()
        local_path = os.path.join(cache_dir, 'emblems', safe_name)

        # Create emblems subdirectory
        os.makedirs(os.path.dirname(local_path), exist_ok=True)

        # Return cached version if it exists
        if os.path.exists(local_path):
            return local_path

        # Download the emblem
        try:
            from qgis.core import QgsBlockingNetworkRequest
            from qgis.PyQt.QtNetwork import QNetworkRequest
            from qgis.PyQt.QtCore import QUrl

            # Re-encode the URL path to handle special characters
            # The + signs in the path represent spaces (URL-encoded as %20 in paths)
            # First replace + with space, then properly encode the path
            decoded_path = parsed.path.replace('+', ' ')
            encoded_path = quote(decoded_path, safe='/')
            encoded_url = f"{parsed.scheme}://{parsed.netloc}{encoded_path}"
            if parsed.query:
                encoded_url += f"?{parsed.query}"

            self._log(f"Downloading emblem from {encoded_url}")
            request = QNetworkRequest(QUrl(encoded_url))
            blocking_request = QgsBlockingNetworkRequest()

            err = blocking_request.get(request)
            if err == QgsBlockingNetworkRequest.NoError:
                reply = blocking_request.reply()
                content = bytes(reply.content())
                if content and len(content) > 0:
                    with open(local_path, 'wb') as f:
                        f.write(content)
                    self._log(f"Cached emblem to {local_path}")
                    return local_path
            else:
                self._log(f"Failed to download emblem: error code {err}", Qgis.Warning)
        except Exception as e:
            self._log(f"Failed to download emblem {emblem_url}: {e}", Qgis.Warning)

        return None

    def apply_emblem_style(self, layer: QgsVectorLayer, emblem_url: str):
        """Apply a unit emblem as the marker symbol for all features.

        Args:
            layer: Layer to style
            emblem_url: URL to the emblem image
        """
        from qgis.core import QgsRasterMarkerSymbolLayer

        if not emblem_url:
            self._log("No emblem URL provided, using default style", Qgis.Warning)
            self._apply_simple_style(layer)
            return

        # Download and cache the emblem
        local_path = self._download_emblem(emblem_url)

        if local_path and os.path.exists(local_path):
            try:
                # Create marker symbol with the emblem image
                symbol = QgsMarkerSymbol()
                symbol.deleteSymbolLayer(0)  # Remove default layer

                raster_layer = QgsRasterMarkerSymbolLayer(local_path)
                raster_layer.setSize(8)  # Larger size for emblems (in mm)
                symbol.appendSymbolLayer(raster_layer)

                renderer = QgsSingleSymbolRenderer(symbol)
                layer.setRenderer(renderer)
                layer.triggerRepaint()

                self._log(f"Applied unit emblem style from {local_path}")
            except Exception as e:
                self._log(f"Error applying emblem style: {e}", Qgis.Warning)
                self._apply_simple_style(layer)
        else:
            self._log(f"Could not download emblem, using default style", Qgis.Warning)
            self._apply_simple_style(layer)

    def _apply_icon_categorized_style(self, layer: QgsVectorLayer):
        """Apply categorized style based on icon field, using actual PNG icons.

        Args:
            layer: Layer to style
        """
        from qgis.core import QgsRasterMarkerSymbolLayer

        categories = []

        # Get unique icon values from layer
        icons = set()
        for feature in layer.getFeatures():
            icon = feature['icon']
            if icon:
                icons.add(icon)

        self._log(f"Found {len(icons)} unique icon types, downloading...")

        # Fallback colors for icons that fail to download
        base_colors = [
            '#e74c3c', '#3498db', '#2ecc71', '#f39c12', '#9b59b6',
            '#1abc9c', '#e67e22', '#34495e', '#16a085', '#c0392b',
            '#2980b9', '#27ae60', '#d35400', '#8e44ad', '#17a2b8'
        ]

        # Create categories with actual icons
        for i, icon in enumerate(sorted(icons)):
            # Extract a short name from the icon path
            icon_name = icon.split('/')[-1].replace('.png', '').replace('_', ' ') if icon else 'Unknown'

            # Try to download and use the actual icon
            local_icon_path = self._download_icon(icon)

            if local_icon_path and os.path.exists(local_icon_path):
                # Use the actual PNG icon
                symbol = QgsMarkerSymbol()
                symbol.deleteSymbolLayer(0)  # Remove default layer

                raster_layer = QgsRasterMarkerSymbolLayer(local_icon_path)
                raster_layer.setSize(6)  # Size in mm
                symbol.appendSymbolLayer(raster_layer)
            else:
                # Fallback to colored circle
                color = base_colors[i % len(base_colors)]
                symbol = QgsMarkerSymbol.createSimple({
                    'name': 'circle',
                    'color': color,
                    'size': '3',
                    'outline_color': '#333333',
                    'outline_width': '0.4'
                })

            category = QgsRendererCategory(
                icon,
                symbol,
                icon_name[:30]  # Truncate long names
            )
            categories.append(category)

        # Add default category for unmatched
        default_symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '#bdc3c7',
            'size': '3',
            'outline_color': '#333333',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(
            None,
            default_symbol,
            'Other'
        ))

        renderer = QgsCategorizedSymbolRenderer('icon', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()

        self._log(f"Applied icon styling with {len(categories)} categories")

    def _apply_simple_style(self, layer: QgsVectorLayer):
        """Apply a simple marker style to a layer.

        Args:
            layer: Layer to style
        """
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '#3498db',
            'size': '3',
            'outline_color': '#2c3e50',
            'outline_width': '0.4'
        })

        renderer = QgsSingleSymbolRenderer(symbol)
        layer.setRenderer(renderer)
        layer.triggerRepaint()

    def _add_open_link_action(
        self,
        layer: QgsVectorLayer,
        is_placemarks: bool = True
    ):
        """Add an action to open source links in browser.

        Args:
            layer: Layer to add action to
            is_placemarks: Whether this is a placemark layer
        """
        if is_placemarks:
            # For placemarks, open GeoConfirmed page with coordinates
            action = QgsAction(
                QgsAction.GenericPython,
                "Open in GeoConfirmed",
                '''
import webbrowser
# Open GeoConfirmed at this location
lat = [%latitude%]
lon = [%longitude%]
webbrowser.open(f"https://geoconfirmed.org/ukraine?lat={lat}&lng={lon}&zoom=15")
''',
                "",
                False,
                "Open location in GeoConfirmed website",
                {"Feature", "Canvas"}
            )
        else:
            # For mentions, open the source link
            action = QgsAction(
                QgsAction.GenericPython,
                "Open Source",
                '''
import webbrowser
url = "[%link%]"
if url and url.strip():
    webbrowser.open(url)
''',
                "",
                False,
                "Open source link in browser",
                {"Feature", "Canvas"}
            )

        layer.actions().addAction(action)

    def enable_temporal(self, layer: QgsVectorLayer):
        """Enable temporal properties on a layer.

        Args:
            layer: Layer to configure
        """
        temporal_props = layer.temporalProperties()
        temporal_props.setIsActive(True)
        temporal_props.setMode(
            QgsVectorLayerTemporalProperties.ModeFeatureDateTimeInstantFromField
        )
        temporal_props.setStartField('date')

        # Set accumulate mode so features stay visible
        temporal_props.setAccumulateFeatures(True)

        self._log(f"Enabled temporal properties on layer {layer.name()}")

    def zoom_to_layer(self, layer: QgsVectorLayer):
        """Zoom the map canvas to a layer's extent.

        Args:
            layer: Layer to zoom to
        """
        if layer and layer.isValid() and layer.featureCount() > 0:
            self.iface.mapCanvas().setExtent(layer.extent())
            self.iface.mapCanvas().refresh()

    def get_latest_date_created(self, layer: QgsVectorLayer) -> Optional[str]:
        """Get the most recent date_created value from a layer.

        Args:
            layer: Vector layer to inspect

        Returns:
            ISO format datetime string of the most recent date_created,
            or None if no valid values found
        """
        if not layer or not layer.isValid():
            return None

        # Check if the layer has the date_created field
        field_idx = layer.fields().indexFromName('date_created')
        if field_idx < 0:
            self._log("Layer does not have 'date_created' field", Qgis.Warning)
            return None

        max_date = None
        for feature in layer.getFeatures():
            date_created = feature['date_created']
            if date_created and (max_date is None or str(date_created) > max_date):
                max_date = str(date_created)

        self._log(f"Latest date_created in layer: {max_date}")
        return max_date

    def get_geoconfirmed_layers(self) -> List[Tuple[QgsVectorLayer, str]]:
        """Find all GeoConfirmed layers in the current project.

        Identifies layers by checking for the 'geoconfirmed_conflict' custom property
        or by name pattern matching.

        Returns:
            List of tuples: (layer, conflict_name)
        """
        geoconfirmed_layers = []

        for layer in QgsProject.instance().mapLayers().values():
            if not isinstance(layer, QgsVectorLayer):
                continue

            # Check for custom property first (most reliable)
            conflict = layer.customProperty('geoconfirmed_conflict')
            if conflict:
                geoconfirmed_layers.append((layer, conflict))
                continue

            # Fallback: check layer name pattern
            if layer.name().startswith('GeoConfirmed -'):
                # Try to extract conflict name from layer name
                # Format: "GeoConfirmed - {conflict name}"
                parts = layer.name().split(' - ', 1)
                if len(parts) > 1:
                    # We don't know the shortName, so use the display name
                    geoconfirmed_layers.append((layer, parts[1]))

        self._log(f"Found {len(geoconfirmed_layers)} GeoConfirmed layers")
        return geoconfirmed_layers

    def append_features_to_layer(
        self,
        layer: QgsVectorLayer,
        items: List[Dict],
        is_placemarks: bool = True
    ) -> int:
        """Append new features to an existing layer.

        Works with both memory layers and file-based layers (GeoPackage).

        Args:
            layer: Existing vector layer to append to
            items: List of new data items from API
            is_placemarks: Whether these are placemarks

        Returns:
            Number of features successfully added
        """
        if not layer or not layer.isValid():
            self._log("Cannot append to invalid layer", Qgis.Critical)
            return 0

        if not items:
            self._log("No items to append")
            return 0

        # Create features from items
        layer_fields = layer.fields()
        features = []

        for item in items:
            feature = self._create_feature(item, layer_fields, is_placemarks)
            if feature:
                features.append(feature)

        if not features:
            self._log("No valid features created from items", Qgis.Warning)
            return 0

        self._log(f"Appending {len(features)} features to layer '{layer.name()}'")

        # Start editing (works for both memory and file-based layers)
        if not layer.isEditable():
            layer.startEditing()

        # Add features
        added_count = 0
        for feature in features:
            if layer.addFeature(feature):
                added_count += 1

        # Commit changes
        if layer.commitChanges():
            self._log(f"Successfully added {added_count} features")
            layer.updateExtents()
            layer.triggerRepaint()
        else:
            errors = layer.commitErrors()
            self._log(f"Failed to commit features: {errors}", Qgis.Warning)
            layer.rollBack()
            return 0

        return added_count

    def set_layer_conflict_property(self, layer: QgsVectorLayer, conflict_short_name: str):
        """Set the conflict custom property on a layer.

        This allows the plugin to identify GeoConfirmed layers and know
        which conflict they belong to.

        Args:
            layer: Vector layer to tag
            conflict_short_name: Short name of the conflict (e.g., 'Ukraine')
        """
        if layer and layer.isValid():
            layer.setCustomProperty('geoconfirmed_conflict', conflict_short_name)
            self._log(f"Set geoconfirmed_conflict property to '{conflict_short_name}' on layer '{layer.name()}'")
