# -*- coding: utf-8 -*-
"""
GeoConfirmed Dialog Controller
Handles UI logic and data fetching coordination.

Alpha version with faction filtering and statistics support.
"""

import os
from datetime import datetime, date
from typing import Optional, List, Dict

from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt, QDate, QThread, pyqtSignal
from qgis.PyQt.QtWidgets import (
    QDialog, QFileDialog, QMessageBox, QApplication, QCheckBox, QTreeWidgetItem
)
from qgis.PyQt.QtGui import QColor
from qgis.core import QgsProject, Qgis, QgsRectangle, QgsCoordinateReferenceSystem, QgsMapLayerProxyModel
from qgis.gui import QgsMapLayerComboBox

from .api.client import GeoConfirmedClient
from .utils.layer_manager import LayerManager

# Load the UI file
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'geoconfirmed_dialog_base.ui'))


class FetchWorker(QThread):
    """Worker thread for fetching data from API."""

    finished = pyqtSignal(list)
    error = pyqtSignal(str)
    progress = pyqtSignal(int, int)
    status_update = pyqtSignal(str)

    def __init__(
        self,
        client: GeoConfirmedClient,
        fetch_type: str,
        conflict: str,
        date_from: Optional[date] = None,
        date_to: Optional[date] = None,
        sources: Optional[List[str]] = None,
        filters: Optional[Dict] = None,
        keywords: Optional[str] = None,
        location: Optional[str] = None,
        orbat_node_ids: Optional[List[int]] = None
    ):
        super().__init__()
        self.client = client
        self.fetch_type = fetch_type
        self.conflict = conflict
        self.date_from = date_from
        self.date_to = date_to
        self.sources = sources
        self.filters = filters or {}
        self.keywords = keywords
        self.location = location
        self.orbat_node_ids = orbat_node_ids
        self._cancelled = False

    def run(self):
        """Execute the fetch operation."""
        try:
            if self.fetch_type == 'placemarks':
                items = self.client.get_all_placemarks(
                    self.conflict,
                    keywords=self.keywords,
                    location=self.location,
                    orbat_node_ids=self.orbat_node_ids,
                    progress_callback=self._on_progress
                )
            else:
                items = self.client.get_all_mentions(
                    conflict=self.conflict,
                    sources=self.sources,
                    doubt_filter=self.filters.get('doubt_filter'),
                    wrong_filter=self.filters.get('wrong_filter'),
                    processed_filter=self.filters.get('processed_filter'),
                    include_unmarked=self.filters.get('include_unmarked'),
                    progress_callback=self._on_progress
                )

            if self.date_from or self.date_to:
                items = self._filter_by_date(items)

            if not self._cancelled:
                self.finished.emit(items)

        except Exception as e:
            if not self._cancelled:
                self.error.emit(str(e))

    def _on_progress(self, current: int, total: int):
        if not self._cancelled:
            self.progress.emit(current, total)

    def _filter_by_date(self, items: List[Dict]) -> List[Dict]:
        filtered = []
        for item in items:
            item_date_str = item.get('date') or item.get('dateCreated')
            if not item_date_str:
                continue
            try:
                if 'T' in item_date_str:
                    item_date = datetime.fromisoformat(item_date_str.replace('Z', '+00:00')).date()
                else:
                    item_date = datetime.strptime(item_date_str[:10], '%Y-%m-%d').date()
                if self.date_from and item_date < self.date_from:
                    continue
                if self.date_to and item_date > self.date_to:
                    continue
                filtered.append(item)
            except (ValueError, TypeError):
                filtered.append(item)
        return filtered

    def cancel(self):
        self._cancelled = True
        self.client.cancel_pending_requests()


class UpdateWorker(QThread):
    """Worker thread for updating a layer with new placemarks."""

    finished = pyqtSignal(list)
    error = pyqtSignal(str)
    progress = pyqtSignal(int, int, int)
    status_update = pyqtSignal(str)

    def __init__(self, client: GeoConfirmedClient, conflict: str, since_date_created: str):
        super().__init__()
        self.client = client
        self.conflict = conflict
        self.since_date_created = since_date_created
        self._cancelled = False

    def run(self):
        try:
            self.status_update.emit(f"Fetching new placemarks since {self.since_date_created[:19]}...")
            items = self.client.get_new_placemarks(self.conflict, self.since_date_created, progress_callback=self._on_progress)
            if not self._cancelled:
                self.finished.emit(items)
        except Exception as e:
            if not self._cancelled:
                self.error.emit(str(e))

    def _on_progress(self, new_count: int, batch_num: int, total_server: int):
        if not self._cancelled:
            self.progress.emit(new_count, batch_num, total_server)
            self.status_update.emit(f"Found {new_count} new placemarks (batch {batch_num})...")

    def cancel(self):
        self._cancelled = True
        self.client.cancel_pending_requests()


class GeoConfirmedDialog(QDialog, FORM_CLASS):
    """Dialog for querying GeoConfirmed data."""

    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.setupUi(self)

        self.iface = iface
        self.client = GeoConfirmedClient(self)
        self.layer_manager = LayerManager(iface)
        self.worker: Optional[FetchWorker] = None
        self.update_worker: Optional[UpdateWorker] = None
        self.conflicts: List[Dict] = []
        self.current_conflict_details: Optional[Dict] = None
        self.faction_checkboxes: List[QCheckBox] = []
        self._pending_search_query: Optional[str] = None
        self._update_target_layer = None
        self._orbat_cache: Dict[str, List[Dict]] = {}
        self._orbat_node_map: Dict[int, QTreeWidgetItem] = {}
        self._orbat_id_to_name: Dict[int, str] = {}
        self._orbat_id_to_emblem: Dict[int, str] = {}

        self._setup_connections()
        self._setup_defaults()
        self._load_conflicts()
        self._populate_existing_layers()

    def _setup_connections(self):
        self.btnRefreshConflicts.clicked.connect(self._load_conflicts)
        self.btnFetch.clicked.connect(self._start_fetch)
        self.btnCancel.clicked.connect(self._cancel_fetch)
        self.btnBrowseOutput.clicked.connect(self._browse_output)
        self.btnUpdate.clicked.connect(self._start_update)
        self.btnRefreshLayers.clicked.connect(self._populate_existing_layers)
        self.comboExistingLayers.currentIndexChanged.connect(self._on_existing_layer_changed)
        self.btnSelectAllFactions.clicked.connect(self._select_all_factions)
        self.btnSelectNoneFactions.clicked.connect(self._select_no_factions)
        self.comboConflict.currentIndexChanged.connect(self._on_conflict_changed)
        self.client.error_occurred.connect(self._on_error)
        self.lineUnitSearch.textChanged.connect(self._filter_orbat_tree)
        self.btnClearUnitSelection.clicked.connect(self._clear_unit_selection)
        self.treeUnits.itemChanged.connect(self._on_unit_item_changed)
        self._setup_spatial_filter()
        # Connect collapsible sections to resize dialog
        self.groupBoxSearch.collapsedStateChanged.connect(self._on_section_collapsed)
        self.groupBoxUnits.collapsedStateChanged.connect(self._on_section_collapsed)
        self.groupBoxSpatial.collapsedStateChanged.connect(self._on_section_collapsed)
        self.groupBoxFactions.collapsedStateChanged.connect(self._on_section_collapsed)
        self.groupBoxStatistics.collapsedStateChanged.connect(self._on_section_collapsed)

    def _setup_defaults(self):
        today = QDate.currentDate()
        self.dateTo.setDate(today)
        self.dateFrom.setDate(today.addMonths(-1))

    def _setup_spatial_filter(self):
        layout = self.gridLayoutSpatial
        original_combo = self.comboSpatialLayer
        self.spatialLayerCombo = QgsMapLayerComboBox(self)
        self.spatialLayerCombo.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.spatialLayerCombo.setToolTip("Select reference layer for spatial filter")
        self.spatialLayerCombo.setSizePolicy(original_combo.sizePolicy())
        layout.replaceWidget(original_combo, self.spatialLayerCombo)
        original_combo.hide()
        original_combo.deleteLater()
        self.comboSpatialType.currentIndexChanged.connect(self._on_spatial_type_changed)
        self._on_spatial_type_changed()

    def _on_spatial_type_changed(self):
        filter_type = self.comboSpatialType.currentText()
        show_layer = filter_type in ("Around", "Within Layer")
        self.labelSpatialLayer.setVisible(show_layer)
        self.spatialLayerCombo.setVisible(show_layer)
        show_radius = filter_type == "Around"
        self.labelRadius.setVisible(show_radius)
        self.spinRadius.setVisible(show_radius)

    def _get_spatial_filter_params(self) -> dict:
        filter_text = self.comboSpatialType.currentText()
        type_map = {"Not Spatial": "none", "Around": "around", "Canvas Extent": "canvas", "Within Layer": "within"}
        filter_type = type_map.get(filter_text, "none")
        params = {'filter_type': filter_type, 'reference_layer': None, 'radius_meters': 0, 'canvas_extent': None, 'canvas_crs': None}
        if filter_type == "around":
            params['reference_layer'] = self.spatialLayerCombo.currentLayer()
            params['radius_meters'] = self.spinRadius.value()
        elif filter_type == "within":
            params['reference_layer'] = self.spatialLayerCombo.currentLayer()
        elif filter_type == "canvas":
            canvas = self.iface.mapCanvas()
            params['canvas_extent'] = canvas.extent()
            params['canvas_crs'] = canvas.mapSettings().destinationCrs()
        return params

    def _load_conflicts(self):
        self.labelStatus.setText("Loading conflicts...")
        self.btnRefreshConflicts.setEnabled(False)
        QApplication.processEvents()
        try:
            conflicts = self.client.get_conflicts()
            if conflicts:
                self.conflicts = conflicts
                self._populate_conflicts()
                self.labelStatus.setText(f"Loaded {len(conflicts)} conflicts")
            else:
                self.labelStatus.setText("No conflicts found")
        except Exception as e:
            self._on_error(f"Failed to load conflicts: {str(e)}")
        finally:
            self.btnRefreshConflicts.setEnabled(True)

    def _populate_conflicts(self):
        self.comboConflict.clear()
        self.comboConflict.addItem("-- Select a conflict --", None)
        for conflict in self.conflicts:
            name = conflict.get('name', 'Unknown')
            short_name = conflict.get('shortName', '')
            display_name = f"{name} ({short_name})" if short_name else name
            self.comboConflict.addItem(display_name, conflict)

    def _on_conflict_changed(self, index: int):
        conflict = self.comboConflict.currentData()
        if conflict:
            start_date = conflict.get('startDate')
            if start_date:
                try:
                    qdate = QDate.fromString(start_date[:10], 'yyyy-MM-dd')
                    if qdate.isValid():
                        self.dateFrom.setDate(qdate)
                except Exception:
                    pass
            self._load_conflict_details(conflict.get('shortName', ''))
            self._load_orbat_for_conflict(conflict.get('shortName', ''))

    def _load_conflict_details(self, conflict_name: str):
        if not conflict_name:
            return
        self.labelStatus.setText(f"Loading factions for {conflict_name}...")
        QApplication.processEvents()
        try:
            details = self.client.get_conflict_details(conflict_name)
            if details:
                self.current_conflict_details = details
                factions = details.get('factions', [])
                self.layer_manager.set_faction_lookup(factions)
                self._populate_faction_checkboxes(factions)
                self.labelStatus.setText(f"Loaded {len(factions)} factions for {conflict_name}")
        except Exception as e:
            self.labelStatus.setText(f"Failed to load factions: {str(e)}")

    def _load_orbat_for_conflict(self, conflict_name: str):
        if not conflict_name:
            self.treeUnits.clear()
            self._orbat_node_map.clear()
            return
        if conflict_name in self._orbat_cache:
            self._populate_orbat_tree(self._orbat_cache[conflict_name])
            return
        if not self.current_conflict_details:
            return
        factions = self.current_conflict_details.get('factions', [])
        if not factions:
            return
        self.labelStatus.setText(f"Loading ORBAT for {conflict_name}...")
        QApplication.processEvents()
        try:
            all_hierarchies = []
            for faction in factions:
                orbat_ids = faction.get('orbatIds', [])
                if orbat_ids:
                    hierarchies = self.client.get_faction_orbat_hierarchy(orbat_ids)
                    all_hierarchies.extend(hierarchies)
            self._orbat_cache[conflict_name] = all_hierarchies
            self._populate_orbat_tree(all_hierarchies)
            self.labelStatus.setText(f"Loaded ORBAT for {conflict_name}")
        except Exception as e:
            self.labelStatus.setText(f"Failed to load ORBAT: {str(e)}")

    def _populate_orbat_tree(self, hierarchies: List[Dict]):
        self.treeUnits.clear()
        self._orbat_node_map.clear()
        self._orbat_id_to_name.clear()
        self._orbat_id_to_emblem.clear()
        self.treeUnits.blockSignals(True)
        for hierarchy in hierarchies:
            self._add_orbat_node_to_tree(hierarchy, None, [])
        self.treeUnits.blockSignals(False)

    def _add_orbat_node_to_tree(self, node: Dict, parent_item: Optional[QTreeWidgetItem], parent_path: List[str]):
        from urllib.parse import unquote
        name = node.get('name', 'Unknown')
        node_id = node.get('id')
        if parent_item is None:
            item = QTreeWidgetItem(self.treeUnits)
        else:
            item = QTreeWidgetItem(parent_item)
        item.setText(0, name)
        item.setData(0, Qt.UserRole, node_id)
        item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
        item.setCheckState(0, Qt.Unchecked)
        if node_id is not None:
            self._orbat_node_map[node_id] = item
            current_path = parent_path + [name]
            self._orbat_id_to_name[node_id] = name
            emblems = node.get('emblems', [])
            patches = node.get('patches', [])
            tactical_markings = node.get('tacticalMarkings', [])
            emblem_url = None
            if emblems and len(emblems) > 0:
                emblem_url = unquote(emblems[0])
            elif patches and len(patches) > 0:
                emblem_url = unquote(patches[0])
            elif tactical_markings and len(tactical_markings) > 0:
                emblem_url = unquote(tactical_markings[0])
            if emblem_url:
                self._orbat_id_to_emblem[node_id] = emblem_url
        children = node.get('children', [])
        current_path = parent_path + [name]
        for child in children:
            self._add_orbat_node_to_tree(child, item, current_path)

    def _filter_orbat_tree(self, search_text: str):
        search_lower = search_text.lower().strip()
        def set_item_visible(item: QTreeWidgetItem, visible: bool):
            item.setHidden(not visible)
        def filter_item(item: QTreeWidgetItem) -> bool:
            item_text = item.text(0).lower()
            item_matches = search_lower in item_text if search_lower else True
            child_matches = False
            for i in range(item.childCount()):
                child = item.child(i)
                if filter_item(child):
                    child_matches = True
            visible = item_matches or child_matches
            set_item_visible(item, visible)
            if child_matches and search_lower:
                item.setExpanded(True)
            return visible
        for i in range(self.treeUnits.topLevelItemCount()):
            item = self.treeUnits.topLevelItem(i)
            filter_item(item)

    def _clear_unit_selection(self):
        self.treeUnits.blockSignals(True)
        def clear_item(item: QTreeWidgetItem):
            item.setCheckState(0, Qt.Unchecked)
            for i in range(item.childCount()):
                clear_item(item.child(i))
        for i in range(self.treeUnits.topLevelItemCount()):
            clear_item(self.treeUnits.topLevelItem(i))
        self.treeUnits.blockSignals(False)
        self.lineUnitSearch.clear()

    def _on_unit_item_changed(self, item: QTreeWidgetItem, column: int):
        pass

    def _get_selected_orbat_node_ids(self) -> List[int]:
        selected_ids = []
        def collect_checked(item: QTreeWidgetItem):
            if item.checkState(0) == Qt.Checked:
                node_id = item.data(0, Qt.UserRole)
                if node_id is not None:
                    selected_ids.append(node_id)
            for i in range(item.childCount()):
                collect_checked(item.child(i))
        for i in range(self.treeUnits.topLevelItemCount()):
            collect_checked(self.treeUnits.topLevelItem(i))
        return selected_ids

    def _get_selected_emblem_url(self) -> Optional[str]:
        selected_ids = self._get_selected_orbat_node_ids()
        for node_id in selected_ids:
            emblem_url = self._orbat_id_to_emblem.get(node_id)
            if emblem_url:
                return emblem_url
        return None

    def _populate_faction_checkboxes(self, factions: List[Dict]):
        for cb in self.faction_checkboxes:
            cb.deleteLater()
        self.faction_checkboxes.clear()
        layout = self.gridLayoutFactions
        while layout.count():
            item = layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()
        row = 0
        col = 0
        for faction in factions:
            name = faction.get('name', 'Unknown')
            color = faction.get('color', '#666666')
            cb = QCheckBox(name)
            cb.setChecked(True)
            cb.setStyleSheet(f"QCheckBox {{ padding-left: 5px; }} QCheckBox::indicator:checked {{ background-color: {color}; border: 1px solid #333; }}")
            cb.setProperty('faction_name', name)
            cb.setProperty('faction_color', color)
            layout.addWidget(cb, row, col)
            self.faction_checkboxes.append(cb)
            col += 1
            if col >= 2:
                col = 0
                row += 1

    def _select_all_factions(self):
        for cb in self.faction_checkboxes:
            cb.setChecked(True)

    def _select_no_factions(self):
        for cb in self.faction_checkboxes:
            cb.setChecked(False)

    def _get_selected_factions(self) -> List[str]:
        if not not self.groupBoxFactions.isCollapsed():
            return []
        selected = []
        for cb in self.faction_checkboxes:
            if cb.isChecked():
                selected.append(cb.property('faction_name'))
        return selected

    def _browse_output(self):
        conflict = self.comboConflict.currentData()
        default_name = f"geoconfirmed_{conflict.get('shortName', 'data') if conflict else 'data'}.gpkg"
        file_path, _ = QFileDialog.getSaveFileName(self, "Save GeoPackage", default_name, "GeoPackage (*.gpkg)")
        if file_path:
            if not file_path.endswith('.gpkg'):
                file_path += '.gpkg'
            self.lineOutputPath.setText(file_path)

    def _populate_existing_layers(self):
        self.comboExistingLayers.clear()
        self.comboExistingLayers.addItem("-- Select a layer --", None)
        layers = self.layer_manager.get_geoconfirmed_layers()
        for layer, conflict in layers:
            display_name = f"{layer.name()} ({layer.featureCount()} features)"
            self.comboExistingLayers.addItem(display_name, (layer, conflict))
        has_layers = len(layers) > 0
        self.btnUpdate.setEnabled(has_layers and self.comboExistingLayers.currentIndex() > 0)

    def _on_existing_layer_changed(self, index: int):
        data = self.comboExistingLayers.currentData()
        self.btnUpdate.setEnabled(data is not None)

    def _start_update(self):
        data = self.comboExistingLayers.currentData()
        if not data:
            QMessageBox.warning(self, "No Layer Selected", "Please select a GeoConfirmed layer to update.")
            return
        layer, conflict = data
        if not layer or not layer.isValid():
            QMessageBox.warning(self, "Invalid Layer", "The selected layer is no longer valid. Please refresh the layer list.")
            self._populate_existing_layers()
            return
        latest_date = self.layer_manager.get_latest_date_created(layer)
        if not latest_date:
            QMessageBox.warning(self, "Cannot Determine Date", "Could not determine the most recent date from the layer.")
            return
        conflict_short_name = layer.customProperty('geoconfirmed_conflict')
        if not conflict_short_name:
            for c in self.conflicts:
                if c.get('name') == conflict or c.get('shortName') == conflict:
                    conflict_short_name = c.get('shortName')
                    break
        if not conflict_short_name:
            conflict_short_name = conflict
        self._update_target_layer = layer
        self._set_fetching_state(True)
        self.progressBar.setMaximum(0)
        self.labelStatus.setText(f"Checking for new placemarks since {latest_date[:19]}...")
        self.client.reset_cancelled()
        self.update_worker = UpdateWorker(self.client, conflict_short_name, latest_date)
        self.update_worker.progress.connect(self._on_update_progress)
        self.update_worker.finished.connect(self._on_update_finished)
        self.update_worker.error.connect(self._on_error)
        self.update_worker.status_update.connect(self._on_status_update)
        self.update_worker.start()

    def _on_update_progress(self, new_count: int, batch_num: int, total_server: int):
        self.progressBar.setMaximum(0)
        self.labelStatus.setText(f"Found {new_count} new placemarks (checking batch {batch_num})...")

    def _on_update_finished(self, items: List[Dict]):
        self._set_fetching_state(False)
        layer = self._update_target_layer
        self._update_target_layer = None
        if not items:
            self.labelStatus.setText("Layer is up to date")
            self.progressBar.setValue(0)
            QMessageBox.information(self, "Up to Date", "No new placemarks found. The layer is already up to date.")
            return
        if not layer or not layer.isValid():
            self._on_error("Target layer is no longer valid")
            return
        self.labelStatus.setText(f"Adding {len(items)} new features to layer...")
        QApplication.processEvents()
        try:
            added_count = self.layer_manager.append_features_to_layer(layer, items, is_placemarks=True)
            if added_count > 0:
                self.labelStatus.setText(f"Added {added_count} new features to '{layer.name()}'")
                self.iface.messageBar().pushMessage("GeoConfirmed", f"Added {added_count} new features to layer", level=Qgis.Success, duration=5)
                self._populate_existing_layers()
            else:
                self.labelStatus.setText("Failed to add features to layer")
                QMessageBox.warning(self, "Update Failed", "Failed to add new features to the layer.")
        except Exception as e:
            self._on_error(f"Failed to update layer: {str(e)}")
        self.progressBar.setValue(self.progressBar.maximum() if self.progressBar.maximum() > 0 else 100)

    def _start_fetch(self):
        conflict = self.comboConflict.currentData()
        if not conflict:
            QMessageBox.warning(self, "No Conflict Selected", "Please select a conflict to query.")
            return
        if self.radioGeoPackage.isChecked() and not self.lineOutputPath.text():
            QMessageBox.warning(self, "No Output File", "Please select an output file for the GeoPackage.")
            return
        spatial_type = self.comboSpatialType.currentText()
        if spatial_type == "Around":
            if not self.spatialLayerCombo.currentLayer():
                QMessageBox.warning(self, "Validation Error", "Please select a reference layer for 'Around' spatial filter.")
                return
            if self.spinRadius.value() <= 0:
                QMessageBox.warning(self, "Validation Error", "Radius must be greater than 0 for 'Around' spatial filter.")
                return
        elif spatial_type == "Within Layer":
            if not self.spatialLayerCombo.currentLayer():
                QMessageBox.warning(self, "Validation Error", "Please select a reference layer for 'Within Layer' spatial filter.")
                return
        conflict_name = conflict.get('shortName', '')
        keywords_query = self.lineSearch.text().strip() or None
        location_query = self.lineLocation.text().strip() if hasattr(self, 'lineLocation') else None
        orbat_node_ids = None
        if not self.groupBoxUnits.isCollapsed():
            selected_units = self._get_selected_orbat_node_ids()
            if selected_units:
                orbat_node_ids = selected_units
        self._pending_keywords = keywords_query
        self._pending_location = location_query
        self._pending_orbat_units = orbat_node_ids
        date_from = None
        date_to = None
        if not self.checkAllData.isChecked():
            date_from = self.dateFrom.date().toPyDate()
            date_to = self.dateTo.date().toPyDate()
        self._set_fetching_state(True)
        self.progressBar.setMaximum(0)
        filter_parts = []
        if keywords_query:
            filter_parts.append(f"keywords='{keywords_query}'")
        if location_query:
            filter_parts.append(f"location='{location_query}'")
        if orbat_node_ids:
            filter_parts.append(f"units={len(orbat_node_ids)}")
        if filter_parts:
            self.labelStatus.setText(f"Fetching placemarks with {', '.join(filter_parts)}...")
        else:
            self.labelStatus.setText("Fetching all placemarks...")
        self.worker = FetchWorker(self.client, 'placemarks', conflict_name, date_from=date_from, date_to=date_to, keywords=keywords_query, location=location_query, orbat_node_ids=orbat_node_ids)
        self.worker.progress.connect(self._on_progress)
        self.worker.finished.connect(self._on_fetch_finished)
        self.worker.error.connect(self._on_error)
        self.worker.status_update.connect(self._on_status_update)
        self.worker.start()

    def _cancel_fetch(self):
        if self.worker and self.worker.isRunning():
            self.worker.cancel()
            self.worker.wait()
            self.worker = None
        if self.update_worker and self.update_worker.isRunning():
            self.update_worker.cancel()
            self.update_worker.wait()
            self.update_worker = None
        self._update_target_layer = None
        self._set_fetching_state(False)
        self.labelStatus.setText("Cancelled")
        self.progressBar.setValue(0)

    def _set_fetching_state(self, fetching: bool):
        self.btnFetch.setEnabled(not fetching)
        self.btnCancel.setEnabled(fetching)
        self.comboConflict.setEnabled(not fetching)
        self.groupBoxTimeframe.setEnabled(not fetching)
        self.groupBoxSearch.setEnabled(not fetching)
        self.groupBoxSpatial.setEnabled(not fetching)
        self.groupBoxFactions.setEnabled(not fetching)
        self.groupBoxOutput.setEnabled(not fetching)
        self.groupBoxUpdate.setEnabled(not fetching)
        if not fetching:
            data = self.comboExistingLayers.currentData()
            self.btnUpdate.setEnabled(data is not None)

    def _on_progress(self, current: int, total: int):
        self.progressBar.setMaximum(total)
        self.progressBar.setValue(current)
        self.labelStatus.setText(f"Fetching... {current} / {total}")

    def _on_status_update(self, message: str):
        self.labelStatus.setText(message)

    def _on_fetch_finished(self, items: List[Dict]):
        self._set_fetching_state(False)
        if not items:
            self.labelStatus.setText("No data found")
            QMessageBox.information(self, "No Data", "No data was found matching your criteria.")
            return
        original_count = len(items)
        keywords = getattr(self, '_pending_keywords', None)
        location = getattr(self, '_pending_location', None)
        if keywords or location:
            filter_parts = []
            if keywords:
                filter_parts.append(f"keywords='{keywords}'")
            if location:
                filter_parts.append(f"location='{location}'")
            self.labelStatus.setText(f"Server-side filter ({', '.join(filter_parts)}): {len(items)} items found")
        spatial_params = self._get_spatial_filter_params()
        if spatial_params.get('filter_type') != 'none':
            pre_spatial_count = len(items)
            self.labelStatus.setText(f"Applying spatial filter to {len(items)} items...")
            QApplication.processEvents()
            items = self.layer_manager.filter_items_by_spatial(items, **spatial_params)
            if not items:
                self.labelStatus.setText("No data matches spatial filter")
                QMessageBox.information(self, "No Data", "No data matches the spatial filter criteria.")
                return
            self.labelStatus.setText(f"Spatial filter: {len(items)} of {pre_spatial_count} items match")
        pre_faction_count = len(items)
        selected_factions = self._get_selected_factions()
        if not self.groupBoxFactions.isCollapsed() and selected_factions:
            items = self.layer_manager.filter_items_by_factions(items, selected_factions)
            self.labelStatus.setText(f"Filtered to {len(items)} of {pre_faction_count} features...")
            if not items:
                self.labelStatus.setText("No data after faction filter")
                QMessageBox.information(self, "No Data", "No data matches the selected faction filters.")
                return
        else:
            self.labelStatus.setText(f"Processing {len(items)} features...")
        QApplication.processEvents()
        if not self.groupBoxStatistics.isCollapsed():
            self._display_statistics(items)
        try:
            conflict = self.comboConflict.currentData()
            layer_name = f"GeoConfirmed - {conflict.get('name', 'Data')}"
            if self.radioGeoPackage.isChecked():
                output_path = self.lineOutputPath.text()
                layer = self.layer_manager.create_geopackage_layer(items, output_path, layer_name, is_placemarks=True, orbat_lookup=self._orbat_id_to_name)
            else:
                layer = self.layer_manager.create_memory_layer(items, layer_name, is_placemarks=True, orbat_lookup=self._orbat_id_to_name)
            if layer and layer.isValid():
                conflict_short_name = conflict.get('shortName', '')
                if conflict_short_name:
                    self.layer_manager.set_layer_conflict_property(layer, conflict_short_name)
                if self.checkApplyStyling.isChecked():
                    use_emblems = self.checkUseUnitEmblems.isChecked()
                    emblem_url = self._get_selected_emblem_url() if use_emblems else None
                    if use_emblems and emblem_url:
                        self.layer_manager.apply_emblem_style(layer, emblem_url)
                    elif use_emblems and not emblem_url:
                        QMessageBox.warning(self, "No Unit Emblem Available", "No emblem available for selected unit(s). Using default styling.")
                        self.layer_manager.apply_categorized_style(layer, is_placemarks=True)
                    else:
                        self.layer_manager.apply_categorized_style(layer, is_placemarks=True)
                if self.checkEnableTemporal.isChecked():
                    self.layer_manager.enable_temporal(layer)
                QgsProject.instance().addMapLayer(layer)
                self._zoom_to_layer(layer)
                self._populate_existing_layers()
                self.labelStatus.setText(f"Loaded {len(items)} features to '{layer_name}'")
                self.iface.messageBar().pushMessage("GeoConfirmed", f"Loaded {len(items)} features", level=Qgis.Success, duration=5)
            else:
                raise Exception("Failed to create layer")
        except Exception as e:
            self._on_error(f"Failed to create layer: {str(e)}")
        self.progressBar.setValue(self.progressBar.maximum())

    def _display_statistics(self, items: List[Dict]):
        stats = self.layer_manager.compute_statistics(items)
        html = []
        html.append(f"<b>Total Events:</b> {stats['total']}<br><br>")
        html.append("<b>By Status:</b><br>")
        html.append(f"&nbsp;&nbsp;Active: {stats['by_status']['active']}<br>")
        html.append(f"&nbsp;&nbsp;Destroyed: {stats['by_status']['destroyed']}<br><br>")
        html.append("<b>By Faction:</b><br>")
        sorted_factions = sorted(stats['by_faction'].items(), key=lambda x: x[1], reverse=True)
        for faction, count in sorted_factions[:10]:
            pct = (count / stats['total'] * 100) if stats['total'] > 0 else 0
            html.append(f"&nbsp;&nbsp;{faction}: {count} ({pct:.1f}%)<br>")
        html.append("<br><b>By Equipment Type:</b><br>")
        sorted_equipment = sorted(stats['by_equipment'].items(), key=lambda x: x[1], reverse=True)
        for equip, count in sorted_equipment[:10]:
            pct = (count / stats['total'] * 100) if stats['total'] > 0 else 0
            html.append(f"&nbsp;&nbsp;{equip}: {count} ({pct:.1f}%)<br>")
        self.textStatistics.setHtml(''.join(html))

    def _on_error(self, message: str):
        self._set_fetching_state(False)
        self.labelStatus.setText(f"Error: {message}")
        self.iface.messageBar().pushMessage("GeoConfirmed", message, level=Qgis.Warning, duration=10)

    def _zoom_to_layer(self, layer):
        from qgis.core import QgsCoordinateTransform
        if not layer or not layer.isValid() or layer.featureCount() == 0:
            return
        canvas = self.iface.mapCanvas()
        layer_extent = layer.extent()
        if layer.crs() != canvas.mapSettings().destinationCrs():
            transform = QgsCoordinateTransform(layer.crs(), canvas.mapSettings().destinationCrs(), QgsProject.instance())
            layer_extent = transform.transformBoundingBox(layer_extent)
        layer_extent.scale(1.1)
        canvas.setExtent(layer_extent)
        canvas.refresh()

    def closeEvent(self, event):
        self._cancel_fetch()
        super().closeEvent(event)

    def showEvent(self, event):
        """Resize dialog to fit content when shown."""
        super().showEvent(event)
        # Use timer to allow layout to settle before adjusting size
        from qgis.PyQt.QtCore import QTimer
        QTimer.singleShot(0, self._adjust_dialog_size)

    def _on_section_collapsed(self, collapsed: bool):
        """Resize dialog when a collapsible section changes state."""
        from qgis.PyQt.QtCore import QTimer
        QTimer.singleShot(0, self._adjust_dialog_size)

    def _adjust_dialog_size(self):
        """Adjust dialog size to fit content."""
        self.adjustSize()
