# -*- coding: utf-8 -*-
"""
GeoConfirmed API Client
Handles all communication with the GeoConfirmed API using QGIS network manager.
"""

import json
from typing import Optional, List, Dict, Any, Callable
from urllib.parse import urlencode, quote

from qgis.PyQt.QtCore import QObject, pyqtSignal, QUrl
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkReply
from qgis.core import QgsNetworkAccessManager, QgsMessageLog, Qgis, QgsBlockingNetworkRequest


class GeoConfirmedClient(QObject):
    """Client for the GeoConfirmed API."""

    BASE_URL = "https://geoconfirmed.org/api"

    # Signals
    conflicts_loaded = pyqtSignal(list)
    conflict_details_loaded = pyqtSignal(dict)
    placemarks_loaded = pyqtSignal(list, int)  # items, total_count
    mentions_loaded = pyqtSignal(list, int)  # items, total_count
    error_occurred = pyqtSignal(str)
    progress_updated = pyqtSignal(int, int)  # current, total

    def __init__(self, parent: Optional[QObject] = None):
        """Initialize the API client.

        Args:
            parent: Parent QObject
        """
        super().__init__(parent)
        self._cancelled = False

    def _log(self, message: str, level: Qgis.MessageLevel = Qgis.Info):
        """Log a message to QGIS message log.

        Args:
            message: Message to log
            level: Log level
        """
        QgsMessageLog.logMessage(message, "GeoConfirmed", level)

    def _make_request(self, url: str) -> QNetworkRequest:
        """Create a network request with proper headers.

        Args:
            url: URL to request

        Returns:
            Configured QNetworkRequest
        """
        request = QNetworkRequest(QUrl(url))
        request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
        request.setRawHeader(b"Accept", b"application/json")
        request.setRawHeader(b"User-Agent", b"QGIS-GeoConfirmed-Plugin/1.0")
        return request

    def _blocking_get(self, url: str) -> Optional[Dict[str, Any]]:
        """Perform a blocking GET request.

        Args:
            url: URL to fetch

        Returns:
            Parsed JSON response or None
        """
        request = self._make_request(url)
        blocking_request = QgsBlockingNetworkRequest()

        err = blocking_request.get(request)

        if err != QgsBlockingNetworkRequest.NoError:
            error_msg = f"Network error: {blocking_request.errorMessage()}"
            self._log(error_msg, Qgis.Warning)
            self.error_occurred.emit(error_msg)
            return None

        reply = blocking_request.reply()
        try:
            data = bytes(reply.content()).decode('utf-8')
            return json.loads(data)
        except json.JSONDecodeError as e:
            error_msg = f"JSON parse error: {str(e)}"
            self._log(error_msg, Qgis.Warning)
            self.error_occurred.emit(error_msg)
            return None

    def _blocking_post(self, url: str, body: Optional[Dict] = None) -> Optional[Dict[str, Any]]:
        """Perform a blocking POST request.

        Args:
            url: URL to post to
            body: Optional JSON body to send

        Returns:
            Parsed JSON response or None
        """
        request = self._make_request(url)
        blocking_request = QgsBlockingNetworkRequest()

        # Prepare body data
        body_data = json.dumps(body or {}).encode('utf-8')

        err = blocking_request.post(request, body_data)

        if err != QgsBlockingNetworkRequest.NoError:
            error_msg = f"Network error: {blocking_request.errorMessage()}"
            self._log(error_msg, Qgis.Warning)
            self.error_occurred.emit(error_msg)
            return None

        reply = blocking_request.reply()
        try:
            data = bytes(reply.content()).decode('utf-8')
            return json.loads(data)
        except json.JSONDecodeError as e:
            error_msg = f"JSON parse error: {str(e)}"
            self._log(error_msg, Qgis.Warning)
            self.error_occurred.emit(error_msg)
            return None

    def get_conflicts(self, callback: Optional[Callable[[List[Dict]], None]] = None):
        """Fetch list of all available conflicts.

        Args:
            callback: Optional callback function receiving list of conflicts
        """
        url = f"{self.BASE_URL}/Conflict"
        self._log(f"Fetching conflicts from {url}")

        data = self._blocking_get(url)
        if data is not None:
            conflicts = data if isinstance(data, list) else []
            self._log(f"Loaded {len(conflicts)} conflicts")
            self.conflicts_loaded.emit(conflicts)
            if callback:
                callback(conflicts)
        return data

    def get_conflict_details(self, conflict_name: str,
                             callback: Optional[Callable[[Dict], None]] = None):
        """Fetch details for a specific conflict including factions.

        Args:
            conflict_name: Short name of the conflict
            callback: Optional callback function receiving conflict details
        """
        url = f"{self.BASE_URL}/Conflict/{quote(conflict_name)}"
        self._log(f"Fetching conflict details from {url}")

        data = self._blocking_get(url)
        if data is not None:
            self.conflict_details_loaded.emit(data)
            if callback:
                callback(data)
        return data

    def _build_search_filter(
        self,
        keywords: Optional[str] = None,
        location: Optional[str] = None,
        orbat_node_ids: Optional[List[int]] = None
    ) -> Dict:
        """Build a filter body for server-side search.

        Args:
            keywords: Search keywords (searches Description, Name, Gear)
            location: Location filter - either coordinates (e.g., "50.123,30.456")
                     or text search (city/region/country via PlusCode field)
            orbat_node_ids: List of ORBAT node IDs to filter by units

        Returns:
            Filter body dict for the v2 API
        """
        rules = []

        # Keyword search across Description, Name, Gear (OR within keywords)
        # Note: Units is handled separately via ORBAT tree selection
        if keywords:
            keyword_rules = [
                {
                    "field": {"name": "Description"},
                    "operator": "LIKE",
                    "valueType": "System.String",
                    "value": keywords
                },
                {
                    "field": {"name": "Name"},
                    "operator": "LIKE",
                    "valueType": "System.String",
                    "value": keywords
                },
                {
                    "field": {"name": "Gear"},
                    "operator": "LIKE",
                    "valueType": "System.String",
                    "value": keywords
                }
            ]
            # If only keywords (no location, no orbat), add as OR rules directly
            # Otherwise wrap in OR group so it can be ANDed with other filters
            if not location and not orbat_node_ids:
                rules.extend(keyword_rules)
            else:
                # Wrap keyword rules in an OR group
                rules.append({
                    "rules": keyword_rules,
                    "operator": "Or"
                })

        # Location search - detect if coordinates or text
        if location:
            # Check if location looks like coordinates (e.g., "50.123,30.456")
            location_stripped = location.strip()
            is_coordinates = False
            if ',' in location_stripped:
                parts = location_stripped.split(',')
                if len(parts) == 2:
                    try:
                        float(parts[0].strip())
                        float(parts[1].strip())
                        is_coordinates = True
                    except ValueError:
                        pass

            if is_coordinates:
                # Use Location field with = operator for coordinate search
                location_rule = {
                    "field": {"name": "Location"},
                    "operator": "=",
                    "valueType": "System.String",
                    "value": location_stripped
                }
            else:
                # Use PlusCode field with LIKE for text search (city/region)
                location_rule = {
                    "field": {"name": "PlusCode"},
                    "operator": "LIKE",
                    "valueType": "System.String",
                    "value": location
                }

            if not keywords:
                rules.append(location_rule)
            else:
                # Location is ANDed with keywords group
                rules.append(location_rule)

        # ORBAT unit filter
        if orbat_node_ids:
            orbat_rule = {
                "field": {"name": "OrbatNodes"},
                "operator": "IN",
                "valueType": "Geoconfirmed.Shared.QueryBuilder.QueryCondition+IntList, Geoconfirmed.Shared, Version=1.0.0.0, Culture=neutral, PublicKeyToken=null",
                "value": orbat_node_ids
            }
            rules.append(orbat_rule)

        if not rules:
            return {}

        # Determine logical operator based on number of filter types
        filter_count = sum([bool(keywords), bool(location), bool(orbat_node_ids)])
        logical_op = "And" if filter_count > 1 else "Or"

        return {
            "filter": {
                "rules": rules,
                "operator": logical_op
            }
        }

    def get_placemarks(
        self,
        conflict: str,
        skip: int = 0,
        take: int = 100,
        keywords: Optional[str] = None,
        location: Optional[str] = None,
        orbat_node_ids: Optional[List[int]] = None,
        callback: Optional[Callable[[List[Dict], int], None]] = None
    ) -> Optional[Dict]:
        """Fetch placemarks for a conflict using v2 API with full details.

        Args:
            conflict: Conflict short name
            skip: Number of items to skip (pagination offset)
            take: Number of items to take (page size)
            keywords: Optional keyword search (Description, Name, Gear)
            location: Optional location filter (PlusCode - city/region/country)
            orbat_node_ids: Optional list of ORBAT node IDs to filter by units
            callback: Optional callback receiving (items, total_count)

        Returns:
            Raw API response dict with full placemark details
        """
        url = f"{self.BASE_URL}/Placemark/v2/{quote(conflict)}/{skip}/{take}"

        # Build request body with optional filters
        body = self._build_search_filter(keywords=keywords, location=location, orbat_node_ids=orbat_node_ids)

        filter_desc = []
        if keywords:
            filter_desc.append(f"keywords='{keywords}'")
        if location:
            filter_desc.append(f"location='{location}'")
        if orbat_node_ids:
            filter_desc.append(f"units={len(orbat_node_ids)} nodes")

        if filter_desc:
            self._log(f"Fetching placemarks (v2) with {', '.join(filter_desc)} from {url}")
        else:
            self._log(f"Fetching placemarks (v2) from {url}")

        data = self._blocking_post(url, body)

        if data is not None:
            items = data.get('items', [])
            count = data.get('count', len(items))
            self._log(f"Loaded {len(items)} placemarks with details (total: {count})")
            self.placemarks_loaded.emit(items, count)
            if callback:
                callback(items, count)
        return data

    def get_all_placemarks(
        self,
        conflict: str,
        batch_size: int = 500,
        keywords: Optional[str] = None,
        location: Optional[str] = None,
        orbat_node_ids: Optional[List[int]] = None,
        progress_callback: Optional[Callable[[int, int], None]] = None
    ) -> List[Dict]:
        """Fetch all placemarks for a conflict using v2 API, handling pagination.

        The v2 API returns full details for each placemark in a single batch request,
        eliminating the need for individual enrichment calls.

        Args:
            conflict: Conflict short name
            batch_size: Number of results per batch
            keywords: Optional keyword search (Description, Name, Gear)
            location: Optional location filter (PlusCode - city/region/country)
            orbat_node_ids: Optional list of ORBAT node IDs to filter by units
            progress_callback: Optional callback for progress updates (current, total)

        Returns:
            List of all placemarks with full details
        """
        all_items = []
        skip = 0

        # First request to get total count
        data = self.get_placemarks(conflict, skip, batch_size, keywords=keywords, location=location, orbat_node_ids=orbat_node_ids)
        if data is None:
            return []

        total_count = data.get('count', 0)
        items = data.get('items', [])
        all_items.extend(items)

        filter_desc = []
        if keywords:
            filter_desc.append(f"keywords='{keywords}'")
        if location:
            filter_desc.append(f"location='{location}'")
        if orbat_node_ids:
            filter_desc.append(f"units={len(orbat_node_ids)} nodes")
        if filter_desc:
            self._log(f"Server-side filter ({', '.join(filter_desc)}) found {total_count} matching items")

        if progress_callback:
            progress_callback(len(all_items), total_count)
        self.progress_updated.emit(len(all_items), total_count)

        # Fetch remaining batches
        while len(all_items) < total_count:
            if self._cancelled:
                break

            skip = len(all_items)
            data = self.get_placemarks(conflict, skip, batch_size, keywords=keywords, location=location, orbat_node_ids=orbat_node_ids)
            if data is None:
                break

            items = data.get('items', [])
            if not items:
                break

            all_items.extend(items)

            if progress_callback:
                progress_callback(len(all_items), total_count)
            self.progress_updated.emit(len(all_items), total_count)

        filter_suffix = f" ({', '.join(filter_desc)})" if filter_desc else ""
        self._log(f"Fetched {len(all_items)} placemarks for {conflict}{filter_suffix}")
        return all_items

    def get_new_placemarks(
        self,
        conflict: str,
        since_date_created: str,
        batch_size: int = 500,
        progress_callback: Optional[Callable[[int, int, int], None]] = None
    ) -> List[Dict]:
        """Fetch only new placemarks added since a given dateCreated timestamp.

        The v2 API returns items sorted by dateCreated descending (newest first).
        This method fetches batches starting from skip=0 and stops when it
        encounters an item with dateCreated <= since_date_created.

        Args:
            conflict: Conflict short name
            since_date_created: ISO format datetime string of the most recent
                               item in the existing layer
            batch_size: Number of results per batch
            progress_callback: Optional callback for progress updates (new_count, batch_num, total_server)
                              Note: total_server is the total count on server, not new items

        Returns:
            List of new placemarks (those with dateCreated > since_date_created)
        """
        new_items = []
        skip = 0
        batch_num = 0
        found_existing = False

        self._log(f"Fetching new placemarks since {since_date_created}")

        while not found_existing:
            if self._cancelled:
                break

            batch_num += 1
            data = self.get_placemarks(conflict, skip, batch_size)
            if data is None:
                break

            total_count = data.get('count', 0)
            items = data.get('items', [])

            if not items:
                break

            for item in items:
                item_date_created = item.get('dateCreated', '')

                # Compare datetime strings (ISO format compares correctly as strings)
                if item_date_created > since_date_created:
                    new_items.append(item)
                else:
                    # Found an item that already exists in our layer
                    found_existing = True
                    self._log(f"Found existing item at dateCreated={item_date_created}")
                    break

            if progress_callback:
                progress_callback(len(new_items), batch_num, total_count)

            # If we processed all items in this batch and didn't find existing
            # data, continue to the next batch
            if not found_existing:
                skip += batch_size
                # Safety check: if we've fetched more than exists, stop
                if skip >= total_count:
                    break

        self._log(f"Found {len(new_items)} new placemarks for {conflict}")
        return new_items

    def get_mentions(
        self,
        page: int = 0,
        page_size: int = 100,
        conflict: Optional[str] = None,
        search: Optional[str] = None,
        sources: Optional[List[str]] = None,
        in_slack_filter: Optional[bool] = None,
        processed_filter: Optional[bool] = None,
        doubt_filter: Optional[bool] = None,
        wrong_filter: Optional[bool] = None,
        include_unmarked: Optional[bool] = None,
        callback: Optional[Callable[[List[Dict], int], None]] = None
    ) -> Optional[Dict]:
        """Fetch mentions with filters.

        Args:
            page: Page number (0-indexed)
            page_size: Number of results per page
            conflict: Filter by conflict name
            search: Text search query
            sources: List of source platforms to filter
            in_slack_filter: Filter by Slack status
            processed_filter: Filter by processed status
            doubt_filter: Filter by doubt status
            wrong_filter: Filter by wrong status
            include_unmarked: Include untagged mentions
            callback: Optional callback receiving (items, total_count)

        Returns:
            Raw API response dict
        """
        url = f"{self.BASE_URL}/Mention/{page}/{page_size}"

        # Build query parameters
        params = {}
        if conflict:
            params['conflict'] = conflict
        if search:
            params['search'] = search
        if in_slack_filter is not None:
            params['inSlackFilter'] = str(in_slack_filter).lower()
        if processed_filter is not None:
            params['processedFilter'] = str(processed_filter).lower()
        if doubt_filter is not None:
            params['doubtFilter'] = str(doubt_filter).lower()
        if wrong_filter is not None:
            params['wrongFilter'] = str(wrong_filter).lower()
        if include_unmarked is not None:
            params['includeUnmarked'] = str(include_unmarked).lower()

        # Handle sources as multiple parameters
        if sources:
            source_params = '&'.join([f'sources={quote(s)}' for s in sources])
            if params:
                url = f"{url}?{urlencode(params)}&{source_params}"
            else:
                url = f"{url}?{source_params}"
        elif params:
            url = f"{url}?{urlencode(params)}"

        self._log(f"Fetching mentions from {url}")

        data = self._blocking_get(url)
        if data is not None:
            items = data.get('items', [])
            count = data.get('count', len(items))
            self._log(f"Loaded {len(items)} mentions (total: {count})")
            self.mentions_loaded.emit(items, count)
            if callback:
                callback(items, count)
        return data

    def get_all_mentions(
        self,
        conflict: Optional[str] = None,
        search: Optional[str] = None,
        sources: Optional[List[str]] = None,
        in_slack_filter: Optional[bool] = None,
        processed_filter: Optional[bool] = None,
        doubt_filter: Optional[bool] = None,
        wrong_filter: Optional[bool] = None,
        include_unmarked: Optional[bool] = None,
        page_size: int = 500,
        progress_callback: Optional[Callable[[int, int], None]] = None
    ) -> List[Dict]:
        """Fetch all mentions with filters, handling pagination.

        Args:
            conflict: Filter by conflict name
            search: Text search query
            sources: List of source platforms to filter
            in_slack_filter: Filter by Slack status
            processed_filter: Filter by processed status
            doubt_filter: Filter by doubt status
            wrong_filter: Filter by wrong status
            include_unmarked: Include untagged mentions
            page_size: Number of results per page
            progress_callback: Optional callback for progress updates

        Returns:
            List of all mentions
        """
        all_items = []
        page = 0

        # First request to get total count
        data = self.get_mentions(
            page, page_size, conflict, search, sources,
            in_slack_filter, processed_filter, doubt_filter,
            wrong_filter, include_unmarked
        )
        if data is None:
            return []

        total_count = data.get('count', 0)
        items = data.get('items', [])
        all_items.extend(items)

        if progress_callback:
            progress_callback(len(all_items), total_count)
        self.progress_updated.emit(len(all_items), total_count)

        # Fetch remaining pages
        while len(all_items) < total_count:
            page += 1
            data = self.get_mentions(
                page, page_size, conflict, search, sources,
                in_slack_filter, processed_filter, doubt_filter,
                wrong_filter, include_unmarked
            )
            if data is None:
                break

            items = data.get('items', [])
            if not items:
                break

            all_items.extend(items)

            if progress_callback:
                progress_callback(len(all_items), total_count)
            self.progress_updated.emit(len(all_items), total_count)

        self._log(f"Fetched all {len(all_items)} mentions")
        return all_items

    def get_orbat(self, orbat_id: int) -> Optional[Dict]:
        """Fetch ORBAT detail by ID.

        Args:
            orbat_id: The ORBAT ID from faction's orbatIds array

        Returns:
            ORBAT detail dict containing orbatNodeId
        """
        url = f"{self.BASE_URL}/Orbat/{orbat_id}"
        self._log(f"Fetching ORBAT from {url}")
        return self._blocking_get(url)

    def get_orbat_node(self, orbat_node_id: int) -> Optional[Dict]:
        """Fetch ORBAT node hierarchy by node ID.

        Args:
            orbat_node_id: The root ORBAT node ID

        Returns:
            Hierarchical dict with 'children' arrays containing nested nodes.
            Each node has: id, name, children (recursive)
        """
        url = f"{self.BASE_URL}/Orbat/node/{orbat_node_id}"
        self._log(f"Fetching ORBAT node hierarchy from {url}")
        return self._blocking_get(url)

    def get_faction_orbat_hierarchy(self, orbat_ids: List[int]) -> List[Dict]:
        """Fetch complete ORBAT hierarchies for a faction's orbat IDs.

        Args:
            orbat_ids: List of ORBAT IDs from faction

        Returns:
            List of root ORBAT nodes with full hierarchies
        """
        hierarchies = []
        for orbat_id in orbat_ids:
            orbat = self.get_orbat(orbat_id)
            if orbat and 'orbatNodeId' in orbat:
                node_hierarchy = self.get_orbat_node(orbat['orbatNodeId'])
                if node_hierarchy:
                    hierarchies.append(node_hierarchy)
        return hierarchies

    def cancel_pending_requests(self):
        """Cancel all pending network requests."""
        self._cancelled = True

    def reset_cancelled(self):
        """Reset the cancelled flag."""
        self._cancelled = False


def build_search_query(
    keywords: Optional[List[str]] = None,
    must_include: Optional[List[str]] = None,
    must_exclude: Optional[List[str]] = None,
    any_of: Optional[List[str]] = None
) -> str:
    """Build a boolean search query string.

    The GeoConfirmed API supports boolean operators:
    - & for AND
    - | for OR
    - ! for NOT
    - () for grouping

    Args:
        keywords: Basic keywords (ANDed together)
        must_include: Keywords that must be present (ANDed)
        must_exclude: Keywords to exclude (NOTed)
        any_of: Keywords where any match is ok (ORed)

    Returns:
        Formatted search query string

    Example:
        build_search_query(
            must_include=['drone'],
            must_exclude=['Ukrainian'],
            any_of=['FPV', 'Lancet']
        )
        Returns: "drone & (FPV | Lancet) & !Ukrainian"
    """
    parts = []

    # Add required keywords
    if must_include:
        parts.extend(must_include)

    if keywords:
        parts.extend(keywords)

    # Add OR group
    if any_of and len(any_of) > 1:
        or_group = ' | '.join(any_of)
        parts.append(f'({or_group})')
    elif any_of:
        parts.extend(any_of)

    # Build base query with AND
    query = ' & '.join(parts) if parts else ''

    # Add exclusions
    if must_exclude:
        exclusions = ' & '.join([f'!{term}' for term in must_exclude])
        if query:
            query = f'{query} & {exclusions}'
        else:
            query = exclusions

    return query
