# -*- coding: utf-8 -*-
"""
GeoConfirmed Dialog Controller
Handles UI logic and data fetching coordination.

Alpha version with faction filtering and statistics support.
"""

import os
from datetime import datetime, date
from typing import Optional, List, Dict

from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt, QDate, QThread, pyqtSignal
from qgis.PyQt.QtWidgets import (
    QDialog, QFileDialog, QMessageBox, QApplication, QCheckBox, QTreeWidgetItem
)
from qgis.PyQt.QtGui import QColor
from qgis.core import QgsProject, Qgis, QgsRectangle, QgsCoordinateReferenceSystem, QgsMapLayerProxyModel
from qgis.gui import QgsMapLayerComboBox

from .api.client import GeoConfirmedClient
from .utils.layer_manager import LayerManager

# Load the UI file
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'geoconfirmed_dialog_base.ui'))


class FetchWorker(QThread):
    """Worker thread for fetching data from API."""

    finished = pyqtSignal(list)
    error = pyqtSignal(str)
    progress = pyqtSignal(int, int)
    status_update = pyqtSignal(str)

    def __init__(
        self,
        client: GeoConfirmedClient,
        fetch_type: str,
        conflict: str,
        date_from: Optional[date] = None,
        date_to: Optional[date] = None,
        sources: Optional[List[str]] = None,
        filters: Optional[Dict] = None,
        keywords: Optional[str] = None,
        location: Optional[str] = None,
        orbat_node_ids: Optional[List[int]] = None
    ):
        super().__init__()
        self.client = client
        self.fetch_type = fetch_type
        self.conflict = conflict
        self.date_from = date_from
        self.date_to = date_to
        self.sources = sources
        self.filters = filters or {}
        self.keywords = keywords  # Server-side keyword search
        self.location = location  # Server-side location filter
        self.orbat_node_ids = orbat_node_ids  # Server-side unit filter
        self._cancelled = False

    def run(self):
        """Execute the fetch operation."""
        try:
            if self.fetch_type == 'placemarks':
                # v2 API with server-side filtering
                items = self.client.get_all_placemarks(
                    self.conflict,
                    keywords=self.keywords,
                    location=self.location,
                    orbat_node_ids=self.orbat_node_ids,
                    progress_callback=self._on_progress
                )
            else:  # mentions
                items = self.client.get_all_mentions(
                    conflict=self.conflict,
                    sources=self.sources,
                    doubt_filter=self.filters.get('doubt_filter'),
                    wrong_filter=self.filters.get('wrong_filter'),
                    processed_filter=self.filters.get('processed_filter'),
                    include_unmarked=self.filters.get('include_unmarked'),
                    progress_callback=self._on_progress
                )

            # Filter by date if specified (client-side filtering)
            if self.date_from or self.date_to:
                items = self._filter_by_date(items)

            if not self._cancelled:
                self.finished.emit(items)

        except Exception as e:
            if not self._cancelled:
                self.error.emit(str(e))

    def _on_progress(self, current: int, total: int):
        """Handle progress updates."""
        if not self._cancelled:
            self.progress.emit(current, total)

    def _filter_by_date(self, items: List[Dict]) -> List[Dict]:
        """Filter items by date range.

        Args:
            items: List of items to filter

        Returns:
            Filtered list
        """
        filtered = []
        for item in items:
            item_date_str = item.get('date') or item.get('dateCreated')
            if not item_date_str:
                continue

            try:
                # Parse ISO date format
                if 'T' in item_date_str:
                    item_date = datetime.fromisoformat(
                        item_date_str.replace('Z', '+00:00')
                    ).date()
                else:
                    item_date = datetime.strptime(
                        item_date_str[:10], '%Y-%m-%d'
                    ).date()

                # Check date range
                if self.date_from and item_date < self.date_from:
                    continue
                if self.date_to and item_date > self.date_to:
                    continue

                filtered.append(item)
            except (ValueError, TypeError):
                # Include items with unparseable dates
                filtered.append(item)

        return filtered

    def cancel(self):
        """Cancel the fetch operation."""
        self._cancelled = True
        self.client.cancel_pending_requests()


class UpdateWorker(QThread):
    """Worker thread for updating a layer with new placemarks."""

    finished = pyqtSignal(list)
    error = pyqtSignal(str)
    progress = pyqtSignal(int, int, int)  # new_count, batch_num, total_server
    status_update = pyqtSignal(str)

    def __init__(
        self,
        client: GeoConfirmedClient,
        conflict: str,
        since_date_created: str
    ):
        super().__init__()
        self.client = client
        self.conflict = conflict
        self.since_date_created = since_date_created
        self._cancelled = False

    def run(self):
        """Execute the incremental fetch operation."""
        try:
            self.status_update.emit(f"Fetching new placemarks since {self.since_date_created[:19]}...")

            items = self.client.get_new_placemarks(
                self.conflict,
                self.since_date_created,
                progress_callback=self._on_progress
            )

            if not self._cancelled:
                self.finished.emit(items)

        except Exception as e:
            if not self._cancelled:
                self.error.emit(str(e))

    def _on_progress(self, new_count: int, batch_num: int, total_server: int):
        """Handle progress updates."""
        if not self._cancelled:
            self.progress.emit(new_count, batch_num, total_server)
            self.status_update.emit(f"Found {new_count} new placemarks (batch {batch_num})...")

    def cancel(self):
        """Cancel the fetch operation."""
        self._cancelled = True
        self.client.cancel_pending_requests()


class GeoConfirmedDialog(QDialog, FORM_CLASS):
    """Dialog for querying GeoConfirmed data."""

    def __init__(self, iface, parent=None):
        """Constructor.

        Args:
            iface: QGIS interface instance
            parent: Parent widget
        """
        super().__init__(parent)
        self.setupUi(self)

        self.iface = iface
        self.client = GeoConfirmedClient(self)
        self.layer_manager = LayerManager(iface)
        self.worker: Optional[FetchWorker] = None
        self.update_worker: Optional[UpdateWorker] = None
        self.conflicts: List[Dict] = []
        self.current_conflict_details: Optional[Dict] = None
        self.faction_checkboxes: List[QCheckBox] = []
        self._pending_search_query: Optional[str] = None
        self._update_target_layer = None  # Layer being updated
        self._orbat_cache: Dict[str, List[Dict]] = {}  # Cache ORBAT by conflict
        self._orbat_node_map: Dict[int, QTreeWidgetItem] = {}  # Map node ID to tree item
        self._orbat_id_to_name: Dict[int, str] = {}  # Map node ID to full name path
        self._orbat_id_to_emblem: Dict[int, str] = {}  # Map node ID to emblem/patch URL

        self._setup_connections()
        self._setup_defaults()
        self._load_conflicts()
        self._populate_existing_layers()

    def _setup_connections(self):
        """Set up signal/slot connections."""
        # Buttons
        self.btnRefreshConflicts.clicked.connect(self._load_conflicts)
        self.btnFetch.clicked.connect(self._start_fetch)
        self.btnCancel.clicked.connect(self._cancel_fetch)
        self.btnBrowseOutput.clicked.connect(self._browse_output)

        # Update layer buttons
        self.btnUpdate.clicked.connect(self._start_update)
        self.btnRefreshLayers.clicked.connect(self._populate_existing_layers)
        self.comboExistingLayers.currentIndexChanged.connect(
            self._on_existing_layer_changed
        )

        # Faction filter buttons
        self.btnSelectAllFactions.clicked.connect(self._select_all_factions)
        self.btnSelectNoneFactions.clicked.connect(self._select_no_factions)

        # Conflict selection changes
        self.comboConflict.currentIndexChanged.connect(
            self._on_conflict_changed
        )

        # API client signals
        self.client.error_occurred.connect(self._on_error)

        # ORBAT tree connections
        self.lineUnitSearch.textChanged.connect(self._filter_orbat_tree)
        self.btnClearUnitSelection.clicked.connect(self._clear_unit_selection)
        self.treeUnits.itemChanged.connect(self._on_unit_item_changed)

        # Spatial filter setup
        self._setup_spatial_filter()

    def _setup_defaults(self):
        """Set up default values."""
        # Set date range defaults
        today = QDate.currentDate()
        self.dateTo.setDate(today)
        self.dateFrom.setDate(today.addMonths(-1))

    def _setup_spatial_filter(self):
        """Set up the spatial filter widgets."""
        # Replace the placeholder QComboBox with QgsMapLayerComboBox
        # Get the layout and position of the original combobox
        layout = self.gridLayoutSpatial
        original_combo = self.comboSpatialLayer

        # Create the QgsMapLayerComboBox replacement
        self.spatialLayerCombo = QgsMapLayerComboBox(self)
        self.spatialLayerCombo.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.spatialLayerCombo.setToolTip("Select reference layer for spatial filter")
        self.spatialLayerCombo.setSizePolicy(original_combo.sizePolicy())

        # Replace the original combo in the layout
        layout.replaceWidget(original_combo, self.spatialLayerCombo)
        original_combo.hide()
        original_combo.deleteLater()

        # Connect spatial type change to show/hide relevant widgets
        self.comboSpatialType.currentIndexChanged.connect(self._on_spatial_type_changed)

        # Initialize visibility (default is "Not Spatial" so hide layer/radius)
        self._on_spatial_type_changed()

    def _on_spatial_type_changed(self):
        """Handle spatial filter type change - show/hide relevant widgets."""
        filter_type = self.comboSpatialType.currentText()

        # Show layer selector for "Around" and "Within Layer"
        show_layer = filter_type in ("Around", "Within Layer")
        self.labelSpatialLayer.setVisible(show_layer)
        self.spatialLayerCombo.setVisible(show_layer)

        # Show radius only for "Around"
        show_radius = filter_type == "Around"
        self.labelRadius.setVisible(show_radius)
        self.spinRadius.setVisible(show_radius)

    def _get_spatial_filter_params(self) -> dict:
        """Get current spatial filter parameters.

        :returns: Dictionary with spatial filter parameters
        """
        filter_text = self.comboSpatialType.currentText()

        # Map UI text to filter_type values
        type_map = {
            "Not Spatial": "none",
            "Around": "around",
            "Canvas Extent": "canvas",
            "Within Layer": "within"
        }
        filter_type = type_map.get(filter_text, "none")

        params = {
            'filter_type': filter_type,
            'reference_layer': None,
            'radius_meters': 0,
            'canvas_extent': None,
            'canvas_crs': None
        }

        if filter_type == "around":
            params['reference_layer'] = self.spatialLayerCombo.currentLayer()
            params['radius_meters'] = self.spinRadius.value()
        elif filter_type == "within":
            params['reference_layer'] = self.spatialLayerCombo.currentLayer()
        elif filter_type == "canvas":
            canvas = self.iface.mapCanvas()
            params['canvas_extent'] = canvas.extent()
            params['canvas_crs'] = canvas.mapSettings().destinationCrs()

        return params

    def _load_conflicts(self):
        """Load available conflicts from API."""
        self.labelStatus.setText("Loading conflicts...")
        self.btnRefreshConflicts.setEnabled(False)
        QApplication.processEvents()

        try:
            conflicts = self.client.get_conflicts()
            if conflicts:
                self.conflicts = conflicts
                self._populate_conflicts()
                self.labelStatus.setText(
                    f"Loaded {len(conflicts)} conflicts"
                )
            else:
                self.labelStatus.setText("No conflicts found")
        except Exception as e:
            self._on_error(f"Failed to load conflicts: {str(e)}")
        finally:
            self.btnRefreshConflicts.setEnabled(True)

    def _populate_conflicts(self):
        """Populate the conflict dropdown."""
        self.comboConflict.clear()
        self.comboConflict.addItem("-- Select a conflict --", None)

        for conflict in self.conflicts:
            name = conflict.get('name', 'Unknown')
            short_name = conflict.get('shortName', '')
            display_name = f"{name} ({short_name})" if short_name else name
            self.comboConflict.addItem(display_name, conflict)

    def _on_conflict_changed(self, index: int):
        """Handle conflict selection change.

        Args:
            index: Selected index
        """
        conflict = self.comboConflict.currentData()
        if conflict:
            # Update date range based on conflict dates
            start_date = conflict.get('startDate')
            if start_date:
                try:
                    qdate = QDate.fromString(start_date[:10], 'yyyy-MM-dd')
                    if qdate.isValid():
                        self.dateFrom.setDate(qdate)
                except Exception:
                    pass

            # Load conflict details for factions (Alpha feature)
            self._load_conflict_details(conflict.get('shortName', ''))

            # Load ORBAT for units filter
            self._load_orbat_for_conflict(conflict.get('shortName', ''))

    def _load_conflict_details(self, conflict_name: str):
        """Load detailed conflict info including factions.

        Args:
            conflict_name: Short name of the conflict
        """
        if not conflict_name:
            return

        self.labelStatus.setText(f"Loading factions for {conflict_name}...")
        QApplication.processEvents()

        try:
            details = self.client.get_conflict_details(conflict_name)
            if details:
                self.current_conflict_details = details
                factions = details.get('factions', [])

                # Update layer manager faction lookup
                self.layer_manager.set_faction_lookup(factions)

                # Populate faction checkboxes
                self._populate_faction_checkboxes(factions)

                self.labelStatus.setText(
                    f"Loaded {len(factions)} factions for {conflict_name}"
                )
        except Exception as e:
            self.labelStatus.setText(f"Failed to load factions: {str(e)}")

    def _load_orbat_for_conflict(self, conflict_name: str):
        """Load ORBAT hierarchy for the selected conflict.

        Args:
            conflict_name: Short name of the conflict
        """
        if not conflict_name:
            self.treeUnits.clear()
            self._orbat_node_map.clear()
            return

        # Check cache first
        if conflict_name in self._orbat_cache:
            self._populate_orbat_tree(self._orbat_cache[conflict_name])
            return

        # Need to load from factions
        if not self.current_conflict_details:
            return

        factions = self.current_conflict_details.get('factions', [])
        if not factions:
            return

        self.labelStatus.setText(f"Loading ORBAT for {conflict_name}...")
        QApplication.processEvents()

        try:
            all_hierarchies = []
            for faction in factions:
                orbat_ids = faction.get('orbatIds', [])
                if orbat_ids:
                    hierarchies = self.client.get_faction_orbat_hierarchy(orbat_ids)
                    all_hierarchies.extend(hierarchies)

            # Cache and populate
            self._orbat_cache[conflict_name] = all_hierarchies
            self._populate_orbat_tree(all_hierarchies)
            self.labelStatus.setText(f"Loaded ORBAT for {conflict_name}")
        except Exception as e:
            self.labelStatus.setText(f"Failed to load ORBAT: {str(e)}")

    def _populate_orbat_tree(self, hierarchies: List[Dict]):
        """Populate the ORBAT tree widget.

        Args:
            hierarchies: List of root ORBAT nodes with children
        """
        self.treeUnits.clear()
        self._orbat_node_map.clear()
        self._orbat_id_to_name.clear()
        self._orbat_id_to_emblem.clear()

        # Temporarily block signals to avoid triggering itemChanged during population
        self.treeUnits.blockSignals(True)

        for hierarchy in hierarchies:
            self._add_orbat_node_to_tree(hierarchy, None, [])

        self.treeUnits.blockSignals(False)

    def _add_orbat_node_to_tree(self, node: Dict, parent_item: Optional[QTreeWidgetItem], parent_path: List[str]):
        """Recursively add ORBAT node and children to tree.

        Args:
            node: ORBAT node dict with name, id, children, patches, emblems
            parent_item: Parent tree item or None for root
            parent_path: List of parent names for building full path
        """
        from urllib.parse import unquote

        name = node.get('name', 'Unknown')
        node_id = node.get('id')

        if parent_item is None:
            item = QTreeWidgetItem(self.treeUnits)
        else:
            item = QTreeWidgetItem(parent_item)

        item.setText(0, name)
        item.setData(0, Qt.UserRole, node_id)
        item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
        item.setCheckState(0, Qt.Unchecked)

        # Store in map for quick lookup
        if node_id is not None:
            self._orbat_node_map[node_id] = item
            # Build full path name (e.g., "55th Artillery Brigade > 1st Battalion")
            current_path = parent_path + [name]
            # Store just the unit name, but could use ' > '.join(current_path) for full path
            self._orbat_id_to_name[node_id] = name

            # Extract emblem/patch/tactical marking URL (prefer emblems, then patches, then tactical markings)
            emblems = node.get('emblems', [])
            patches = node.get('patches', [])
            tactical_markings = node.get('tacticalMarkings', [])

            emblem_url = None
            # URLs are URL-encoded, decode them
            if emblems and len(emblems) > 0:
                emblem_url = unquote(emblems[0])
            elif patches and len(patches) > 0:
                emblem_url = unquote(patches[0])
            elif tactical_markings and len(tactical_markings) > 0:
                emblem_url = unquote(tactical_markings[0])

            if emblem_url:
                self._orbat_id_to_emblem[node_id] = emblem_url

        # Recursively add children
        children = node.get('children', [])
        current_path = parent_path + [name]
        for child in children:
            self._add_orbat_node_to_tree(child, item, current_path)

    def _filter_orbat_tree(self, search_text: str):
        """Filter the ORBAT tree based on search text.

        Args:
            search_text: Text to search for (case-insensitive)
        """
        search_lower = search_text.lower().strip()

        def set_item_visible(item: QTreeWidgetItem, visible: bool):
            item.setHidden(not visible)

        def filter_item(item: QTreeWidgetItem) -> bool:
            """Returns True if item or any child matches."""
            item_text = item.text(0).lower()
            item_matches = search_lower in item_text if search_lower else True

            # Check children recursively
            child_matches = False
            for i in range(item.childCount()):
                child = item.child(i)
                if filter_item(child):
                    child_matches = True

            # Item is visible if it matches or has matching children
            visible = item_matches or child_matches
            set_item_visible(item, visible)

            # Expand if has matching children
            if child_matches and search_lower:
                item.setExpanded(True)

            return visible

        # Apply filter to all top-level items
        for i in range(self.treeUnits.topLevelItemCount()):
            item = self.treeUnits.topLevelItem(i)
            filter_item(item)

    def _clear_unit_selection(self):
        """Clear all unit selections in the tree."""
        self.treeUnits.blockSignals(True)

        def clear_item(item: QTreeWidgetItem):
            item.setCheckState(0, Qt.Unchecked)
            for i in range(item.childCount()):
                clear_item(item.child(i))

        for i in range(self.treeUnits.topLevelItemCount()):
            clear_item(self.treeUnits.topLevelItem(i))

        self.treeUnits.blockSignals(False)
        self.lineUnitSearch.clear()

    def _on_unit_item_changed(self, item: QTreeWidgetItem, column: int):
        """Handle unit tree item check state change.

        Args:
            item: Changed tree item
            column: Changed column
        """
        # Could add logic here to check/uncheck children
        pass

    def _get_selected_orbat_node_ids(self) -> List[int]:
        """Get list of selected ORBAT node IDs.

        Returns:
            List of checked node IDs
        """
        selected_ids = []

        def collect_checked(item: QTreeWidgetItem):
            if item.checkState(0) == Qt.Checked:
                node_id = item.data(0, Qt.UserRole)
                if node_id is not None:
                    selected_ids.append(node_id)
            for i in range(item.childCount()):
                collect_checked(item.child(i))

        for i in range(self.treeUnits.topLevelItemCount()):
            collect_checked(self.treeUnits.topLevelItem(i))

        return selected_ids

    def _get_selected_emblem_url(self) -> Optional[str]:
        """Get the emblem URL for the first selected unit that has one.

        Returns:
            Emblem URL string or None if no emblem available
        """
        selected_ids = self._get_selected_orbat_node_ids()

        # Find the first selected unit with an emblem
        for node_id in selected_ids:
            emblem_url = self._orbat_id_to_emblem.get(node_id)
            if emblem_url:
                return emblem_url

        return None

    def _populate_faction_checkboxes(self, factions: List[Dict]):
        """Populate the faction filter checkboxes.

        Args:
            factions: List of faction dicts from API
        """
        # Clear existing checkboxes
        for cb in self.faction_checkboxes:
            cb.deleteLater()
        self.faction_checkboxes.clear()

        # Get the grid layout
        layout = self.gridLayoutFactions

        # Clear existing items from layout
        while layout.count():
            item = layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()

        # Add checkbox for each faction (2 columns)
        row = 0
        col = 0
        for faction in factions:
            name = faction.get('name', 'Unknown')
            color = faction.get('color', '#666666')

            cb = QCheckBox(name)
            cb.setChecked(True)  # Default to checked

            # Style the checkbox with faction color indicator
            cb.setStyleSheet(f"""
                QCheckBox {{
                    padding-left: 5px;
                }}
                QCheckBox::indicator:checked {{
                    background-color: {color};
                    border: 1px solid #333;
                }}
            """)

            cb.setProperty('faction_name', name)
            cb.setProperty('faction_color', color)

            layout.addWidget(cb, row, col)
            self.faction_checkboxes.append(cb)

            col += 1
            if col >= 2:
                col = 0
                row += 1

    def _select_all_factions(self):
        """Select all faction checkboxes."""
        for cb in self.faction_checkboxes:
            cb.setChecked(True)

    def _select_no_factions(self):
        """Deselect all faction checkboxes."""
        for cb in self.faction_checkboxes:
            cb.setChecked(False)

    def _get_selected_factions(self) -> List[str]:
        """Get list of selected faction names.

        Returns:
            List of faction names that are checked
        """
        if not self.groupBoxFactions.isChecked():
            return []  # Empty means no filtering

        selected = []
        for cb in self.faction_checkboxes:
            if cb.isChecked():
                selected.append(cb.property('faction_name'))
        return selected

    def _browse_output(self):
        """Open file browser for output path."""
        conflict = self.comboConflict.currentData()
        default_name = f"geoconfirmed_{conflict.get('shortName', 'data') if conflict else 'data'}.gpkg"

        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Save GeoPackage",
            default_name,
            "GeoPackage (*.gpkg)"
        )
        if file_path:
            if not file_path.endswith('.gpkg'):
                file_path += '.gpkg'
            self.lineOutputPath.setText(file_path)

    def _populate_existing_layers(self):
        """Populate the dropdown with existing GeoConfirmed layers."""
        self.comboExistingLayers.clear()
        self.comboExistingLayers.addItem("-- Select a layer --", None)

        layers = self.layer_manager.get_geoconfirmed_layers()

        for layer, conflict in layers:
            display_name = f"{layer.name()} ({layer.featureCount()} features)"
            self.comboExistingLayers.addItem(display_name, (layer, conflict))

        # Enable/disable update button based on available layers
        has_layers = len(layers) > 0
        self.btnUpdate.setEnabled(has_layers and self.comboExistingLayers.currentIndex() > 0)

    def _on_existing_layer_changed(self, index: int):
        """Handle existing layer selection change."""
        data = self.comboExistingLayers.currentData()
        self.btnUpdate.setEnabled(data is not None)

    def _start_update(self):
        """Start updating an existing layer with new placemarks."""
        # Get selected layer
        data = self.comboExistingLayers.currentData()
        if not data:
            QMessageBox.warning(
                self,
                "No Layer Selected",
                "Please select a GeoConfirmed layer to update."
            )
            return

        layer, conflict = data

        if not layer or not layer.isValid():
            QMessageBox.warning(
                self,
                "Invalid Layer",
                "The selected layer is no longer valid. Please refresh the layer list."
            )
            self._populate_existing_layers()
            return

        # Get the most recent date_created from the layer
        latest_date = self.layer_manager.get_latest_date_created(layer)
        if not latest_date:
            QMessageBox.warning(
                self,
                "Cannot Determine Date",
                "Could not determine the most recent date from the layer. "
                "The layer may be missing the 'date_created' field."
            )
            return

        # Determine conflict short name
        # First try custom property, then try to match from loaded conflicts
        conflict_short_name = layer.customProperty('geoconfirmed_conflict')

        if not conflict_short_name:
            # Try to match conflict name to short name from loaded conflicts
            for c in self.conflicts:
                if c.get('name') == conflict or c.get('shortName') == conflict:
                    conflict_short_name = c.get('shortName')
                    break

        if not conflict_short_name:
            # Last resort: use the conflict value directly
            conflict_short_name = conflict

        # Store the target layer for use in _on_update_finished
        self._update_target_layer = layer

        # Update UI state
        self._set_fetching_state(True)
        self.progressBar.setMaximum(0)  # Indeterminate
        self.labelStatus.setText(f"Checking for new placemarks since {latest_date[:19]}...")

        # Reset cancelled flag
        self.client.reset_cancelled()

        # Create and start update worker
        self.update_worker = UpdateWorker(
            self.client,
            conflict_short_name,
            latest_date
        )
        self.update_worker.progress.connect(self._on_update_progress)
        self.update_worker.finished.connect(self._on_update_finished)
        self.update_worker.error.connect(self._on_error)
        self.update_worker.status_update.connect(self._on_status_update)
        self.update_worker.start()

    def _on_update_progress(self, new_count: int, batch_num: int, total_server: int):
        """Handle update progress.

        Args:
            new_count: Number of new items found so far
            batch_num: Current batch number being processed
            total_server: Total count on server (for reference)
        """
        # For updates, we don't know the total ahead of time
        # Show batch progress instead
        self.progressBar.setMaximum(0)  # Keep indeterminate
        self.labelStatus.setText(f"Found {new_count} new placemarks (checking batch {batch_num})...")

    def _on_update_finished(self, items: List[Dict]):
        """Handle update completion.

        Args:
            items: List of new items fetched
        """
        self._set_fetching_state(False)

        layer = self._update_target_layer
        self._update_target_layer = None

        if not items:
            self.labelStatus.setText("Layer is up to date")
            self.progressBar.setValue(0)
            QMessageBox.information(
                self,
                "Up to Date",
                "No new placemarks found. The layer is already up to date."
            )
            return

        if not layer or not layer.isValid():
            self._on_error("Target layer is no longer valid")
            return

        # Add new features to the layer
        self.labelStatus.setText(f"Adding {len(items)} new features to layer...")
        QApplication.processEvents()

        try:
            added_count = self.layer_manager.append_features_to_layer(
                layer,
                items,
                is_placemarks=True
            )

            if added_count > 0:
                self.labelStatus.setText(
                    f"Added {added_count} new features to '{layer.name()}'"
                )
                self.iface.messageBar().pushMessage(
                    "GeoConfirmed",
                    f"Added {added_count} new features to layer",
                    level=Qgis.Success,
                    duration=5
                )

                # Refresh the layer list to show updated feature count
                self._populate_existing_layers()
            else:
                self.labelStatus.setText("Failed to add features to layer")
                QMessageBox.warning(
                    self,
                    "Update Failed",
                    "Failed to add new features to the layer."
                )

        except Exception as e:
            self._on_error(f"Failed to update layer: {str(e)}")

        self.progressBar.setValue(self.progressBar.maximum() if self.progressBar.maximum() > 0 else 100)

    def _start_fetch(self):
        """Start fetching data from API."""
        # Validate inputs
        conflict = self.comboConflict.currentData()
        if not conflict:
            QMessageBox.warning(
                self,
                "No Conflict Selected",
                "Please select a conflict to query."
            )
            return

        if self.radioGeoPackage.isChecked() and not self.lineOutputPath.text():
            QMessageBox.warning(
                self,
                "No Output File",
                "Please select an output file for the GeoPackage."
            )
            return

        # Validate spatial filter settings
        spatial_type = self.comboSpatialType.currentText()
        if spatial_type == "Around":
            if not self.spatialLayerCombo.currentLayer():
                QMessageBox.warning(
                    self,
                    "Validation Error",
                    "Please select a reference layer for 'Around' spatial filter."
                )
                return
            if self.spinRadius.value() <= 0:
                QMessageBox.warning(
                    self,
                    "Validation Error",
                    "Radius must be greater than 0 for 'Around' spatial filter."
                )
                return
        elif spatial_type == "Within Layer":
            if not self.spatialLayerCombo.currentLayer():
                QMessageBox.warning(
                    self,
                    "Validation Error",
                    "Please select a reference layer for 'Within Layer' spatial filter."
                )
                return

        # Prepare parameters
        conflict_name = conflict.get('shortName', '')
        # Get keywords and location for server-side filtering
        keywords_query = self.lineSearch.text().strip() or None
        location_query = self.lineLocation.text().strip() if hasattr(self, 'lineLocation') else None
        # Get selected ORBAT node IDs if Units group is enabled
        orbat_node_ids = None
        if self.groupBoxUnits.isChecked():
            selected_units = self._get_selected_orbat_node_ids()
            if selected_units:
                orbat_node_ids = selected_units
        # Store for status display
        self._pending_keywords = keywords_query
        self._pending_location = location_query
        self._pending_orbat_units = orbat_node_ids

        date_from = None
        date_to = None
        if not self.checkAllData.isChecked():
            date_from = self.dateFrom.date().toPyDate()
            date_to = self.dateTo.date().toPyDate()

        # Update UI state
        self._set_fetching_state(True)
        self.progressBar.setMaximum(0)  # Indeterminate initially

        # Build status message based on filters
        filter_parts = []
        if keywords_query:
            filter_parts.append(f"keywords='{keywords_query}'")
        if location_query:
            filter_parts.append(f"location='{location_query}'")
        if orbat_node_ids:
            filter_parts.append(f"units={len(orbat_node_ids)}")

        if filter_parts:
            self.labelStatus.setText(f"Fetching placemarks with {', '.join(filter_parts)}...")
        else:
            self.labelStatus.setText("Fetching all placemarks...")

        # Create and start worker - v2 API with server-side filtering
        self.worker = FetchWorker(
            self.client,
            'placemarks',
            conflict_name,
            date_from=date_from,
            date_to=date_to,
            keywords=keywords_query,
            location=location_query,
            orbat_node_ids=orbat_node_ids
        )
        self.worker.progress.connect(self._on_progress)
        self.worker.finished.connect(self._on_fetch_finished)
        self.worker.error.connect(self._on_error)
        self.worker.status_update.connect(self._on_status_update)
        self.worker.start()

    def _cancel_fetch(self):
        """Cancel the current fetch or update operation."""
        if self.worker and self.worker.isRunning():
            self.worker.cancel()
            self.worker.wait()
            self.worker = None

        if self.update_worker and self.update_worker.isRunning():
            self.update_worker.cancel()
            self.update_worker.wait()
            self.update_worker = None

        self._update_target_layer = None
        self._set_fetching_state(False)
        self.labelStatus.setText("Cancelled")
        self.progressBar.setValue(0)

    def _set_fetching_state(self, fetching: bool):
        """Update UI for fetching state.

        Args:
            fetching: Whether a fetch is in progress
        """
        self.btnFetch.setEnabled(not fetching)
        self.btnCancel.setEnabled(fetching)
        self.comboConflict.setEnabled(not fetching)
        self.groupBoxTimeframe.setEnabled(not fetching)
        self.groupBoxSearch.setEnabled(not fetching)
        self.groupBoxSpatial.setEnabled(not fetching)
        self.groupBoxFactions.setEnabled(not fetching)
        self.groupBoxOutput.setEnabled(not fetching)
        self.groupBoxUpdate.setEnabled(not fetching)

        # Also enable/disable the update button based on layer selection
        if not fetching:
            data = self.comboExistingLayers.currentData()
            self.btnUpdate.setEnabled(data is not None)

    def _on_progress(self, current: int, total: int):
        """Handle progress updates.

        Args:
            current: Current count
            total: Total count
        """
        self.progressBar.setMaximum(total)
        self.progressBar.setValue(current)
        self.labelStatus.setText(f"Fetching... {current} / {total}")

    def _on_status_update(self, message: str):
        """Handle status update messages.

        Args:
            message: Status message
        """
        self.labelStatus.setText(message)

    def _on_fetch_finished(self, items: List[Dict]):
        """Handle fetch completion.

        Args:
            items: Fetched items
        """
        self._set_fetching_state(False)

        if not items:
            self.labelStatus.setText("No data found")
            QMessageBox.information(
                self,
                "No Data",
                "No data was found matching your criteria."
            )
            return

        original_count = len(items)

        # Server-side filters were already applied - display results
        keywords = getattr(self, '_pending_keywords', None)
        location = getattr(self, '_pending_location', None)
        if keywords or location:
            filter_parts = []
            if keywords:
                filter_parts.append(f"keywords='{keywords}'")
            if location:
                filter_parts.append(f"location='{location}'")
            self.labelStatus.setText(
                f"Server-side filter ({', '.join(filter_parts)}): {len(items)} items found"
            )

        # Apply spatial filter
        spatial_params = self._get_spatial_filter_params()
        if spatial_params.get('filter_type') != 'none':
            pre_spatial_count = len(items)
            self.labelStatus.setText(f"Applying spatial filter to {len(items)} items...")
            QApplication.processEvents()
            items = self.layer_manager.filter_items_by_spatial(items, **spatial_params)
            if not items:
                self.labelStatus.setText("No data matches spatial filter")
                QMessageBox.information(
                    self,
                    "No Data",
                    "No data matches the spatial filter criteria."
                )
                return
            self.labelStatus.setText(
                f"Spatial filter: {len(items)} of {pre_spatial_count} items match"
            )

        # Apply faction filter if enabled (Alpha feature)
        pre_faction_count = len(items)
        selected_factions = self._get_selected_factions()

        # Only filter if faction groupbox is checked AND at least one faction is selected
        if self.groupBoxFactions.isChecked() and selected_factions:
            items = self.layer_manager.filter_items_by_factions(
                items, selected_factions
            )
            self.labelStatus.setText(
                f"Filtered to {len(items)} of {pre_faction_count} features..."
            )

            if not items:
                self.labelStatus.setText("No data after faction filter")
                QMessageBox.information(
                    self,
                    "No Data",
                    "No data matches the selected faction filters."
                )
                return
        else:
            self.labelStatus.setText(f"Processing {len(items)} features...")

        QApplication.processEvents()

        # Compute and display statistics (Alpha feature)
        if self.groupBoxStatistics.isChecked():
            self._display_statistics(items)

        try:
            # Determine output
            conflict = self.comboConflict.currentData()
            layer_name = f"GeoConfirmed - {conflict.get('name', 'Data')}"

            if self.radioGeoPackage.isChecked():
                # Export to GeoPackage
                output_path = self.lineOutputPath.text()
                layer = self.layer_manager.create_geopackage_layer(
                    items,
                    output_path,
                    layer_name,
                    is_placemarks=True,
                    orbat_lookup=self._orbat_id_to_name
                )
            else:
                # Create memory layer
                layer = self.layer_manager.create_memory_layer(
                    items,
                    layer_name,
                    is_placemarks=True,
                    orbat_lookup=self._orbat_id_to_name
                )

            if layer and layer.isValid():
                # Set the conflict custom property for update functionality
                conflict_short_name = conflict.get('shortName', '')
                if conflict_short_name:
                    self.layer_manager.set_layer_conflict_property(
                        layer, conflict_short_name
                    )

                # Apply styling if requested
                if self.checkApplyStyling.isChecked():
                    # Check if unit emblems should be used
                    use_emblems = self.checkUseUnitEmblems.isChecked()
                    emblem_url = self._get_selected_emblem_url() if use_emblems else None

                    if use_emblems and emblem_url:
                        self.layer_manager.apply_emblem_style(layer, emblem_url)
                    elif use_emblems and not emblem_url:
                        # User wanted emblems but none available - show warning
                        QMessageBox.warning(
                            self,
                            "No Unit Emblem Available",
                            "No emblem, patch, or tactical marking exists for the selected unit(s).\n\n"
                            "You can contribute unit emblems at:\n"
                            "https://geoconfirmed.org/\n\n"
                            "Using default styling instead."
                        )
                        self.layer_manager.apply_categorized_style(
                            layer,
                            is_placemarks=True
                        )
                    else:
                        self.layer_manager.apply_categorized_style(
                            layer,
                            is_placemarks=True
                        )

                # Enable temporal if requested
                if self.checkEnableTemporal.isChecked():
                    self.layer_manager.enable_temporal(layer)

                # Add to project
                QgsProject.instance().addMapLayer(layer)

                # Zoom to layer extent (with CRS transform)
                self._zoom_to_layer(layer)

                # Refresh the existing layers dropdown
                self._populate_existing_layers()

                self.labelStatus.setText(
                    f"Loaded {len(items)} features to '{layer_name}'"
                )
                self.iface.messageBar().pushMessage(
                    "GeoConfirmed",
                    f"Loaded {len(items)} features",
                    level=Qgis.Success,
                    duration=5
                )
            else:
                raise Exception("Failed to create layer")

        except Exception as e:
            self._on_error(f"Failed to create layer: {str(e)}")

        self.progressBar.setValue(self.progressBar.maximum())

    def _display_statistics(self, items: List[Dict]):
        """Display statistics for the fetched items.

        Args:
            items: List of placemark items
        """
        stats = self.layer_manager.compute_statistics(items)

        # Build HTML for display
        html = []
        html.append(f"<b>Total Events:</b> {stats['total']}<br><br>")

        # Status breakdown
        html.append("<b>By Status:</b><br>")
        html.append(f"&nbsp;&nbsp;Active: {stats['by_status']['active']}<br>")
        html.append(f"&nbsp;&nbsp;Destroyed: {stats['by_status']['destroyed']}<br><br>")

        # Faction breakdown (sorted by count)
        html.append("<b>By Faction:</b><br>")
        sorted_factions = sorted(
            stats['by_faction'].items(),
            key=lambda x: x[1],
            reverse=True
        )
        for faction, count in sorted_factions[:10]:  # Top 10
            pct = (count / stats['total'] * 100) if stats['total'] > 0 else 0
            html.append(f"&nbsp;&nbsp;{faction}: {count} ({pct:.1f}%)<br>")

        # Equipment breakdown (sorted by count)
        html.append("<br><b>By Equipment Type:</b><br>")
        sorted_equipment = sorted(
            stats['by_equipment'].items(),
            key=lambda x: x[1],
            reverse=True
        )
        for equip, count in sorted_equipment[:10]:  # Top 10
            pct = (count / stats['total'] * 100) if stats['total'] > 0 else 0
            html.append(f"&nbsp;&nbsp;{equip}: {count} ({pct:.1f}%)<br>")

        self.textStatistics.setHtml(''.join(html))

    def _on_error(self, message: str):
        """Handle error.

        Args:
            message: Error message
        """
        self._set_fetching_state(False)
        self.labelStatus.setText(f"Error: {message}")
        self.iface.messageBar().pushMessage(
            "GeoConfirmed",
            message,
            level=Qgis.Warning,
            duration=10
        )

    def _zoom_to_layer(self, layer):
        """Zoom to layer extent with proper CRS transformation.

        Args:
            layer: Layer to zoom to
        """
        from qgis.core import QgsCoordinateTransform

        if not layer or not layer.isValid() or layer.featureCount() == 0:
            return

        canvas = self.iface.mapCanvas()
        layer_extent = layer.extent()

        # Transform extent from layer CRS to canvas CRS if different
        if layer.crs() != canvas.mapSettings().destinationCrs():
            transform = QgsCoordinateTransform(
                layer.crs(),
                canvas.mapSettings().destinationCrs(),
                QgsProject.instance()
            )
            layer_extent = transform.transformBoundingBox(layer_extent)

        # Add 10% buffer around the extent
        layer_extent.scale(1.1)

        canvas.setExtent(layer_extent)
        canvas.refresh()

    def closeEvent(self, event):
        """Handle dialog close.

        Args:
            event: Close event
        """
        self._cancel_fetch()
        super().closeEvent(event)
