# -*- coding: utf-8 -*-
"""
GeoConfirmed QGIS Plugin
Main plugin class that handles QGIS integration.
"""

import os

from qgis.PyQt.QtCore import QTranslator, QCoreApplication, QSettings, QLocale
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

from .geoconfirmed_dialog import GeoConfirmedDialog


class GeoConfirmed:
    """QGIS Plugin Implementation for GeoConfirmed data access."""

    def __init__(self, iface):
        """Constructor.

        Args:
            iface: An interface instance that provides the hook to
                   manipulate the QGIS application at runtime.
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # Initialize locale
        locale = QSettings().value('locale/userLocale', QLocale.system().name())
        if locale:
            locale = locale[0:2]  # Get language code (e.g., 'en' from 'en_US')

        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            f'geoconfirmed_{locale}.qm'
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr('&GeoConfirmed')
        self.toolbar = self.iface.addToolBar('GeoConfirmed')
        self.toolbar.setObjectName('GeoConfirmed')

        self.dialog = None

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        Args:
            message: String to translate

        Returns:
            Translated string
        """
        return QCoreApplication.translate('GeoConfirmed', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None
    ):
        """Add a toolbar icon to the toolbar.

        Args:
            icon_path: Path to the icon for this action
            text: Text that shows in menu items for this action
            callback: Function to be called when the action is triggered
            enabled_flag: Enable/disable flag for this action
            add_to_menu: Add to the plugin menu
            add_to_toolbar: Add to the plugin toolbar
            status_tip: Optional status tip shown in status bar
            whats_this: Optional what's this text
            parent: Parent widget for the new action

        Returns:
            The action that was created
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(self.menu, action)

        self.actions.append(action)
        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        # Try ICO first (GeoConfirmed favicon), then SVG, PNG, then fallback
        icon_path = os.path.join(self.plugin_dir, 'resources', 'icon.ico')
        if not os.path.exists(icon_path):
            icon_path = os.path.join(self.plugin_dir, 'resources', 'icon.svg')
        if not os.path.exists(icon_path):
            icon_path = os.path.join(self.plugin_dir, 'resources', 'icon.png')
        if not os.path.exists(icon_path):
            icon_path = ':/images/themes/default/mActionAddWfsLayer.svg'

        self.add_action(
            icon_path,
            text=self.tr('Query GeoConfirmed Data'),
            callback=self.run,
            parent=self.iface.mainWindow(),
            status_tip=self.tr('Query geolocated conflict data from GeoConfirmed.org'),
            whats_this=self.tr(
                'Opens a dialog to query and import conflict data '
                'from the GeoConfirmed API'
            )
        )

    def unload(self):
        """Remove the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(self.tr('&GeoConfirmed'), action)
            self.iface.removeToolBarIcon(action)

        # Remove the toolbar
        del self.toolbar

        # Clean up dialog
        if self.dialog:
            self.dialog.close()
            self.dialog = None

    def run(self):
        """Run method that performs all the real work."""
        # Always recreate the dialog to ensure UI changes are loaded
        # (During development; can cache after UI is stable)
        if self.dialog is not None:
            self.dialog.close()
            self.dialog.deleteLater()

        self.dialog = GeoConfirmedDialog(self.iface, parent=self.iface.mainWindow())

        # Show the dialog (non-modal so user can interact with map)
        self.dialog.show()

        # Raise and activate the dialog
        self.dialog.raise_()
        self.dialog.activateWindow()
