# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Geocod
                                 A QGIS plugin
 Este plugin automatiza a geocodificação de endereços dispostos em CSV e cria um shapefile deste pontos
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-11-18
        copyright            : (C) 2024 by Edson Tadeu da Silva Pinto
        email                : e.tadeu.eb@ime.eb.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
*   Plugin Geocod para QGIS                                                *
*                                                                          *
*   Este plugin adapta código da biblioteca geocoder                       *                         
*   (https://github.com/DenisCarriere/geocoder)                            *
*   licenciada sob MIT License.                                            *
 ***************************************************************************/
"""

__author__ = 'Edson Tadeu da Silva Pinto'
__date__ = '2024-11-18'
__copyright__ = '(C) 2024 by Edson Tadeu da Silva Pinto'
__revision__ = '$Format:%H$'

import os
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterField,
                       QgsProcessingParameterFile,
                       QgsVectorLayer,
                       QgsFeatureSink,
                       QgsFields,
                       QgsField,
                       QgsGeometry,
                       QgsPointXY,
                       QgsFeature,
                       QgsWkbTypes,
                       QgsCoordinateReferenceSystem)
from .geocoder_local import ArcgisQuery
import pandas as pd
from qgis.PyQt.QtCore import QVariant, QCoreApplication
from qgis.PyQt.QtGui import QIcon

import certifi
import os
os.environ['REQUESTS_CA_BUNDLE'] = certifi.where()

class GeocodificacaoAlgorithm(QgsProcessingAlgorithm):
    INPUT_CSV = 'INPUT_CSV'
    ADDRESS_FIELD = 'ADDRESS_FIELD'
    OUTPUT_LAYER = 'OUTPUT_LAYER'
    
    def initAlgorithm(self, config=None):
        # Parâmetro para o arquivo CSV
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_CSV,
                'Address file (CSV)',
                extension='csv'
            )
        )
        # Parâmetro para a coluna que contém os endereços
        self.addParameter(
            QgsProcessingParameterField(
                self.ADDRESS_FIELD,
                'Address column',
                parentLayerParameterName=self.INPUT_CSV,
                type=QgsProcessingParameterField.String
            )
        )
        # Saída como camada vetorial
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_LAYER,
                self.tr('Geocoded Points')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # Obter o arquivo CSV de entrada
        input_csv = self.parameterAsFile(parameters, self.INPUT_CSV, context)
        address_field = self.parameterAsString(parameters, self.ADDRESS_FIELD, context)
        
        # Criar uma nova camada de saída de pontos
        crs = QgsCoordinateReferenceSystem("EPSG:4326")  # WGS 84
        output_layer = QgsVectorLayer(f"Point?crs={crs.authid()}", "Geocoded Points", "memory")
        output_layer_pr = output_layer.dataProvider()

        # Definir campos para a camada de saída
        fields = QgsFields()
        fields.append(QgsField("address", QVariant.String))
        fields.append(QgsField("latitude", QVariant.Double))
        fields.append(QgsField("longitude", QVariant.Double))
        output_layer_pr.addAttributes(fields)
        output_layer.updateFields()

        (sink, dest_id) = self.parameterAsSink(parameters, 
                                self.OUTPUT_LAYER,
                                context,
                                output_layer.fields(),
                                QgsWkbTypes.Point,
                                output_layer.sourceCrs())

        # Leitura do arquivo CSV com pandas
        feedback.setProgressText('Loading the file...')
        data = pd.read_csv(input_csv)

        total = 100.0 / len(data) if len(data) > 0 else 0

        # Processamento de cada endereço na coluna selecionada
        feedback.setProgressText('Searching for addresses...')
        for index, row in data.iterrows():
            address = row[address_field]

            if not isinstance(address, str) or not address.strip():
                feedback.pushInfo(f"\nEmpty address on line {index + 1}, ignored.")
                continue
            
            # Geocodificação do endereço
            location = ArcgisQuery(address)
            if location and location.latlng:
                lat, lng = location.latlng
                feedback.pushInfo(f"\nThe address {address} has the coordinates {lat}, {lng}.")
                # Criar uma nova feição de ponto
                feature = QgsFeature()
                feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(lng, lat)))
                feature.setAttributes([address, lat, lng])
                output_layer_pr.addFeature(feature)
            else:
                feedback.pushInfo(f"\nUnable to geocode address: {address}")
            
            feedback.setProgress(int((index + 1) * total))

        # Atualização da camada de saída
        output_layer.updateExtents()

        # Adição das feições na camada output
        for feat in output_layer.getFeatures():
            sink.addFeature(feat, QgsFeatureSink.FastInsert)
        
        return {self.OUTPUT_LAYER: output_layer}

    def name(self):
        return 'Geocoding'

    def displayName(self):
        return self.tr(self.name())

    def group(self):
        return self.tr(self.groupId())

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return GeocodificacaoAlgorithm()

    def shortHelpString(self):
        return self.tr("""
    <p>This algorithm performs the geocoding of addresses contained in a <b>CSV</b> file.</p>

    <h3>🪶 Instructions:</h3>
    <ol>
    <li>Select the CSV file containing the addresses (each line must contain a valid address).</li>
    <li>Choose the column that contains the addresses.</li>
    <li>Run the algorithm and wait for the creation of the geocoded points layer.</li>
    </ol>

    <h3>📄 CSV File Format:</h3>
    <p>For the plugin to work correctly, the input CSV file must:</p>
    <ul>
    <li>Use <b>commas</b> as column separators.</li>
    <li>Contain a column of addresses in the standardized format shown below.</li>
    </ul>

    <h3>🏷️ Address Column Format:</h3>
    <p>During execution, you will be asked to select the <b>address column</b>. This column must contain addresses formatted as:</p>

    <pre>"Rua Saladino de Castro, 1575 - Centro, Arapoti, Paraná"</pre>

    <p><b>Example CSV content:</b></p>
    <pre>
    Locais, endereços, obs
    Local 1, "Rua Bolívia, 151 - Vila Nova, Barbosa Ferraz, Paraná", Hospital 1
    Local 2, "Rua Curitiba, 165 - Jardim América, Assis Chateaubriand, Paraná", Hospital 2
    Local 3, "Avenida Thiago Peixoto, 274 - Batel, Antonina, Paraná", Hospital 3
    </pre>

    <h3>📤 Output:</h3>
    <p>A point layer will be generated in the coordinate reference system <b>EPSG:4326 (WGS 84)</b>, containing the following fields:</p>
    <ul>
    <li><b>address</b> — Original address</li>
    <li><b>latitude</b> — Decimal latitude</li>
    <li><b>longitude</b> — Decimal longitude</li>
    </ul>

    <h3>📎 Notes:</h3>
    <ul>
    <li>Empty fields will be ignored.</li>
    <li>The plugin uses the <b>ArcGIS Geocoding</b> service through the <b>geocoder</b> library.</li>
    </ul>

    <p>© 2024 Edson Tadeu da Silva Pinto — Adapted from the geocoder library (MIT License).</p>
    """)

    
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'geocod.jfif'))
