"""
Algoritmo de processamento para fusionamento de bandas Landsat 5, Landsat 8, CBERS-4A e Sentinel-2
Implementa pansharpening para Landsat 8 e composição para Landsat 5, CBERS-4A e Sentinel-2
"""

from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingMultiStepFeedback,
    QgsProcessingParameterRasterLayer,
    QgsProcessingParameterRasterDestination,
    QgsProcessingParameterCrs,
    QgsProcessingParameterEnum,
    QgsCoordinateReferenceSystem
)
import processing


class LandsatFusionAlgorithm(QgsProcessingAlgorithm):
    """
    Algoritmo para fusionar bandas Landsat 5, Landsat 8, CBERS-4A e Sentinel-2
    - Landsat 8: Fusiona B4, B5, B6 com B8 usando pansharpening
    - Landsat 5: Combina B3, B4, B5 sem pansharpening
    - CBERS-4A: Combina B5, B4, B3 sem pansharpening
    - Sentinel-2: Combina B11, B8, B4 sem pansharpening
    """

    # Constantes de parâmetros
    SENSOR_TYPE = 'SENSOR_TYPE'
    
    # Landsat 8
    INPUT_L8_B4 = 'INPUT_L8_B4'
    INPUT_L8_B5 = 'INPUT_L8_B5'
    INPUT_L8_B6 = 'INPUT_L8_B6'
    INPUT_L8_B8 = 'INPUT_L8_B8'
    
    # Landsat 5
    INPUT_L5_B3 = 'INPUT_L5_B3'
    INPUT_L5_B4 = 'INPUT_L5_B4'
    INPUT_L5_B5 = 'INPUT_L5_B5'
    
    # CBERS-4A
    INPUT_CB4A_B3 = 'INPUT_CB4A_B3'
    INPUT_CB4A_B4 = 'INPUT_CB4A_B4'
    INPUT_CB4A_B5 = 'INPUT_CB4A_B5'
    
    # Sentinel-2
    INPUT_S2_B4 = 'INPUT_S2_B4'
    INPUT_S2_B8 = 'INPUT_S2_B8'
    INPUT_S2_B11 = 'INPUT_S2_B11'
    
    OUTPUT_CRS = 'OUTPUT_CRS'
    OUTPUT_RASTER = 'OUTPUT_RASTER'

    def initAlgorithm(self, config=None):
        """Define os parâmetros de entrada e saída do algoritmo"""
        
        # Seleção de sensor - com opções explícitas
        sensor_param = QgsProcessingParameterEnum(
            self.SENSOR_TYPE,
            'Sensor',
            options=['Landsat 8', 'Landsat 5', 'CBERS-4A', 'Sentinel-2'],
            defaultValue=0,
            allowMultiple=False
        )
        self.addParameter(sensor_param)
        
        # Parâmetros Landsat 8
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L8_B4,
                'Landsat 8 - Banda B4 (Red)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L8_B5,
                'Landsat 8 - Banda B5 (Near Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L8_B6,
                'Landsat 8 - Banda B6 (Short Wave Infrared 1)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L8_B8,
                'Landsat 8 - Banda B8 (Panchromatic)',
                defaultValue=None,
                optional=True
            )
        )
        
        # Parâmetros Landsat 5
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L5_B3,
                'Landsat 5 - Banda B3 (Red)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L5_B4,
                'Landsat 5 - Banda B4 (Near Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_L5_B5,
                'Landsat 5 - Banda B5 (Short Wave Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        # Parâmetros CBERS-4A
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_CB4A_B3,
                'CBERS-4A - Banda B3 (Red)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_CB4A_B4,
                'CBERS-4A - Banda B4 (Near Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_CB4A_B5,
                'CBERS-4A - Banda B5 (Short Wave Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        # Parâmetros Sentinel-2
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_S2_B4,
                'Sentinel-2 - Banda B4 (Red)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_S2_B8,
                'Sentinel-2 - Banda B8 (Near Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_S2_B11,
                'Sentinel-2 - Banda B11 (Short Wave Infrared)',
                defaultValue=None,
                optional=True
            )
        )
        
        # Parâmetro de projeção de saída
        self.addParameter(
            QgsProcessingParameterCrs(
                self.OUTPUT_CRS,
                'Sistema de Coordenadas de Saída',
                defaultValue='EPSG:4674'
            )
        )
        
        # Parâmetro de saída
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT_RASTER,
                'Raster Processado',
                createByDefault=True,
                defaultValue=None
            )
        )

    def processAlgorithm(self, parameters, context, model_feedback):
        """
        Executa o algoritmo de fusionamento
        
        Args:
            parameters: Dicionário com os parâmetros de entrada
            context: Contexto de processamento
            model_feedback: Feedback para atualizar progresso
            
        Returns:
            Dicionário com os resultados do processamento
        """
        # Obter tipo de sensor - tentar métodos diferentes
        try:
            sensor_type = parameters.get(self.SENSOR_TYPE)
            if sensor_type is not None and isinstance(sensor_type, int):
                pass
            else:
                sensor_type = self.parameterAsEnum(parameters, self.SENSOR_TYPE, context)
        except:
            sensor_type = 0
        
        sensor_type = int(sensor_type)
        
        # Log para debug
        if model_feedback:
            sensor_names = {0: "Landsat 8", 1: "Landsat 5", 2: "CBERS-4A", 3: "Sentinel-2"}
            sensor_name = sensor_names.get(sensor_type, "Desconhecido")
            model_feedback.pushInfo(f'Processando: {sensor_name} (ID: {sensor_type})')
        
        # Rotear para o método correto
        if sensor_type == 0:
            return self.process_landsat8(parameters, context, model_feedback)
        elif sensor_type == 1:
            return self.process_landsat5(parameters, context, model_feedback)
        elif sensor_type == 2:
            return self.process_cbers4a(parameters, context, model_feedback)
        elif sensor_type == 3:
            return self.process_sentinel2(parameters, context, model_feedback)
        else:
            raise ValueError(f'Tipo de sensor invalido: {sensor_type}')

    def process_landsat8(self, parameters, context, model_feedback):
        """Processa bandas Landsat 8 com pansharpening"""
        
        # Obter bandas - tentar métodos diferentes
        try:
            b4 = parameters.get(self.INPUT_L8_B4)
            if b4 is None:
                b4 = self.parameterAsRasterLayer(parameters, self.INPUT_L8_B4, context)
        except:
            b4 = None
        
        try:
            b5 = parameters.get(self.INPUT_L8_B5)
            if b5 is None:
                b5 = self.parameterAsRasterLayer(parameters, self.INPUT_L8_B5, context)
        except:
            b5 = None
        
        try:
            b6 = parameters.get(self.INPUT_L8_B6)
            if b6 is None:
                b6 = self.parameterAsRasterLayer(parameters, self.INPUT_L8_B6, context)
        except:
            b6 = None
        
        try:
            b8 = parameters.get(self.INPUT_L8_B8)
            if b8 is None:
                b8 = self.parameterAsRasterLayer(parameters, self.INPUT_L8_B8, context)
        except:
            b8 = None
        
        # Debug log
        if model_feedback:
            model_feedback.pushInfo(f'Landsat 8 - B4: {b4}, B5: {b5}, B6: {b6}, B8: {b8}')
        
        # Validar parâmetros Landsat 8
        if b4 is None or b5 is None or b6 is None or b8 is None:
            raise ValueError('Todas as bandas Landsat 8 (B4, B5, B6, B8) são obrigatórias')
        
        # Criar feedback com múltiplos passos
        feedback = QgsProcessingMultiStepFeedback(3, model_feedback)
        results = {}
        outputs = {}

        # Passo 1: Construir raster virtual com as bandas B4, B5, B6
        feedback.pushInfo('Landsat 8: Construindo raster virtual com bandas B4, B5, B6...')
        alg_params = {
            'ADD_ALPHA': False,
            'ASSIGN_CRS': None,
            'EXTRA': '',
            'INPUT': [b6.source(), b5.source(), b4.source()],
            'PROJ_DIFFERENCE': False,
            'RESAMPLING': 0,  # Nearest Neighbour
            'RESOLUTION': 0,  # Average
            'SEPARATE': True,
            'SRC_NODATA': '',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        
        outputs['VirtualRaster'] = processing.run(
            'gdal:buildvirtualraster',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}

        # Passo 2: Aplicar pansharpening
        feedback.pushInfo('Landsat 8: Aplicando pansharpening...')
        alg_params = {
            'EXTRA': '',
            'OPTIONS': '',
            'PANCHROMATIC': b8.source(),
            'RESAMPLING': 2,  # Cubic (4x4 Kernel)
            'SPECTRAL': outputs['VirtualRaster']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        
        outputs['Pansharpening'] = processing.run(
            'gdal:pansharp',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )

        feedback.setCurrentStep(2)
        if feedback.isCanceled():
            return {}

        # Passo 3: Reprojetar para o CRS selecionado
        feedback.pushInfo('Landsat 8: Reprojetando para o sistema de coordenadas selecionado...')
        
        output_crs = self.parameterAsCrs(parameters, self.OUTPUT_CRS, context)
        
        alg_params = {
            'DATA_TYPE': 0,  # Use Input Layer Data Type
            'EXTRA': '',
            'INPUT': outputs['Pansharpening']['OUTPUT'],
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,  # Nearest Neighbour
            'SOURCE_CRS': None,
            'TARGET_CRS': output_crs,
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': parameters.get(self.OUTPUT_RASTER, '')
        }
        
        outputs['Reproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )
        
        results[self.OUTPUT_RASTER] = outputs['Reproject']['OUTPUT']
        
        feedback.pushInfo('Landsat 8: Fusionamento concluído com sucesso!')
        return results

    def process_landsat5(self, parameters, context, model_feedback):
        """Processa bandas Landsat 5 sem pansharpening"""
        
        # Obter bandas - tentar métodos diferentes
        try:
            b3 = parameters.get(self.INPUT_L5_B3)
            if b3 is None:
                b3 = self.parameterAsRasterLayer(parameters, self.INPUT_L5_B3, context)
        except:
            b3 = None
        
        try:
            b4 = parameters.get(self.INPUT_L5_B4)
            if b4 is None:
                b4 = self.parameterAsRasterLayer(parameters, self.INPUT_L5_B4, context)
        except:
            b4 = None
        
        try:
            b5 = parameters.get(self.INPUT_L5_B5)
            if b5 is None:
                b5 = self.parameterAsRasterLayer(parameters, self.INPUT_L5_B5, context)
        except:
            b5 = None
        
        # Debug log
        if model_feedback:
            model_feedback.pushInfo(f'Landsat 5 - B3: {b3}, B4: {b4}, B5: {b5}')
        
        # Validar parâmetros Landsat 5
        if b3 is None or b4 is None or b5 is None:
            raise ValueError('Todas as bandas Landsat 5 (B3, B4, B5) são obrigatórias')
        
        # Criar feedback com múltiplos passos
        feedback = QgsProcessingMultiStepFeedback(2, model_feedback)
        results = {}
        outputs = {}

        # Passo 1: Construir raster virtual com as bandas B3, B4, B5
        feedback.pushInfo('Landsat 5: Construindo raster virtual com bandas B3, B4, B5...')
        alg_params = {
            'ADD_ALPHA': False,
            'ASSIGN_CRS': None,
            'EXTRA': '',
            'INPUT': [b5.source(), b4.source(), b3.source()],
            'PROJ_DIFFERENCE': False,
            'RESAMPLING': 0,  # Nearest Neighbour
            'RESOLUTION': 0,  # Average
            'SEPARATE': True,
            'SRC_NODATA': '',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        
        outputs['VirtualRaster'] = processing.run(
            'gdal:buildvirtualraster',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}

        # Passo 2: Reprojetar para o CRS selecionado
        feedback.pushInfo('Landsat 5: Reprojetando para o sistema de coordenadas selecionado...')
        
        output_crs = self.parameterAsCrs(parameters, self.OUTPUT_CRS, context)
        
        alg_params = {
            'DATA_TYPE': 0,  # Use Input Layer Data Type
            'EXTRA': '',
            'INPUT': outputs['VirtualRaster']['OUTPUT'],
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,  # Nearest Neighbour
            'SOURCE_CRS': None,
            'TARGET_CRS': output_crs,
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': parameters.get(self.OUTPUT_RASTER, '')
        }
        
        outputs['Reproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )
        
        results[self.OUTPUT_RASTER] = outputs['Reproject']['OUTPUT']
        
        feedback.pushInfo('Landsat 5: Processamento concluído com sucesso!')
        return results

    def process_cbers4a(self, parameters, context, model_feedback):
        """Processa bandas CBERS-4A sem pansharpening - Sequência 5-4-3"""
        
        # Obter bandas
        b3 = self.parameterAsRasterLayer(parameters, self.INPUT_CB4A_B3, context)
        b4 = self.parameterAsRasterLayer(parameters, self.INPUT_CB4A_B4, context)
        b5 = self.parameterAsRasterLayer(parameters, self.INPUT_CB4A_B5, context)
        
        # Validar parâmetros CBERS-4A
        if b3 is None or b4 is None or b5 is None:
            raise ValueError('Todas as bandas CBERS-4A (B3, B4, B5) são obrigatórias')
        
        # Criar feedback com múltiplos passos
        feedback = QgsProcessingMultiStepFeedback(2, model_feedback)
        results = {}
        outputs = {}

        # Passo 1: Construir raster virtual com as bandas B5, B4, B3 (sequência especificada)
        feedback.pushInfo('CBERS-4A: Construindo raster virtual com bandas B5, B4, B3...')
        alg_params = {
            'ADD_ALPHA': False,
            'ASSIGN_CRS': None,
            'EXTRA': '',
            'INPUT': [b5.source(), b4.source(), b3.source()],
            'PROJ_DIFFERENCE': False,
            'RESAMPLING': 0,  # Nearest Neighbour
            'RESOLUTION': 0,  # Average
            'SEPARATE': True,
            'SRC_NODATA': '',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        
        outputs['VirtualRaster'] = processing.run(
            'gdal:buildvirtualraster',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}

        # Passo 2: Reprojetar para o CRS selecionado
        feedback.pushInfo('CBERS-4A: Reprojetando para o sistema de coordenadas selecionado...')
        
        output_crs = self.parameterAsCrs(parameters, self.OUTPUT_CRS, context)
        
        alg_params = {
            'DATA_TYPE': 0,  # Use Input Layer Data Type
            'EXTRA': '',
            'INPUT': outputs['VirtualRaster']['OUTPUT'],
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,  # Nearest Neighbour
            'SOURCE_CRS': None,
            'TARGET_CRS': output_crs,
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': parameters.get(self.OUTPUT_RASTER, '')
        }
        
        outputs['Reproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )
        
        results[self.OUTPUT_RASTER] = outputs['Reproject']['OUTPUT']
        
        feedback.pushInfo('CBERS-4A: Processamento concluído com sucesso!')
        return results

    def process_sentinel2(self, parameters, context, model_feedback):
        """Processa bandas Sentinel-2 sem pansharpening - Sequência 11-8-4"""
        
        # Obter bandas
        b4 = self.parameterAsRasterLayer(parameters, self.INPUT_S2_B4, context)
        b8 = self.parameterAsRasterLayer(parameters, self.INPUT_S2_B8, context)
        b11 = self.parameterAsRasterLayer(parameters, self.INPUT_S2_B11, context)
        
        # Validar parâmetros Sentinel-2
        if b4 is None or b8 is None or b11 is None:
            raise ValueError('Todas as bandas Sentinel-2 (B4, B8, B11) são obrigatórias')
        
        # Criar feedback com múltiplos passos
        feedback = QgsProcessingMultiStepFeedback(2, model_feedback)
        results = {}
        outputs = {}

        # Passo 1: Construir raster virtual com as bandas B11, B8, B4 (sequência especificada)
        feedback.pushInfo('Sentinel-2: Construindo raster virtual com bandas B11, B8, B4...')
        alg_params = {
            'ADD_ALPHA': False,
            'ASSIGN_CRS': None,
            'EXTRA': '',
            'INPUT': [b11.source(), b8.source(), b4.source()],
            'PROJ_DIFFERENCE': False,
            'RESAMPLING': 0,  # Nearest Neighbour
            'RESOLUTION': 0,  # Average
            'SEPARATE': True,
            'SRC_NODATA': '',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        
        outputs['VirtualRaster'] = processing.run(
            'gdal:buildvirtualraster',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}

        # Passo 2: Reprojetar para o CRS selecionado
        feedback.pushInfo('Sentinel-2: Reprojetando para o sistema de coordenadas selecionado...')
        
        output_crs = self.parameterAsCrs(parameters, self.OUTPUT_CRS, context)
        
        alg_params = {
            'DATA_TYPE': 0,  # Use Input Layer Data Type
            'EXTRA': '',
            'INPUT': outputs['VirtualRaster']['OUTPUT'],
            'MULTITHREADING': False,
            'NODATA': None,
            'OPTIONS': '',
            'RESAMPLING': 0,  # Nearest Neighbour
            'SOURCE_CRS': None,
            'TARGET_CRS': output_crs,
            'TARGET_EXTENT': None,
            'TARGET_EXTENT_CRS': None,
            'TARGET_RESOLUTION': None,
            'OUTPUT': parameters.get(self.OUTPUT_RASTER, '')
        }
        
        outputs['Reproject'] = processing.run(
            'gdal:warpreproject',
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True
        )
        
        results[self.OUTPUT_RASTER] = outputs['Reproject']['OUTPUT']
        
        feedback.pushInfo('Sentinel-2: Processamento concluído com sucesso!')
        return results

    def name(self):
        """Nome interno do algoritmo"""
        return 'landsatfusion'

    def displayName(self):
        """Nome exibido do algoritmo"""
        return 'GeoCAR Vision (L5 & L8 & CBERS-4A & S2)'

    def group(self):
        """Grupo do algoritmo"""
        return 'ForgeGeo'

    def groupId(self):
        """ID do grupo"""
        return 'forgegeo'

    def shortHelpString(self):
        """Texto de ajuda curto"""
        return """
        Plugin para fusionamento de bandas Landsat 5, Landsat 8, CBERS-4A e Sentinel-2.
        
        <b>Landsat 8:</b>
        - Fusiona bandas B4, B5, B6 com B8 usando pansharpening
        - Aumenta resolução de 30m para 15m
        
        <b>Landsat 5:</b>
        - Combina bandas B3, B4, B5
        - Mantém resolução de 30m
        
        <b>CBERS-4A:</b>
        - Combina bandas B5, B4, B3 (sequência especificada)
        - Mantém resolução de 64m
        
        <b>Sentinel-2:</b>
        - Combina bandas B11, B8, B4 (sequência especificada)
        - Mantém resolução de 20m (B11) e 10m (B8, B4)
        
        Desenvolvido por ForgeGeo/Geoserviço
        """

    def createInstance(self):
        """Cria uma nova instância do algoritmo"""
        return LandsatFusionAlgorithm()
